# Soya 3D tutorial
# Copyright (C) 2001-2002 Bertrand 'blam!' LAMY
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


# ---------------------
# Lesson 104: Landscape
# ---------------------

# This lesson shows how to make a landscape (also known as heightmap)

import random, os, os.path, sys, time

import soya.model
import soya.soya3d
import soya.land


soya.init()

data_dir = os.path.join(os.path.dirname(os.path.abspath(sys.argv[0])), "data")

soya.model.Image   .PATH = os.path.join(data_dir, "images")
soya.model.Material.PATH = os.path.join(data_dir, "materials")
soya.soya3d.World  .PATH = os.path.join(data_dir, "worlds")
soya.model.Shape   .PATH = os.path.join(data_dir, "shapes")

# Create a world
world = soya.soya3d.World()

# Add a light
light = soya.soya3d.Light(world)
light.set_xyz(0.0, 15.0, 0.0)


# Land creation
# -------------
#   There are 2 ways to create a landscape:
#   1) give each height value
#   2) create a landscape from an image (white pixels are upper)

land_size = 33
land = soya.land.Land(land_size)
# Land_size is the size of the landscape. Landscapes are always squares
# and their size must always be like this : (2^n)+1

# The following lines set randomized value for each height
i = 0
while (i < land_size):
  j = 0
  while (j < land_size):
    land.set_height(i, j, random.random())
    j = j + 1
  i = i + 1

# Multiply all the heights by 4
land.multiply_height(4.0)

# Now we must set a Material to the Land. Land must always have a Material
# even if it is None (None is a white Material)
i = 0
while (i < land_size):
  j = 0
  while (j < land_size):
    land.set_material(i, j, None)
    j = j + 1
  i = i + 1


# Above lines are quite useless. Now we will see how to create a Land from
# an image and how to texture it quickly


# Create a new Land
land = soya.land.Land(0)
# The size of 0 will be overriden by the size of the image
land.from_image(soya.model.Image("map1.tga"))
# Heights range from 0.0 (black pixels) to 1.0 (white pixels)
land.multiply_height(8.0)

# Create 2 Materials
material1 = soya.model.Material()
material1.tex_filename  = "block2.tga"

material2 = soya.model.Material()
material2.tex_filename  = "metal1.tga"

#   land.set_material_layer(MATERIAL, FROM, TO)
# Set to all points that have a height between FROM and TO the material
# MATERIAL
land.set_material_layer(material1, 0.0, 6.0)
land.set_material_layer(material2, 6.0, 8.0)

# You can also use:
#   land.set_material_layer_angle(MATERIAL, HEIGHT_FROM, HEIGHT_TO, ANGLE_FROM, ANGLE_TO)
# This function works just has the previous one but add 2 more arguments:
# ANGLE_FROM and ANGLE_TO.
# This means that all points of the land will have the given material
# only if:
# - its height is between HEIGHT_FROM and HEIGHT_TO
# - its normal makes an angle with the horizontal plane that is between ANGLE_FROM
# and ANGLE_TO (expressed in degrees).
land.set_material_layer_angle(material1, 0.0, 8.0, 0.0, 20.0)


# Now we will see some Land attributes:

land.scale_factor = 1.5
# This is the distance between 2 consecutive heights that you have given
# to define the Land. This is quite like doing the following:
#   world.scale(1.5, 1.0, 1.5)
# Default value is 1.5

land.texture_factor = 1.0
# This factor is applied to texture coordinates when rendering the Land.
# Default value is 1.0

land.map_size = 8
land.split_factor = 4.0
# These 2 values influence the behaviour of the level of detail algorithm
# (it means that there are more triangles to draw the Land near the Camera
# and less far from the Camera).
# The higher split_factor is, the better precision you have (it means more
# triangles to draw the Land even at long distance from Camera).
# map_size represents the size of a map. A map is a square part of the Land
# that computes its visibility and precision.
# map_size and split_factor values can change FPS (rendering speed). Values
# above are default ones (I get the best FPS with these ones on my computer)


# Land is a shape (and so can't be added in a World but must be set as shape)
world.set_shape(land)


# Add a camera and a loop to render

camera = soya.soya3d.Camera(world)
camera.set_xyz (16.0, 8.0, 0.0)
camera.look_at (soya.soya3d.Point (world, 16.0, 6.0, 10.0))
soya.set_root_widget(camera)


while(1):
  soya.render()

  time.sleep(0.1)
  
# Use Up, Down, Left, Right arrows to move the camera

  for event in soya.process_event():
    if event[0] == soya.KEYDOWN:
      if event[1] == soya.K_UP:
        camera.translate(0.0, 0.0, 0.5)
      elif event[1] == soya.K_DOWN:
        camera.translate(0.0, 0.0, -0.5)
      elif event[1] == soya.K_LEFT:
        camera.translate(0.5, 0.0, 0.0)
      elif event[1] == soya.K_RIGHT:
        camera.translate(-0.5, 0.0, 0.0)
      elif event[1] == soya.K_q:
        sys.exit()


# Other Land functions not aborded in this tutorial (easy to understand):
# -----------------------------------------------------------------------
#   height = land.get_height(x, z)
#   material = land.get_material(x, z)
#   land.add_height(value)
#     [x and z are integers, value and height are floats]


# TO DO exercice:
# ---------------
# make a Land editor

