///###////////////////////////////////////////////////////////////////////////
//
// Burton Computer Corporation
// http://www.burton-computer.com
// $Id: PhraseBuilder.h,v 1.6 2003/10/24 15:59:34 bburton Exp $
//
// Copyright (C) 2000 Burton Computer Corporation
// ALL RIGHTS RESERVED
//
// This program is open source software; you can redistribute it
// and/or modify it under the terms of the Q Public License (QPL)
// version 1.0. Use of this software in whole or in part, including
// linking it (modified or unmodified) into other programs is
// subject to the terms of the QPL.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// Q Public License for more details.
//
// You should have received a copy of the Q Public License
// along with this program; see the file LICENSE.txt.  If not, visit
// the Burton Computer Corporation or CoolDevTools web site
// QPL pages at:
//
//    http://www.burton-computer.com/qpl.html
//

#ifndef _PhraseBuilder_h
#define _PhraseBuilder_h

#include "AbstractPhraseBuilder.h"

class PhraseBuilder : public AbstractPhraseBuilder
{
public:
  PhraseBuilder(int max_words)
    : m_offsets(new string::size_type[max_words]),
      m_numWords(0),
      m_maxWords(max_words),
      m_maxChars(0),
      m_minWords(1),
      m_last(0)
  {
    resetOffsetsArray();
  }

  ~PhraseBuilder()
  {
  }

  void setMinWords(int value)
  {
    m_minWords = value;
  }

  void setMaxWords(int value)
  {
    m_maxWords = value;
    resetOffsetsArray();
  }

  int getMinWords()
  {
    return m_minWords;
  }

  int getMaxWords()
  {
    return m_maxWords;
  }

  int getWordCount() const
  {
    if (m_numWords > m_maxWords) {
      for (int num_words = m_maxWords + 1; num_words <= m_numWords; ++num_words) {
        int index = getPrevIndex(m_last, num_words - 1);
        int length = m_buffer.length() - m_offsets[index];
        if (length > m_maxChars) {
          return num_words - 1;
        }
      }
    }
    return m_numWords;
  }

  void setMaxChars(int value)
  {
    m_maxChars = value;
    resetOffsetsArray();
  }

  int getMaxChars() const
  {
    return m_maxChars;
  }

  void addWord(const string &word)
  {
    assert(m_shiftCounter >= 0);
    if (--m_shiftCounter < 0) {
      shift();
      resetShiftCounter();
    }

    m_numWords = min(m_numWords + 1, m_maxOffsets);
    m_last = getNextIndex(m_last);

    m_buffer += ' ';
    m_offsets[m_last] = m_buffer.length();
    m_buffer += word;

    assert(m_last >= 0);
    assert(m_last < m_maxOffsets);
  }

  const char *getPhrase(int num_words)
  {
    assert(num_words <= m_maxOffsets);
    assert(num_words <= m_numWords);

    int index = getPrevIndex(m_last, num_words - 1);
    return m_buffer.c_str() + m_offsets[index];
  }

  void clear()
  {
    m_last = 0;
    m_numWords = 0;
    m_buffer.erase();
    resetShiftCounter();
  }

private:
  void resetShiftCounter()
  {
    m_shiftCounter = 10 * m_maxOffsets;
  }

  int getNextIndex(int index) const
  {
    return (index + 1) % m_maxOffsets;
  }

  int getPrevIndex(int index,
                   int offset) const
  {
    return (index + m_maxOffsets - offset) % m_maxOffsets;
  }

  void resetOffsetsArray()
  {
    m_maxOffsets = max(m_maxWords, ((m_maxChars + 1) / 2));
    m_offsets.set(new string::size_type[m_maxOffsets]);
    clear();
  }

  void shift()
  {
    assert(m_numWords > 0);

    int index = getPrevIndex(m_last, m_numWords - 1);
    string::size_type shift_amount = m_offsets[index];
    if (shift_amount > 0) {
#ifndef NDEBUG
      vector<string> old_strings;
      for (int i = 1; i <= m_numWords; ++i) {
        old_strings.push_back(getPhrase(i));
      }
#endif

      if (is_debug) {
        cerr << "BEFORE SHIFT: " << getPhrase(m_numWords) << endl;
      }

      m_buffer.erase(0, shift_amount);
      for (int i = 0; i < m_numWords; ++i) {
        m_offsets[index] -= shift_amount;
        index = getNextIndex(index);
      }

      if (is_debug) {
        cerr << "AFTER  SHIFT: " << getPhrase(m_numWords) << endl;
      }

      assert(index == getNextIndex(m_last));
#ifndef NDEBUG
      for (int i = 1; i <= m_numWords; ++i) {
        assert(old_strings[i-1] == getPhrase(i));
      }
#endif
    }
  }

private:
  /// Not implemented.
  PhraseBuilder(const PhraseBuilder &);

  /// Not implemented.
  PhraseBuilder& operator=(const PhraseBuilder &);

private:
  string m_buffer;
  NewPtr<string::size_type> m_offsets;
  int m_numWords;
  int m_minWords;
  int m_maxWords;
  int m_maxChars;
  int m_maxOffsets;
  int m_last;
  int m_addCount;
  int m_shiftCounter;
};

#endif // _PhraseBuilder_h
