-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Sem.CompUnit.WalkStatements)

--------------------------------------------------------------------
--  CheckForMutuallyExclusiveBranches
--
--  Implementation Notes:
--    The details of the algorithm used is described in S.P0468.53.49.
--    The set of ancestor conditional branches and the set of
--    the closest sequences of statements eminating from a conditional
--    branch node are constructed using SeqAlgebra.Seq objects.
--------------------------------------------------------------------
procedure CheckForMutuallyExclusiveBranches
  (GivenNode, PrecedingNode : in     STree.SyntaxNode;
   TheHeap                  : in out Heap.HeapRecord;
   AreMutuallyExclusive     :    out Boolean) is
   AncestorCondBranches : SeqAlgebra.Seq;
   SetOfSeqOfStatements : SeqAlgebra.Seq;
   BranchNode           : STree.SyntaxNode;
   CondAncestor         : STree.SyntaxNode;
   CommonAncestor       : STree.SyntaxNode;
   GivenNodeSeqStat     : STree.SyntaxNode;
   PrecedingNodeSeqStat : STree.SyntaxNode;
   Iter                 : STree.Iterator;

   function LocateChildOfType (Node      : STree.SyntaxNode;
                               ChildType : SPSymbols.SPSymbol) return STree.SyntaxNode
   --# global in STree.Table;
   is
      Child : STree.SyntaxNode;
   begin
      Child := Child_Node (Node);

      while Child /= STree.NullNode and then Syntax_Node_Type (Node => Child) /= ChildType loop
         Child := Next_Sibling (Child);
      end loop;
      return Child;
   end LocateChildOfType;

   procedure FindRecursiveBranches (Node      : in STree.SyntaxNode;
                                    FindType  : in SPSymbols.SPSymbol;
                                    BranchSet : in SeqAlgebra.Seq)
   --# global in     STree.Table;
   --#        in out Statistics.TableUsage;
   --#        in out TheHeap;
   --# derives Statistics.TableUsage,
   --#         TheHeap               from *,
   --#                                    BranchSet,
   --#                                    FindType,
   --#                                    Node,
   --#                                    STree.Table,
   --#                                    TheHeap;
   is
      Child        : STree.SyntaxNode;
      NextInstance : STree.SyntaxNode;
      RecurseOver  : SPSymbols.SPSymbol;
      Iter         : STree.Iterator;
   begin
      -- Determine the type of node we are recursing over.
      RecurseOver := Syntax_Node_Type (Node => Node);

      -- Find all children from the given node and search for a node
      -- of the FindType only on children which are not recursive.
      -- For if and case statements a parent has at most 1 recursive
      -- child node.  Only traverse a recursive child node after all
      -- its siblings have been traversed.  The process is then
      -- repeated until a node without a recursive node is encountered.
      -- when all the children of the node have been processed the
      -- loop terminates.
      -- The traversal of the syntax tree is not pre-order but
      -- the order in which the nodes are placed into the BranchSet
      -- is unimportant.
      NextInstance := Node;
      while NextInstance /= STree.NullNode loop
         Child        := Child_Node (NextInstance);
         NextInstance := STree.NullNode;
         while Child /= STree.NullNode loop
            if Syntax_Node_Type (Node => Child) = RecurseOver then
               -- There is at most one instance of a recursive child node.
               NextInstance := Child;
            else
               Iter := Find_First_Node (Node_Kind    => FindType,
                                        From_Root    => Child,
                                        In_Direction => STree.Down);
               if not STree.IsNull (Iter) then
                  -- Only add the set of branches if a
                  -- node of the FindType is present.
                  SeqAlgebra.AddMember (TheHeap, BranchSet, Natural (STree.NodeToRef (Get_Node (It => Iter))));
               end if;
            end if;
            Child := Next_Sibling (Child);
         end loop;
      end loop;
   end FindRecursiveBranches;

   procedure FindIfBranches (IfNode    : in STree.SyntaxNode;
                             BranchSet : in SeqAlgebra.Seq)
   --# global in     STree.Table;
   --#        in out Statistics.TableUsage;
   --#        in out TheHeap;
   --# derives Statistics.TableUsage,
   --#         TheHeap               from *,
   --#                                    BranchSet,
   --#                                    IfNode,
   --#                                    STree.Table,
   --#                                    TheHeap;
   is
      CurrentChild : STree.SyntaxNode;
   begin
      -- Process "then" part.
      CurrentChild := LocateChildOfType (IfNode, SPSymbols.sequence_of_statements);

      if CurrentChild /= STree.NullNode then
         -- there should always be a then part otherwise the
         -- syntax tree is invalid, but no error is raised here
         -- as it will be reported elsewhere by the Examiner.
         SeqAlgebra.AddMember (TheHeap, BranchSet, Natural (STree.NodeToRef (CurrentChild)));
      end if;

      -- Process the "else" part if one exists
      CurrentChild := LocateChildOfType (IfNode, SPSymbols.else_part);
      CurrentChild := LocateChildOfType (CurrentChild, SPSymbols.sequence_of_statements);

      if CurrentChild /= STree.NullNode then
         -- Only add the branch if the else sequence of statements exist.
         SeqAlgebra.AddMember (TheHeap, BranchSet, Natural (STree.NodeToRef (CurrentChild)));
      end if;

      -- Process the elsif part if one exists.
      CurrentChild := LocateChildOfType (IfNode, SPSymbols.elsif_part);

      if CurrentChild /= STree.NullNode then
         FindRecursiveBranches (CurrentChild, SPSymbols.sequence_of_statements, BranchSet);
      end if;
   end FindIfBranches;

   procedure FindCaseBranches (CaseNode  : in STree.SyntaxNode;
                               BranchSet : in SeqAlgebra.Seq)
   --# global in     STree.Table;
   --#        in out Statistics.TableUsage;
   --#        in out TheHeap;
   --# derives Statistics.TableUsage,
   --#         TheHeap               from *,
   --#                                    BranchSet,
   --#                                    CaseNode,
   --#                                    STree.Table,
   --#                                    TheHeap;
   is
      CurrentChild : STree.SyntaxNode;
   begin
      -- Process "others" part if it exists.
      CurrentChild := LocateChildOfType (CaseNode, SPSymbols.others_part);
      CurrentChild := LocateChildOfType (CurrentChild, SPSymbols.sequence_of_statements);

      if CurrentChild /= STree.NullNode then
         SeqAlgebra.AddMember (TheHeap, BranchSet, Natural (STree.NodeToRef (CurrentChild)));
      end if;

      -- Process the alternatives part if one exists.
      CurrentChild := LocateChildOfType (CaseNode, SPSymbols.alternatives);

      if CurrentChild /= STree.NullNode then
         FindRecursiveBranches (CurrentChild, SPSymbols.sequence_of_statements, BranchSet);
      end if;
   end FindCaseBranches;

   function FindContainingSequenceOfStatements
     (Node                 : STree.SyntaxNode;
      SetOfSeqOfStatements : SeqAlgebra.Seq)
     return                 STree.SyntaxNode
   --# global in STree.Table;
   --#        in TheHeap;
   is
      Iter              : STree.Iterator;
      SeqStatementsNode : STree.SyntaxNode;
   begin
      Iter              :=
        Find_First_Node (Node_Kind    => SPSymbols.sequence_of_statements,
                         From_Root    => Node,
                         In_Direction => STree.Up);
      SeqStatementsNode := Get_Node (It => Iter);

      while not (STree.IsNull (Iter)
                   or else SeqAlgebra.IsMember (TheHeap, SetOfSeqOfStatements, Natural (STree.NodeToRef (SeqStatementsNode)))) loop
         Iter              := STree.NextNode (Iter);
         SeqStatementsNode := Get_Node (It => Iter);
      end loop;

      return SeqStatementsNode;
   end FindContainingSequenceOfStatements;

begin
   SeqAlgebra.CreateSeq (TheHeap, AncestorCondBranches);
   SeqAlgebra.CreateSeq (TheHeap, SetOfSeqOfStatements);
   Iter := Find_First_Branch_Node (From_Root    => PrecedingNode,
                                   In_Direction => STree.Up);

   -- Determine the set of Ancestor If and Case branch nodes
   -- of the Preceding Node.
   while not STree.IsNull (Iter) loop
      BranchNode := Get_Node (It => Iter);

      case Syntax_Node_Type (Node => BranchNode) is
         -- Only if and case statement branches create
         -- mutually exclusive sequences of statements
         when SPSymbols.if_statement | SPSymbols.case_statement =>
            SeqAlgebra.AddMember (TheHeap, AncestorCondBranches, Natural (STree.NodeToRef (BranchNode)));
         when others =>
            null;
      end case;

      Iter := STree.NextNode (Iter);
   end loop;

   if SeqAlgebra.IsEmptySeq (TheHeap, AncestorCondBranches) then
      -- The PrecedingNode has no if or case branches and therefore
      -- cannot be on a mutually exclusive branch to the GivenNode
      AreMutuallyExclusive := False;
   else
      -- Find the closest if or case branch common to the
      -- PrecedingNode and the GivenNode.
      -- As we traverse up the tree from the GivenNode this will
      -- ensure that the closest common conditional branch node
      -- is located.
      Iter           := Find_First_Branch_Node (From_Root    => GivenNode,
                                                In_Direction => STree.Up);
      CommonAncestor := STree.NullNode;

      while not (STree.IsNull (Iter)) and CommonAncestor = STree.NullNode loop
         -- The AncestorCondBranches set only contains conditional
         -- branch nodes. No need to check again here for type of branch.
         CondAncestor := Get_Node (It => Iter);
         if SeqAlgebra.IsMember (TheHeap, AncestorCondBranches, Natural (STree.NodeToRef (CondAncestor))) then
            CommonAncestor := CondAncestor;
         else
            Iter := STree.NextNode (Iter);
         end if;
      end loop;

      if CommonAncestor = STree.NullNode then
         -- The GivenNode and the PrecedingNode have no conditional
         -- branches in common and therefore are not mutually exclusive.
         AreMutuallyExclusive := False;
      else
         -- Determine the set of mutually exclusive branches from the
         -- closest common if or case statement ancestor.
         -- Both the GivenNode and the PrecedingNode will be contained
         -- within a sequence of statements.  Only the branches which
         -- contain sequences of statements are considered and the
         -- nodes representing the sequence of statements form the set.

         if Syntax_Node_Type (Node => CommonAncestor) = SPSymbols.if_statement then
            FindIfBranches (CommonAncestor, SetOfSeqOfStatements);
         else
            FindCaseBranches (CommonAncestor, SetOfSeqOfStatements);
         end if;

         -- Find the sequence of statements which contains the GivenNode.
         -- Such a node must exist.
         GivenNodeSeqStat := FindContainingSequenceOfStatements (GivenNode, SetOfSeqOfStatements);

         -- Find the sequence of statements which contains the PrecedingNode.
         -- Such a node must exist.
         PrecedingNodeSeqStat := FindContainingSequenceOfStatements (PrecedingNode, SetOfSeqOfStatements);

         AreMutuallyExclusive := GivenNodeSeqStat /= PrecedingNodeSeqStat;
      end if;
   end if;

   SeqAlgebra.DisposeOfSeq (TheHeap, AncestorCondBranches);
   SeqAlgebra.DisposeOfSeq (TheHeap, SetOfSeqOfStatements);
end CheckForMutuallyExclusiveBranches;
