-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

-- Overview:
-- Checks a Package Specification for Sem on down pass through
-- TreeProcessor.  Starts at node package_specification.  May directly raise
-- errors for: mismatch between initial and terminal identifier.
-- Other errors may be raised indirectly by wf_package_annotation which
-- are called from here.
--------------------------------------------------------------------------------

with SLI;

separate (Sem.CompUnit)
procedure Wf_Package_Specification
  (Node          : in STree.SyntaxNode;
   Ident_Str     : in LexTokenManager.Lex_String;
   Pack_Sym      : in Dictionary.Symbol;
   Current_Scope : in Dictionary.Scopes) is
   type Enclosing_Scope_Types is (In_Library, In_Package, In_Procedure);
   Enclosing_Scope_Type                                               : Enclosing_Scope_Types;
   End_Pos_Node, Anno_Node, Vis_Part_Node, Priv_Part_Node, Ident_Node : STree.SyntaxNode;
   Pack_Vis_Scope                                                     : Dictionary.Scopes;

   ------------------------------

   procedure Find_Key_Nodes
     (Node           : in     STree.SyntaxNode;
      Anno_Node      :    out STree.SyntaxNode;
      Ident_Node     :    out STree.SyntaxNode;
      Vis_Part_Node  :    out STree.SyntaxNode;
      Priv_Part_Node :    out STree.SyntaxNode)
   --# global in STree.Table;
   --# derives Anno_Node,
   --#         Ident_Node,
   --#         Priv_Part_Node,
   --#         Vis_Part_Node  from Node,
   --#                             STree.Table;
   is
   begin
      -- ASSUME Node = package_specification
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Node) = SPSymbols.package_specification,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Node = package_specification in Find_Key_Nodes");

      Anno_Node := Next_Sibling (Current_Node => Child_Node (Current_Node => Node));
      -- ASSUME Anno_Node = package_annotation
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Anno_Node) = SPSymbols.package_annotation,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Anno_Node = package_annotation in Find_Key_Nodes");

      Ident_Node := Child_Node (Current_Node => Child_Node (Current_Node => Node));
      -- ASSUME Ident_Node = dotted_simple_name OR identifier
      if Syntax_Node_Type (Node => Ident_Node) = SPSymbols.dotted_simple_name then
         -- ASSUME Ident_Node = dotted_simple_name
         -- declaring a child package
         Ident_Node := Last_Child_Of (Start_Node => Ident_Node);
      elsif Syntax_Node_Type (Node => Ident_Node) /= SPSymbols.identifier then
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Ident_Node = dotted_simple_name OR identifier in Find_Key_Nodes");
      end if;
      -- ASSUME Ident_Node = identifier
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Ident_Node) = SPSymbols.identifier,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Ident_Node = identifier in Find_Key_Nodes");

      Vis_Part_Node := Child_Node (Current_Node => Next_Sibling (Current_Node => Anno_Node));
      -- ASSUME Vis_Part_Node = visible_part_rep
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Vis_Part_Node) = SPSymbols.visible_part_rep,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Vis_Part_Node = visible_part_rep in Find_Key_Nodes");

      Priv_Part_Node := Next_Sibling (Current_Node => Next_Sibling (Current_Node => Anno_Node));
      -- ASSUME Priv_Part_Node = private_part OR dotted_simple_name
      if Syntax_Node_Type (Node => Priv_Part_Node) = SPSymbols.private_part then
         -- ASSUME Private_Part = private_part
         Priv_Part_Node := Child_Node (Current_Node => Priv_Part_Node);
      elsif Syntax_Node_Type (Node => Priv_Part_Node) = SPSymbols.dotted_simple_name then
         -- ASSUME Priv_Part_Node = dotted_simple_name
         Priv_Part_Node := STree.NullNode;
      else
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Priv_Part_Node = private_part OR dotted_simple_name in Find_Key_Nodes");
      end if;
      -- ASSUME Priv_Part_Node = basic_declarative_item_rep OR hidden_part OR NULL
      SystemErrors.RT_Assert
        (C       => Priv_Part_Node = STree.NullNode
           or else Syntax_Node_Type (Node => Priv_Part_Node) = SPSymbols.basic_declarative_item_rep
           or else Syntax_Node_Type (Node => Priv_Part_Node) = SPSymbols.hidden_part,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Private_Part = basic_declarative_item_rep OR hidden_part OR NULL in Find_Key_Nodes");
   end Find_Key_Nodes;

   ------------------------------

   procedure Find_Enclosing_Scope_Type (Scope : in Dictionary.Scopes)
   --# global in     Dictionary.Dict;
   --#           out Enclosing_Scope_Type;
   --# derives Enclosing_Scope_Type from Dictionary.Dict,
   --#                                   Scope;
   is
   begin
      if Dictionary.IsGlobalScope (Scope) then
         Enclosing_Scope_Type := In_Library;
      elsif Dictionary.IsPackage (Dictionary.GetRegion (Scope)) then
         Enclosing_Scope_Type := In_Package;
      else
         Enclosing_Scope_Type := In_Procedure;
      end if;
   end Find_Enclosing_Scope_Type;

   ------------------------------

   procedure Wf_Anno
     (Node          : in STree.SyntaxNode;
      Pack_Sym      : in Dictionary.Symbol;
      Scope_Type    : in Enclosing_Scope_Types;
      Current_Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Pack_Sym,
   --#                                         Scope_Type,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Pack_Sym,
   --#                                         Scope_Type,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
      is separate;

   ------------------------------

   procedure Wf_Visible
     (Node          : in STree.SyntaxNode;
      Pack_Sym      : in Dictionary.Symbol;
      Current_Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Pack_Sym,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Pack_Sym,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
      is separate;

   ------------------------------

   procedure Check_Closing_Identifier (End_Name_Node, Ident_Node : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         End_Name_Node,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
   is
      Start_Ident_Node : STree.SyntaxNode;
      End_Ident_Node   : STree.SyntaxNode;
   begin
      -- ASSUME End_Name_Node = dotted_simple_name
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => End_Name_Node) = SPSymbols.dotted_simple_name,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect End_Name_Node = dotted_simple_name in Check_Closing_Identifier");
      -- ASSUME Ident_Node = identifier
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Ident_Node) = SPSymbols.identifier,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Ident_Node = identifier in Check_Closing_Identifier");

      Start_Ident_Node := Ident_Node;
      End_Ident_Node   := Last_Child_Of (Start_Node => End_Name_Node);
      loop
         -- ASSUME Start_Ident_Node = identifier
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Start_Ident_Node) = SPSymbols.identifier,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Start_Ident_Node = identifier in Check_Closing_Identifier");
         -- ASSUME End_Ident_Node = identifier
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => End_Ident_Node) = SPSymbols.identifier,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect End_Ident_Node = identifier in Check_Closing_Identifier");

         -- check identifiers at current positions:
         if LexTokenManager.Lex_String_Case_Insensitive_Compare
           (Lex_Str1 => Node_Lex_String (Node => Start_Ident_Node),
            Lex_Str2 => Node_Lex_String (Node => End_Ident_Node)) /=
           LexTokenManager.Str_Eq then
            ErrorHandler.Semantic_Error
              (Err_Num   => 58,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => End_Ident_Node),
               Id_Str    => Node_Lex_String (Node => Start_Ident_Node));
            exit;
         end if;

         -- move on to next identifiers:
         Start_Ident_Node := Next_Sibling (Current_Node => Parent_Node (Current_Node => Start_Ident_Node));
         End_Ident_Node   := Next_Sibling (Current_Node => Parent_Node (Current_Node => End_Ident_Node));

         -- finished when both exhausted:
         exit when Syntax_Node_Type (Node => Start_Ident_Node) /= SPSymbols.identifier
           and then Syntax_Node_Type (Node => End_Ident_Node) /= SPSymbols.identifier;

         -- check if only one exhausted (length mismatch):
         if Syntax_Node_Type (Node => Start_Ident_Node) /= SPSymbols.identifier
           or else Syntax_Node_Type (Node => End_Ident_Node) /= SPSymbols.identifier then
            ErrorHandler.Semantic_Error
              (Err_Num   => 615,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => End_Name_Node),
               Id_Str    => LexTokenManager.Null_String);
            exit;
         end if;
      end loop;
   end Check_Closing_Identifier;

   ------------------------------

   procedure Check_Deferred_Items (End_Node : in STree.SyntaxNode;
                                   Pack_Sym : in Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         End_Node,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Pack_Sym,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
   is
      Error_Position : LexTokenManager.Token_Position;
      It             : Dictionary.Iterator;
      Sym            : Dictionary.Symbol;
   begin
      -- ASSUME End_Node = dotted_simple_name
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => End_Node) = SPSymbols.dotted_simple_name,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect End_Node = dotted_simple_name in Check_Deferred_Items");

      Error_Position := Node_Position (Node => Child_Node (End_Node));
      It             := Dictionary.FirstDeferredConstant (Pack_Sym);
      while not Dictionary.IsNullIterator (It) loop
         Sym := Dictionary.CurrentSymbol (It);
         if not Dictionary.IsDeclared (Sym) then
            ErrorHandler.Semantic_Error
              (Err_Num   => 26,
               Reference => ErrorHandler.No_Reference,
               Position  => Error_Position,
               Id_Str    => Dictionary.GetSimpleName (Sym));
         end if;
         It := Dictionary.NextSymbol (It);
      end loop;

      It := Dictionary.FirstPrivateType (Pack_Sym);
      while not Dictionary.IsNullIterator (It) loop
         Sym := Dictionary.CurrentSymbol (It);
         if not Dictionary.IsDeclared (Sym) then
            ErrorHandler.Semantic_Error
              (Err_Num   => 27,
               Reference => ErrorHandler.No_Reference,
               Position  => Error_Position,
               Id_Str    => Dictionary.GetSimpleName (Sym));
         end if;
         It := Dictionary.NextSymbol (It);
      end loop;
   end Check_Deferred_Items;

   ------------------------------

   procedure CheckTypesCanBeUsed (PackSym : in Dictionary.Symbol;
                                  ErrNode : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrNode,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         PackSym,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
      is separate;

   ------------------------------

   procedure CheckModes (Node     : in STree.SyntaxNode;
                         Pack_Sym : in Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out TheHeap;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Pack_Sym,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         Statistics.TableUsage,
   --#         TheHeap                    from *,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Pack_Sym,
   --#                                         STree.Table,
   --#                                         TheHeap;
      is separate;

   ------------------------------

   procedure Wf_Private (Node  : in STree.SyntaxNode;
                         Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
      is separate;

   ------------------------------

   procedure Check_Body_Required_By_Spark (Node     : in STree.SyntaxNode;
                                           Pack_Sym : in Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Pack_Sym,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
   is
      It : Dictionary.Iterator;
   begin
      -- ASSUME Node = dotted_simple_name OR hidden_part
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Node) = SPSymbols.dotted_simple_name
           or else Syntax_Node_Type (Node => Node) = SPSymbols.hidden_part,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Node = dotted_simple_name OR hidden_part in Wf_Package_Specification");
      It := Dictionary.FirstInitializedOwnVariable (Pack_Sym);
      while not Dictionary.IsNullIterator (It) loop
         if Dictionary.IsDeclared (Dictionary.CurrentSymbol (It))
           and then not Dictionary.VariableIsInitialized (Dictionary.CurrentSymbol (It)) then
            case CommandLineData.Content.Language_Profile is
               when CommandLineData.SPARK83 =>
                  ErrorHandler.Semantic_Warning
                    (Err_Num  => 407,
                     Position => Node_Position (Node => Node),
                     Id_Str   => LexTokenManager.Null_String);
               when CommandLineData.SPARK95 | CommandLineData.SPARK2005 =>
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 607,
                     Reference => ErrorHandler.No_Reference,
                     Position  => Node_Position (Node => Node),
                     Id_Str    => LexTokenManager.Null_String);
            end case;
            exit;
         end if;
         It := Dictionary.NextSymbol (It);
      end loop;
   end Check_Body_Required_By_Spark;

   ------------------------------

   procedure CheckStateCanBeInitialized (PackSym  : in Dictionary.Symbol;
                                         AnnoNode : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from AnnoNode,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         PackSym,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
      is separate;

begin -- Wf_Package_Specification

   -- ASSUME Node = package_specification
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Node) = SPSymbols.package_specification,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Node = package_specification in Wf_Package_Specification");

   Find_Key_Nodes
     (Node           => Node,
      Anno_Node      => Anno_Node,
      Ident_Node     => Ident_Node,
      Vis_Part_Node  => Vis_Part_Node,
      Priv_Part_Node => Priv_Part_Node);
   -- ASSUME Anno_Node = package_annotation
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Anno_Node) = SPSymbols.package_annotation,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Anno_Node = package_annotation in Wf_Package_Specification");
   -- ASSUME Ident_Node = identifier
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Ident_Node) = SPSymbols.identifier,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Ident_Node = identifier in Wf_Package_Specification");
   -- ASSUME Vis_Part_Node = visible_part_rep
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Vis_Part_Node) = SPSymbols.visible_part_rep,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Vis_Part_Node = visible_part_rep in Wf_Package_Specification");

   Find_Enclosing_Scope_Type (Scope => Current_Scope); -- tells us where package is being declared

   Pack_Vis_Scope := Dictionary.VisibleScope (Pack_Sym);

   --# assert True;
   Wf_Anno (Node          => Anno_Node,
            Pack_Sym      => Pack_Sym,
            Scope_Type    => Enclosing_Scope_Type,
            Current_Scope => Pack_Vis_Scope);

   Wf_Visible (Node          => Vis_Part_Node,
               Pack_Sym      => Pack_Sym,
               Current_Scope => Pack_Vis_Scope);

   --# assert True;

   -- ASSUME Priv_Part_Node = basic_declarative_item_rep OR hidden_part OR NULL
   if Priv_Part_Node = STree.NullNode then
      -- ASSUME Priv_Part_Node = NULL
      -- no private part
      Check_Closing_Identifier (End_Name_Node => Last_Sibling_Of (Start_Node => Anno_Node),
                                Ident_Node    => Ident_Node);
      Check_Deferred_Items (End_Node => Last_Sibling_Of (Start_Node => Anno_Node),
                            Pack_Sym => Pack_Sym);
      CheckTypesCanBeUsed (Pack_Sym, Last_Sibling_Of (Start_Node => Anno_Node));
   elsif Syntax_Node_Type (Node => Priv_Part_Node) = SPSymbols.hidden_part then
      -- ASSUME Priv_Part_Node = hidden_part
      Dictionary.AddPrivatePart
        (Pack_Sym,
         Dictionary.Location'(Start_Position => Node_Position (Node => Priv_Part_Node),
                              End_Position   => Node_Position (Node => Priv_Part_Node)),
         True);
      ErrorHandler.Hidden_Text
        (Position => Node_Position (Node => Priv_Part_Node),
         Unit_Str => Ident_Str,
         Unit_Typ => SPSymbols.private_part);
      CheckTypesCanBeUsed (Pack_Sym, Priv_Part_Node);
   elsif Syntax_Node_Type (Node => Priv_Part_Node) = SPSymbols.basic_declarative_item_rep then
      -- ASSUME Priv_Part_Node = basic_declarative_item_rep
      Dictionary.AddPrivatePart
        (Pack_Sym,
         Dictionary.Location'(Start_Position => Node_Position (Node => Priv_Part_Node),
                              End_Position   => Node_Position (Node => Priv_Part_Node)),
         False);
      Wf_Private (Node  => Priv_Part_Node,
                  Scope => Dictionary.PrivateScope (Pack_Sym));

      -- check that private types resolved into scalars have not
      -- invalidated parameter declarations of exported procedures
      if CommandLineData.Content.Language_Profile = CommandLineData.SPARK83 then
         -- guarded because '83 rules different from 95 and 2005
         CheckModes (Node     => Vis_Part_Node,
                     Pack_Sym => Pack_Sym);
      end if;
      Check_Closing_Identifier (End_Name_Node => Next_Sibling (Current_Node => Priv_Part_Node),
                                Ident_Node    => Ident_Node);
      Check_Deferred_Items (End_Node => Next_Sibling (Current_Node => Priv_Part_Node),
                            Pack_Sym => Pack_Sym);
      CheckTypesCanBeUsed (Pack_Sym, Next_Sibling (Priv_Part_Node));
   else
      SystemErrors.Fatal_Error
        (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Private_Part = basic_declarative_item_rep OR hidden_part OR NULL in Wf_Package_Specification");
   end if;

   -- check for cases where package requires a body
   --# assert True;
   if Enclosing_Scope_Type = In_Library and then not Dictionary.PackageRequiresBody (Pack_Sym) then
      End_Pos_Node := Last_Sibling_Of (Start_Node => Anno_Node);
      -- ASSUME End_Pos_Node = private_part OR dotted_simple_name
      if Syntax_Node_Type (Node => End_Pos_Node) = SPSymbols.private_part then
         -- ASSUME End_Pos_Node = private_part
         -- declaring a library package for which Ada rules do not demand a body so
         -- checks must be made to see if Spark rules require a body
         End_Pos_Node := Child_Node (Current_Node => End_Pos_Node);
         -- ASSUME End_Pos_Node = basic_declarative_item_rep OR hidden_part
         if Syntax_Node_Type (Node => End_Pos_Node) = SPSymbols.basic_declarative_item_rep then
            End_Pos_Node := Next_Sibling (Current_Node => End_Pos_Node);
         elsif Syntax_Node_Type (Node => End_Pos_Node) /= SPSymbols.hidden_part then
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect End_Pos_Node = basic_declarative_item_rep OR hidden_part in Wf_Package_Specification");
         end if;
      elsif Syntax_Node_Type (Node => End_Pos_Node) /= SPSymbols.dotted_simple_name then
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect End_Pos_Node = private_part OR dotted_simple_name in Wf_Package_Specification");
      end if;
      -- ASSUME End_Pos_Node = dotted_simple_name OR hidden_part
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => End_Pos_Node) = SPSymbols.dotted_simple_name
           or else Syntax_Node_Type (Node => End_Pos_Node) = SPSymbols.hidden_part,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect End_Pos_Node = dotted_simple_name OR hidden_part in Wf_Package_Specification");
      Check_Body_Required_By_Spark (Node     => End_Pos_Node,
                                    Pack_Sym => Pack_Sym);
   end if;

   CheckStateCanBeInitialized (Pack_Sym, Anno_Node);
   CheckAnnouncedTypesDeclared
     (PackSym => Pack_Sym,
      Scope   => Dictionary.VisibleScope (Pack_Sym),
      Node    => Last_Sibling_Of (Start_Node => Anno_Node));

   --  The cross-references for the own variables are generated
   --  after the full semantic analysis of the package
   --  specification because we need to know if an own variable is
   --  actually an abstract own variable or a visble concrete own
   --  variable. If it is an abstract own variable, the own
   --  variable is considered as a declaration, if it is a visible
   --  concrete own variable, the own variable is a usage of the
   --  visible concrete variable that will be declared later in the
   --  same package specification. If the declaration of the
   --  concrete variable only appears in the package body, the own
   --  variable is considered as an abstract declaration.
   if ErrorHandler.Generate_SLI and then Child_Node (Anno_Node) /= STree.NullNode then
      SLI.Generate_Xref_Own
        (Comp_Unit  => ContextManager.Ops.Current_Unit,
         Parse_Tree => Child_Node (Anno_Node),
         Scope      => Pack_Vis_Scope);
   end if;

end Wf_Package_Specification;
