-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Sem.CompUnit)
procedure Wf_Qualified_Expression
  (Node    : in     STree.SyntaxNode;
   Scope   : in     Dictionary.Scopes;
   E_Stack : in out ExpStack.ExpStackType) is

   Name_Exp, Exp_Result : Exp_Record;
   Exp_Value            : Maths.Value;
   Store_Rep            : LexTokenManager.Lex_String;
   Exp_Node             : STree.SyntaxNode;
begin
   -- ASSUME Node = qualified_expression OR annotation_qualified_expression
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Node) = SPSymbols.qualified_expression
        or else Syntax_Node_Type (Node => Node) = SPSymbols.annotation_qualified_expression,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Node = qualified_expression OR annotation_qualified_expression in Wf_Qualified_Expression");

   Exp_Node := Next_Sibling (Current_Node => Child_Node (Current_Node => Node));
   -- ASSUME Exp_Node = aggregate            OR extension_aggregate            OR expression OR
   --                   annotation_aggregate OR annotation_extension_aggregate OR annotation_expression
   if Syntax_Node_Type (Node => Exp_Node) = SPSymbols.expression
     or else Syntax_Node_Type (Node => Exp_Node) = SPSymbols.annotation_expression then
      -- ASSUME Exp_Node = expression OR annotation_expression
      -- we are dealing with a simple qualified expression
      ExpStack.Pop (Exp_Result, E_Stack);
      ExpStack.Pop (Name_Exp, E_Stack);
      case Name_Exp.Sort is
         when Is_Type_Mark =>
            -- check that constraining type mark is not unconstrained
            if Dictionary.IsUnconstrainedArrayType (Name_Exp.Type_Symbol) then
               ErrorHandler.Semantic_Error
                 (Err_Num   => 39,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Node),
                  Id_Str    => LexTokenManager.Null_String);
               ExpStack.Push (UnknownTypeRecord, E_Stack);
            elsif Dictionary.CompatibleTypes (Scope, Name_Exp.Type_Symbol, Exp_Result.Type_Symbol) then
               Maths.StorageRep (Exp_Result.Range_RHS, Store_Rep);
               if Dictionary.IsPredefinedStringType (Exp_Result.Type_Symbol) then
                  if Exp_Result.Range_RHS = Maths.NoValue then -- must be a parameter
                     ErrorHandler.Semantic_Error
                       (Err_Num   => 39,
                        Reference => ErrorHandler.No_Reference,
                        Position  => Node_Position (Node => Exp_Node),
                        Id_Str    => LexTokenManager.Null_String);
                  elsif LexTokenManager.Lex_String_Case_Insensitive_Compare
                    (Lex_Str1 => Dictionary.GetScalarAttributeValue
                       (False,
                        LexTokenManager.Last_Token,
                        Dictionary.CurrentSymbol (Dictionary.FirstArrayIndex (Name_Exp.Type_Symbol))),
                     Lex_Str2 => Store_Rep) /=
                    LexTokenManager.Str_Eq then
                     if Syntax_Node_Type (Node => Node) = SPSymbols.annotation_qualified_expression then
                        -- in annotation
                        ErrorHandler.Semantic_Error
                          (Err_Num   => 399,
                           Reference => ErrorHandler.No_Reference,
                           Position  => Node_Position (Node => Node),
                           Id_Str    => LexTokenManager.Null_String);
                     else
                        ErrorHandler.Semantic_Error
                          (Err_Num   => 402,
                           Reference => ErrorHandler.No_Reference,
                           Position  => Node_Position (Node => Node),
                           Id_Str    => LexTokenManager.Null_String);
                     end if;
                  end if;
               elsif Illegal_Unconstrained (Name_Exp.Type_Symbol, Exp_Result.Type_Symbol) then
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 418,
                     Reference => ErrorHandler.No_Reference,
                     Position  => Node_Position (Node => Exp_Node),
                     Id_Str    => LexTokenManager.Null_String);
               end if;
               ConstraintCheck
                 (Exp_Result.Value,
                  Exp_Value,
                  Syntax_Node_Type (Node => Node) = SPSymbols.annotation_qualified_expression,
                  Name_Exp.Type_Symbol,
                  Node_Position (Node => Exp_Node));

               Exp_Result.Value                 := Exp_Value;
               Exp_Result.Type_Symbol           := Name_Exp.Type_Symbol;
               Exp_Result.Is_AVariable          := False;
               Exp_Result.Is_An_Entire_Variable := False;
               ExpStack.Push (Exp_Result, E_Stack);
            else
               ErrorHandler.Semantic_Error
                 (Err_Num   => 38,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Exp_Node),
                  Id_Str    => LexTokenManager.Null_String);
               ExpStack.Push (UnknownTypeRecord, E_Stack);
            end if;
         when Is_Unknown =>
            -- any error from unknown prefix will already have been raised
            ExpStack.Push (UnknownTypeRecord, E_Stack);
         when others =>
            ExpStack.Push (UnknownTypeRecord, E_Stack);
            ErrorHandler.Semantic_Error
              (Err_Num   => 95,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Node),
               Id_Str    => LexTokenManager.Null_String);
      end case;
   elsif Syntax_Node_Type (Node => Exp_Node) /= SPSymbols.aggregate
     and then Syntax_Node_Type (Node => Exp_Node) /= SPSymbols.annotation_aggregate
     and then Syntax_Node_Type (Node => Exp_Node) /= SPSymbols.extension_aggregate
     and then Syntax_Node_Type (Node => Exp_Node) /= SPSymbols.annotation_extension_aggregate then
      SystemErrors.Fatal_Error
        (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Exp_Node = aggregate OR extension_aggregate OR expression OR annotation_aggregate OR annotation_extension_aggregate OR annotation_expression in Wf_Qualified_Expression");
   end if;
end Wf_Qualified_Expression;
