-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Sem.CompUnit.Wf_Subtype_Declaration)
procedure Wf_Ravenscar_Subtype
  (Id_Str          : in LexTokenManager.Lex_String;
   Type_Sym        : in Dictionary.Symbol;
   Scope           : in Dictionary.Scopes;
   Id_Node         : in STree.SyntaxNode;
   Constraint_Node : in STree.SyntaxNode) is

   The_Subtype : Dictionary.Symbol;
   Assoc_Node  : STree.SyntaxNode;

   procedure Process_Expression
     (Exp_Node    : in STree.SyntaxNode;
      Formal_Sym  : in Dictionary.Symbol;
      Type_Sym    : in Dictionary.Symbol;
      Subtype_Sym : in Dictionary.Symbol;
      Scope       : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#           out AggregateStack.State;
   --#           out TheHeap;
   --# derives AggregateStack.State,
   --#         STree.Table,
   --#         TheHeap                    from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         Exp_Node,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         Dictionary.Dict,
   --#         LexTokenManager.State      from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         Exp_Node,
   --#                                         Formal_Sym,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         Subtype_Sym,
   --#                                         Type_Sym &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Exp_Node,
   --#                                         Formal_Sym,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         Subtype_Sym,
   --#                                         Type_Sym &
   --#         SLI.State                  from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Exp_Node,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         Statistics.TableUsage      from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         Exp_Node,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         STree.Table;
   is
      Result                : Exp_Record;
      Unwanted_Seq          : SeqAlgebra.Seq;
      Unused_Component_Data : ComponentManager.ComponentData;
      Static_Value          : LexTokenManager.Lex_String;
      Pragma_Kind           : Dictionary.RavenscarPragmasWithValue;
      Value_Rep             : LexTokenManager.Lex_String;
   begin
      -- ASSUME Exp_Node = expression
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Exp_Node) = SPSymbols.expression,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Exp_Node = expression in Process_Expression");

      Heap.Initialize (TheHeap);
      ComponentManager.Initialise (Unused_Component_Data);
      SeqAlgebra.CreateSeq (TheHeap, Unwanted_Seq);
      --# accept Flow, 10, Unused_Component_Data, "Expected ineffective assignment";
      WalkExpression
        (Exp_Node                => Exp_Node,
         Scope                   => Scope,
         Type_Context            => Dictionary.GetUnknownTypeMark,
         Context_Requires_Static => False,
         Result                  => Result,
         Ref_Var                 => Unwanted_Seq,
         Component_Data          => Unused_Component_Data);
      --# end accept;
      SeqAlgebra.DisposeOfSeq (TheHeap, Unwanted_Seq);

      AssignmentCheck (Node_Position (Node => Exp_Node), Scope, Dictionary.GetType (Formal_Sym), Result);

      if Result.Is_Static then
         Maths.StorageRep (Result.Value, Static_Value);
         Dictionary.AddDiscriminantConstraintStaticValue
           (ProtectedOrTaskSubtype => Subtype_Sym,
            Comp_Unit              => ContextManager.Ops.Current_Unit,
            Declaration            => Dictionary.Location'(Start_Position => Node_Position (Node => Exp_Node),
                                                           End_Position   => Node_Position (Node => Exp_Node)),
            TheValue               => Static_Value);
         if Dictionary.SetsPriority (Formal_Sym) then
            if Dictionary.GetTypeHasPragma (Type_Sym, Dictionary.Priority) then
               Pragma_Kind := Dictionary.Priority;
            else
               -- must be
               Pragma_Kind := Dictionary.InterruptPriority;
            end if;
            CheckPriorityRange
              (Error_Sym   => Subtype_Sym,
               Scope       => Scope,
               Pragma_Kind => Pragma_Kind,
               Err_Pos     => Node_Position (Node => Exp_Node),
               Value       => Result.Value,
               Value_Rep   => Value_Rep);
            -- Value_Rep is either a storage rep of a valid value or a null string; we can always add it to dict
            Dictionary.SetSubtypePriority (Subtype_Sym, Value_Rep);
         end if;
      elsif Dictionary.TypeIsAccess (Dictionary.GetType (Formal_Sym)) then
         Dictionary.AddDiscriminantConstraintAccessedObject
           (ProtectedOrTaskSubtype => Subtype_Sym,
            Comp_Unit              => ContextManager.Ops.Current_Unit,
            Declaration            => Dictionary.Location'(Start_Position => Node_Position (Node => Exp_Node),
                                                           End_Position   => Node_Position (Node => Exp_Node)),
            TheObject              => Result.Variable_Symbol);
         -- N.B. VariableSymbol is the accessed _variable_ name, put there by wf_attribute_designator
      else
         -- not static and not a protected types so must be wrong
         ErrorHandler.Semantic_Error
           (Err_Num   => 36,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Exp_Node),
            Id_Str    => LexTokenManager.Null_String);
      end if;
   end Process_Expression;

   ------------------------------------------------------------------------

   procedure Handle_Named_Association
     (Node        : in STree.SyntaxNode;
      Type_Sym    : in Dictionary.Symbol;
      Subtype_Sym : in Dictionary.Symbol;
      Scope       : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         Subtype_Sym,
   --#                                         Type_Sym &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         Subtype_Sym,
   --#                                         Type_Sym;
   is
      It              : Dictionary.Iterator;
      Expression_Node : STree.SyntaxNode;
   begin
      -- ASSUME Node = named_argument_association
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Node) = SPSymbols.named_argument_association,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Node = named_argument_association in Handle_Named_Association");

      CheckNamedAssociation (TheFormals             => Type_Sym,
                             Scope                  => Scope,
                             NamedArgumentAssocNode => Node);

      -- Loop through all the formals
      It := Dictionary.FirstKnownDiscriminant (Type_Sym);
      while not Dictionary.IsNullIterator (It) loop
         Expression_Node := FindActualNode (For_Formal                => Dictionary.CurrentSymbol (It),
                                            Named_Argument_Assoc_Node => Node);
         -- ASSUME Expression_Node = expression OR NULL
         if Syntax_Node_Type (Node => Expression_Node) = SPSymbols.expression then
            -- ASSUME Expression_Node = expression
            Process_Expression
              (Exp_Node    => Expression_Node,
               Formal_Sym  => Dictionary.CurrentSymbol (It),
               Type_Sym    => Type_Sym,
               Subtype_Sym => Subtype_Sym,
               Scope       => Scope);
         elsif Expression_Node /= STree.NullNode then
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Expression_Node = expression OR NULL in Handle_Named_Association");
         end if;
         It := Dictionary.NextSymbol (It);
      end loop;
   end Handle_Named_Association;

   ------------------------------------------------------------------------

   procedure Handle_Positional_Association
     (Node        : in STree.SyntaxNode;
      Type_Sym    : in Dictionary.Symbol;
      Subtype_Sym : in Dictionary.Symbol;
      Scope       : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         Subtype_Sym,
   --#                                         Type_Sym &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         Subtype_Sym,
   --#                                         Type_Sym;
   is
      Expression_Node : STree.SyntaxNode;
      FormalIt        : Dictionary.Iterator;
      ActualIt        : STree.Iterator;
      FormalParameter : Dictionary.Symbol;
   begin
      -- ASSUME Assoc_Node = positional_argument_association
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Node) = SPSymbols.positional_argument_association,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Node = positional_argument_association in Handle_Positional_Association");

      FormalIt := Dictionary.FirstKnownDiscriminant (Type_Sym);

      ActualIt := Find_First_Node (Node_Kind    => SPSymbols.expression,
                                   From_Root    => Node,
                                   In_Direction => STree.Down);

      while not Dictionary.IsNullIterator (FormalIt) and not STree.IsNull (ActualIt) loop

         FormalParameter := Dictionary.CurrentSymbol (FormalIt);
         Expression_Node := Get_Node (It => ActualIt);

         Process_Expression
           (Exp_Node    => Expression_Node,
            Formal_Sym  => FormalParameter,
            Type_Sym    => Type_Sym,
            Subtype_Sym => Subtype_Sym,
            Scope       => Scope);

         FormalIt := Dictionary.NextSymbol (FormalIt);
         ActualIt := STree.NextNode (ActualIt);
      end loop;

      if not Dictionary.IsNullIterator (FormalIt) or not STree.IsNull (ActualIt) then
         ErrorHandler.Semantic_Error_Sym
           (Err_Num   => 893,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Node),
            Sym       => Type_Sym,
            Scope     => Scope);
      end if;
   end Handle_Positional_Association;

begin -- Wf_Ravenscar_Subtype

   -- ASSUME Constraint_Node = index_or_discriminant_constraint
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Constraint_Node) = SPSymbols.index_or_discriminant_constraint,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Constraint_Node = index_or_discriminant_constraint in Wf_Ravenscar_Subtype");
   -- ASSUME Id_Node = identifier
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Id_Node) = SPSymbols.identifier,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Id_Node = identifier in Wf_Ravenscar_Subtype");

   Dictionary.AddTaskOrProtectedSubtype
     (Name        => Id_Str,
      Parent      => Type_Sym,
      Comp_Unit   => ContextManager.Ops.Current_Unit,
      Declaration => Dictionary.Location'(Start_Position => Node_Position (Node => Id_Node),
                                          End_Position   => Node_Position (Node => Id_Node)),
      Scope       => Scope,
      Context     => Dictionary.ProgramContext,
      TheSubtype  => The_Subtype);
   if ErrorHandler.Generate_SLI then
      SLI.Generate_Xref_Symbol
        (Comp_Unit      => ContextManager.Ops.Current_Unit,
         Parse_Tree     => Id_Node,
         Symbol         => The_Subtype,
         Is_Declaration => True);
   end if;
   Assoc_Node := Child_Node (Current_Node => Child_Node (Current_Node => Constraint_Node));
   -- ASSUME Assoc_Node = named_argument_association OR positional_argument_association
   if Syntax_Node_Type (Node => Assoc_Node) = SPSymbols.named_argument_association then
      -- ASSUME Assoc_Node = named_argument_association
      Handle_Named_Association (Node        => Assoc_Node,
                                Type_Sym    => Type_Sym,
                                Subtype_Sym => The_Subtype,
                                Scope       => Scope);
   elsif Syntax_Node_Type (Node => Assoc_Node) = SPSymbols.positional_argument_association then
      -- ASSUME Assoc_Node = positional_argument_association
      Handle_Positional_Association (Node        => Assoc_Node,
                                     Type_Sym    => Type_Sym,
                                     Subtype_Sym => The_Subtype,
                                     Scope       => Scope);
   else
      SystemErrors.Fatal_Error
        (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Assoc_Node = named_argument_association OR positional_argument_association in Wf_Ravenscar_Subtype");
   end if;
   -- Check that subtype has a priority, if one has not been set then inherit parent's
   if LexTokenManager.Lex_String_Case_Insensitive_Compare
     (Lex_Str1 => Dictionary.GetTypePriority (The_Subtype),
      Lex_Str2 => LexTokenManager.Null_String) =
     LexTokenManager.Str_Eq then
      Dictionary.SetSubtypePriority (The_Subtype, Dictionary.GetTypePriority (Type_Sym));
   end if;
end Wf_Ravenscar_Subtype;
