-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Sem.CompUnit)
procedure Wf_Full_Type_Declaration
  (Node           : in     STree.SyntaxNode;
   Scope          : in     Dictionary.Scopes;
   Component_Data : in out ComponentManager.ComponentData) is
   Ident_Node               : STree.SyntaxNode;
   Ident_Str                : LexTokenManager.Lex_String;
   Sym                      : Dictionary.Symbol;
   Type_Definition_Node     : STree.SyntaxNode;
   Type_Declared_As_Limited : Boolean;
   Type_Declared_As_Tagged  : Boolean;
   Error_Found              : Boolean;

   -------------------------------

   function Is_Private_Type_Resolution (Sym   : Dictionary.Symbol;
                                        Scope : Dictionary.Scopes) return Boolean
   --# global in Dictionary.Dict;
   is
   begin
      return (not Dictionary.IsDeclared (Sym)
                and then not Dictionary.IsVisibleScope (Scope)
                and then Dictionary.IsType (Sym)
                and then Dictionary.TypeIsPrivate (Sym)
                and then (Dictionary.GetRegion (Scope) = Dictionary.GetRegion (Dictionary.GetScope (Sym))));
   end Is_Private_Type_Resolution;

   -------------------------------

   procedure Empty_Type_Check (Dec_Loc      : in     LexTokenManager.Token_Position;
                               Lower, Upper : in out LexTokenManager.Lex_String)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dec_Loc,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Lower,
   --#                                         SPARK_IO.File_Sys,
   --#                                         Upper &
   --#         Lower,
   --#         Upper                      from LexTokenManager.State,
   --#                                         Lower,
   --#                                         Upper;
   is
      Unused         : Maths.ErrorCode;
      Maths_Result   : Maths.Value;
      Range_Is_Empty : Boolean;
   begin
      if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Lower,
                                                              Lex_Str2 => LexTokenManager.Null_String) /=
        LexTokenManager.Str_Eq
        and then LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Upper,
                                                                      Lex_Str2 => LexTokenManager.Null_String) /=
        LexTokenManager.Str_Eq then
         --# accept Flow, 10, Unused, "Expected ineffective assignment";
         Maths.Lesser (Maths.ValueRep (Upper), Maths.ValueRep (Lower),
                       --to get
                       Maths_Result, Unused); -- not used because it can only be ok or type mismatch
         Maths.ValueToBool (Maths_Result,
                            --to get
                            Range_Is_Empty, Unused);
         --# end accept;
         if Range_Is_Empty then
            Lower := LexTokenManager.Null_String;
            Upper := LexTokenManager.Null_String;
            ErrorHandler.Semantic_Error
              (Err_Num   => 416,
               Reference => ErrorHandler.No_Reference,
               Position  => Dec_Loc,
               Id_Str    => LexTokenManager.Null_String);
         end if;
      end if;
      --# accept Flow, 33, Unused, "Expected to be neither referenced nor exported";
   end Empty_Type_Check;

   -------------------------------

   procedure Wf_Integer
     (Node       : in STree.SyntaxNode;
      Scope      : in Dictionary.Scopes;
      Ident_Node : in STree.SyntaxNode;
      Dec_Loc    : in LexTokenManager.Token_Position)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --#           out Aggregate_Stack.State;
   --# derives Aggregate_Stack.State,
   --#         LexTokenManager.State,
   --#         STree.Table,
   --#         TheHeap                    from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         Dictionary.Dict            from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dec_Loc,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dec_Loc,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dec_Loc,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         Statistics.TableUsage      from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.integer_type_definition and
   --#   Syntax_Node_Type (Ident_Node, STree.Table) = SP_Symbols.identifier;
   --# post STree.Table = STree.Table~;
      is separate;

   -------------------------------

   procedure Wf_Derived
     (Node       : in STree.SyntaxNode;
      Scope      : in Dictionary.Scopes;
      Ident_Node : in STree.SyntaxNode;
      Dec_Loc    : in LexTokenManager.Token_Position)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Aggregate_Stack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives Aggregate_Stack.State,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         Dictionary.Dict            from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dec_Loc,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dec_Loc,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.derived_type_definition and
   --#   Syntax_Node_Type (Ident_Node, STree.Table) = SP_Symbols.identifier;
   --# post STree.Table = STree.Table~;
      is separate;

   -------------------------------

   procedure Wf_Modular
     (Node       : in STree.SyntaxNode;
      Scope      : in Dictionary.Scopes;
      Ident_Node : in STree.SyntaxNode;
      Dec_Loc    : in LexTokenManager.Token_Position)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Aggregate_Stack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives Aggregate_Stack.State,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         Dictionary.Dict            from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dec_Loc,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dec_Loc,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.modular_type_definition and
   --#   Syntax_Node_Type (Ident_Node, STree.Table) = SP_Symbols.identifier;
   --# post STree.Table = STree.Table~;
      is separate;

   -------------------------------

   procedure Wf_Enum
     (Node       : in STree.SyntaxNode;
      Scope      : in Dictionary.Scopes;
      Ident_Node : in STree.SyntaxNode;
      Dec_Loc    : in LexTokenManager.Token_Position)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dictionary.Dict,
   --#         LexTokenManager.State      from ContextManager.Ops.Unit_Stack,
   --#                                         Dec_Loc,
   --#                                         Dictionary.Dict,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dec_Loc,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         SLI.State                  from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dec_Loc,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         STree.Table;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.enumeration_type_definition and
   --#   Syntax_Node_Type (Ident_Node, STree.Table) = SP_Symbols.identifier;
      is separate;

   -------------------------------

   procedure Wf_Record
     (Node                        : in STree.SyntaxNode;
      Scope                       : in Dictionary.Scopes;
      Ident_Node                  : in STree.SyntaxNode;
      Dec_Loc                     : in LexTokenManager.Token_Position;
      Extends                     : in Dictionary.Symbol;
      Private_Type_Being_Resolved : in Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         STree.Table                from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dec_Loc,
   --#                                         Dictionary.Dict,
   --#                                         Extends,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Private_Type_Being_Resolved,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dec_Loc,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Extends,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Private_Type_Being_Resolved,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         SLI.State                  from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dec_Loc,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Extends,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Private_Type_Being_Resolved,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
   --# pre (Syntax_Node_Type (Node, STree.Table) = SP_Symbols.record_type_definition or
   --#        Syntax_Node_Type (Node, STree.Table) = SP_Symbols.record_type_extension) and
   --#   Syntax_Node_Type (Ident_Node, STree.Table) = SP_Symbols.identifier;
   --# post STree.Table = STree.Table~;
      is separate;

   -------------------------------

   procedure Wf_Real
     (Node       : in STree.SyntaxNode;
      Scope      : in Dictionary.Scopes;
      Ident_Node : in STree.SyntaxNode;
      Dec_Loc    : in LexTokenManager.Token_Position)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --#           out Aggregate_Stack.State;
   --# derives Aggregate_Stack.State,
   --#         LexTokenManager.State,
   --#         STree.Table,
   --#         TheHeap                    from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         Dictionary.Dict            from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dec_Loc,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dec_Loc,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dec_Loc,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         Statistics.TableUsage      from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.real_type_definition and
   --#   Syntax_Node_Type (Ident_Node, STree.Table) = SP_Symbols.identifier;
   --# post STree.Table = STree.Table~;
      is separate;

   -------------------------------

   procedure Wf_Type_Extension
     (Node                        : in STree.SyntaxNode;
      Scope                       : in Dictionary.Scopes;
      Ident_Node                  : in STree.SyntaxNode;
      Dec_Loc                     : in LexTokenManager.Token_Position;
      Private_Type_Being_Resolved : in Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         STree.Table                from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dec_Loc,
   --#                                         Dictionary.Dict,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Private_Type_Being_Resolved,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dec_Loc,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Private_Type_Being_Resolved,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         SLI.State                  from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dec_Loc,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Private_Type_Being_Resolved,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.type_extension and
   --#   Syntax_Node_Type (Ident_Node, STree.Table) = SP_Symbols.identifier;
   --# post STree.Table = STree.Table~;
      is separate;

   -------------------------------

   procedure Wf_Protected_Type_Declaration
     (Node           : in     STree.SyntaxNode;
      Scope          : in     Dictionary.Scopes;
      Component_Data : in out ComponentManager.ComponentData)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Aggregate_Stack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives Aggregate_Stack.State,
   --#         Component_Data,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         Component_Data,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Component_Data,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.protected_type_declaration;
   --# post STree.Table = STree.Table~;
      is separate;

   -------------------------------

   procedure Wf_Task_Type_Declaration (Node  : in STree.SyntaxNode;
                                       Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Aggregate_Stack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives Aggregate_Stack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.task_type_declaration;
   --# post STree.Table = STree.Table~;
      is separate;

   -------------------------------

   procedure Wf_Type_Definition
     (Node                        : in STree.SyntaxNode;
      Scope                       : in Dictionary.Scopes;
      Ident_Node                  : in STree.SyntaxNode;
      Dec_Loc                     : in LexTokenManager.Token_Position;
      Private_Type_Being_Resolved : in Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Aggregate_Stack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives Aggregate_Stack.State,
   --#         Statistics.TableUsage,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         Dictionary.Dict,
   --#         STree.Table                from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dec_Loc,
   --#                                         Dictionary.Dict,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Private_Type_Being_Resolved,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dec_Loc,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Private_Type_Being_Resolved,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         LexTokenManager.State      from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dec_Loc,
   --#                                         Dictionary.Dict,
   --#                                         Ident_Node,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.type_definition and
   --#   Syntax_Node_Type (Ident_Node, STree.Table) = SP_Symbols.identifier;
   --# post STree.Table = STree.Table~;
   is
      Type_Node                : STree.SyntaxNode;
      Unused_Array_Type_Symbol : Dictionary.Symbol;
      Unused_Errors_Found      : Boolean;
   begin
      Type_Node := Child_Node (Current_Node => Node);
      -- ASSUME Type_Node = enumeration_type_definition OR integer_type_definition OR real_type_definition OR
      --                    array_type_definition OR record_type_definition OR modular_type_definition OR
      --                    type_extension OR derived_type_definition
      case Syntax_Node_Type (Node => Type_Node) is
         when SP_Symbols.enumeration_type_definition =>
            -- ASSUME Type_Node = enumeration_type_definition
            Wf_Enum (Node       => Type_Node,
                     Scope      => Scope,
                     Ident_Node => Ident_Node,
                     Dec_Loc    => Dec_Loc);

         when SP_Symbols.real_type_definition =>
            -- ASSUME Type_Node = real_type_definition
            Wf_Real (Node       => Type_Node,
                     Scope      => Scope,
                     Ident_Node => Ident_Node,
                     Dec_Loc    => Dec_Loc);

         when SP_Symbols.record_type_definition =>
            -- ASSUME Type_Node = record_type_definition
            Wf_Record
              (Node                        => Type_Node,
               Scope                       => Scope,
               Ident_Node                  => Ident_Node,
               Dec_Loc                     => Dec_Loc,
               Extends                     => Dictionary.NullSymbol,
               Private_Type_Being_Resolved => Private_Type_Being_Resolved);

         when SP_Symbols.integer_type_definition =>
            -- ASSUME Type_Node = integer_type_definition
            Wf_Integer (Node       => Type_Node,
                        Scope      => Scope,
                        Ident_Node => Ident_Node,
                        Dec_Loc    => Dec_Loc);

         when SP_Symbols.derived_type_definition =>
            -- ASSUME Type_Node = derived_type_definition
            Wf_Derived (Node       => Type_Node,
                        Scope      => Scope,
                        Ident_Node => Ident_Node,
                        Dec_Loc    => Dec_Loc);

         when SP_Symbols.modular_type_definition =>
            -- ASSUME Type_Node = modular_type_definition
            Wf_Modular (Node       => Type_Node,
                        Scope      => Scope,
                        Ident_Node => Ident_Node,
                        Dec_Loc    => Dec_Loc);

         when SP_Symbols.array_type_definition =>
            -- ASSUME Type_Node = array_type_definition
            --# accept Flow, 10, Unused_Errors_Found, "Expected ineffective assignment" &
            --#        Flow, 10, Unused_Array_Type_Symbol, "Expected ineffective assignment";
            Wf_Array_Type_Definition
              (Node         => Type_Node,
               Scope        => Scope,
               Ident_Node   => Ident_Node,
               Dec_Loc      => Dec_Loc,
               Is_Generic   => False,
               Errors_Found => Unused_Errors_Found,
               The_Array    => Unused_Array_Type_Symbol);
            --# end accept;
         when SP_Symbols.type_extension =>
            -- ASSUME Type_Node = type_extension
            Wf_Type_Extension
              (Node                        => Type_Node,
               Scope                       => Scope,
               Ident_Node                  => Ident_Node,
               Dec_Loc                     => Dec_Loc,
               Private_Type_Being_Resolved => Private_Type_Being_Resolved);
         when others =>
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Type_Node = enumeration_type_definition OR integer_type_definition OR " &
                 "real_type_definition OR array_type_definition OR record_type_definition OR modular_type_definition OR " &
                 "type_extension OR derived_type_definition in Wf_Type_Definition");
      end case;
      --# accept Flow, 33, Unused_Errors_Found, "Expected to be neither referenced nor exported" &
      --#        Flow, 33, Unused_Array_Type_Symbol, "Expected to be neither referenced nor exported";
   end Wf_Type_Definition;

   -------------------------------

   function Valid_Scope_For_Task_Or_Protected_Type_Declaration (Scope : Dictionary.Scopes) return Boolean
   --# global in Dictionary.Dict;
   is
   begin
      return (Dictionary.IsVisibleScope (Scope) or else Dictionary.IsPrivateScope (Scope))
        and then Dictionary.IsPackage (Dictionary.GetRegion (Scope));
   end Valid_Scope_For_Task_Or_Protected_Type_Declaration;

begin -- Wf_Full_Type_Declaration
   Ident_Node := Child_Node (Current_Node => Node);
   -- ASSUME Ident_Node = identifier OR task_type_declaration OR protected_type_declaration
   case Syntax_Node_Type (Node => Ident_Node) is
      when SP_Symbols.identifier =>
         -- ASSUME Ident_Node = identifier
         Ident_Str := Node_Lex_String (Node => Ident_Node);
         Sym       :=
           Dictionary.LookupItem
           (Name              => Ident_Str,
            Scope             => Scope,
            Context           => Dictionary.ProofContext,
            Full_Package_Name => False);

         if Sym = Dictionary.NullSymbol
           or else (Dictionary.IsTypeMark (Sym)
                      and then Dictionary.TypeIsAnnounced (Sym)
                      and then not Dictionary.IsDeclared (Sym))
           or else Is_Private_Type_Resolution (Sym   => Sym,
                                               Scope => Scope) then
            Type_Definition_Node := Next_Sibling (Current_Node => Ident_Node);
            -- ASSUME Type_Definition_Node = type_definition
            SystemErrors.RT_Assert
              (C       => Syntax_Node_Type (Node => Type_Definition_Node) = SP_Symbols.type_definition,
               Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Type_Definition_Node = type_definition in Wf_Full_Type_Declaration");
            if Sym = Dictionary.NullSymbol then
               -- initial type declaration
               Wf_Type_Definition
                 (Node                        => Type_Definition_Node,
                  Scope                       => Scope,
                  Ident_Node                  => Ident_Node,
                  Dec_Loc                     => Node_Position (Node => Node),
                  Private_Type_Being_Resolved => Sym);
            else
               Type_Declared_As_Limited := Dictionary.IsLimitedPrivateType (Sym);
               Type_Declared_As_Tagged  := Dictionary.TypeIsTagged (Sym);
               Error_Found              := False;
               Wf_Type_Definition
                 (Node                        => Type_Definition_Node,
                  Scope                       => Scope,
                  Ident_Node                  => Ident_Node,
                  Dec_Loc                     => Node_Position (Node => Node),
                  Private_Type_Being_Resolved => Sym);
               if Dictionary.IsUnconstrainedArrayType (Sym) then
                  if Dictionary.TypeIsAnnounced (Sym) then
                     Error_Found := True;
                     ErrorHandler.Semantic_Error
                       (Err_Num   => 311,
                        Reference => ErrorHandler.No_Reference,
                        Position  => Node_Position (Node => Ident_Node),
                        Id_Str    => Ident_Str);
                  else -- private type
                     Error_Found := True;
                     ErrorHandler.Semantic_Error
                       (Err_Num   => 331,
                        Reference => ErrorHandler.No_Reference,
                        Position  => Node_Position (Node => Ident_Node),
                        Id_Str    => Ident_Str);
                  end if;
               end if;

               if not Type_Declared_As_Limited and then Dictionary.TypeIsLimited (Sym, Dictionary.GlobalScope) then
                  Error_Found := True;
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 332,
                     Reference => ErrorHandler.No_Reference,
                     Position  => Node_Position (Node => Ident_Node),
                     Id_Str    => Ident_Str);
               end if;

               -- initial declaration tagged but completion is not
               if Type_Declared_As_Tagged and then not Dictionary.TypeIsTagged (Sym) then
                  Error_Found := True;
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 821,
                     Reference => ErrorHandler.No_Reference,
                     Position  => Node_Position (Node => Ident_Node),
                     Id_Str    => Ident_Str);
               end if;

               -- reverse case; this could be made legal but is not allowed for now
               if not Type_Declared_As_Tagged and then Dictionary.TypeIsTagged (Sym) then
                  Error_Found := True;
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 830,
                     Reference => ErrorHandler.No_Reference,
                     Position  => Node_Position (Node => Ident_Node),
                     Id_Str    => Ident_Str);
               end if;

               if not Error_Found then
                  STree.Set_Node_Lex_String (Sym  => Sym,
                                             Node => Ident_Node);
               end if;

            end if;
         else
            ErrorHandler.Semantic_Error
              (Err_Num   => 10,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Ident_Node),
               Id_Str    => Ident_Str);
         end if;

      when SP_Symbols.task_type_declaration =>
         -- ASSUME Ident_Node = task_type_declaration
         if CommandLineData.Ravenscar_Selected then
            if Valid_Scope_For_Task_Or_Protected_Type_Declaration (Scope => Scope) then
               Wf_Task_Type_Declaration (Node  => Ident_Node,
                                         Scope => Scope);
            else
               ErrorHandler.Semantic_Error
                 (Err_Num   => 987,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Ident_Node),
                  Id_Str    => LexTokenManager.Null_String);
            end if;
         else -- declaration not allowed
            ErrorHandler.Semantic_Error
              (Err_Num   => 850,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Ident_Node),
               Id_Str    => LexTokenManager.Null_String);
         end if;

      when SP_Symbols.protected_type_declaration =>
         -- ASSUME Ident_Node = protected_type_declaration
         if CommandLineData.Ravenscar_Selected then
            if Valid_Scope_For_Task_Or_Protected_Type_Declaration (Scope => Scope) then
               Wf_Protected_Type_Declaration (Node           => Ident_Node,
                                              Scope          => Scope,
                                              Component_Data => Component_Data);
            else
               ErrorHandler.Semantic_Error
                 (Err_Num   => 987,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Ident_Node),
                  Id_Str    => LexTokenManager.Null_String);
            end if;
         else -- declaration not allowed
            ErrorHandler.Semantic_Error
              (Err_Num   => 850,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Ident_Node),
               Id_Str    => LexTokenManager.Null_String);
         end if;

      when others =>
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Ident_Node = identifier OR task_type_declaration OR " &
              "protected_type_declaration in Wf_Full_Type_Declaration");
   end case;
end Wf_Full_Type_Declaration;
