#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>

/* general program headers */
#include "specimen.h"
#include "patch.h"
#include "mixer.h"

/* gui specific headers */
#include "gui.h"
#include "waveform.h"
#include "sample-editor.h"
#include "sample-selector.h"
#include "bank-ops.h"
#include "audio-settings.h"
#include "playmode-settings.h"
#include "filter-settings.h"
#include "adsr-settings.h"

/* parameter button codes */
typedef enum param_s {
     PARAM_FILTER,
     PARAM_MODE,
     PARAM_ADSR
} param_t;

/* main window */
static GtkWidget *window;
static GtkWidget *window_vbox;
static GtkWidget *master_vbox;

/* controls */
static GtkWidget *master_vol_spin;
static GtkWidget *chan_spin;
static GtkWidget *note_spin;
static GtkWidget *vol_spin;
static GtkWidget *pan_spin;
static GtkWidget *file_button;
static GtkWidget *file_label;
static GtkWidget *play_button;
static GtkWidget *patch_option_menu;
static GtkWidget *cut_spin;
static GtkWidget *cut_by_spin;
static GtkWidget *range_check;
static GtkWidget *range_low_spin;
static GtkWidget *range_up_spin;
static GtkWidget *waveform;

/* main menu */
static GtkWidget *menu_file;
static GtkWidget *menu_file_item;
static GtkWidget *menu_file_new_bank;
static GtkWidget *menu_file_open_bank;
static GtkWidget *menu_file_save_bank;
static GtkWidget *menu_file_save_bank_as;
static GtkWidget *menu_file_quit;
static GtkWidget *menu_file_vsep;
static GtkWidget *menu_settings;
static GtkWidget *menu_settings_item;
static GtkWidget *menu_settings_audio;

/* global adjustments */
static GtkAdjustment *master_vol_adj;
static GtkAdjustment *chan_adj;
static GtkAdjustment *note_adj;
static GtkAdjustment *vol_adj;
static GtkAdjustment *pan_adj;
static GtkAdjustment *cut_adj;
static GtkAdjustment *cut_by_adj;
static GtkAdjustment *range_low_adj;
static GtkAdjustment *range_up_adj;

/* forward declarations */
static void set_current_patch(int id);

/* refreshes list of current patches */
static int update_interface(int index)
{
     static int count = 0;
     static GtkWidget **item = NULL;
     static GtkWidget *menu = NULL;
     int *patch; /* array of active patches gets dumped here */
     int i;
     char *name;

     /* destroy any menu items */
     for (i = 0; i < count; i++)
	  if (item[i] != NULL)
	       gtk_widget_destroy(item[i]);

     /* destroy menu */
     if (menu != NULL)
	  gtk_widget_destroy(menu);

     /* create new menu */
     menu = gtk_menu_new();

     /* get array of active patches */
     count = patch_dump(&patch);
     if (count < 0)
	  debug("Failed to update interface, couldn't get array of active patches!\n");

     if (count == 0) {
	  /* better to attach an empty menu than no menu at all */
	  gtk_option_menu_set_menu(GTK_OPTION_MENU(patch_option_menu), menu);

	  gtk_label_set_text(GTK_LABEL(file_label), "Load Sample");

	  set_current_patch(-1);

	  return 0;
     }

     /* create array of menu items */
     item = malloc(sizeof(GtkMenuItem *) * count);
     if (item == NULL) {
	  errmsg("Failed to allocate space for patch menu!\n");
	  exit(FUBAR);
     }

     /* create a menu item for each patch id with appropriate callback */
     for (i = 0; i < count; i++) {
	  if ((name = patch_name_get(patch[i])) == NULL) {
	       errmsg("Couldn't find name for patch %d.\n", patch[i]);
	       exit(FUBAR);
	  }
	  item[i] = gtk_menu_item_new_with_label(name);
	  gtk_menu_shell_append(GTK_MENU_SHELL(menu), item[i]);
	  g_signal_connect_swapped(G_OBJECT(item[i]), "activate",
				   G_CALLBACK(set_current_patch), (gpointer) patch[i]);
	  gtk_widget_show(item[i]);
	  if (name != NULL)
	       free(name);
     }

     /* free patch array */
     if (patch != NULL)
	  free(patch);
     
     /* attach new menu to patch option-menu */
     gtk_option_menu_set_menu(GTK_OPTION_MENU(patch_option_menu), menu);

     /* activate the requested menu item */
     g_signal_emit_by_name(G_OBJECT(item[index]), "activate");
     gtk_option_menu_set_history(GTK_OPTION_MENU(patch_option_menu), index);

     return count;
}

/* when the 'x' button is clicked on the titlebar */
static gboolean cb_delete(GtkWidget *widget, GdkEvent *event, gpointer data)
{
     /* we return false, causing the "destroy" event to be released on "widget" */
     return FALSE;
}

static void cb_quit(GtkWidget *widget, thread_data_t *thread_data)
{
     debug("Issuing quit command\n");
     pthread_mutex_lock(thread_data->mutex);
     thread_data->quit = 1;
     pthread_mutex_unlock(thread_data->mutex);
     debug("Shutting down GUI\n");
     gtk_main_quit();
}

static void cb_master_vol_set(GtkWidget *widget, GtkSpinButton *spin)
{
     float volume;
     int val;

     volume = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(spin));
     val = mixer_set_volume(volume / 100.0);
     if (val < 0)
	  errmsg("Failed to set master volume to %.3f\n", volume);

     debug("Master volume set to %.3f\n", volume);
}

static void cb_chan_set(GtkWidget *widget, GtkSpinButton *spin)
{
     int channel;
     int val;
     int cp;

     if ((cp = get_current_patch()) < 0)
	  return;

     channel = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spin));
     if ((val = patch_channel_set(cp, channel-1)) < 0) {
	  errmsg("Failed to set patch %d to channel %d (%s)\n", cp, val,
		 patch_strerror(channel));
	  return;
     }
     debug("Patch %d channel set to %d\n", cp, channel-1);
}

static void cb_note_set(GtkWidget *widget, GtkSpinButton *spin)
{
     int note;
     int val;
     int cp;

     if ((cp = get_current_patch()) < 0)
	  return;

     note = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spin));
     if ((val = patch_note_set(cp, note)) < 0) {
	  errmsg("Failed to set patch %d to note %d (%s)\n", cp, note,
		 patch_strerror(val));
     }
     debug("Patch %d now listening on note %d\n", cp, note);

     /* make sure range selection widgets can't do stupid stuff */
     gtk_spin_button_set_range(GTK_SPIN_BUTTON(range_low_spin), 0, note);
     gtk_spin_button_set_range(GTK_SPIN_BUTTON(range_up_spin), note, 127);
}

static void cb_range_set(GtkWidget *button, gpointer data)
{
     int cp;

     if ((cp = get_current_patch()) < 0)
	  return;

     if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button))) {
	  gtk_widget_set_sensitive(range_low_spin, TRUE);
	  gtk_widget_set_sensitive(range_up_spin, TRUE);
	  patch_range_set(cp, 1);
	  debug("Patch %d now listening to a range of notes\n", cp); 
     } else {
	  gtk_widget_set_sensitive(range_low_spin, FALSE);
	  gtk_widget_set_sensitive(range_up_spin, FALSE);
	  patch_range_set(cp, 0);
	  debug("Patch %d note listening to an individual note\n", cp);
     }
}

static void cb_lower_note_set(GtkWidget *widget)
{
     int note, val;
     int cp;

     if ((cp = get_current_patch()) < 0)
	  return;

     note = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(widget));
     if ((val = patch_lower_note_set(cp, note)) < 0) {
	  debug("Unable to set patch %d lower note to %d (%s)\n", cp, note, patch_strerror(val));
	  return;
     }
     debug("Patch %d lower note set to %d\n", cp, note);
}

static void cb_upper_note_set(GtkWidget *widget)
{
     int note, val;
     int cp;

     if ((cp = get_current_patch()) < 0)
	  return;

     note = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(widget));
     if ((val = patch_upper_note_set(cp, note)) < 0) {
	  debug("Unable to set patch %d upper note to %d (%s)\n", cp, note, patch_strerror(val));
	  return;
     }
     debug("Patch %d upper note set to %d\n", cp, note);
}

static void cb_vol_set(GtkWidget *widget)
{
     float volume;
     int val;
     int cp;

     if ((cp = get_current_patch()) < 0)
	  return;

     volume = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(widget));
     volume /= 100;
     if ((val = patch_volume_set(cp, volume)) < 0) {
	  debug("Unable to set patch %d volume to %.3f (%s)\n", cp, volume, patch_strerror(val));
	  return;
     }
     debug("Patch %d volume set to %.3f\n", cp, volume);
}

static void cb_pan_set(GtkWidget *widget)
{
     float pan;
     int val;
     int cp;

     if ((cp = get_current_patch()) < 0)
	  return;

     pan = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(widget));
     pan /= 100;
     if ((val = patch_pan_set(cp, pan)) < 0) {
	  debug("Unable to set patch %d panning to %.3f (%s)\n", cp, pan, patch_strerror(val));
	  return;
     }
     debug("Patch %d pan set to %.3f\n", cp, pan);
}

static void cb_cut_set(GtkWidget *widget, gpointer arg)
{
     int val;
     int cut;
     int cp;

     if ((cp = get_current_patch()) < 0)
	  return;

     cut = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(cut_spin));
     if ((val = patch_cut_set(cp, cut)) < 0) {
	  debug("Unable to set patch %d cut signal to %d (%s)\n", cp, cut, patch_strerror(val));
	  return;
     }
     debug("Patch %d cut signal set to %d\n", cp, cut);
}

static void cb_cut_by_set(GtkWidget *widget, gpointer arg)
{
     int val;
     int cut_by;
     int cp;

     if ((cp = get_current_patch()) < 0)
	  return;

     cut_by = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(cut_by_spin));
     if ((val = patch_cut_by_set(cp, cut_by)) < 0) {
	  debug("Unable to set patch %d cut_by signal to %d (%s)\n", cp, cut_by, patch_strerror(val));
	  return;
     }
     debug("Patch %d cut_by signal set to %d\n", cp, cut_by);
}

/* enables ok button if there is input in entry, disables otherwise */
static gboolean cb_action_name_verify(GtkWidget *entry, GdkEventKey *event, GtkDialog *dialog)
{
     int val = strlen((char *)gtk_entry_get_text(GTK_ENTRY(entry)));
     
     if (val)
	  gtk_dialog_set_response_sensitive(dialog, GTK_RESPONSE_ACCEPT, TRUE);
     else
	  gtk_dialog_set_response_sensitive(dialog, GTK_RESPONSE_ACCEPT, FALSE);

     return FALSE; /* make sure normal event handlers grab signal */
}

static void cb_action_add(GtkWidget *menu_item, GtkWidget *main_window)
{
     GtkWidget *dialog;
     GtkWidget *entry;
     int val;
     int index;

     /* dialog box */
     dialog = gtk_dialog_new_with_buttons("Add Patch",
					  GTK_WINDOW(main_window),
					  GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
					  GTK_STOCK_OK,
					  GTK_RESPONSE_ACCEPT,
					  GTK_STOCK_CANCEL,
					  GTK_RESPONSE_REJECT,
					  NULL);
     gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_ACCEPT);

     /* create entry box */
     entry = gtk_entry_new_with_max_length(PATCH_MAX_NAME);
     gtk_entry_set_text(GTK_ENTRY(entry), "Patch Name");
     gtk_entry_set_activates_default(GTK_ENTRY(entry), TRUE);

     /* I wish I had a "changed" event... */
     g_signal_connect(G_OBJECT(entry), "key-press-event", G_CALLBACK(cb_action_name_verify), (gpointer)dialog);
     g_signal_connect(G_OBJECT(entry), "key-release-event", G_CALLBACK(cb_action_name_verify), (gpointer)dialog);
     gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), entry, FALSE, FALSE, 0);
     gtk_widget_show(entry);

     /* guaranteed to have a string in there if response is 'accept'
	due to sensitivity callback */
     if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_ACCEPT) {
	  val = patch_create((char *)gtk_entry_get_text(GTK_ENTRY(entry)));
	  if (val < 0) {
	       errmsg("Failed to create a new patch (%s).\n", patch_strerror(val));
	       return;
	  }
	  index = patch_display_index_get(val);

	  update_interface(index);
     }
	       
     gtk_widget_destroy(dialog);
}

static void cb_action_duplicate(GtkWidget *menu_item, GtkWidget *main_window)
{
     int val, index;
     int cp;
     
     if ((cp = get_current_patch()) < 0) {
	  debug("Duplicate what, jackass?\n");
	  return;
     }
     
     val = patch_duplicate(cp);
     if (val < 0) {
	  errmsg("Failed to create a new patch (%s).\n", patch_strerror(val));
	  return;
     }
     index = patch_display_index_get(val);

     update_interface(index);
}

static void cb_action_rename(GtkWidget *menu_item, GtkWidget *main_window)
{
     GtkWidget *dialog;
     GtkWidget *entry;
     int val;
     int index;
     int cp;

     if ((cp = get_current_patch()) < 0) {
	  debug("No patches to rename, infidel.\n");
	  return;
     }

     /* dialog box */
     dialog = gtk_dialog_new_with_buttons("Rename Patch",
					  GTK_WINDOW(main_window),
					  GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
					  GTK_STOCK_OK,
					  GTK_RESPONSE_ACCEPT,
					  GTK_STOCK_CANCEL,
					  GTK_RESPONSE_REJECT,
					  NULL);
     gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_ACCEPT);

     /* create entry box */
     entry = gtk_entry_new_with_max_length(PATCH_MAX_NAME);
     gtk_entry_set_text(GTK_ENTRY(entry), patch_name_get(cp));
     gtk_entry_set_activates_default(GTK_ENTRY(entry), TRUE);

     /* I wish I had a "changed" event... */
     g_signal_connect(G_OBJECT(entry), "key-press-event", G_CALLBACK(cb_action_name_verify), (gpointer)dialog);
     g_signal_connect(G_OBJECT(entry), "key-release-event", G_CALLBACK(cb_action_name_verify), (gpointer)dialog);
     gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), entry, FALSE, FALSE, 0);
     gtk_widget_show(entry);

     /* guaranteed to have a string in there if response is 'accept'
	due to sensitivity callback */
     if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_ACCEPT) {
	  val = patch_name_set(cp, (char *)gtk_entry_get_text(GTK_ENTRY(entry)));
	  if (val < 0) {
	       errmsg("Failed to rename patch (%s).\n", patch_strerror(val));
	       return;
	  }
	  
	  index = patch_display_index_get(cp);
	  update_interface(index);
     }
	       
     gtk_widget_destroy(dialog);
}

static void cb_action_remove(GtkWidget *menu_item, gpointer data)
{
     int val;
     int cp;
     int index;

     if ((cp = get_current_patch()) < 0) {
	  debug("No patches to remove, you dolt.\n");
	  return;
     }

     index = patch_display_index_get(cp);
     if ((val = patch_destroy(cp)) < 0) {
	  errmsg("Error removing patch %d (%s).\n", cp, patch_strerror(val));
	  return;
     }

     if (index == 0)
	  update_interface(index);
     else
	  update_interface(index-1);
}

/* ripped off from gtk-2.0 tutorial, makes menu popup when button is clicked */
static gboolean cb_action_popup(GtkWidget *menu, GdkEvent *event)
{
     GdkEventButton *bevent;

     if (event->type == GDK_BUTTON_PRESS) {
	  bevent = (GdkEventButton *) event; 
	  gtk_menu_popup (GTK_MENU (menu), NULL, NULL, NULL, NULL,
			  bevent->button, bevent->time);
	  /* Tell calling code that we have handled this event; the buck
	   * stops here. */
	  return TRUE;
     }

     /* Tell calling code that we have not handled this event; pass it on. */
     return FALSE;
}

static void cb_sample_load(GtkWidget *button, gpointer data)
{
     int cp;
     int index;

     if ((cp = get_current_patch()) < 0)
	  return;

     sample_selector_show();
     index = patch_display_index_get(cp);
     update_interface(index);
}

static void cb_menu_file_new_bank(GtkWidget *widget, gpointer data)
{
     if (bank_ops_new() == 0)
	  update_interface(0);
}

static void cb_menu_file_open_bank(GtkWidget *widget, gpointer data)
{
     if (bank_ops_open() == 0)
	  update_interface(0);
}

static void cb_menu_file_save_bank(GtkWidget *widget, gpointer data)
{
     bank_ops_save();
}

static void cb_menu_file_save_bank_as(GtkWidget *widget, gpointer data)
{
     bank_ops_save_as();
}

static void cb_menu_settings_audio(GtkWidget *widget, gpointer data)
{
     audio_settings_show(window);
}

/* this is my favorite function */
static gboolean cb_play(GtkWidget *widget, GdkEvent *event, gpointer data)
{
     int cp;

     if ((cp = get_current_patch()) < 0)
	  return FALSE;

     if (event->type == GDK_BUTTON_PRESS || event->type == GDK_KEY_PRESS)
	  patch_activate_by_id(cp);
     else if (event->type == GDK_BUTTON_RELEASE || event->type == GDK_KEY_RELEASE)
	  patch_deactivate_by_id(cp);

     return FALSE;
}


static gboolean cb_waveform_button_press(GtkWidget *wf, GdkEvent *event, gpointer data)
{
     int cp;

     if ((cp = get_current_patch()) < 0)
	  return FALSE;

     if (event->type == GDK_BUTTON_PRESS)
	  sample_editor_show();

     /* this doesn't work, I need to get the waveform to refresh
      * whenever it's parent toplevel is idle */
     gtk_widget_queue_draw(wf); 
     return FALSE;
}


static void cb_param_clicked(GtkWidget *widget, param_t param)
{
     int cp;

     if ((cp = get_current_patch()) < 0)
	  return;

     if (param == PARAM_FILTER)
	  filter_settings_show();
     else if (param == PARAM_MODE)
	  playmode_settings_show();
     else if (param == PARAM_ADSR)
	  adsr_settings_show();

     return;
}

static void cb_realize(GtkWidget *widget, gpointer data)
{
     /* probably gonna put bank file to open (passed from command
      * line) here */
     return;
}

int init_gui(thread_data_t *thread_data)
{
     /* frame stuff */
     GtkWidget *frame;
     GtkWidget *frame_vbox;
     GtkWidget *frame_hbox;

     /* misc */
     GtkWidget *button;
     GtkWidget *hbox;
     GtkWidget *label;
     GtkWidget *vsep;
     GtkWidget *image;
     GtkWidget *menubar;
     GtkWidget *menu;
     GtkWidget *item;

     debug("Initializing GUI\n");

     /****************************MAIN WINDOW****************************************/
     window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
     gtk_window_set_title(GTK_WINDOW(window), "Specimen");
     gtk_window_set_resizable(GTK_WINDOW(window), FALSE);
     g_signal_connect(G_OBJECT(window), "delete-event", G_CALLBACK(cb_delete), NULL);
     g_signal_connect(G_OBJECT(window), "destroy", G_CALLBACK(cb_quit), (gpointer)thread_data);
     g_signal_connect(G_OBJECT(window), "realize", G_CALLBACK(cb_realize), NULL);

     /* setup the window's main vbox */
     window_vbox = gtk_vbox_new(FALSE, 0);
     gtk_container_add(GTK_CONTAINER(window), window_vbox);

     /* the menubar */
     menubar = gtk_menu_bar_new();
     gtk_box_pack_start(GTK_BOX(window_vbox), menubar, FALSE, FALSE, 0);
     gtk_widget_show(menubar);

     /* file menu */
     menu_file = gtk_menu_new();
     menu_file_item = gtk_menu_item_new_with_label("File");
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(menu_file_item), menu_file);
     gtk_menu_bar_append(GTK_MENU_BAR(menubar), menu_file_item);
     gtk_widget_show(menu_file_item);

     menu_file_new_bank = gtk_menu_item_new_with_label("New Bank");
     gtk_menu_shell_append(GTK_MENU_SHELL(menu_file), menu_file_new_bank);
     g_signal_connect(G_OBJECT(menu_file_new_bank), "activate",
		      G_CALLBACK(cb_menu_file_new_bank), NULL);
     gtk_widget_show(menu_file_new_bank);

     menu_file_open_bank = gtk_menu_item_new_with_label("Open Bank...");
     gtk_menu_shell_append(GTK_MENU_SHELL(menu_file), menu_file_open_bank);
     g_signal_connect(G_OBJECT(menu_file_open_bank), "activate",
		      G_CALLBACK(cb_menu_file_open_bank), NULL);
     gtk_widget_show(menu_file_open_bank);

     menu_file_save_bank = gtk_menu_item_new_with_label("Save Bank");
     gtk_menu_shell_append(GTK_MENU_SHELL(menu_file), menu_file_save_bank);
     g_signal_connect(G_OBJECT(menu_file_save_bank), "activate",
		      G_CALLBACK(cb_menu_file_save_bank), NULL);
     gtk_widget_show(menu_file_save_bank);

     menu_file_save_bank_as = gtk_menu_item_new_with_label("Save Bank As...");
     gtk_menu_shell_append(GTK_MENU_SHELL(menu_file), menu_file_save_bank_as);
     g_signal_connect(G_OBJECT(menu_file_save_bank_as), "activate",
		      G_CALLBACK(cb_menu_file_save_bank_as), NULL);
     gtk_widget_show(menu_file_save_bank_as);

     menu_file_vsep = gtk_menu_item_new();
     gtk_menu_shell_append(GTK_MENU_SHELL(menu_file), menu_file_vsep);
     gtk_widget_show(menu_file_vsep);

     menu_file_quit = gtk_menu_item_new_with_label("Quit");
     gtk_menu_shell_append(GTK_MENU_SHELL(menu_file), menu_file_quit);
     g_signal_connect(G_OBJECT(menu_file_quit), "activate",
		      G_CALLBACK(cb_quit), (gpointer)thread_data);
     gtk_widget_show(menu_file_quit);

     /* settings menu */
     menu_settings = gtk_menu_new();
     menu_settings_item = gtk_menu_item_new_with_label("Settings");
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(menu_settings_item), menu_settings);
     gtk_menu_bar_append(GTK_MENU_BAR(menubar), menu_settings_item);
     gtk_widget_show(menu_settings_item);

     menu_settings_audio = gtk_menu_item_new_with_label("Audio");
     gtk_menu_shell_append(GTK_MENU_SHELL(menu_settings), menu_settings_audio);
     g_signal_connect(G_OBJECT(menu_settings_audio), "activate",
		      G_CALLBACK(cb_menu_settings_audio), NULL);
     gtk_widget_show(menu_settings_audio);

     /* setup the main window's master vbox */
     master_vbox = gtk_vbox_new(FALSE, SPACING);
     gtk_box_pack_start(GTK_BOX(window_vbox), master_vbox, FALSE, FALSE, 0);
     gtk_container_set_border_width(GTK_CONTAINER(master_vbox), SPACING);

     /****************************MASTER FRAME***************************************/
     frame = gtk_frame_new("Master");
     gtk_box_pack_start(GTK_BOX(master_vbox), frame, FALSE, FALSE, 0);

     /* master vbox */
     frame_vbox = gtk_vbox_new(FALSE, SPACING);
     gtk_container_set_border_width(GTK_CONTAINER(frame_vbox), SPACING);
     gtk_container_add(GTK_CONTAINER(frame), frame_vbox);

     /*---------------------------ROW ONE-------------------------------------------*/
     frame_hbox = gtk_hbox_new(FALSE, SPACING);
     gtk_box_pack_start(GTK_BOX(frame_vbox), frame_hbox, FALSE, FALSE, 0);

     /* patch option-menu label */
     label = gtk_label_new("Patch:");
     gtk_box_pack_start(GTK_BOX(frame_hbox), label, FALSE, FALSE, 0);
     gtk_widget_show(label);

     /* patch option-menu */
     patch_option_menu = gtk_option_menu_new();
     gtk_box_pack_start(GTK_BOX(frame_hbox), patch_option_menu, FALSE, FALSE, 0);
     gtk_widget_show(patch_option_menu);

     /* patch option-menu action button menu */
     menu = gtk_menu_new();

     /*-add-*/
     item = gtk_menu_item_new_with_label("Add");
     gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
     g_signal_connect(G_OBJECT(item), "activate", G_CALLBACK(cb_action_add),
		      (gpointer)window);
     gtk_widget_show(item);

     /*-duplicate-*/
     item = gtk_menu_item_new_with_label("Duplicate");
     gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
     g_signal_connect(G_OBJECT(item), "activate", G_CALLBACK(cb_action_duplicate),
		      (gpointer)window);
     gtk_widget_show(item);

     /*-rename-*/
     item = gtk_menu_item_new_with_label("Rename");
     gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
     g_signal_connect(G_OBJECT(item), "activate", G_CALLBACK(cb_action_rename),
		      (gpointer)window);
     gtk_widget_show(item);

     /*-remove-*/
     item = gtk_menu_item_new_with_label("Remove");
     gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
     g_signal_connect(G_OBJECT(item), "activate", G_CALLBACK(cb_action_remove),
		      (gpointer) NULL);
     gtk_widget_show(item);

     /* patch option-menu action button */
     button = gtk_button_new_with_label("Action");
     gtk_box_pack_end(GTK_BOX(frame_hbox), button, FALSE, FALSE, 0);
     g_signal_connect_swapped(G_OBJECT(button), "event", G_CALLBACK(cb_action_popup),
			      G_OBJECT(menu));
     gtk_widget_show(button);
     
     /*--------end row one--------*/
     gtk_widget_show(frame_hbox);

     /*---------------------------ROW TWO-------------------------------------------*/
     frame_hbox = gtk_hbox_new(FALSE, SPACING);
     gtk_box_pack_start(GTK_BOX(frame_vbox), frame_hbox, FALSE, FALSE, 0);

     /* volume label */
     label = gtk_label_new("Volume:");
     gtk_box_pack_start(GTK_BOX(frame_hbox), label, FALSE, FALSE, 0);
     gtk_widget_show(label);

     /* master volume spinbutton adjustment */
     master_vol_adj = (GtkAdjustment *) gtk_adjustment_new(50, 0, 100, 1, 0, 0);

     /* master volume spinbutton */
     master_vol_spin = gtk_spin_button_new(master_vol_adj, 0, 0);
     gtk_box_pack_start(GTK_BOX(frame_hbox), master_vol_spin, FALSE, FALSE, 0);
     gtk_widget_show(master_vol_spin);
     g_signal_connect(G_OBJECT(master_vol_adj), "value-changed",
		      G_CALLBACK(cb_master_vol_set), (gpointer)master_vol_spin);
     g_signal_emit_by_name(G_OBJECT(master_vol_adj), "value-changed");

     /*--------end row two--------*/
     gtk_widget_show(frame_hbox);

     /********end master frame*****/
     gtk_widget_show(frame_vbox);
     gtk_widget_show(frame);

     /***************************SAMPLE FRAME***********************************/
     frame = gtk_frame_new("Sample");
     gtk_box_pack_start(GTK_BOX(master_vbox), frame, FALSE, FALSE, 0);

     /* master vbox */
     frame_vbox = gtk_vbox_new(FALSE, SPACING);
     gtk_container_set_border_width(GTK_CONTAINER(frame_vbox), SPACING);
     gtk_container_add(GTK_CONTAINER(frame), frame_vbox);

     /*----------------------------ROW ONE----------------------------------------*/
     frame_hbox = gtk_hbox_new(FALSE, SPACING);
     gtk_box_pack_start(GTK_BOX(frame_vbox), frame_hbox, FALSE, FALSE, 0);

     /* file label */
     label = gtk_label_new("File:");
     gtk_box_pack_start(GTK_BOX(frame_hbox), label, FALSE, FALSE, 0);
     gtk_widget_show(label);
     
     /* file browsing button */
     file_button = gtk_button_new();
     hbox = gtk_hbox_new(FALSE, SPACING);
     image = gtk_image_new_from_file(PIXMAPSDIR"open.png");
     file_label = gtk_label_new("Load Sample");
     vsep = gtk_vseparator_new();

     gtk_box_pack_start(GTK_BOX(hbox), file_label, FALSE, FALSE, 0);
     gtk_box_pack_start(GTK_BOX(hbox), vsep, FALSE, FALSE, 0);
     gtk_box_pack_start(GTK_BOX(hbox), image, FALSE, FALSE, 0);
     gtk_container_add(GTK_CONTAINER(file_button), hbox);
     gtk_box_pack_start(GTK_BOX(frame_hbox), file_button, FALSE, FALSE, 0);

     g_signal_connect(G_OBJECT(file_button), "clicked", G_CALLBACK(cb_sample_load), NULL);
		      
     gtk_widget_show(file_label);
     gtk_widget_show(vsep);
     gtk_widget_show(image);
     gtk_widget_show(hbox);
     gtk_widget_show(file_button);

     /* play button */
     play_button = gtk_button_new_with_label("Play");
     gtk_box_pack_start(GTK_BOX(frame_hbox), play_button, FALSE, FALSE, 0);
     g_signal_connect(G_OBJECT(play_button), "event",
		      G_CALLBACK(cb_play), NULL);
     gtk_widget_show(play_button);

     /*---------------end row one--------------*/
     gtk_widget_show(frame_hbox);

     /*----------------------------ROW TWO----------------------------------------*/
     frame_hbox = gtk_hbox_new(FALSE, SPACING);
     gtk_box_pack_start(GTK_BOX(frame_vbox), frame_hbox, FALSE, FALSE, 0);

     /* volume label */
     label = gtk_label_new("Volume:");
     gtk_box_pack_start(GTK_BOX(frame_hbox), label, FALSE, FALSE, 0);
     gtk_widget_show(label);

     /* volume spin adjustment */
     vol_adj = (GtkAdjustment *)gtk_adjustment_new(100, 0, 100, 1, 0, 0);

     /* volume spin */
     vol_spin = gtk_spin_button_new(vol_adj, 0, 0);
     gtk_box_pack_start(GTK_BOX(frame_hbox), vol_spin, FALSE, FALSE, 0);
     g_signal_connect_swapped(G_OBJECT(vol_adj), "value-changed",
			      G_CALLBACK(cb_vol_set), (gpointer)vol_spin);
     gtk_widget_show(vol_spin);

     /* pan label */
     label = gtk_label_new("Pan:");
     gtk_box_pack_start(GTK_BOX(frame_hbox), label, FALSE, FALSE, 0);
     gtk_widget_show(label);

     /* pan spin adjustment */
     pan_adj = (GtkAdjustment *)gtk_adjustment_new(0, -100, 100, 1, 0, 0);

     /* pan spin button */
     pan_spin = gtk_spin_button_new(pan_adj, 0, 0);
     gtk_box_pack_start(GTK_BOX(frame_hbox), pan_spin, FALSE, FALSE, 0);
     g_signal_connect_swapped(G_OBJECT(pan_adj), "value-changed",
			      G_CALLBACK(cb_pan_set), (gpointer)pan_spin);
     gtk_widget_show(pan_spin);
			      
     /*---------------end row two--------------*/
     gtk_widget_show(frame_hbox);

     /*----------------------------ROW THREE----------------------------------------*/
     frame_hbox = gtk_hbox_new(FALSE, SPACING);
     gtk_box_pack_start(GTK_BOX(frame_vbox), frame_hbox, FALSE, FALSE, 0);

     /* cut label */
     label = gtk_label_new("Cut:");
     gtk_box_pack_start(GTK_BOX(frame_hbox), label, FALSE, FALSE, 0);
     gtk_widget_show(label);

     /* cut adjustment */
     cut_adj = (GtkAdjustment *)gtk_adjustment_new(0, 0, 99, 1, 0, 0);

     /* cut spin button */
     cut_spin = gtk_spin_button_new(cut_adj, 0, 0);
     gtk_box_pack_start(GTK_BOX(frame_hbox), cut_spin, FALSE, FALSE, 0);
     g_signal_connect(G_OBJECT(cut_adj), "value-changed", G_CALLBACK(cb_cut_set), (gpointer)NULL);
     gtk_widget_show(cut_spin);


     /* cut by label */
     label = gtk_label_new("Cut by:");
     gtk_box_pack_start(GTK_BOX(frame_hbox), label, FALSE, FALSE, 0);
     gtk_widget_show(label);

     /* cut by adjustment */
     cut_by_adj = (GtkAdjustment *)gtk_adjustment_new(0, 0, 99, 1, 0, 0);
     
     /* cut by spin button */
     cut_by_spin = gtk_spin_button_new(cut_by_adj, 0, 0);
     gtk_box_pack_start(GTK_BOX(frame_hbox), cut_by_spin, FALSE, FALSE, 0);
     g_signal_connect(G_OBJECT(cut_by_adj), "value-changed", G_CALLBACK(cb_cut_by_set), (gpointer)NULL);
     gtk_widget_show(cut_by_spin);

     /*---------------end row three--------------*/
     gtk_widget_show(frame_hbox);

     /*----------------------------ROW FOUR ----------------------------------------*/
     frame_hbox = gtk_hbox_new(FALSE, SPACING);
     gtk_box_pack_start(GTK_BOX(frame_vbox), frame_hbox, FALSE, FALSE, 0);

     /* waveform preview */
     waveform = waveform_new(get_current_patch(), 256, 64, FALSE);
     gtk_box_pack_start(GTK_BOX(frame_hbox), waveform, FALSE, FALSE, 0);
     g_signal_connect(G_OBJECT(waveform), "button-press-event", G_CALLBACK(cb_waveform_button_press), NULL);
     gtk_widget_show(waveform);

     /*---------------end row four--------------*/
     gtk_widget_show(frame_hbox);

     /**************end sample frame************/
     gtk_widget_show(frame_vbox);
     gtk_widget_show(frame);

     /***************************MIDI FRAME***********************************/
     frame = gtk_frame_new("Midi");
     gtk_box_pack_start(GTK_BOX(master_vbox), frame, FALSE, FALSE, 0);

     /* master vbox */
     frame_vbox = gtk_vbox_new(FALSE, SPACING);
     gtk_container_set_border_width(GTK_CONTAINER(frame_vbox), SPACING);
     gtk_container_add(GTK_CONTAINER(frame), frame_vbox);

     /*----------------------------ROW ONE----------------------------------------*/
     frame_hbox = gtk_hbox_new(FALSE, SPACING);
     gtk_box_pack_start(GTK_BOX(frame_vbox), frame_hbox, FALSE, FALSE, 0);

     /* channel label */
     label = gtk_label_new("Channel:");
     gtk_box_pack_start(GTK_BOX(frame_hbox), label, FALSE, FALSE, 0);
     gtk_widget_show(label);

     /* channel spinbutton adjustment */
     chan_adj = (GtkAdjustment *) gtk_adjustment_new(1, 1, 16, 1, 0, 0);
     
     /* channel spinbutton */
     chan_spin = gtk_spin_button_new(chan_adj, 0, 0);
     gtk_box_pack_start(GTK_BOX(frame_hbox), chan_spin, FALSE, FALSE, 0);
     gtk_widget_show(chan_spin);

     /* bind to proper callback */
     g_signal_connect(G_OBJECT(chan_adj), "value_changed",
		      G_CALLBACK(cb_chan_set), (gpointer)chan_spin);

     /* note label */
     label = gtk_label_new("Note:");
     gtk_box_pack_start(GTK_BOX(frame_hbox), label, FALSE, FALSE, 0);
     gtk_widget_show(label);

     /* note spinbutton adjustment */
     note_adj = (GtkAdjustment *)gtk_adjustment_new(60, 0, 127, 1, 0, 0);

     /* note spinbutton */
     note_spin = gtk_spin_button_new(note_adj, 0, 0);
     gtk_box_pack_start(GTK_BOX(frame_hbox), note_spin, FALSE, FALSE, 0);
     gtk_widget_show(note_spin);

     /* bind to proper callback */
     g_signal_connect(G_OBJECT(note_adj), "value_changed",
		      G_CALLBACK(cb_note_set), (gpointer)note_spin);
     
     /*---------------end row one--------------*/
     gtk_widget_show(frame_hbox);

     /*----------------------------ROW TWO----------------------------------------*/
     frame_hbox = gtk_hbox_new(FALSE, SPACING);
     gtk_box_pack_start(GTK_BOX(frame_vbox), frame_hbox, FALSE, FALSE, 0);

     /* range check box */
     range_check = gtk_check_button_new_with_label("Range:");
     gtk_box_pack_start(GTK_BOX(frame_hbox), range_check, FALSE, FALSE, 0);
     g_signal_connect(G_OBJECT(range_check), "toggled", G_CALLBACK(cb_range_set), NULL);
     gtk_widget_show(range_check);

     /* range lower spinbutton */
     range_low_adj = (GtkAdjustment *)gtk_adjustment_new(60, 0, 60, 1, 0, 0);
     range_low_spin = gtk_spin_button_new(range_low_adj, 0, 0);
     gtk_box_pack_start(GTK_BOX(frame_hbox), range_low_spin, FALSE, FALSE, 0);
     g_signal_connect_swapped(G_OBJECT(range_low_adj), "value-changed",
			      G_CALLBACK(cb_lower_note_set), (gpointer)range_low_spin);
     gtk_widget_show(range_low_spin);

     /* "to" label */
     label = gtk_label_new("to");
     gtk_box_pack_start(GTK_BOX(frame_hbox), label, FALSE, FALSE, 0);
     gtk_widget_show(label);

     /* range upper spinbutton */
     range_up_adj = (GtkAdjustment *)gtk_adjustment_new(60, 60, 127, 1, 0, 0);
     range_up_spin = gtk_spin_button_new(range_up_adj, 0, 0);
     gtk_box_pack_start(GTK_BOX(frame_hbox), range_up_spin, FALSE, FALSE, 0);
     g_signal_connect_swapped(G_OBJECT(range_up_adj), "value-changed",
			      G_CALLBACK(cb_upper_note_set), (gpointer)range_up_spin);
     gtk_widget_show(range_up_spin);

     /*---------------end row two--------------*/
     gtk_widget_show(frame_hbox);

     /**************end midi frame************/
     gtk_widget_show(frame_vbox);
     gtk_widget_show(frame);


     /***************PARAMETERS FRAME***************/
     frame = gtk_frame_new("Parameters");
     gtk_box_pack_start(GTK_BOX(master_vbox), frame, FALSE, FALSE, 0);

     /* master vbox */
     frame_vbox = gtk_vbox_new(FALSE, SPACING);
     gtk_container_set_border_width(GTK_CONTAINER(frame_vbox), SPACING);
     gtk_container_add(GTK_CONTAINER(frame), frame_vbox);

     /*------------ROW ONE-------------------*/
     frame_hbox = gtk_hbox_new(FALSE, SPACING);
     gtk_box_pack_start(GTK_BOX(frame_vbox), frame_hbox, FALSE, FALSE, 0);

     /* mode button */
     button = gtk_button_new_with_label("Mode");
     g_signal_connect(G_OBJECT(button), "clicked", G_CALLBACK(cb_param_clicked), (gpointer)PARAM_MODE);
     gtk_box_pack_start(GTK_BOX(frame_hbox), button, FALSE, FALSE, 0);
     gtk_widget_show(button);

     /* filter button */
     button = gtk_button_new_with_label("Filter");
     g_signal_connect(G_OBJECT(button), "clicked", G_CALLBACK(cb_param_clicked), (gpointer)PARAM_FILTER);
     gtk_box_pack_start(GTK_BOX(frame_hbox), button, FALSE, FALSE, 0);
     gtk_widget_show(button);

     /* filter button */
     button = gtk_button_new_with_label("ADSR");
     g_signal_connect(G_OBJECT(button), "clicked", G_CALLBACK(cb_param_clicked), (gpointer)PARAM_ADSR);
     gtk_box_pack_start(GTK_BOX(frame_hbox), button, FALSE, FALSE, 0);
     gtk_widget_show(button);

     /*-------------end row one---------------*/
     gtk_widget_show(frame_hbox);

     /************end parameters frame************/
     gtk_widget_show(frame_vbox);
     gtk_widget_show(frame);
     
     /**************end window******************/
     gtk_widget_show(master_vbox);
     gtk_widget_show(window_vbox);
     gtk_widget_show(window);

     /* initialize children */
     sample_editor_init(window);
     audio_settings_init(window);
     filter_settings_init(window);
     playmode_settings_init(window);
     adsr_settings_init(window);

     /* initialize */
     update_interface(0);

     /* success */
     return 0;
}

static int current_patch = -1;

/* make all patch related controls sensitive or insensitive */
static void set_sensitive(int val)
{
     gtk_widget_set_sensitive(chan_spin, val);
     gtk_widget_set_sensitive(file_button, val);
     gtk_widget_set_sensitive(note_spin, val);
     gtk_widget_set_sensitive(menu_file_save_bank, val);
     gtk_widget_set_sensitive(menu_file_save_bank_as, val);
     gtk_widget_set_sensitive(play_button, val);
     gtk_widget_set_sensitive(vol_spin, val);
     gtk_widget_set_sensitive(pan_spin, val);
     gtk_widget_set_sensitive(cut_spin, val);
     gtk_widget_set_sensitive(cut_by_spin, val);
     gtk_widget_set_sensitive(range_check, val);
     gtk_widget_set_sensitive(range_low_spin, val);
     gtk_widget_set_sensitive(range_up_spin, val);
}

/* sets current patch and updates widgets accordingly */
static void set_current_patch(int id)
{
     gdouble chan, vol, note, range, lower_note, \
	  upper_note, pan, cut, cut_by, frames;
     char *sample_name;

     if (id < 0) {
	  debug("No active patches\n");
	  set_sensitive(FALSE);
	  current_patch = -1;
	  waveform_set_patch(WAVEFORM(waveform), -1);
	  return;
     }

     if (!patch_verify(id)) {
	  errmsg("Patch %d is invalid or misconfigured, not switching to it\n", id);
	  return;
     }

     chan = (gdouble)patch_channel_get(id);
     note = (gdouble)patch_note_get(id);
     range = (gdouble)patch_range_get(id);
     lower_note = (gdouble)patch_lower_note_get(id);
     upper_note = (gdouble)patch_upper_note_get(id);
     vol = (gdouble)patch_volume_get(id);
     pan = (gdouble)patch_pan_get(id);
     cut = (gdouble)patch_cut_get(id);
     cut_by = (gdouble)patch_cut_by_get(id);
     frames = (gdouble)patch_frames_get(id);
     sample_name = patch_sample_name_get(id);

     debug("Changing current patch to: %d, index: %d\n", id, patch_display_index_get(id));
     if (current_patch < 0)
	  set_sensitive(TRUE);
     current_patch = id;
 
     gtk_spin_button_set_value(GTK_SPIN_BUTTON(chan_spin), chan+1);
     gtk_spin_button_set_value(GTK_SPIN_BUTTON(note_spin), note);
     gtk_spin_button_set_value(GTK_SPIN_BUTTON(range_low_spin), lower_note);
     gtk_spin_button_set_value(GTK_SPIN_BUTTON(range_up_spin), upper_note);
     gtk_spin_button_set_value(GTK_SPIN_BUTTON(vol_spin), vol*100);
     gtk_spin_button_set_value(GTK_SPIN_BUTTON(pan_spin), pan*100);
     gtk_spin_button_set_value(GTK_SPIN_BUTTON(cut_spin), cut);
     gtk_spin_button_set_value(GTK_SPIN_BUTTON(cut_by_spin), cut_by);

     if (*sample_name == '\0')
	  gtk_label_set_text(GTK_LABEL(file_label), "Load Sample");
     else
	  gtk_label_set_text(GTK_LABEL(file_label), basename(sample_name));
     if (sample_name != NULL)
	  free(sample_name);

     if (range > 0) {
	  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(range_check), TRUE);
	  gtk_widget_set_sensitive(range_low_spin, TRUE);
	  gtk_widget_set_sensitive(range_up_spin, TRUE);
     } else {
	  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(range_check), FALSE);
	  gtk_widget_set_sensitive(range_low_spin, FALSE);
	  gtk_widget_set_sensitive(range_up_spin, FALSE);
     }

     /* make pretty pictures */
     waveform_set_patch(WAVEFORM(waveform), id);

     return;
}

int get_current_patch()
{
     return current_patch;
}
