/*
 *  SPL - The SPL Programming Language
 *  Copyright (C) 2006 Clifford Wolf <clifford@clifford.at>
 *  Copyright (C) 2007 Raphael Langerhorst <raphael@raphael.g-system.at>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  mod_epfb.c: Hardware Accelerated Raster Graphics Engine on EP9315.
 */

/**
 * SPL EPFB Module
 *
 * This Module provides hardware accelerated graphics on EP9315 processors.
 * It is tailored for the NetBSD/evbarm architecture.
 * If compiling fails, make sure that the file epio.h is in include path.
 */

#include <stdlib.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdio.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <epio.h>

#include <spl.h>
#include <compat.h>
#include <stdio.h>

extern void SPL_ABI(spl_mod_epfb_init)(struct spl_vm *vm, struct spl_module *mod, int restore);
extern void SPL_ABI(spl_mod_epfb_done)(struct spl_vm *vm, struct spl_module *mod);

int epfb_fd;

// FONT BITMAPS TAKEN FROM L3D (raster/ras3_sw.cc)

// L3D realtime 3D library, explained in book "Linux 3D Graphics Programming"
// Copyright (C) 2000  Norman Lin
// Contact: nlin@linux3dgraphicsprogramming.org (alt. nlin@geocities.com)
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License along
// with this program; if not, write to the Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.

///////////////////////////////////////////////////////////////////////////


static const unsigned char period[] =
  {0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

static const unsigned char space[] =
  {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

static const unsigned char digits[][13] = {
  {0x0,0x0, 0x18,0x3C,0x66,0xC3,0xC3,0xC3,0xC3,0xC3,0x66,0x3C,0x18},
  {0x0,0x0, 0x7F,0x7F,0xC,0xC,0xC,0xC,0xC,0xC,0x6C,0x3C,0xC},
  {0x0,0x0, 0xFF,0xFF,0xC0,0x60,0x30,0x18,0xC,0x6,0xC6,0x66,0x3C},
  {0x0,0x0, 0x7E,0xC3,0x3,0x6,0xC,0x38,0xC,0x6,0x3,0xC3,0x7E},
  {0x0,0x0, 0x6,0x6,0x6,0x6,0xFF,0xFF,0xC6,0xC6,0x66,0x36,0x1E},
  {0x0,0x0, 0x7E,0xFF,0xC3,0x3,0x3,0xFF,0xFE,0xC0,0xC0,0xC0,0xFE},
  {0x0,0x0, 0x7E,0xFF,0xC3,0xC3,0xC3,0xFF,0xFE,0xC0,0xC0,0xC0,0x7E},
  {0x0,0x0, 0x60,0x60,0x60,0x60,0x30,0x18,0xC,0x6,0x3,0x3,0xFF},
  {0x0,0x0, 0x7E,0xFF,0xC3,0xC3,0xC3,0x7E,0xC3,0xC3,0xC3,0xC3,0x7E},
  {0x0,0x0, 0x7E,0xFF,0xC3,0x3,0x3,0x7F,0xC3,0xC3,0xC3,0xC3,0x7E}
};


static const unsigned char upper_case_letters[][13] = {
  {0x00, 0x00, 0xc3, 0xc3, 0xc3, 0xc3, 0xff, 0xc3, 0xc3, 0xc3, 0x66, 0x3c, 0x18},
  {0x00, 0x00, 0xfe, 0xc7, 0xc3, 0xc3, 0xc7, 0xfe, 0xc7, 0xc3, 0xc3, 0xc7, 0xfe},
  {0x00, 0x00, 0x7e, 0xe7, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xe7, 0x7e},
  {0x00, 0x00, 0xfc, 0xce, 0xc7, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3, 0xc7, 0xce, 0xfc},
  {0x00, 0x00, 0xff, 0xc0, 0xc0, 0xc0, 0xc0, 0xfc, 0xc0, 0xc0, 0xc0, 0xc0, 0xff},
  {0x00, 0x00, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xfc, 0xc0, 0xc0, 0xc0, 0xff},
  {0x00, 0x00, 0x7e, 0xe7, 0xc3, 0xc3, 0xcf, 0xc0, 0xc0, 0xc0, 0xc0, 0xe7, 0x7e},
  {0x00, 0x00, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3, 0xff, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3},
  {0x00, 0x00, 0x7e, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x7e},
  {0x00, 0x00, 0x7c, 0xee, 0xc6, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06},
  {0x00, 0x00, 0xc3, 0xc6, 0xcc, 0xd8, 0xf0, 0xe0, 0xf0, 0xd8, 0xcc, 0xc6, 0xc3},
  {0x00, 0x00, 0xff, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0},
  {0x00, 0x00, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3, 0xdb, 0xff, 0xff, 0xe7, 0xc3},
  {0x00, 0x00, 0xc7, 0xc7, 0xcf, 0xcf, 0xdf, 0xdb, 0xfb, 0xf3, 0xf3, 0xe3, 0xe3},
  {0x00, 0x00, 0x7e, 0xe7, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3, 0xe7, 0x7e},
  {0x00, 0x00, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xfe, 0xc7, 0xc3, 0xc3, 0xc7, 0xfe},
  {0x00, 0x00, 0x3f, 0x6e, 0xdf, 0xdb, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3, 0x66, 0x3c},
  {0x00, 0x00, 0xc3, 0xc6, 0xcc, 0xd8, 0xf0, 0xfe, 0xc7, 0xc3, 0xc3, 0xc7, 0xfe},
  {0x00, 0x00, 0x7e, 0xe7, 0x03, 0x03, 0x07, 0x7e, 0xe0, 0xc0, 0xc0, 0xe7, 0x7e},
  {0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xff},
  {0x00, 0x00, 0x7e, 0xe7, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3},
  {0x00, 0x00, 0x18, 0x3c, 0x3c, 0x66, 0x66, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3},
  {0x00, 0x00, 0xc3, 0xe7, 0xff, 0xff, 0xdb, 0xdb, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3},
  {0x00, 0x00, 0xc3, 0x66, 0x66, 0x3c, 0x3c, 0x18, 0x3c, 0x3c, 0x66, 0x66, 0xc3},
  {0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3c, 0x3c, 0x66, 0x66, 0xc3},
  {0x00, 0x00, 0xff, 0xc0, 0xc0, 0x60, 0x30, 0x7e, 0x0c, 0x06, 0x03, 0x03, 0xff}
};

/* lower case letters added by raphael */
static const unsigned char lower_case_letters[][13] = {
  {0x00, 0x00, 0x7f, 0xc3, 0xc3, 0xc3, 0xc7, 0x7b, 0x00, 0x00, 0x00, 0x00, 0x00}, // a
  {0x00, 0x00, 0xff, 0xc3, 0xc3, 0xc3, 0xc3, 0xfe, 0xc0, 0xc0, 0xc0, 0xc0, 0x00}, // b
  {0x00, 0x00, 0x7e, 0xc3, 0xc0, 0xc0, 0xc3, 0x7e, 0x00, 0x00, 0x00, 0x00, 0x00}, // c
  {0x00, 0x00, 0x7f, 0xc3, 0xc3, 0xc3, 0xc3, 0x7f, 0x03, 0x03, 0x03, 0x03, 0x00}, // d
  {0x00, 0x00, 0x7e, 0xc0, 0xc0, 0xff, 0xc2, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00}, // e
  {0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3e, 0x18, 0x1a, 0x1e, 0x00}, // f
  {0xfe, 0x03, 0x7f, 0xc3, 0xc3, 0xc3, 0xc3, 0x7f, 0x00, 0x00, 0x00, 0x00, 0x00}, // g
  {0x00, 0x00, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3, 0xfe, 0xc0, 0xc0, 0xc0, 0xc0, 0x00}, // h
  {0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x18, 0x00, 0x00}, // i
  {0x30, 0x58, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x18, 0x00, 0x00}, // j
  {0x00, 0x00, 0xc3, 0xc6, 0xcc, 0xd8, 0xf0, 0xd8, 0xcc, 0xc6, 0xc0, 0xc0, 0x00}, // k
  {0x00, 0x00, 0x1c, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x00}, // l
  {0x00, 0x00, 0xdb, 0xdb, 0xdb, 0xdb, 0xdb, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00}, // m
  {0x00, 0x00, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00}, // n
  {0x00, 0x00, 0x7e, 0xc3, 0xc3, 0xc3, 0xc3, 0x7e, 0x00, 0x00, 0x00, 0x00, 0x00}, // o
  {0xc0, 0xc0, 0xff, 0xc3, 0xc3, 0xc3, 0xc3, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00}, // p
  {0x03, 0x03, 0x7f, 0xc3, 0xc3, 0xc3, 0xc3, 0x7f, 0x00, 0x00, 0x00, 0x00, 0x00}, // q
  {0x00, 0x00, 0x30, 0x30, 0x30, 0x30, 0x38, 0x2e, 0x00, 0x00, 0x00, 0x00, 0x00}, // r
  {0x00, 0x00, 0x1e, 0x23, 0x06, 0x08, 0x31, 0x1e, 0x00, 0x00, 0x00, 0x00, 0x00}, // s
  {0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3e, 0x18, 0x18, 0x18, 0x00}, // t
  {0x00, 0x00, 0x7f, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3, 0x00, 0x00, 0x00, 0x00, 0x00}, // u
  {0x00, 0x00, 0x18, 0x3c, 0x24, 0x66, 0x42, 0xc3, 0x00, 0x00, 0x00, 0x00, 0x00}, // v
  {0x00, 0x00, 0x6d, 0xdb, 0xdb, 0xdb, 0xc3, 0xc3, 0x00, 0x00, 0x00, 0x00, 0x00}, // w
  {0x00, 0x00, 0xc3, 0x66, 0x3c, 0x18, 0x3c, 0x66, 0xc3, 0x00, 0x00, 0x00, 0x00}, // x
  {0x60, 0x30, 0x18, 0x3c, 0x24, 0x66, 0x42, 0xc3, 0x00, 0x00, 0x00, 0x00, 0x00}, // y
  {0x00, 0x00, 0xff, 0xc0, 0x60, 0x18, 0x06, 0x03, 0xff, 0x00, 0x00, 0x00, 0x00}  // z
};

unsigned char font_bitmaps[256][13];

void epfb_init_fontbitmaps()
{
  //- create font bitmaps
  int i,j;

  for(i=0; i<26; i++) {
    for(j=0; j<13; j++) {
      font_bitmaps['A' + i][j] = upper_case_letters[i][j];
      font_bitmaps['a' + i][j] = lower_case_letters[i][j];
    }
  }

  for(i=0; i<10; i++) {
    for(j=0; j<13; j++) {
      font_bitmaps['0' + i][j] = digits[i][j];
    }
  }

  for(j=0; j<13; j++) {
    font_bitmaps[' '][j] = space[j];
  }

  for(j=0; j<13; j++) {
    font_bitmaps['.'][j] = period[j];
  }
}


void epfb_draw_text(int x, int y, const char *text, int col) {

#define FONT_Y_SIZE 13
#define FONT_X_SIZE 8
#define FONT_SPACING 2

  int font_y, font_x;

  struct epfb_pixelset pixel;
  pixel.col = col;

  // raphael 2007-06-28: we need to start at Y_SIZE - 1 (font_y=1) and go to 0 (font_y == Y_SIZE)
  for(font_y=1; font_y <= FONT_Y_SIZE; font_y++) {

    pixel.y = y + font_y;
    pixel.x = x;
    unsigned int c;
    for(c=0; c<strlen(text); c++) {
      int current_bit;
      for(font_x=0, current_bit=0x80;
          font_x < FONT_X_SIZE;
          font_x++, current_bit >>= 1)
      {
        if(font_bitmaps[text[c]][FONT_Y_SIZE-font_y] & current_bit)
	  ioctl(epfb_fd,EPFB_PIXELSET,&pixel);
	pixel.x++;
      }
      pixel.x += FONT_SPACING;
    }
  }
}


// END OF FONT BITMAPS BY NORMAN LIN

static struct spl_node *handler_epfb_clear(struct spl_task *task, void *data)
{
  if (ioctl(epfb_fd,EPFB_CLEAR)==-1)
    spl_report(SPL_REPORT_RUNTIME|SPL_REPORT_WARNING, task, "clear ioctl failed\n");
  return 0;
}

/**
 * Set the colour of a single pixel on screen.
 */
// builtin epfb_pixelset(x,y,col)

static struct spl_node *handler_epfb_pixelset(struct spl_task *task, void *data)
{
  struct epfb_pixelset pixel;
  pixel.x = spl_clib_get_int(task);
  pixel.y = spl_clib_get_int(task);
  pixel.col = spl_clib_get_int(task);

  if (ioctl(epfb_fd,EPFB_PIXELSET,&pixel) == -1)
    spl_report(SPL_REPORT_RUNTIME|SPL_REPORT_WARNING, task, "pixelset ioctl failed\n");
  return 0;
}

/**
 * Draw a line from (x1/y1) to (x2/y2) on screen.
 */
// builtin epfb_linedraw(startx,starty,stopx,stopy,col)

static struct spl_node *handler_epfb_linedraw(struct spl_task *task, void *data)
{
  struct epfb_linedraw line;
  line.startx = spl_clib_get_int(task);
  line.starty = spl_clib_get_int(task);
  line.stopx = spl_clib_get_int(task);
  line.stopy = spl_clib_get_int(task);
  line.col = spl_clib_get_int(task);

  if (ioctl(epfb_fd,EPFB_LINEDRAW,&line)==-1)
    spl_report(SPL_REPORT_RUNTIME|SPL_REPORT_WARNING, task, "line draw ioctl failed\n");
  return 0;
}

/**
 * Rectangular blockfill from (x/y) with given width and height.
 */
// builtin epfb_blockfill(x,y,width,height,col)

static struct spl_node *handler_epfb_blockfill(struct spl_task *task, void *data)
{
  struct epfb_blockfill block;
  block.startx = spl_clib_get_int(task);
  block.starty = spl_clib_get_int(task);
  block.width = spl_clib_get_int(task);
  block.height = spl_clib_get_int(task);

  /*
  if (block.height <= 0)
	return 0;
  block.height--;
  */

  block.col = spl_clib_get_int(task);

  if (ioctl(epfb_fd,EPFB_BLOCKFILL,&block)==-1)
    spl_report(SPL_REPORT_RUNTIME|SPL_REPORT_WARNING, task, "block fill ioctl failed\n");
  return 0;
}

/**
 * Renders text at (x,y) with given colour.
 */
// builtin epfb_drawtext(x,y,text,col)

static struct spl_node *handler_epfb_drawtext(struct spl_task *task, void *data)
{
  int x = spl_clib_get_int(task);
  int y = spl_clib_get_int(task);
  char* text = spl_clib_get_string(task);
  int col = spl_clib_get_int(task);
  epfb_draw_text(x,y,text,col);
  return 0;
}

/**
 * Setup Hardware Cursor.
 * The cursor must be enabled with [epfb_cursor_enable] as well.
 * The cursor position can be specified with [epfb_cursor_moveto].
 */
// builtin epfb_cursor_setup(width,height,col1,col2,blink1,blink2,blinkrate)

static struct spl_node *handler_epfb_cursor_setup(struct spl_task *task, void *data)
{
  struct epfb_cursor_setup cursor;
  cursor.width = spl_clib_get_int(task);
  cursor.height = spl_clib_get_int(task);
  cursor.col1 = spl_clib_get_int(task);
  cursor.col2 = spl_clib_get_int(task);
  cursor.blink1 = spl_clib_get_int(task);
  cursor.blink2 = spl_clib_get_int(task);
  cursor.blinkrate = spl_clib_get_int(task);

  if (ioctl(epfb_fd,EPFB_CURSOR_SETUP,&cursor)==-1)
    spl_report(SPL_REPORT_RUNTIME|SPL_REPORT_WARNING, task, "setting up cursor failed\n");
  return 0;
}

/**
 * Set pixel value for hardware cursor.
 * The value must be in the range of 0 to 3 (00 to 11).
 * 0 means transparent, 1 means invert of video stream,
 * 2 means (blink) colour 1, 3 means (blink) colour 2.
 */
// builtin epfb_cursor_pixel(x,y,value)

static struct spl_node *handler_epfb_cursor_pixel(struct spl_task *task, void *data)
{
  struct epfb_cursor_pixelset pixel;
  pixel.x = spl_clib_get_int(task);
  pixel.y = spl_clib_get_int(task);
  pixel.value = spl_clib_get_int(task);
  if (ioctl(epfb_fd,EPFB_CURSOR_PIXELSET,&pixel)==-1)
    spl_report(SPL_REPORT_RUNTIME|SPL_REPORT_WARNING, task, "failed setting cursor enable state\n");
  return 0;
}

/**
 * Enable or disable hardware cursor.
 * 1 is enable, 0 is disable.
 */
// builtin epfb_cursor_enable(enable)

static struct spl_node *handler_epfb_cursor_enable(struct spl_task *task, void *data)
{
  int enable = spl_clib_get_int(task);
  if (ioctl(epfb_fd,EPFB_CURSOR_ENABLE,&enable)==-1)
    spl_report(SPL_REPORT_RUNTIME|SPL_REPORT_WARNING, task, "failed setting cursor enable state\n");
  return 0;
}


/**
 * Set Hardware Cursor Position on screen.
 */
// builtin epfb_cursor_moveto(x,y)

static struct spl_node *handler_epfb_cursor_moveto(struct spl_task *task, void *data)
{
  struct epfb_cursor_position pos;
  pos.x = spl_clib_get_int(task);
  pos.y = spl_clib_get_int(task);
  if (ioctl(epfb_fd,EPFB_CURSOR_MOVETO,&pos)==-1)
    spl_report(SPL_REPORT_RUNTIME|SPL_REPORT_WARNING, task, "setting cursor position failed\n");
  return 0;
}

/**
 * Get Hardware Cursor Position on screen.
 */
// builtin epfb_cursor_position(x,y)

static struct spl_node *handler_epfb_cursor_position(struct spl_task *task, void *data)
{
  struct epfb_cursor_position pos;
  if (ioctl(epfb_fd,EPFB_CURSOR_POSITION,&pos)==-1)
  {
    spl_report(SPL_REPORT_RUNTIME|SPL_REPORT_WARNING, task, "getting cursor position failed\n");
    return 0;
  }

  struct spl_node* position = spl_get(0);
  spl_create(task, position, "x",  SPL_NEW_INT(pos.x),  SPL_CREATE_LOCAL);
  spl_create(task, position, "y",  SPL_NEW_INT(pos.y),  SPL_CREATE_LOCAL);

  return position;
}

void SPL_ABI(spl_mod_epfb_init)(struct spl_vm *vm, struct spl_module *mod UNUSED, int restore UNUSED)
{
  epfb_fd = open("/dev/epfb0",0,O_RDWR);
  if (epfb_fd == -1)
    epfb_fd = open("/dev/epfb",0,O_RDWR);
  if (epfb_fd == -1)
  {
    // TODO: can we do SPL like error reporting here?
    printf("Failed to open the EPFB device file at /dev/epfb and /dev/epfb0, cannot load epfb module.\n");
    return;
  }
  epfb_init_fontbitmaps();

//  spl_clib_reg(vm, "epfb_output_status", handler_epfb_output_status, 0);
//  spl_clib_reg(vm, "epfb_output_enable", handler_epfb_output_enable, 0);
//  spl_clib_reg(vm, "epfb_info", handler_epfb_devinfo, 0);
  spl_clib_reg(vm, "epfb_clear", handler_epfb_clear, 0);
  spl_clib_reg(vm, "epfb_pixel", handler_epfb_pixelset, 0);
  spl_clib_reg(vm, "epfb_line", handler_epfb_linedraw, 0);
  spl_clib_reg(vm, "epfb_blockfill", handler_epfb_blockfill, 0);
  spl_clib_reg(vm, "epfb_text", handler_epfb_drawtext, 0);

  spl_clib_reg(vm, "epfb_cursor_setup", handler_epfb_cursor_setup, 0);
  spl_clib_reg(vm, "epfb_cursor_pixel", handler_epfb_cursor_pixel, 0);
  spl_clib_reg(vm, "epfb_cursor_enable", handler_epfb_cursor_enable, 0);
  spl_clib_reg(vm, "epfb_cursor_moveto", handler_epfb_cursor_moveto, 0);
  spl_clib_reg(vm, "epfb_cursor_position", handler_epfb_cursor_position, 0);
}

void SPL_ABI(spl_mod_epfb_done)(struct spl_vm *vm UNUSED, struct spl_module *mod UNUSED)
{
  return;
}

