/*
 *  SPL - The SPL Programming Language
 *  Copyright (C) 2004, 2005  Clifford Wolf <clifford@clifford.at>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  compat.h: Some defines and inlines for non-gnu systems
 *
 */

#ifndef SPL_COMPAT_H
#define SPL_COMPAT_H

#include <stdlib.h>

#define UNUSED __attribute__((unused))

#define my_strndupa(s, n)						\
	({								\
		char *__new = (char *) __builtin_alloca ((n) + 1);	\
		strncpy(__new, (s), (n));				\
		__new[(n)] = '\0'; __new;				\
	})

#define my_strndup(s, n)						\
	({								\
		char *__new = (char *) malloc ((n) + 1);		\
		strncpy(__new, (s), (n));				\
		__new[(n)] = '\0'; __new;				\
	})

#define my_alloca(_s) __builtin_alloca(_s)

static inline void *my_memrchr(const void *s, int c, size_t n)
{
	const char *sx = s;
	for (int counter = n; counter >= 0; counter--)
		if (sx[counter] == c) return (void*)(sx+counter);
	return 0;
}

static inline void *my_memmem(const void *haystack, size_t haystacklen,
		const void *needle, size_t needlelen)
{
	if (needlelen <= haystacklen) {
		for (unsigned int i=0; i <= haystacklen-needlelen; i++)
			if (!memcmp(haystack+i, needle, needlelen))
				return (void*)(haystack+i);
	}
	return 0;
}

static inline char *my_strsep(char **s, const char *ct)
{
	char *begin = *s, *end = *s;

	if (!begin)
		return 0;

	while (*end && !strchr(ct, *end))
		end++;

	if (*end) {
		*end = 0;
		*s = end+1;
	} else
		*s = 0;

	return begin;
}

#if defined USEWIN32API

#ifndef COMPAT_H_NO_WIN_INCL
#    include <stdarg.h>
#    include <windef.h>
#    include <winbase.h>
#endif

#  define MY_O_BINARY O_BINARY
#  define my_sleep(sec) Sleep((sec)*1000)

#else

#  define MY_O_BINARY 0
#  define my_sleep sleep

#endif

#if defined USEWIN32API || defined USEIRIXAPI

/* The Win32 and IRIX snprintf functions returns -1 if the target string is to
 * small. So we need this crappy hack to allocate a buffer of the right size..
 */

#  define my_asprintf(__result, ...)					\
	({								\
		int __size = 64, __rc;					\
		*(__result) = malloc(__size + 1);			\
		while (1) {						\
			__rc = snprintf(*(__result), __size + 1,	\
					## __VA_ARGS__);		\
			if (__rc >= 0 && __rc < __size) break;		\
			free(*(__result));				\
			__size *= 2;					\
			*(__result) = malloc(__size + 1);		\
		} 							\
		__rc;							\
	})

#  define my_vasprintf(__result, ...)					\
	({								\
		int __size = 64, __rc;					\
		*(__result) = malloc(__size + 1);			\
		while (1) {						\
			__rc = vsnprintf(*(__result), __size + 1,	\
					## __VA_ARGS__);		\
			if (__rc >= 0 && __rc < __size) break;		\
			free(*(__result));				\
			__size *= 2;					\
			*(__result) = malloc(__size + 1);		\
		} 							\
		__rc;							\
	})

#else

#  define my_asprintf asprintf
#  define my_vasprintf vasprintf

#endif

#if defined USECYGWINAPI || defined USEMACOSXAPI || defined USEWIN32API || defined USEIRIXAPI || defined USEBSDAPI

#  define my_dprintf(__fd, ...)						\
	({								\
		char *__data;						\
		int __rc = my_asprintf(&__data, ## __VA_ARGS__);	\
		for (int __written = 0, __rc2 = 0; __written < __rc;	\
				__written += __rc2) {			\
			__rc2 = write((__fd), __data+__written,		\
					__rc-__written);		\
			if (__rc2 <= 0) break;				\
		}							\
		free(__data); __rc;					\
	})

#include <unistd.h>

static inline ssize_t my_sendfile(int out_fd, int in_fd, off_t *offset, size_t count)
{
	char buffer[count];
	int read_rc = read(in_fd, buffer, count);

	if (read_rc <= 0)
		return read_rc;

	int write_rc = 0;

	while (write_rc < read_rc) {
		int rc = write(out_fd, buffer+write_rc, read_rc-write_rc);
		if (rc <= 0) return rc ? rc : write_rc;
		write_rc += rc;
	}

	return write_rc;
}

#else

#  define my_dprintf dprintf
#  define my_sendfile sendfile

#endif

#endif

