#!/usr/bin/perl
#
# This object represents an event that has happened that a record should be
# kept for.  The current case is when a service changes state (color change).
# An record representing that change is written to the database so that you
# can see a record of these events over time.
#
#   host          - string FQDN of the machine its associated with
#   type          - type of history event (status change, etc...)
#   time          - time (time() format) that the event occurred
#   service       - service name this history is associated with
#   color         - color of the service at time history event was created
#   other         - summary information about the history event
#   service       - service object that cause status change
#
# + new()         - constructor (sets instance vars to arguments passed in)
# + gets/sets()   - magical set/get functions (autoloaded based on func name)
# + display()     - output format and view
#
# History:
# (1) Cleaned up (Ed July 31, 1997);

# $Id: History.pm,v 1.5 2000/12/15 21:01:10 sljohnson Exp $

package Spong::History;

use Spong::HistoryService;

# Constructor.  A simple constructor that just sets all of the instance vars
# according to what was passed in.  All vars should be set on creation.

sub new {
   my( $proto, $host, $type, $time, $service, $color, $other ) = @_;
   my( $class ) = ref($proto) || $proto;
   my $self = {};

   $self->{'host'}    = $host;
   $self->{'type'}    = $type;
   $self->{'time'}    = $time;
   $self->{'service'} = $service;
   $self->{'color'}   = $color;
   $self->{'other'}   = $other;
   $self->{'status'} = new Spong::HistoryService($host, $service, $time );

   bless ($self,$class);
   return $self;
}

# Get/Set methods, nothing fancy here...

sub host { my $var = 'host';
   if( defined $_[1] ) { $_[0]->{$var} = $_[1]; } return $_[0]->{$var}; }
sub type { my $var = 'type';
   if( defined $_[1] ) { $_[0]->{$var} = $_[1]; } return $_[0]->{$var}; }
sub time { my $var = 'time';
   if( defined $_[1] ) { $_[0]->{$var} = $_[1]; } return $_[0]->{$var}; }
sub service { my $var = 'service';
   if( defined $_[1] ) { $_[0]->{$var} = $_[1]; } return $_[0]->{$var}; }
sub color { my $var = 'color';
   if( defined $_[1] ) { $_[0]->{$var} = $_[1]; } return $_[0]->{$var}; }
sub other { my $var = 'other';
   if( defined $_[1] ) { $_[0]->{$var} = $_[1]; } return $_[0]->{$var}; }

# Simple Set method for the status object
sub status {
   return $self->{'status'}
}

# Display summary.  Does both text and html, doesn't make any calls to 
# sub-objects or other helper objects, just spits out the data that it has.
# This is a little different in that there are not different views of the
# data.  The views come into play with the HistoryList in that more days worth
# of history is shown with the more detailed views.

sub display {
   my( $self, $type, $view ) = @_;

   if( $type eq "text" ) { return $self->display_text( $view ); }
   if( $type eq "html" ) { return $self->display_html( $view ); }
}

sub display_text {
   my( $self, $view ) = @_;
   my( $min, $hour ) = (localtime($self->time()))[1,2];
   my $shost = $main::HOSTS{$self->host()}->{'display_name'};

   $shost = $self->host() if $main::HISTORY_FQDN;
   $shost = (split( /\./, $self->host() ))[0] unless $shost;

   print $self->color(), " "x(7-length($self->color()));
   my($type) = substr($self->type,0,1); $type =~ tr/a-z/A-Z/;
   print " $type ";
   print sprintf( "%2.2d:%2.2d  ", $hour, $min );
   print substr( $shost, 0, 12 ), " "x(14-length(substr($shost, 0, 12)));
   print substr( $self->service(), 0, 5 );
   print " "x(7-length(substr( $self->service(), 0, 5 )));
   print $self->other(), "\n";
}

sub display_html {
   my( $self, $view ) = @_;
   my( $min, $hour ) = (localtime($self->time()))[1,2];
   my $shost = $main::HOSTS{$self->host()}->{'display_name'};
   my $color = $self->color();
   my $service = $self->service();
   my $other = $self->other();
   my $time = $self->time();

   $shost = $self->host() if $main::HISTORY_FQDN;
   $shost = (split( /\./, $self->host() ))[0] unless $shost;

   print "<tr><td width=5% align=left valign=top>";

   if( $main::WWW_USE_IMAGES == 1 ) {
      print "<a href=\"" . $main::WWWSPONG . "/histservice/".$self->host()
         . "/$service/$time\">" if ($main::STATUS_HISTORY);
      print "<img src=\"!!WWWGIFS!!/$color.gif\" alt=$color border=0>";
      print "</a>" if ($main::STATUS_HISTORY);
   } else {
      print "<table width=20 border=0 cellspacing=0 cellpadding=0><tr>";
      print "<td width=20 bgcolor=\"" . $main::WWW_COLOR{$color} . "\">"; 
      print "<a href=\"" . $main::WWWSPONG . "/histservice/".$self->host()
          . "/$service/$time\"><font color=\"" . $main::WWW_COLOR{$color}
          . "\">" if ($main::STATUS_HISTORY);
      print "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;";
      print "</a>" if $main::STATUS_HISTORY;
      print "</td></tr></table>";
   }

   print "</td>\n";
   print "<td width=5% align=center>";
   my($type) = substr($self->type,0,1); $type =~ tr/a-z/A-Z/;
   print "$type</td>\n";
   print "<td width=10% align=center valign=top>";
   printf ( "%2.2d:%2.2d</td>", $hour, $min );
   print "<td width=15% align=left valign=top>$shost </td>\n";
   print "<td width=15% align=left valign=top>$service</td>\n";
   print "<td width=51% align=left valign=top>$other</td></tr>\n";
}

1;
