// $Id: ssm_malign.h,v 1.5 2006/06/14 15:13:06 keb Exp $
// =================================================================
//
//    10.04.13   <--  Date of Last Modification.
//                   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//  ----------------------------------------------------------------
//
//  **** Module  :  SSM_MAlign <interface>
//       ~~~~~~~~~
//  **** Project :  SSM
//       ~~~~~~~~~
//  **** Classes :  CSSMMultAlign
//       ~~~~~~~~~
//
//  E. Krissinel, 2003-2013
//
// =================================================================
//

#ifndef  __SSM_MAlign__
#define  __SSM_MAlign__

#include "mmdb/mmdb_manager.h"

#include "ssm_csia.h"
#include "ssm_superpose.h"


namespace ssm  {

  // --------------------------- SMAStruct ---------------------------

  DefineStructure(MAStruct);

  struct MAStruct  {

    SuperposeData  SD;  //!< data for pairwise superposition
    PGraph          G;  //!< reduceable copy of SSE graph
    rvector         P;  //!< SSE matching probabilities
    rvector         Q;  //!< SSE matching scores
    ivector         F;  //!< original vertex numbering
    int           sNo;  //!< serial number of the structure
    int            nV;  //!< number of vertices in G
    int             n;  //!< number of non-zero SSE matchings
    int       nSAlloc;  //!< memory allocation size
    mat44          RT;  //!< rotation-translation matrix
    mat44         RT0;  //!< best rotation-translation matrix
    realtype     Qsum;  //!< sum of Q-scores with other structures
    rvector        x0;  //!< original C-alpha X-coordinates
    rvector        y0;  //!< original C-alpha Y-coordinates
    rvector        z0;  //!< original C-alpha Z-coordinates
    realtype xm,ym,zm;  //!< center of mass
    realtype cx,cy,cz;  //!< consensus center of mass

    int        nalign;  //!< number of aligned residues
    realtype    Rmsd0;  //!< parameter of Q-score

    void    Init ( PCMMDBManager MMDB, PGraph graph,
                   int serNo, int nStruct );
    void    Set  ( PCMMDBManager MMDB, PGraph graph,
                   int serNo, int nStruct );
    void    PrepareSSEMatching();
    Boolean Refine            ( int maxdel, realtype P0,
                                ivector v1, ivector v2 );
    void    Transform         ();
    void    SaveCoordinates   ();
    void    RestoreCoordinates();
    void    CalcCorrelationMatrix ( rmatrix & A, rvector xc,
                                    rvector  yc, rvector zc );
    void    CalcTranslation   ();
    void    GetDirection      ( int atompos, vect3 & v );
    Boolean isMC              ( int pos1, PMAStruct S, int pos2 );
    void    Dispose           ();

  };


  // ----------------------------  MSSEOutput -------------------------

  DefineStructure(MSSEOutput);

  struct MSSEOutput {
    ResName  name1,name2;
    ChainID  chID;
    int      seqNum1,seqNum2;
    int      sseType,loopNo;
    InsCode  insCode1,insCode2;
    Boolean  aligned;
    void  Init        ();
    void  SetSSERange ( PVertex     V );
    void  Copy        ( RMSSEOutput M );
    void  write       ( RCFile f );
    void  read        ( RCFile f );
  };


  // ----------------------------  SMAOutput -------------------------

  DefineStructure(MAOutput);

  struct MAOutput {
    ResName  name;
    ChainID  chID;
    int      seqNum;
    int      sseType;
    InsCode  insCode;
    realtype rmsd;      // not used
    Boolean  aligned;
    void  Init  ();
    void  Fill  ( PCAtom A, PGraph G, Boolean align );
    void  Copy  ( RMAOutput M );
    void  write ( RCFile f );
    void  read  ( RCFile f );
  };

  extern void FreeMSOutput ( PPMAOutput   & MAOut,   int & nrows );
  extern void FreeMSOutput ( PPMSSEOutput & MSSEOut, int & nrows );

  // ---------------------------  CPAMatch ---------------------------

  DefineClass(PAMatch);

  class PAMatch  {

    public :
      ivector  F1,F2;
      rvector  Q;
      realtype Qscore;
      int      mlen;

      PAMatch();
      ~PAMatch();

      void   FreeMemory();
      void    Set      ( ivector v1, ivector v2, int matchlen,
                         realtype matchQ, rvector SSEQ );
      Boolean GetMatch ( ivector v1, ivector v2, int matchlen,
                         realtype & matchQ, rvector SSEQ );
  };


  // --------------------------  PAMatches --------------------------

  DefineClass(PAMatches);

  class PAMatches  {

    public :
      PPPAMatch PA;
      int       nMatches;
      int       nBest;

      PAMatches ();
      ~PAMatches();

      int  AddMatch ( ivector v1, ivector v2, int matchlen,
                      realtype matchQ, rvector SSEQ );
      int  GetMatch ( ivector v1, ivector v2, int matchlen,
                      realtype & matchQ, rvector SSEQ );
      void SetBestMatch ( int mNo );
      realtype GetBestQscore();
      void GetBestMatch ( ivector & v1, ivector & v2, int & matchlen );

    private :
      int nAlloc;

  };


  // ----------------------------  MAMap ----------------------------

 DefineStructure(MAMap);

  struct MAMap  {
    realtype rmsd;
    ivector  map;   // 0:i is mapped onto j:SMAMap[i].map[j]
    void Init   ( int nStruct );
    void Dispose();
  };


  // ---------------------------  MultAlign -------------------------

  typedef void MAProgressFunc ( void * UserData, int stagekey,
                                int progress );
  typedef MAProgressFunc * PMAProgressFunc;

  DefineClass(MultAlign);
  DefineStreamFunctions(MultAlign)

  class MultAlign : public CStream  {

    public :

      MultAlign ();
      MultAlign ( RPCStream Object );
      ~MultAlign();

      void  setProgressFunction  ( void * UserData,
                                   PMAProgressFunc Fnc );

      int   align ( PPCMMDBManager MMDB, psvector selstring,
                    PPGraph Graph, int  nStructures );

      inline int getMaxNofIter() { return maxIter; }

      void  getAlignScores     ( int & n_align, int & n_SSEs,
                                 realtype & rmsd, realtype & Qscore );
      void  getConsensusScores ( rvector & cons_x,
                                 rvector & cons_y,
                                 rvector & cons_z,
                                 int     & cons_len,
                                 rmatrix & m_rmsd,
                                 rmatrix & m_Qscore,
                                 rmatrix & m_seqId );

      int     getNres      ( int structNo );
      Boolean getAlignment ( int structNo, ivector & Ca, int & nres );
      Boolean getTMatrix   ( mat44 & TMatrix, int structNo );

      void   GetMAOutput   ( PPMAOutput & MAOut,
                             int & nrows, int & ncols );
      void   GetMSSEOutput ( PPMSSEOutput & MSSEOut,
                             int & nrows, int & ncols );

      void  WriteMultAlign   ( RCFile f );
      void  WriteSuperposed  ( cpstr fileName );

      void  WriteMatchedSSEs ( RCFile f, Boolean shortTable=False );

      void  read  ( RCFile f );
      void  write ( RCFile f );

    protected :
      PPMAStruct      S; //!< molecular structure data
      PPPAMatches  * PM; //!< pairwise matches database
      int       nStruct; //!< number of structures

      PMAProgressFunc ProgressFunc;
      void *          ProgFuncData;

      rvector        vq; //!< working array  [1..maxNV]
      ivector     v1,v2; //!< working arrays [1..maxNV]
      int         maxNV; //!< maximal number of vertices

      PRECISION    precision;    //!< SSE matching specificity
      CONNECTIVITY connectivity; //!< SSE matching connectivity flag

      realtype  minCont; //!< minimal contact distance
      realtype  maxCont; //!< maximal contact distance
      realtype    Rmsd0; //!< parameter of Q-score
      int       minIter; //!< minimal number of iterations
      int       maxIter; //!< minimal number of iterations
      int   maxHollowIt; //!< maximal allowed number of consequtive
                         /// iterations without quality improvement

      rmatrix     A,Z,V; //!< corr-n matrix, left and right SVD vectors
      rvector         W; //!< singular values

      PMAMap        Map; //!< Map maps 0:i<->j:Map[i].map[j]

      int             Nalign; //!< number of multuply-aligned rows
      int    minNres,maxNres;
      int          nSSEalign; //!< number of multiply-aligned SSEs
      realtype rmsd_achieved; //!< achieved RMSD
      realtype    Q_achieved; //!< achieved Q

      rvector        xc;  //!< consensus X-coordinates
      rvector        yc;  //!< consensus Y-coordinates
      rvector        zc;  //!< consensus Z-coordinates
      rmatrix   mx_rmsd;  //!< matrix of inter-structure rmsds
      rmatrix mx_Qscore;  //!< matrix of inter-structure Q-scores
      rmatrix  mx_seqId;  //!< matrix of inter-structure seq identities

      GraphMatch U;
      Superpose  superpose;

      void     InitMultAlign      ();
      void     FreeMemory         ();
      void     DeleteStructures   ();
      void     DeselectCalphas    ();
      void     SelectCalphas      ();
      void     DeletePAMatches    ();
      void     DeleteMap          ();
      void     AllocateMap        ();
      void     printStats         ();
      void     AlignSSEs          ();
      void     GetSSEMatchingStats();
      void     GetBestMatch       ( PMAStruct S1, PMAStruct S2 );
      Boolean  RefineGraphs       ();
      int      MakeFirstGuess     ();
      void     CalcRMSD           ( int mappos   );
      Boolean  EvaluateMapping    ( PCMContact C );
      void     CorrespondContacts ( realtype contDist );
      void     CalcConsensus      ();
      realtype MatchQuality2      ( int Nalgn, realtype dist2 );
      realtype MatchQuality       ( int Nalgn, int N1, int N2,
                                               realtype dist2 );
      int      OptimizeAlignments ();
      void     SortStructures     ();
      void     CalcConsensusScores();
      int      AlignCalphas       ();
      int      CalcRotation       ( mat44 & R );

    protected :
      int Map_nrows;     // number of rows in Map;
      int nStructAlloc;  // allocated number of structures

  };

}

#endif
