/*
 * Copyright (C) 2005  Stig Venaas <venaas@uninett.no>
 * $Id: ssmpingd.c,v 1.16 2005/11/29 16:27:26 sv Exp $
 *
 * Contributions:
 * Solaris support by Alexander Gall <gall@switch.ch>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 */

#include "ssmping.h"

void zerosrcinterface(struct msghdr *msgh) {
    struct cmsghdr *cmsgh;
    
    switch (((struct sockaddr *)(msgh->msg_name))->sa_family) {
    case AF_INET:
	{
#ifdef IP_PKTINFO
	    struct in_pktinfo *pktinfo;
	    
	    for (cmsgh = CMSG_FIRSTHDR(msgh); cmsgh; cmsgh = CMSG_NXTHDR(msgh, cmsgh))
		if ((cmsgh->cmsg_level == IPPROTO_IP) &&
		    (cmsgh->cmsg_type == IP_PKTINFO)) {
		    pktinfo = (struct in_pktinfo *)CMSG_DATA(cmsgh);
		    pktinfo->ipi_ifindex = 0;
		    break;
		}
#endif
	    /* if not defined we only got source address (I think) */
	    return;
	}
    case AF_INET6:
	{
	    struct in6_pktinfo *pktinfo;

	    for (cmsgh = CMSG_FIRSTHDR(msgh); cmsgh; cmsgh = CMSG_NXTHDR(msgh, cmsgh))
		if ((cmsgh->cmsg_level == IPPROTO_IPV6) &&
		    (cmsgh->cmsg_type == IPV6_PKTINFO)) {
		    pktinfo = (struct in6_pktinfo *)CMSG_DATA(cmsgh);
		    pktinfo->ipi6_ifindex = 0;
		    break;
		}
	    return;
	}
    }
}

int findgroup(char *buf, int len, struct sockaddr_storage *addr) {
    uint16_t t, l, tmp;
    char *v, *p = buf;
    
    while (p - buf + 4 <= len) {
	memcpy(&tmp, p, 2);
        t = ntohs(tmp);
        p += 2;
	memcpy(&tmp, p, 2);
        l = ntohs(tmp);
        p += 2;
        if (l) {
            if (p - buf + l > len)
                return -1;
            v = p;
            p += l;
        }
	if (t == SSMPING_GROUP) {
	    if (*v == 1) { /* IPv4 */
		memcpy(&((struct sockaddr_in *)addr)->sin_addr, v + 1, 4);
		return 0;
	    }
	    if (*v == 2) { /* IPv6 */
		memcpy(&((struct sockaddr_in6 *)addr)->sin6_addr, v + 1, 16);
		return 0;
	    }
	    return -1;
	}
    }
    return -1;
}

int addrok(struct sockaddr *addr) {
    switch(addr->sa_family) {
    case AF_INET:
	{
	    struct sockaddr_in *a = (struct sockaddr_in *)addr;
	    return *(((uint8_t *)&a->sin_addr) + 3) == 234;
	}
    case AF_INET6:
	{
	    struct sockaddr_in6 *a = (struct sockaddr_in6 *)addr, b;
	    inet_pton(AF_INET6, "ff3e::4321:1234", &b.sin6_addr);
	    return !memcmp(((char *)&a->sin6_addr) + 12, ((char *)&b.sin6_addr) + 12, 4);
	}
    }
    return 0;
}

void respond(int s) {
    static char buf[65535];
    struct sockaddr_storage addr;
    int cnt;
    struct msghdr msgh;
    struct iovec iovec;
    char control[10240];

    memset(&msgh, 0, sizeof(struct msghdr));
    msgh.msg_iov = &iovec;
    msgh.msg_iovlen = 1;
    msgh.msg_control = control;
    msgh.msg_controllen = sizeof(control);
    msgh.msg_name = &addr;
    msgh.msg_namelen = sizeof(addr);

    memset(&iovec, 0, sizeof(struct iovec));
    iovec.iov_base = (caddr_t)buf;
    iovec.iov_len = sizeof(buf);

    cnt = recvmsg(s, &msgh, 0);
    if (cnt <= 0)
	return;
    if (*buf != SSMPING_REQUEST) /* not a request, possibly our own reply */
	return;
    
    *buf = SSMPING_REPLY;

    printf("received request from %s\n", addr2string((struct sockaddr *) &addr, msgh.msg_namelen));

    iovec.iov_len = cnt;

    /* received ancillary data with destination address and interface */
    /* we use that now to specify source address */
    /* setting interface to 0, don't want to specify that */
    zerosrcinterface(&msgh);

    /* send unicast */
    if (sendmsg(s, &msgh, 0) < 0)
	err("sendto");
    
    if (findgroup(buf + 1, cnt - 1, &addr))
	setaddr(&addr, NULL, "ff3e::4321:1234", "232.43.211.234");
    else if (!addrok((struct sockaddr *)&addr)) {
	printf("received request with invalid group address %s\n", addr2string((struct sockaddr *) &addr, msgh.msg_namelen));
	return;
    }
    
    /* send multicast */
    if (sendmsg(s, &msgh, 0) < 0)
	err("sendto");
}

void sethops(int s6, int s4, int hops) {
#ifdef __sun
    uint8_t mcttl = hops;
#else
    int mcttl = hops;
#endif    
    
    if (s6 >= 0) {
	if (setsockopt(s6, IPPROTO_IPV6, IPV6_UNICAST_HOPS, &hops, sizeof(hops)) == -1)
	    err("setsockopt IPV6_UNICAST_HOPS");
	if (setsockopt(s6, IPPROTO_IPV6, IPV6_MULTICAST_HOPS, &hops, sizeof(hops)) == -1)
	    err("setsockopt IPV6_UNICAST_HOPS");
    }
    if (s4 < 0)
	return;
    
    if (setsockopt(s4, IPPROTO_IP, IP_TTL, &hops, sizeof(hops)) == -1)
	err("setsockopt IP_TTL");
    if (setsockopt(s4, IPPROTO_IP, IP_MULTICAST_TTL, &mcttl, sizeof(mcttl)) == -1)
	err("setsockopt IP_MULTICAST_TTL");
}

int setpktinfo(int s, int family) {
    int on = 1;

    switch (family) {
    case AF_INET:
	return setsockopt(s, IPPROTO_IP,
#ifdef IP_PKTINFO
			  IP_PKTINFO,
#else
			  IP_RECVDSTADDR,
#endif
			  &on, sizeof(on));
    case AF_INET6:
	return  setsockopt(s, IPPROTO_IPV6,
#ifdef IPV6_RECVPKTINFO
                   IPV6_RECVPKTINFO,
#else              
                   IPV6_PKTINFO,
#endif             
			   &on, sizeof(on));
    }
    return -1;
}

int main(int argc, char **argv) {
    struct addrinfo hints, *res, *res0;
    int e, s, s4 = -1, s6 = -1, on = 1, max, ndesc;
    fd_set readfds;

    setvbuf(stdout, NULL, _IONBF, 0);
    
    memset(&hints, 0, sizeof(hints));
    hints.ai_socktype = SOCK_DGRAM;
    hints.ai_family = AF_UNSPEC;
    hints.ai_flags = AI_PASSIVE;

    if ((e = getaddrinfo(NULL, "4321", &hints, &res0)))
	errx("getaddrinfo failed: %s\n", gai_strerror(e));
    for (res = res0; res; res = res->ai_next) {
	s = socket(res->ai_family, res->ai_socktype, res->ai_protocol);
	if (s < 0) {
	    err("socket");
	    continue;
	}
	if (setpktinfo(s, res->ai_family) == -1) {
	    err("setpktinfo");
	    continue;
	}

#ifdef IPV6_V6ONLY
	if ((res->ai_family == AF_INET6) &&
	    (setsockopt(s, IPPROTO_IPV6, IPV6_V6ONLY, &on, sizeof(on)) == -1))
	    err("setsockopt IPV6_V6ONLY");
#endif

	if (bind(s, res->ai_addr, res->ai_addrlen) < 0) {
	    err("bind");
	    continue;
	}
	
	switch (res->ai_family) {
	case AF_INET6:
	    s6 = s;
	    break;
	case AF_INET:
	    s4 = s;
	    break;
	default:
	    printf("Only supporting IPv4 and IPv6 families\n");
	}
    }  
    freeaddrinfo(res0);
    
    max = s4 > s6 ? s4 : s6;
    if (max == -1)
	errx("both v4 and v6 binds failed");

    sethops(s6, s4, 64);
    for(;;) {    
	FD_ZERO(&readfds);
	if (s4 >= 0)
	    FD_SET(s4, &readfds);
	if (s6 >= 0)
	    FD_SET(s6, &readfds);

	ndesc = select(max + 1, &readfds, (fd_set *)0, (fd_set *)0, NULL);
	if (ndesc < 1)
	    errx("select returned < 1");

	if (s4 >= 0 && FD_ISSET(s4, &readfds))
	    respond(s4);

	if (s6 >= 0 && FD_ISSET(s6, &readfds))
	    respond(s6);
    }
}
