// StarPlot - A program for interactively viewing 3D maps of stellar positions.
// Copyright (C) 2000  Kevin B. McCarty
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


//
// star.cc - The Star class.
//

#include "star.h"
#include "greek.h"

#define NEED_FULL_NAMES
#include "constellations.h"

using std::string;

// For the Star class ------------------------------------------

// Private function

// Function to set a bunch of star data to save typing in the copy constructor
//  and operator= functions.
void Star::Obtain(StringList Names, StringList Membership, StringList Comments,
		  SolidAngle GridPosn, double Distance, double Diameter,
		  double PrimaryDistance, double Magnitude, SpecClass Spectrum,
		  size_t Place, unsigned int XPixel, unsigned int YPixel,
		  unsigned int RPixel, bool SLabelDraw)
{
  sNames = Names, sMembership = Membership, sComments = Comments;
  sGridPosn = GridPosn, sDistance = Distance, sDiameter = Diameter;
  sPrimaryDistance = PrimaryDistance, sMagnitude = Magnitude;
  sSpectrum = Spectrum, sPlace = Place, xPixel = XPixel, yPixel = YPixel;
  rPixel = RPixel; sLabelDraw = SLabelDraw;
  return;
}


// Public functions

// Default constructor
Star::Star()
{
  Obtain(StringList(), StringList(), StringList(), SolidAngle(0,0),
	 0.0, 0.0, 0.0, 0.0, SpecClass(), 0, 0, 0, 0, true);
}

// Copy constructor
Star::Star(const Star &s)
{
  Obtain(s.sNames, s.sMembership, s.sComments, s.sGridPosn, s.sDistance,
	 s.sDiameter, s.sPrimaryDistance, s.sMagnitude, s.sSpectrum,
	 s.sPlace, s.xPixel, s.yPixel, s.rPixel, s.sLabelDraw);
}

// Assignment operator
Star & Star::operator = (const Star &s)
{
  if (! (this == &s)) { // no point in copying if objects already the same
    Obtain(s.sNames, s.sMembership, s.sComments, s.sGridPosn, s.sDistance,
	   s.sDiameter, s.sPrimaryDistance, s.sMagnitude, s.sSpectrum,
	   s.sPlace, s.xPixel, s.yPixel, s.rPixel, s.sLabelDraw); 
  }
  return *this;
}


// Constructor to create a Star from a formatted text record.  Text record
//  is assumed to be in the following format.  Line breaks are permitted,
//  but only after the semicolon ending a field.
//
//   StarName1,StarName2,...;RA(h,m,s);Dec(d,m,s);Distance(L-Y);Diameter(L-Y);
//   SpectralClass;Magnitude;PrimaryDistance(L-Y);
//   Membership;Comments
//
//  Example (Proxima Centauri):
//
//   Proxima Cen,Alpha Cen C,Rigil Kent C;14,32,0;-62,49,0;4.24;0;M5e V;
//   15.49;0.252;Alpha Cen triple system;Nearest star, known flare star [etc]
//
//  (The diameter field should be zero in the record if not specifically
//  known; it will then be calculated from the spectral class and magnitude.)
//
//  Note: Other field and subfield separators than ';' and ',' may be assumed
//  by changing the definitions of FIELD_DELIMITER and SUBFIELD_DELIMITER
//  in "star.h".
//
//  The ctor "fastconversion" argument, if set to true, will cause only the
//  most fundamental attributes about the star (position and basic
//  spectral class) to be written.  This is for speed.  The default is false.
//
//  The "nameconvert" argument, if true, causes any constellation abbreviations
//  in the name, e.g. "UMa", to be translated into the full genitive form,
//  e.g. "Ursae Majoris".

#define FIELDS(n) ((fields.size() > (n)) ? fields[(n)] : string(""))

Star::Star(const string &record, bool fastconversion, bool nameconvert)
{
  // First, tokenize record into fields
  StringList fields(record, FIELD_DELIMITER);
  fields.stripspace();
  double RA, Dec;

  // Now, extract data from each field.  First the fields for a fast
  //  conversion (where all we want is to see if the star passes the filters):

  // Right ascension and declination: convert from h,m,s and
  //  +/-d,m,s to decimal format, then put into SolidAngle sGridPosn
  RA = starstrings::str_to_ra(FIELDS(1), SUBFIELD_DELIMITER);
  Dec = starstrings::str_to_dec(FIELDS(2), SUBFIELD_DELIMITER);
  sGridPosn = SolidAngle(RA, Dec);

  // Other fields
  sDistance = starmath::atof(FIELDS(3));
  sSpectrum = SpecClass(FIELDS(5));

  if (fastconversion) /* then bail out here */
    return;

  // Then everything else:

  sSpectrum.initialize();
  sMagnitude = starmath::atof(FIELDS(6));

  sNames = StringList(FIELDS(0), SUBFIELD_DELIMITER);
  sNames.stripspace();
  sNames.utf8ize();

  // translate constellation abbrevs. to genitive names
  if (nameconvert) {
    int constel = -1; // cache constellation name once known
    iterate (StringList, sNames, name_ptr) {
      if (constel < 0) {
        for (int i = 0; i < NUM_CONSTELLATIONS; i++) {
	  if (starstrings::find_and_replace(*name_ptr,
	      string(" ") + constellations[i], string(" ") + constelnames[i])) {
	    constel = i;
	    goto next_name;
	  }
	}
      }
      else { // constel is known
	starstrings::find_and_replace(*name_ptr,
				      string(" ") + constellations[constel],
				      string(" ") + constelnames[constel]);
      }
      next_name: continue;
    }
  }

  sDiameter = starmath::atof(FIELDS(4));
  if (sDiameter <= 0.0)
    sDiameter = sSpectrum.diameter(sMagnitude);

  sPrimaryDistance = starmath::atof(FIELDS(7));
  sMembership = StringList(FIELDS(8), SUBFIELD_DELIMITER);
  sMembership.stripspace();
  sMembership.utf8ize();
  sComments.push_back(FIELDS(9));
  sComments.stripspace();
  sComments.utf8ize();

  sPlace = xPixel = yPixel = rPixel = 0;
}


// function to determine whether the star should be included in a StarArray
//  with the set of rules given in "rules".
//
//  Note: If star is a secondary star "too close" to its primary, it should
//  be be put into the array ONLY if it passes the filter but its primary
//  doesn't.  This filtering occurs in StarArray::Read(), NOT here.

bool Star::PassesFilter(const Rules &rules) const
{
  // Filtering by magnitude is already done by the first filter in
  //  StarArray::Read().

  // Filter by location
  Vector3 relativeLocation = GetStarXYZ() - rules.ChartLocation;
  if (relativeLocation.magnitude() > rules.ChartRadius)
    return false;

  // Filter by spectral class
  if (! rules.StarClasses[SpecHash(GetStarClass().classmajor())])
    return false;

  // if we get down here, star passed all the filters.
  return true;
}


// Display(): plot the star onto the painting device, as viewed from a
//  specified point and orientation.  (We assume that the star coordinates
//  and chart center given are in the same coordinate system.)
//
//  Note: StarViewer is a generic display class defined in viewer.h -- I am
//  trying to keep the graphics-library dependent functions contained in
//  descendant classes of StarViewer (e.g. KDEViewer, GTKViewer, etc.,
//  which will be wrapper classes around the given graphics libraries).
//  This will make the code a bit more confusing, but make the program
//  easier to port.

void Star::Display(const Rules &rules, StarViewer *sv) const
{
  int wincenterX, wincenterY, starBaseY;
  unsigned int windowsize, pixelradius;

  Vector3 relativeLocation;
  double x, y, z;
  color_t barcolor;

  Vector3 center = rules.ChartLocation;
  double radius = rules.ChartRadius;
  SolidAngle orientation = rules.ChartOrientation;
  bool bar = rules.StarBars;

  // Determine radius and center of chart, in pixels
  windowsize = (sv->width() > sv->height()) ? sv->height() : sv->width();
  pixelradius = ROUND(0.4 * windowsize);
  wincenterX = sv->width() / 2;
  wincenterY = sv->height() / 2;

  // Determine star position in "local coordinates", where:
  //  XZ-plane is vertical and perpendicular to the computer screen,
  //   with X-axis pointing out of the screen and tipped DOWNward
  //   by the angle orientation.getTheta()
  //  Y-axis is horizontal, parallel to the screen, and pointing rightward
  //  Distances are in pixels

  relativeLocation = GetStarXYZ() - center;
  x = relativeLocation.getX() * cos(orientation.getPhi())
    + relativeLocation.getY() * sin(orientation.getPhi());
  y = -relativeLocation.getX() * sin(orientation.getPhi())
    + relativeLocation.getY() * cos(orientation.getPhi());
  z = relativeLocation.getZ();
  relativeLocation = Vector3(x,y,z) * pixelradius / radius;

  // Determine 2-D projection of this relative location onto the screen.
  xPixel = wincenterX + ROUND(relativeLocation.getY());
  starBaseY = wincenterY
    + ROUND((relativeLocation.getX() * sin(orientation.getTheta())));
  yPixel = starBaseY
    - ROUND((relativeLocation.getZ() * cos(orientation.getTheta())));
  
  // Determine how large the star should be drawn.
  if (sDiameter * pixelradius / (2 * radius) > STAR_PIXEL_RADIUS) {
    rPixel = ROUND(sDiameter * pixelradius / (2 * radius));
    if (rPixel > pixelradius) rPixel = pixelradius;
  }
  else
    rPixel = 0;

  // Draw the star and accompanying devices (position bar, label).

  barcolor = (relativeLocation.getZ() >= 0.0) ? POSITIVE : NEGATIVE;

  if (bar) {
    if (relativeLocation.getZ() * orientation.getTheta() >= 0.0) {
      // then star is in the hemisphere of the chart facing us, so:
      // draw the reference ellipse on the x-y plane
      sv->setcolor(barcolor);
      sv->setfill(false);
      sv->drawellipse(xPixel, starBaseY, 3, 2);
      // draw the vertical bar
      sv->setcolor(BACKGROUND);
      sv->drawline(xPixel - 1, starBaseY, xPixel - 1, yPixel);
      sv->drawline(xPixel + 1, starBaseY, xPixel + 1, yPixel);
      sv->setcolor(barcolor);
      sv->drawline(xPixel, starBaseY, xPixel, yPixel);
    }

    else {
      // outline the vertical bar
      sv->setcolor(BACKGROUND);
      sv->drawline(xPixel - 1, starBaseY, xPixel - 1, yPixel);
      sv->drawline(xPixel + 1, starBaseY, xPixel + 1, yPixel);
    }
  }

  // draw the star and label
  rPixel = Draw(rules, sv, xPixel, yPixel, rPixel);
  // make sure the effective mouse-click radius isn't too small:
  if (rPixel < STAR_PIXEL_RADIUS) rPixel = STAR_PIXEL_RADIUS;

  if (bar && relativeLocation.getZ() * orientation.getTheta() < 0.0) {
    // draw the vertical bar
    sv->setcolor(barcolor);
    sv->drawline(xPixel, starBaseY, xPixel, yPixel);
    // draw the ellipse
    sv->setcolor(barcolor);
    sv->setfill(false);
    sv->drawellipse(xPixel, starBaseY, 3, 2);
  }

  return;
}


// Draw(): plot the star onto the painting device at a given pixel location,
//  not worrying about coordinates, etc.  Returns the radius of circle drawn,
//  in pixels.

unsigned int Star::Draw(const Rules &rules, StarViewer *sv, 
			int xposn, int yposn, int pixelradius) const
{
  // set size of star to pixelradius, unless pixelradius is <= 0 / not given:

  if (pixelradius <= 0) {
    pixelradius = STAR_PIXEL_RADIUS;

    if (rules.StarDiameters == MK_DIAMETERS) {
      // Increase size of the circle for giant-class stars.
      double mktype = sSpectrum.MKtype();
      if (mktype != 0.0) {
	if (mktype <= 3.5)
	  pixelradius++;
	if (mktype <= 1.5)
	  pixelradius++;
      }
    }

    else if (rules.StarDiameters == MAGNITUDE_DIAMETERS) {
      // Set size of the circle based upon the absolute magnitude.
      if (sMagnitude >= 8.0) pixelradius = 1;
      else if (sMagnitude < -4.0) pixelradius = 5;
      else pixelradius = ROUND(4.0 - sMagnitude / 4.0);
    }
  }

  // Draw the circle
  sv->setfill(true);
  sv->setcolor(sSpectrum.color());
  sv->drawstar(xposn, yposn, pixelradius);  

  if (rules.StarLabels != NO_LABEL) { // number the star appropriately
    string starPlace;
    label_t label = rules.StarLabels;

    // make non-stellar object labels a different color
    sv->setcolor((sSpectrum.classmajor() == '*') ? NON_STELLAR_COLOR : 
		 TEXT_COLOR);

    sv->setfill(false);

    switch (label) {
    case NUMBER_LABEL:
      starPlace = starstrings::itoa(sPlace);
      sv->drawtext(starPlace,
		   xposn + ROUND(pixelradius / M_SQRT2) + 3,
		   yposn - ROUND(pixelradius / M_SQRT2) - 3);
      break;
    case STRING_LABEL:
      sv->drawtext(sNames[0],
		   xposn + ROUND(pixelradius / M_SQRT2) + 3,
		   yposn - ROUND(pixelradius / M_SQRT2) - 3);
      break;
    case LANDMARK_LABEL:
      if (sLabelDraw || sNames[0] == string("Sun"))
	sv->drawtext(sNames[0],
		     xposn + ROUND(pixelradius / M_SQRT2) + 3,
		     yposn - ROUND(pixelradius / M_SQRT2) - 3);
      break;
    default: // do nothing
      break;
    }
  }

  return pixelradius;
}


// function to take star information, format it properly, and append it
//  as strings to a StringList.

StringList Star::GetInfo(const Rules &rules, bool punctuation,
		         char sep) const
{
  StringList output;
  output.push_back(starstrings::itoa(sPlace));		// 0. Label number

  // 1. Most common designation
  output.push_back((sNames.size()) ? sNames[0] : "");
  
  if (sDistance == 0.0) {
    output.push_back("N/A");
    output.push_back("N/A");
  }
  else {
    // 2. { Right ascension | Galactic longitude }
    output.push_back(starstrings::ra_to_str(sGridPosn.getPhi(), sep,
					    rules.CelestialCoords,
					    punctuation));
    // 3. { Declination | Galactic latitude }
    output.push_back(starstrings::dec_to_str(sGridPosn.getTheta(), sep,
					     punctuation));
  }

  output.push_back(starstrings::ftoa(sDistance, 5));	// 4. Distance in L-Y
  output.push_back(sSpectrum.print());			// 5. Spectral class
  output.push_back(starstrings::ftoa(sMagnitude, 4));	// 6. M_v
  if (sMembership.size())
    output.push_back(sMembership[0]);			// 7. Cluster membership
  else
    output.push_back("");
  if (sComments.size())
    output.push_back(sComments[0]);			// 8. Comments
  else
    output.push_back("");
  output.push_back(starstrings::ftoa(sPrimaryDistance, 5));
						// 9. Distance from primary
  citerate_from (StringList, sNames, i, 1)
    output.push_back(*i);			// 10+. Other names for star

  return output;
}

