// StarPlot - A program for interactively viewing 3D maps of stellar positions.
// Copyright (C) 2000  Kevin B. McCarty
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


//
// stararray.h - The Star Array class.  Not much more than a container class
//  for stars, with the ability to read and filter a text data file on the fly.
//

#ifndef _STAR_ARRAY_H_
#define _STAR_ARRAY_H_

#include "../../lib/compat.h"
#include "star.h"
#include <istream>
#include <ostream>
#include <fstream>
#include <vector>
#include <string>
#include <cstdlib>
#include <cstdio>

// A typedef to specify the kind of change made to ArrayRules when
//  setting a new set of rules:
typedef enum { FILE_CHANGE, LOCATION_CHANGE, RADIUS_CHANGE,
	       COORDINATE_CHANGE, FILTER_CHANGE, ORIENTATION_CHANGE,
	       DECORATIONS_CHANGE, NO_CHANGE } changetype_t;

// Specify the character used to separate records in the data files
const char RECORD_DELIMITER = '$';

// and the number of stars on the display to label, if the "Landmark Stars"
//  option is checked
const size_t DISPLAY_N_STARS = 8;

// Do not load more than this number of stars into memory at once, to
//  avoid DOS'ing oneself.  Note that 1000 stars ~ 1 megabyte RAM
const size_t MAX_STARS = 20000;

// offsets of star symbols in legend from the position
// {x = width - 75, y = 0} for spectral classes in the order
// O, B, A, F, G, K, M, D, unknown, non-stellar, Wolf-Rayet
#define LEGEND_OFFSETS_SIZE 11
const unsigned int LEGEND_OFFSETS[LEGEND_OFFSETS_SIZE][2] =
	{ {0, 40}, {0, 55}, {0, 70}, {0, 85}, {30, 40}, {30, 55},
	  {30, 70}, {30, 85}, {80, 115}, {80, 100}, {0, 25} };


class StarArray {
 private:
  typedef std::vector<Star>::iterator iterator;
  
  std::vector<Star> Array;
  size_t TotalStars;
  Rules ArrayRules;

  StarArray(const StarArray &rhs);                 // these should never
  StarArray & operator= (const StarArray &rhs);    // be implemented.

  void toCelestial();  // convert coordinates for all the stars,
  void toGalactic();   //  as specified.  Not public because they should
                       //  only be controlled by StarArray::SetRules().

  void Read(std::ifstream &);   // reads and filters data file
  void Sort();             // sorts the array by x-position in local coords

  // draws a chart legend item
  void drawlegenditem(StarViewer *sv, color_t color, unsigned int number, 
		      unsigned int x, unsigned int y, unsigned int r,
		      const std::string &text, int relx = 10, int rely = 5)
		      const;

  // draw the legend
  void drawlegend(StarViewer *sv) const;

  // functions to draw the chart grid
  void drawgridnorth(StarViewer *sv, unsigned int wX, unsigned int wY,
		     unsigned int pixelradius) const;
  void drawgridsouth(StarViewer *sv, unsigned int wX, unsigned int wY,
		     unsigned int pixelradius) const;
  void drawgridequator(StarViewer *sv, unsigned int wX, unsigned int wY,
		       unsigned int pixelradius) const;

 public:
  StarArray() : Array() { TotalStars = 0; }
  // no other constructors needed, memory management is automatic in STL

  // some traditional STL thingies
  typedef std::vector<Star>::const_iterator const_iterator;
  inline size_t size() const		    { return Array.size(); }
  inline const const_iterator begin() const { return Array.begin(); }
  inline const const_iterator end() const   { return Array.end(); }
  
  inline size_t totalstars() const { return TotalStars; }
  inline Star & operator [] (size_t i) { return Array[i]; }

  // look up stars containing a certain substring in their names and create
  //  a StarArray of them (ignores the Rules member)
  void Search(const std::string &searchstring, StringList filelist,
	      const Rules &rules, bool casesensitive = false,
	      bool exactmatch /* i.e. not a substring */ = false,
	      bool exitonmatch /* i.e. return only one match */ = false);

  // set rules for filtering and update the Array of stars using those rules
  //  (default: if type of change is not supplied, assume the most radical 
  //  change)
  bool SetRules(const Rules &rules, changetype_t ruleschange = FILE_CHANGE);

  // displays all the stars, plus a grid and some other decorations
  void Display(StarViewer *sv) const;

  // plots the stars on an HR diagram in range dimmag - brightmag
  void Diagram(StarViewer *sv, double brightmag, double dimmag) const; 
};

#endif
