// StarPlot - A program for interactively viewing 3D maps of stellar positions.
// Copyright (C) 2000  Kevin B. McCarty
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


#ifndef _STARPLOT_H
#define _STARPLOT_H

#include <gtk/gtk.h>
#include <cmath>
#include <cstdio>
#include <cstdlib>

#include "../classes/stararray.h"
#include "gtkviewer.h"
#include "menu.h"

#ifndef DATADIR
#define DATADIR "/usr/local/share/starplot"
#endif

#ifndef DOCDIR
#define DOCDIR "/usr/local/share/doc/starplot"
#endif

#ifndef BROWSER
#define BROWSER "" // initialize to null string
#endif

// Default program values (used if no .starplotrc file is found in $HOME)

namespace defaults {

  // parameters of the GUI

  const unsigned int  program_width	    = 550;
  const unsigned int  program_height	    = 500;
  const unsigned int  program_hr_width	    = 480; // dimensions of HR diagram
  const unsigned int  program_hr_height	    = 360; //  window
  const unsigned int  program_button_width  = 75;  // global default size for
  const unsigned int  program_button_height = 22;  //  OK / Cancel buttons
  const std::string   program_help_browser  = BROWSER;
  
  // parameters controlling what stars are initially displayed

  const StringList    chart_filenames =
    StringList(1 /* copy of arg2: */, DATADIR "/sample.stars"); 
  const Vector3	      chart_center	= Vector3(0, 0, 0);
  const SolidAngle    chart_orientation = SolidAngle(21 HOURS, +30 DEGREES);
  const double	      chart_radius      = 10 /* light-years */;
  const float	      chart_mag_dim     = +25;
  const float	      chart_mag_bright	= -25;
  const bool	      chart_hide_comps	= true; // hide companion stars
  const bool	      chart_mag_auto	= true; // automatically set dim mag.
  const bool	      chart_classes[10] =  { true, true, true, true, true,
	                                     true, true, true, true, true };
  const bool	      chart_coord_type  = CELESTIAL;

  // parameters controlling initial appearance of the chart
  
  // If you edit the following two lines, you also must reorder the menu items
  //  in menuops.cc so that this default is the first of the starlabel menu
  //  items.  Otherwise the initial toggle setting will not match this default.
  const diameter_t    display_starsize	= MAGNITUDE_DIAMETERS;
  const label_t	      display_label_type= LANDMARK_LABEL;

  const bool	      display_bars	= true;
  const bool	      display_grid	= true;
  const bool	      display_legend	= true;
  const std::string   display_fontname	= "Sans 8";

  // parameters controlling appearance of the HR diagram window
  
  const double	      hr_mag_bright	= -8;
  const double	      hr_mag_dim	= +12;

  // structure containing default chart parameters
  
  const Rules	      chart_rules	= Rules(chart_filenames,
						chart_center,
						chart_orientation,
						chart_radius,
						chart_mag_dim,
						chart_mag_bright,
						chart_hide_comps,
						chart_mag_auto,
						chart_classes,
						chart_coord_type,
						display_starsize,
						display_label_type,
						display_bars,
						display_grid,
						display_legend
					       );
}; // end "defaults" namespace

// Values that need to be shared across object files

namespace globals {

  // parameters
  extern std::string  program_help_browser;
  extern std::string  display_fontname;
  extern Rules	      chart_rules;
  extern double	      hr_mag_bright;
  extern double	      hr_mag_dim;
  
  // data
  extern StarArray    *chart_stararray;
  
  // widgets
  extern GtkWidget    *program_status;	    // status text at bottom of window
  extern GtkTreeView  *program_treeview;    // Chart Data list of stars
  extern GTKViewer    *program_viewer;      // The GTKViewer used by StarPlot
  extern GtkWidget    *program_canvas;      // The GTK_DRAWING_AREA used
  extern GdkPixmap    *program_pixmap;	    // The backing pixmap
  extern GTKViewer    *program_hr_viewer;   // HR Diagram GTKViewer
  extern GtkWidget    *program_hr_canvas;   // HR Diagram GTK_DRAWING_AREA  
  extern GdkPixbuf    *program_icon;	    // used by the window manager
  
}; // end "globals" namespace

#define RESET_GLOBAL(var)    (globals::var = defaults::var)

// Button bar callbacks, defined in menuops.cc

extern void button_zoom(GtkWidget *w, gpointer zoomfactor);
extern void button_rotate(GtkWidget *w, gpointer angle);
extern void button_tilt(GtkWidget *w, gpointer angle);
extern void button_magchange(GtkWidget *w, gpointer magchange);

// Useful adjunct functions

extern bool read_rc_file();
extern bool write_rc_file();
extern void help_select_and_open();
extern double automagnitude (double radius);
extern void my_gtk_star_popup (Star &);
extern void my_gtk_main_menu (GtkWidget *window, GtkWidget **menubar);
extern void my_gtk_push_buttons (GtkWidget *window, GtkWidget **buttonbar);
extern void my_gtk_button_bar (GtkWidget **OKbtn, GtkWidget **Defaultsbtn,
			       GtkWidget **Cancelbtn, GtkWidget *box,
			       int spacing = 20);
extern void my_gtk_popup_button (GtkWidget **Closebtn, GtkWidget *box,
				 GtkWidget *window);
extern void my_gtk_buttons_connect_destroy (GtkWidget *btn1, GtkWidget *btn2,
					    GtkWidget *btn3, GtkWidget *win);
extern void my_gtk_error (GtkWindow *parent_win, const char * error_msg);
extern void hr_redraw_display(GtkWidget *hr_display);
extern void set_widget_font(GtkWidget *pixmap, std::string font_description);
extern void update_info(GtkTreeView *info_treeview);
extern void redraw_all (changetype_t);

// Functions to set menu toggle/radio buttons

template<class T> extern void 
set_item(T *, T, GtkWidget *, bool, changetype_t = DECORATIONS_CHANGE);
extern void flip_toggle_item(bool *, int, bool);
extern void set_toggle_item(bool *, bool, int, bool);
extern void set_label_item(label_t, bool);
extern void set_diameter_item(diameter_t, bool);
extern void set_coord_item(bool, bool);

#endif // #include _STARPLOT_H
