/* Stars -- Displays a Map of the Night Sky
    Copyright (C) September 22, 2002  Walter Brisken

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA*/

#include <stdio.h>
#include <math.h>
#include "coordutils.h"

double formattedtext2double(const gchar *str, const gchar *format)
{
	gint n;
	double h, m, s, ret;

	n = sscanf(str, format, &h, &m, &s);

	if(n == 0) return 0.0;
	ret = fabs(h);
	if(n > 1) ret += m/60.0;
	if(n > 2) ret += s/3600.0;
	if(h < 0) ret = -ret;
	return ret;
}

double text3double(const gchar *_str)
{
	gchar *str;
	gchar ignore[] = ":'\"/dmsDMShH";
	gint i, j;
	double h, m, s, sign=1.0;

	str = g_strdup(_str);
	
	for(i = 0; str[i]; i++)
	{
		if(str[i] == '-')
		{
			str[i] = ' ';
			sign = -1.0;
		}
		else for(j = 0; ignore[j]; j++) if(str[i] == ignore[j])
			str[i] = ' ';
	}

	i = sscanf(str, "%lf%lf%lf", &h, &m, &s);
	g_free(str);
	if(i == 0) return 0;
	if(i > 1) h += m/60.0;
	if(i > 2) h += s/3600.0;
	return h*sign;
}

double text2double(const gchar *str)
{
	gint colons = 0;
	gint len;
	double sign = 1.0;
	gchar str2[100], *str3;
	double h = 0, m, s;
	double ret = -999999.999;
	gint digs;
	
	for(len = 0; (str2[len] = str[len]) != 0; len++) 
		if(str[len] == ':') colons++;

	if(str[0] == '-' || str[0] == '+')
	{
		if(str[0] == '-') sign = -1.0;
		str3 = str2+1;
	}
	else str3 = str2;

	if(colons == 0) 
	{
		for(digs = 0; str3[digs] != 0 && str3[digs] != '.'; digs++);
		switch(digs)
		{
			case 0:
			case 1:
			case 2:
				if(sscanf(str, "%lf", &h) == 1) ret = h;
				break;
			case 4:
				h = 10.0*(str3[0] - '0') + (str3[1] - '0');
				if(sscanf(str3 + 2, "%lf", &m) == 1)
					ret = sign*(h + m/60.0);
				break;
			case 6:
				h += 10.0*(str3[0] - '0') + (str3[1] - '0');
				m =  10.0*(str3[2] - '0') + (str3[3] - '0');
				if(sscanf(str3+4, "%lf", &s) == 1)
					ret = sign*(h + m/60.0 + s/3600.0);
				break;
			case 7:
				h += 10.0*(str3[0] - '0') + (str3[1] - '0');
				m =  10.0*(str3[2] - '0') + (str3[3] - '0');
				if(sscanf(str3+4, "%lf", &s) == 1)
					ret = sign*(h + m/60.0 + s/36000.0);
				break;
			case 8:
				h += 10.0*(str3[0] - '0') + (str3[1] - '0');
				m =  10.0*(str3[2] - '0') + (str3[3] - '0');
				if(sscanf(str3+4, "%lf", &s) == 1)
					ret = sign*(h + m/60.0 + s/360000.0);
				break;
			case 9:
				h += 10.0*(str3[0] - '0') + (str3[1] - '0');
				m =  10.0*(str3[2] - '0') + (str3[3] - '0');
				if(sscanf(str3+4, "%lf", &s) == 1)
					ret = sign*(h + m/60.0 + s/3600000.0);
				break;
		}
	}
	else if(colons == 1)
	{
		if(sscanf(str, "%lf:%lf", &h, &m) == 2)
		{
			ret = fabs(h) + m/60.0;
			if(h < 0) ret = -ret;
		}
	}
	else if(colons == 2)
	{
		if(sscanf(str, "%lf:%lf:%lf", &h, &m, &s) == 3) 
		{
			ret = fabs(h) + m/60.0 + s/3600.0;
			if(h < 0) ret = -ret;
		}
	}

	return ret;
}

void double2text(double num, gchar *str, gchar sep)
{
	gint h, m;

	if(num < 0)
	{
		num = -num;
		str[0] = '-';
		str++;
	}
	h = (int)num;
	num -= h;
	num *= 60.0;
	m = (int)num;
	num -= m;
	num *= 60.0;
	sprintf(str, "%02d%c%02d%c%09.6f", h, sep, m, sep, num);
}

gchar *g_double2text(double num, gchar sep)
{
	gint h, m, s=1;

	if(num < 0)
	{
		num = -num;
		s = -1;
	}
	h = (int)num;
	num -= h;
	num *= 60.0;
	m = (int)num;
	num -= m;
	num *= 60.0;

	if(num > 59.999)
	{
		num -= 60.0;
		if(num < 0.0) num = 0.0;
		m = m+1.0;
	}
	if(m > 59.999)
	{
		m -= 60.0;
		if(m < 0.0) m = 0.0;
		h = h+1;
	}

	g_strdup_printf("%02d%c%02d%c%06.3f", s*h, sep, m, sep, num);
}

gint uranopage(double ra, double dec)
{
	gint page = 1, n;
	double decboundary[] = {84.0, 72.0, 61.0, 50.0, 39.0, 28.0, 17.0, 6.0,
		-6.0, -17.0, -28.0, -39.0, -50.0, -61.0, -72.0, -84.0, -90.1};
	gint radiv[] = {2, 12, 20, 24, 30, 36, 45, 45, 45, 45,
		45, 36, 30, 24, 20, 12, 2};

	for(n = 0; n < 17; n++)
	{
		if(dec > decboundary[n])
			return page + rint((ra * (double)radiv[n])/24.0);
		page += radiv[n];
	}

	return -1;
}
