/* Stars -- Displays a Map of the Night Sky
    Copyright (C) September 22, 2002  Walter Brisken

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA*/

#include <stdio.h>
#include "coordutils.h"
#include "location.h"

#define TODEG(d, m, s) ((d) + (m)/60.0 + (s)/3600.0)


GList *cities=0, *observatories=0;

struct site *newsite(const char *name, double latitiude,
	double longitude, double elevation)
{
	struct site *s;

	s = g_new(struct site, 1);
	s->name = g_strdup(name);
	s->pos = newearthpos(latitiude, longitude, elevation + EARTH_RADIUS);

	return s;
}

void deletesite(struct site *s)
{
	if(!s)
	{
		fprintf(stderr, "Weird, trying to delete null site\n");
		return;
	}
	if(s->name) g_free(s->name);
	if(s->pos) deleteearthpos(s->pos);
	g_free(s);
}

struct earthpos *newearthpos(double latitude, double longitude, 
        double elevation)
{
	struct earthpos *pos;

	pos = g_new(struct earthpos, 1);
	pos->latitude = latitude;
	pos->longitude = longitude;
	pos->elevation = elevation;

	return pos;
}

struct earthpos *copyearthpos(struct earthpos *oldpos)
{
	struct earthpos *pos;

	pos = g_new(struct earthpos, 1);
	pos->latitude = oldpos->latitude;
	pos->longitude = oldpos->longitude;
	pos->elevation = oldpos->elevation;

	return pos;
}

void deleteearthpos(struct earthpos *pos)
{
	if(!pos)
	{
		fprintf(stderr, "Weird, deleting numm earthpos\n");
		return;
	}
	g_free(pos);
}

struct location *newlocation()
{
	struct location *loc;

	loc = g_new(struct location, 1);
	loc->type = 1;
	loc->cityid = 0;
	loc->obsid = 0;
	loc->otherpos = newearthpos(120, 33, 200);

	return loc;
}

struct location *copylocation(const struct location *oldloc)
{
	struct location *loc;

	loc = g_new(struct location, 1);
	loc->type = oldloc->type;
	loc->cityid = oldloc->cityid;
	loc->obsid = oldloc->obsid;
	loc->otherpos = copyearthpos(oldloc->otherpos);

	return loc;
}

void deletelocation(struct location *loc)
{
	if(loc == 0) return;
	if(loc->otherpos) g_free(loc->otherpos);
	g_free(loc);
}

GList **getsitelist(gint sitelistnum)
{
	switch(sitelistnum)
	{
		case CITY_SITES:	
			return &cities;
		case OBS_SITES:	
			return &observatories;
		default:
			fprintf(stderr, "getsitelist : out of range\n");
			return 0;
	}
}

struct site *getsite(gint sitelistnum, gint index)
{
	GList **sites;

	sites = getsitelist(sitelistnum);
	return (struct site *)g_list_nth_data(*sites, index);
}

gint getsitenum(gint sitelistnum, const gchar *name)
{
	GList **sites, *s;
	int i=0;

	sites = getsitelist(sitelistnum);
	if(!sites) 
	{
		fprintf(stderr, "getsitenum called with null site\n");
		return -1;
	}
	
	for(s = *sites; s != 0; s = s->next)
	{
		if(!s->data) break;
		if(strcmp( ((struct site *)(s->data))->name, name) == 0) return i;
		i++;
	}

	fprintf(stderr, "getsitenum: site not found : >%s<\n", name);
	return -1;
}

int addtositelist(gint sitelistnum, struct site *s)
{
	GList **sitelist;
	
	sitelist = getsitelist(sitelistnum);
	if(!sitelist)
	{
		printf("addtositelist failed\n");
		return 0;
	}

	*sitelist = g_list_append(*sitelist, s);

	return 1;
}

void sitecombofill(GtkWidget *combo, int sitelistnum)
{
	GList **sitelist;
	GList *s, *sites = 0;

	sitelist = getsitelist(sitelistnum);
	for(s = *sitelist; s != 0; s = s->next)
	{
		if(!s->data) break;
		sites = g_list_append(sites, g_strdup(((struct site *)(s->data))->name));
	}
	gtk_combo_set_popdown_strings(GTK_COMBO(combo), sites);
}

void initlocations()
{
	addtositelist(CITY_SITES, 
		newsite("Fremont, CA", 37.52851, 121.99712, 0.0));
	addtositelist(CITY_SITES, 
		newsite("Socorro, NM", 34.05515, 106.90422, 1400.0));

	addtositelist(OBS_SITES, 
		newsite("Very Large Array", TODEG(34,4,43.497), 
			TODEG(107,37,3.819), 2124.0));
	addtositelist(OBS_SITES, 
		newsite("Green Bank 140ft", TODEG(38,26,15.409),
			TODEG(79,50,9.613), 880.87));
}

gchar *latitude2text(double latitude)
{
	gchar hemi = 'N';
	int degrees, minutes;
	double seconds;

	if(latitude < 0)
	{
		latitude = -latitude;
		hemi = 'S';
	}
	
	degrees  = latitude;
	latitude = 60.0*(latitude-degrees);
	minutes  = latitude;
	seconds  = 60.0*(latitude-minutes);
	
	return g_strdup_printf("%d %02d %06.3f %c", degrees, minutes, seconds,
		hemi);
}

gchar *longitude2text(double longitude)
{
	gchar hemi = 'W';
	int degrees, minutes;
	double seconds;

	if(longitude < 0)
	{
		longitude = -longitude;
		hemi = 'E';
	}
	
	degrees   = longitude;
	longitude = 60.0*(longitude-degrees);
	minutes   = longitude;
	seconds   = 60.0*(longitude-minutes);
	
	return g_strdup_printf("%d %02d %06.3f %c", degrees, minutes, seconds,
		hemi);
}

double text2latitude(const gchar *str_)
{
	double sign = 1.0;
	int i, j;
	gchar *str;
	const gchar skip[] = "NnDdMm\"\'/\\:,";
	double result;

/* Check if abs() > 10^4.  if so, split 1234567.89 -> 123 45 67.89 */

	str = g_strdup(str_);
	
	for(i = 0; str[i] != 0; i++)
	{
		for(j = 0; skip[j] != 0; j++) if(str[i] == skip[j]) str[i] = ' ';
		if(str[i] == 'S' || str[i] == 's') 
		{
			str[i] = ' ';
			sign = -1.0;
		}
	}

	result = sign * formattedtext2double(str, "%lf%lf%lf");

	g_free(str);

	return result;
}

double text2longitude(const gchar *str_)
{
	double sign = 1.0;
	int i, j;
	gchar *str;
	const gchar skip[] = "WwDdMmSs\"\'/\\:,";
	double result;

	str = g_strdup(str_);
	
	for(i = 0; str[i] != 0; i++)
	{
		for(j = 0; skip[j] != 0; j++) if(str[i] == skip[j]) str[i] = ' ';
		if(str[i] == 'E' || str[i] == 'e') 
		{
			str[i] = ' ';
			sign = -1.0;
		}
	}

	result = sign * formattedtext2double(str, "%lf%lf%lf");

	g_free(str);

	return result;
}

gchar *site2text(struct site *s)
{
	gchar *lat, *lon, *output;

	if(!s)
	{
		fprintf(stderr, "site2text: site is null\n");
		return;
	}
	
	lat = latitude2text(s->pos->latitude);
	lon = longitude2text(s->pos->longitude);
	
	output = g_strdup_printf(" at %s, %s", lat, lon);

	g_free(lat);
	g_free(lon);

	return output;
}
