/* $Id: startalk.c,v 400.1 2002/07/25 08:43:16 sgifford Exp $ */

#define STARTALK_C

#include "config.h"

#include "startalk.h"
#include "stdebug.h"
#include "alarm.h"
#include "fstty.h"
#include "sysdep1.h"

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/uio.h>
#include <fcntl.h>
#include <errno.h>
#include <termios.h>
#include <netinet/in.h>
#include <string.h>
#include <ctype.h>
#include <signal.h>

extern int startalk_timeout;
int startalk_initialized = 0;
int startalk_fd = -1;

/* Writing functions */
int
startalk_write_timeout(int serialfd, const char *buf, int nchars, int timeout)
{
  int rr;
  const char *wpos = buf;
  int prevchars = 0;
  
  if (timeout)
    start_alarm(timeout);
  while ( (rr=write(serialfd,wpos,nchars-prevchars)) != -1)
  {
    if ( (rr == 0) && timeout && check_alarm())
    {
      end_alarm();
      return reterr(-1,"Timeout writing to phone");
    }
    if ( (prevchars + rr) == nchars)
    {
      if (timeout)
        end_alarm();
      return nchars;
    }
    else
    {
      prevchars += rr;
      wpos += rr;
    }
  }

  if (timeout)
    end_alarm();

  if (timeout && check_alarm())
    return reterr(-1,"Timeout writing to phone");
  else
    return reterr(-1,"Error writing to phone:");
}

int
startalk_write(int serialfd, const char *buf, int size)
{
  return startalk_write_timeout(serialfd, buf, size, startalk_timeout);
}

/* Writing functions */
int
startalk_writev_timeout(int serialfd, const struct iovec *iov, int count, int timeout)
{
  int rr;
  if (timeout)
    start_alarm(timeout);
  
  rr = writev(serialfd, iov, count);
  if ( (rr <= 0) )
  {
    if (timeout && check_alarm())
    {
      end_alarm();
      return reterr(-1,"Timeout writing to phone");
    }
    return reterr(-1,"Error writing to phone:");
  }
  else
  {
    if (timeout)
      end_alarm();
    return rr;
  }
}

int
startalk_writev(int serialfd, const struct iovec *iov, int count)
{
  return startalk_writev_timeout(serialfd, iov, count, startalk_timeout);
}

int
startalk_writestr_timeout(int serialfd, const char *buf, int timeout)
{
  return startalk_write_timeout(serialfd,buf,strlen(buf),timeout);
}

int
startalk_writestr(int serialfd, const char *buf)
{
  return startalk_writestr_timeout(serialfd, buf, startalk_timeout);
}

/* Reading functions */
int
startalk_read_timeout(int serialfd, char *buf, int nchars, int timeout)
{
  int prevchars = 0;
  int rr;
  char *rpos = buf;

  if (timeout)
    start_alarm(timeout);
  while ((rr=read(serialfd, rpos, nchars-prevchars)) != -1)
  {
    if ((rr == 0) && timeout && check_alarm())
    {
      end_alarm();
      return reterr(-1,"Timeout reading from phone");
    }
    if ( (prevchars + rr) == nchars)
    {
      return nchars;
    } 
    else
    {
      prevchars += rr;
      rpos += rr;
    }
  }
  if (timeout)
    end_alarm();

  if (timeout && check_alarm())
    return reterr(-1,"Timeout reading from phone");
  else
    return reterr(-1,"Error reading from phone:");
}

int
startalk_read(int serialfd, char *buf, int nchars)
{
  return startalk_read_timeout(serialfd, buf, nchars, startalk_timeout);
}

static char *
strcleandup(char *str)
{
  char *ret;
  unsigned char *r;
  char hexchr[] = "0123456789abcdef";
  unsigned char *ustr;

  ustr = (unsigned char *)str;

  
  if (!(ret = malloc(strlen(str)*4+1)))
    return NULL;
  
  r = (unsigned char *)ret;
  
  r[0]='\0';
  for(;*ustr;ustr++)
  {
    if (*ustr == '\\')
    {
      *r++ = '\\';
      *r++ = '\\';
    }
    else if (isprint(*ustr))
    {
      *r++=*ustr;
    }
    else  if (*ustr == '\r')
    {
      *r++ = '\\';
      *r++ = 'r';
    }
    else if (*ustr == '\n')
    {
      *r++ = '\\';
      *r++ = 'n';
    }
    else
    {
      *r++ = '\\';
      *r++ = 'x';
      *r++ = hexchr[(*ustr >> 4) % 16];
      *r++ = hexchr[(*ustr) % 16];
    }
  }
  *r='\0';
  return ret;
}

int
startalk_readfor(int serialfd, char *str)
{
  unsigned char *p;
  unsigned char c;
  char buf[1024]="";
  int bufpos=0;
  char *err1, *err2;
  
  if (startalk_timeout)
    start_alarm(startalk_timeout);
  p=(unsigned char *)str;
  while (1)
  {
    if (startalk_read_timeout(serialfd, (char *)&c, 1, 0) != 1)
    {
      break;
    }
    debugf(4,"startalk_readfor: read character '%c' (%d or 0x%x)\n",(c>=32 && c<=126)?c:'?',c,c);

    if (bufpos < 1022)
      buf[bufpos++] = c;

    if (c == *p)
    {
      p++;
      if (*p == 0)
      {
        end_alarm();
        return 0;
      }
    }
    else
    {
      p=(unsigned char *)str;
      if (c == *p)
      {
        p++;
        if (*p == 0)
        {
          end_alarm();
          return 0;
        }
      }
    }
    if (startalk_timeout && check_alarm())
    {
      break;
    }
  }

  /* If we haven't returned by now, there was an error. */
  buf[bufpos] = '\0';
  err1=strcleandup(buf);
  err2=strcleandup(str);
  if (startalk_timeout && check_alarm())
  {
    if (err1 && err2)
    {
      debugf(2,"startalk_readfor(%d,\"%s\") failed: Timeout.\n",serialfd,err2);
      errorf("Timeout while waiting for '%s' (received '%s' instead)",err2,err1);
    }
    else
    {
      errorf("Timeout while waiting for string, and malloc error displaying string (wow, bad day...)");
    }
  }
  else
  {
    if (err1 && err2)
    {
      debugf(2,"startalk_readfor(%d,\"%s\") failed: got '%s' instead.\n",serialfd,err2,err1);
      errorf("Got '%s' while waiting for '%s'",err1,err2);
    }
    else
    {
      errorf("Waited for string never received, and malloc error displaying string (wow, bad day...)");
    }
  }
  if (err1)
    free(err1);
  if (err2)
    free(err2);
  end_alarm();
  return 1;
}

int
startalk_writemsg(int serialfd, char *msgbuf, int size)
{
  unsigned char csize;
  unsigned short int csum = 0;
  unsigned char csbuf[2];
  int i;
  struct iovec iov[4];
  int len;
  unsigned char *umsgbuf = (unsigned char *)msgbuf;
  
  /* Header */
  csum = 2;
  iov[0].iov_base = "\xAA\xB0\x00\x02\x00";
  len = iov[0].iov_len = 5;
  
  csum += (csize = size);
  iov[1].iov_base = &csize;
  len += (iov[1].iov_len = 1);
  
  for(i=0;i<size;i++)
    csum += umsgbuf[i];
  iov[2].iov_base = msgbuf;
  len += (iov[2].iov_len = size);
    
  csbuf[0] = csum / 256;
  csbuf[1] = csum % 256;
  iov[3].iov_base = csbuf;
  len += (iov[3].iov_len = 2);

  if (startalk_writev(serialfd,iov,4) != len)
    return reterr(-1,"Error writing command to phone");
  
  if (debuglevel(3))
  {
    debugf(3,"--------Sent to Phone--------\n");
    dumpbuf_start(iov[0].iov_base, iov[0].iov_len);
    for (i = 1; i < 4; i++)
      dumpbuf_cont(iov[i].iov_base, iov[i].iov_len);
    dumpbuf_end();
    debugf(3,"-----------------------------\n");
  }
  debugf(3,"Sending checksum %d (0x%04X)\n",csum,csum);

  return 0;
}

int
startalk_readmsg(int serialfd, char *msgbuf, int maxsize, int *msgsize)
{
  int msglen;
  unsigned short int csum = 0;
  int i;
  unsigned char *nextbyte = (unsigned char *)msgbuf;
  unsigned char *umsgbuf = (unsigned char *)msgbuf;

  if (maxsize < 8)
    return errorf("Buffer too small to read ANYTHING from phone (%d byte buffer)",maxsize);

  debugf(3,"Scanning for beginning of packet, character 0xAA\n");
  if (startalk_readfor(serialfd,"\xAA") != 0)
    return errorf("Error finding command from phone");

  *nextbyte='\xAA';
  nextbyte++;
  debugf(3,"Reading header (5 bytes)\n");
  if (startalk_read(serialfd,(char *)nextbyte,5) != 5)
    return errorf("Error reading from phone");
  debugf(3,"Read header!\n");

  nextbyte += 5;
  msglen = umsgbuf[5];
  debugf(3,"Remainder of packet is %d bytes.\n",msglen);
  if (msglen > 0)
  {
    if ( (msglen + 8) > maxsize)
      return errorf("Buffer too small to read from phone (%d byte buffer, %d byte message)",maxsize,msglen+8);
      
    if (startalk_read(serialfd,(char *)nextbyte,msglen) != msglen)
      return errorf("Error reading from phone");

    nextbyte += msglen;
    csum = 0;
    for(i=2;i<(msglen+6);i++)
    {
      csum += umsgbuf[i];
    }

    if (startalk_read(serialfd,(char *)nextbyte,2) != 2)
      return errorf("Error reading from phone");

    if (debuglevel(3))
    {
      debugf(3,"-----Received from Phone-----\n");
      dumpbuf(msgbuf,msglen+8);
      debugf(3,"-----------------------------\n");
    }
    
    if ( ((csum % 256) == nextbyte[1]) && ((csum / 256) == nextbyte[0]) )
      debugf(3,"Checksums matched!  %d == %d\n",csum,nextbyte[0]*256+nextbyte[1]);
    else
      return errorf("Bad checksum on incoming data packet.  Calculated checksum (%d) != received checksum (%d)",csum,nextbyte[0]*256+nextbyte[1]);

  }
  else
  {
    debugf(3,"Received zero-length response.\n");
  }
  
  if (msgsize)
    *msgsize = msglen + 8;
  return 0;
}

int
startalk_open(char *devname)
{
  int serialfd;
#ifdef OLD_SERIALSETUP
  struct termios newtio;
#endif
#ifdef STTY_SERIALSETUP
  extern char *stty;
#endif

  debugf(2,"Opening serial port...");
  serialfd = open(devname,O_RDWR|O_NOCTTY);
  /* Don't allow fd's less than 3.
   * These shouldn't happen normally, and can cause weird
   * problems if they do.
   */
  if (serialfd < 0)
    return reterr(-1,"Error opening serial port:");
  else if (serialfd < 3)
    return reterr(-1,"Serial port fd too low.");
  debugf(2,"Opened!\nSetting up tty...\n");

#ifdef MINICOM_SERIALSETUP
  m_dtrtoggle(serialfd,1);
  m_setparms(serialfd,"19200","N","8","1",1,0);
#else /* Must be STTY_SERIALSETUP or OLD_SERIALSETUP */
#ifdef OLD_SERIALSETUP
  bzero(&newtio, sizeof(newtio));
  if (tcgetattr(serialfd, &newtio) < 0)
    return reterr(-1,"Error getting serial port settings:");

  cfmakeraw(&newtio);
  newtio.c_cflag = newtio.c_cflag | CRTSCTS;

  /* Cause DTR to be dropped */
  cfsetospeed(&newtio,B0);
  cfsetispeed(&newtio,B0);

  if (tcsetattr(serialfd, TCSANOW, &newtio) < 0)
    return reterr(-1,"Error setting up serial port:");
  sleep(1);
#endif

  /* Send a break? */
  tcsendbreak(serialfd, 0);
  
#ifdef STTY_SERIALSETUP
  debugf(3,"Setting up terminal with stty %s\n",stty);
  if (fstty(serialfd,stty) < 0)
    return reterr(-1,"Error setting up terminal");
#else /* Must be OLD_SERIALSETUP */
  /* Set up the real serial speed. */
  cfsetospeed(&newtio,B19200);
  cfsetispeed(&newtio,B19200);
  if (tcsetattr(serialfd, TCSANOW, &newtio) < 0)
    return reterr(-1,"Error setting up serial port:");

  /* Flush */
#endif
  tcflush(serialfd, TCIOFLUSH);
#endif /* MINICOM_SERIALSETUP */

  debugf(2,"Set up!\n");
  if (debuglevel(3))
  {
    debugf(3,"--------Terminal Settings--------\n");
    if (fstty(serialfd,"-a 1>&2") < 0)
      debugf(3,"Warning: Unable to get terminal settings\n");
    debugf(3,"---------------------------------\n");
  }

  return serialfd;
}

void
startalk_prepfor_exit(void)
{
  sigset_t sst;
  
  debugf(3,"Exit prep via startalk_prepfor_exit...\n");
  clearerrors();

  /* Lower the timeout so we're not waiting forever to reset the
   * phone. */
  startalk_timeout = 3;

  /* If we get these signals while we're trying to unitialize,
   * just abort. */
  signal(SIGHUP,SIG_DFL);
  signal(SIGINT,SIG_DFL);
  signal(SIGQUIT,SIG_DFL);
  signal(SIGTERM,SIG_DFL);
  signal(SIGPIPE,SIG_DFL);
  signal(SIGSEGV,SIG_DFL);
  signal(SIGALRM,SIG_DFL);
  sigemptyset(&sst);
  sigaddset(&sst,SIGHUP);
  sigaddset(&sst,SIGINT);
  sigaddset(&sst,SIGQUIT);
  sigaddset(&sst,SIGTERM);
  sigaddset(&sst,SIGPIPE);
  sigaddset(&sst,SIGSEGV);
  sigaddset(&sst,SIGALRM);
  /* We can handle these signals a second time, now that we've
   * set them back to SIG_DFL. */
  sigprocmask(SIG_UNBLOCK, &sst, NULL);

  startalk_uninitialize();
  debugf(3,"Done prepping for exit in startalk_exit!\n");
}

void
startalk_uninitialize()
{
  debugf(3,"Uninitializing phone...\n");
  if (startalk_initialized && (startalk_fd != -1))
    startalk_reset(startalk_fd);
  debugf(3,"Uninitialized!\n");
}

int
startalk_initialize(int serialfd)
{
  extern char *modeminit;
  unsigned char c;
  
  debugf(2,"Initializing phone.\n");
  
  debugf(2,"First, send a data-mode reset to make sure we're not in data mode.\n");
  (void) startalk_reset(serialfd);
  clearerrors();

  sleep(1);
  debugf(2,"Sending \\r...");
  if ((startalk_write(serialfd,"\r",1) != 1))
    return errorf("Couldn't write CR to serial port:");
  debugf(2,"Sent!\n");

  sleep(1);
  debugf(2,"Sending ATZ\\r...");
  if ((startalk_write(serialfd,"ATZ\r",4) != 4))
    return errorf("Couldn't write ATZ to serial port:");
  debugf(2,"Sent!\n");

  /* Now just read until we don't see anything for awhile. */
  debugf(2,"Reading everything from buffer...\n");
  while (startalk_read_timeout(serialfd,(char *)&c,1,2) == 1)
    debugf(4,"Read character '%c' (%d or 0x%02x)\n",(c>=32 && c<=126)?c:'?',c,c);
  debugf(2,"Read everything!\n");
  clearerrors();

  sleep(1);
  debugf(2,"Sending \\r...");
  if ((startalk_write(serialfd,"\r",1) != 1))
    return errorf("Couldn't write CR to serial port:");
  debugf(2,"Sent!\n");

  sleep(1);
  debugf(2,"Sending init string '%s'...",modeminit);
  if ((startalk_write(serialfd,modeminit,strlen(modeminit)) != strlen(modeminit)) || (startalk_write(serialfd,"\r",1) != 1) )
    return errorf("Couldn't write modem init string '%s' to serial port:",modeminit);
  debugf(2,"Sent!\nWaiting for OK...");
  if (startalk_readfor(serialfd,"OK\r\n") != 0)
    return errorf("Protocol error connecting to phone (no OK after init string '%s' sent)",modeminit);
  debugf(2,"Got OK!\n");

  debugf(2,"Sending AT...");
  if (startalk_write(serialfd,"AT\r",3) != 3)
    return errorf("Couldn't write AT command to serial port:");
  debugf(2,"Waiting for OK...");
  if (startalk_readfor(serialfd,"OK\r\n") != 0)
    return errorf("Protocol error connecting to phone");
  debugf(2,"Got OK!\n");

  startalk_initialized = 1;
  startalk_fd = serialfd;
  atexit(startalk_uninitialize);

  return 0;
}


int
startalk_datamode(int serialfd)
{
  char buf[8192];

  debugf(2,"Entering data mode.\n");
  debugf(2,"Sending AT&F8...");
  if (startalk_write(serialfd,"AT&F8\r",7) != 7)
    return errorf("Couldn't write to serial port:");
  debugf(2,"Sent!\n");
  debugf(2,"Waiting for OK...");
  if (startalk_readfor(serialfd,"OK\r\n") != 0)
    return errorf("Protocol error putting phone in datamode");
  debugf(2,"Got OK!\n");

  /* From here on out, we're just winging it. */
  /* OK, some actual data to decode.  What is all this? */
  debugf(3,"Asking phone to enter data mode...\n");
  if (startalk_readmsg(serialfd,buf,8191,NULL) != 0)
    return errorf("error putting phone in data mode");
  debugf(3,"Phone entered data mode!\n");

  debugf(3,"datamode step 2...\n");
  if (startalk_writemsg(serialfd,"\x1F",1) < 0)
    return errorf("error putting phone in data mode");
  debugf(3,"finished step 2!\n");

  debugf(3,"datamode step 3...\n");
  if (startalk_readmsg(serialfd,buf,8191,NULL) != 0)
    return errorf("error putting phone in data mode");
  debugf(3,"finished step 3!\n");

  debugf(3,"datamode step 4...\n");
  if (startalk_writemsg(serialfd,"\x96",1) < 0)
    return errorf("error putting phone in data mode");
  debugf(3,"finished step 4!\n");

  debugf(3,"datamode step 5...\n");
  if (startalk_readmsg(serialfd,buf,8191,NULL) != 0)
    return errorf("error putting phone in data mode");
  debugf(3,"finished step 5!\n");

  debugf(3,"Data mode entered succesfully!\n");
  
  return 0;
}

int
startalk_reset(int serialfd)
{
  startalk_initialized=0;
  startalk_fd = -1;
  
  debugf(2,"Sending reset command...\n");
  if (startalk_writemsg(serialfd,"\x3F",1) != 0)
    return reterr(-1,"Couldn't read from serial port while resetting:");
  debugf(2,"Phone reset!\n");

  return 0;
}

int
startalk_dump_cmd(int serialfd, char *msgbuf, int size)
{
  char buf[8192];
  int respsize;

  printf("-----Command-----\n");
  dumpbuf(msgbuf,size);
  
  if (startalk_writemsg(serialfd,msgbuf,size) != 0)
    return errorf("sending command");
  if (startalk_readmsg(serialfd,buf,8192,&respsize) != 0)
    return errorf("reading response");

  printf("-----Response-----\n");
  dumpbuf(buf,respsize);

  return 0;
}

int
startalk_run_cmd(int serialfd, char *msgbuf, int size, char *resbuf, int *ressize)
{
  if (startalk_writemsg(serialfd,msgbuf,size) != 0)
    return errorf("running command");
  if (startalk_readmsg(serialfd,resbuf,*ressize,ressize) != 0)
    return errorf("running command");

  return 0;
}
