
/* $Id: stdebug.c,v 400.1 2002/07/25 08:43:16 sgifford Exp $ */

#include "stdebug.h"

#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>

static int cur_debug_level = 0;

static char *err[100];
static int numerr = 0;

#define DUMPBUF_COLS 6
static int dumppos;
static char ascbuf[DUMPBUF_COLS+1];
static int warnings = 0;

int
dumpbuf(const char *buf, int size)
{
  dumpbuf_start(buf, size);
  return dumpbuf_end();
}

int
dumpbuf_cont(const char *buf, int size)
{
  int i;
  const unsigned char *ubuf;
  unsigned char ic;

  ubuf = (unsigned char *)buf;
  /* Do a nice, attractive dump. */
  for (i=0;i<size;i++)
  {
    if ( (dumppos%DUMPBUF_COLS) == 0)
    {
      fprintf(stderr,"%5d: ",dumppos);
    }
    
    ic=(unsigned char)ubuf[i];
    fprintf(stderr,"%02X ",ic);

    if ( (ic >= '!') && (ic <= '~') )
    {
      ascbuf[dumppos%DUMPBUF_COLS] = ic;
    }
    else
    {
      ascbuf[dumppos%DUMPBUF_COLS]=' ';
    }
    if ( (dumppos%DUMPBUF_COLS) == (DUMPBUF_COLS-1))
    {
      fprintf(stderr,"| %s\n",ascbuf);
    }
    dumppos++;
  }

  return 0;
}

int
dumpbuf_start(const char *buf, int size)
{
  dumppos = 0;
  ascbuf[DUMPBUF_COLS] = '\0';
  if (buf)
    dumpbuf_cont(buf, size);

  return 0;
}

int
dumpbuf_end(void)
{
  int i;
  if (dumppos%DUMPBUF_COLS)
  {
    ascbuf[dumppos%DUMPBUF_COLS] = '\0';
    for(i=(dumppos%DUMPBUF_COLS);i<=DUMPBUF_COLS-1;i++)
    {
      fprintf(stderr,"   ");
    }
    fprintf(stderr,"| %s\n",ascbuf);
  }
  dumppos = 0;

  return 0;
}

int
setdebuglevel(int newlevel)
{
  int oldlevel;

  oldlevel = cur_debug_level;
  if (newlevel != -1)
  {
    cur_debug_level = newlevel;
  }
  return oldlevel;
}

int
debuglevel(int testlevel)
{
  return (cur_debug_level >= testlevel);
}

int
debugf(int minlevel, const char *format, ...)
{
  va_list a;
  int n;
  
  if (debuglevel(minlevel))
  {
    va_start(a,format);
    n=vfprintf(stderr,format,a);
    fflush(stderr);
    va_end(a);
    return n;
  }
  else
    return 0;
}

void warnordie(const char *note, const char *format, va_list a);

void
die(const char *format, ...)
{
  va_list a;

  va_start(a, format);
  warnordie("",format,a);
  va_end(a);
  exit(1);
}

void
warn(const char *format, ...)
{
  va_list a;

  va_start(a, format);
  warnordie("WARNING: ",format,a);
  va_end(a);
  warnings++;
  clearerrors();
}

void
finished(void)
{
  if (warnings > 0)
    exit(126);
  else
    exit(0);
}

void
errdump(const char *format, ...)
{
  va_list a;

  va_start(a, format);
  warnordie("",format,a);
  va_end(a);
}

void
warnordie(const char *note, const char *format, va_list a)
{
  int i;

  if (numerr)
  {
    fprintf(stderr,"%s%s\n",note,err[0]);
    for (i=1;i<numerr;i++)
    {
      fprintf(stderr,"\twhile %s\n",err[i]);
    }
    fprintf(stderr,"\twhile ");
  }
    
  if (format)
  {
    vfprintf(stderr, format, a);
    fprintf(stderr,"\n");
  }
}


int
errorf(const char *format, ...)
{
  va_list a;
  int n;
  
  va_start(a, format);
  n=vreterr(-1,format,a);
  va_end(a);
  return n;
}

int
vreterr(int ret, const char *format, va_list a)
{
  int l;

  err[numerr] = malloc(8192);
  vsnprintf(err[numerr], 8191, format, a);

  /* If it ends with a :, tack strerror(errno)) on the end. */
  if ( ((l=strlen(err[numerr])) > 0)
       && (err[numerr][l-1] == ':')
       && (l < 8190) )
  {
    err[numerr][l] = ' ';
    err[numerr][l+1] = '\0';
    strncat(err[numerr], strerror(errno), 8191 - l);
  }
  err[numerr][8191] = 0;
  numerr++;

  return ret;
}
  
int
reterr(int ret, const char *format, ...)
{
  va_list a;
  
  va_start(a, format);
  vreterr(ret, format, a);  
  va_end(a); 
  return ret;
}

void
clearerrors(void)
{
  while (numerr > 0)
  {
    free(err[numerr-1]);
    numerr--;
  }
}
