# -*- coding: utf-8 -*-
#   StartUp Manager - Tool for editing settings for Grub and Usplash
#   Copyright (C) 2006-2007  Jimmy Rönnholm
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

import sys
import os
import shutil
import re
import time
from threading import Thread
import locale
import gettext

import gnome.ui
import gobject
try:
    import pygtk
    pygtk.require("2.0")
except:
    pass
try:
    import gtk
    import gtk.glade
except:
    sys.exit(1)

import startupmanager
from startupmanager.base import Grub, Usplash

APP = 'startupmanager'
DIR = '/usr/share/locale'

try:
    locale.setlocale(locale.LC_ALL, '')
except:
    print "Could not set locale, falling back to standard locale"
gettext.bindtextdomain(APP, DIR)
gettext.textdomain(APP)
_ = gettext.gettext

class Config:
    """Contain the data we need to the widgets."""
    glade_dir = 'startupmanager.glade'
    version = '1.0'

def convert_resolution(number):
    resolution = number[0]
    if resolution == "0":
        xres = "640"
        yres = "480"
    elif resolution == "1":
        xres = "800"
        yres = "600"
    elif resolution == "2":
        xres = "1024"
        yres = "768"
    elif resolution == "3":
        xres = "1280"
        yres = "1024"
    else:
        xres = "1600"
        yres = "1200"
    return xres, yres

def convert_color(color, wanted_type, revert):
    """Convert color name to combo box index."""
    if wanted_type == 0:
        color_dict = {
        "black" : 0 ,
        "blue" : 1 ,
        "green" : 2 ,
        "cyan" : 3 ,
        "red" : 4 ,
        "magenta" : 5 ,
        "brown" : 6 ,
        "light-gray" : 7
        }
    if wanted_type == 1:
        color_dict = {
        "black" : 0 ,
        "blue" : 1 ,
        "green" : 2 ,
        "cyan" : 3 ,
        "red" : 4 ,
        "magenta" : 5 ,
        "brown" : 6 ,
        "light-gray" : 7 ,
        "dark-gray" : 8 ,
        "light-blue" : 9 ,
        "light-green" : 10 ,
        "light-cyan" : 11 ,
        "light-red" : 12 ,
        "light-magenta" : 13 ,
        "yellow" : 14 ,
        "white" : 15
        }
    if revert:
        reverse_color_dict = dict((v,k) for (k,v) in color_dict.iteritems())
        return reverse_color_dict[color]
    else:
        return color_dict[color]

def convert_vga(vga, wanted, revert=False):
    """Convert vga number to combo box index."""
    dic = {
    769 : "00" ,
    771 : "10" ,
    773 : "20" ,
    775 : "30" ,
    796 : "40" ,
    758 : "01" ,
    788 : "11" ,
    791 : "21" ,
    794 : "31" ,
    798 : "41" ,
    786 : "02" ,
    789 : "12" ,
    792 : "22" ,
    795 : "32" ,
    799 : "42"
    }
    if revert:
        reverse_dic = dict((v,k) for (k,v) in dic.iteritems())
        return reverse_dic[vga]
    if dic.has_key(vga):
        settings = dic[vga]
    else:
        settings = dic[769]
    return int(settings[wanted])

class StartupThread(Thread):
    def __init__ (self):
        Thread.__init__(self)

    def run(self):
        self.grub = Grub()

class ShutdownThread(Thread):
    def __init__ (self, grub, usplash=None):
        Thread.__init__(self)
        self.grub = grub
        self.usplash = usplash

    def run(self):
        if self.usplash:
            self.usplash.do_shutdown_tasks()
        self.grub.do_shutdown_tasks()

class FloppyThread(Thread):
    def __init__ (self, grub, action):
        Thread.__init__(self)
        self.grub = grub
        self.action = action
        self.ret = None

    def run(self):
        if self.action == "format":
            self.ret = self.grub.format_floppy()
        if self.action == "write":
            self.ret = self.grub.make_floppy()

class SumGui:

    file_path = os.path.split(startupmanager.__file__)[0]

    def on_default_boot_combo_changed(self, widget):
        active = self.default_boot_combo.get_active()
        self.grub.set_default_boot(active)

    def on_timeout_check_toggled(self, widget):
        active = self.timeout_check.get_active()
        timeout = self.timeout_spinner.get_value_as_int()
        self.grub.set_timeout(active, timeout)

    def on_timeout_spinner_value_changed(self, widget):
        active = self.timeout_check.get_active()
        timeout = self.timeout_spinner.get_value_as_int()
        self.grub.set_timeout(active, timeout)

    def on_resolution_combo_changed(self, widget):
        number = str(self.resolution_combo.get_active()) + \
                 str(self.color_depth_combo.get_active())
        if not self.limited_grub:
            self.grub.set_resolution(convert_vga(number, 1, True))
        if self.use_usplash:
            self.usplash.set_resolution(convert_resolution(number))

    def on_color_depth_combo_changed(self, widget):
        number = str(self.resolution_combo.get_active()) + \
                 str(self.color_depth_combo.get_active())
        self.grub.set_resolution(convert_vga(number, 1, True))

    def on_boot_text_check_toggled(self, widget):
        active = self.boot_text_check.get_active()
        self.grub.set_boot_text_visible(active)

    def on_grub_menu_check_toggled(self, widget):
        active = self.grub_menu_check.get_active()
        self.grub.set_menu_visible(active)

    def on_grub_color_check_toggled(self, widget):
        active = self.grub_color_check.get_active()
        self.grub.set_use_colors(active)

    def on_normal_background_combo_changed(self, widget):
        color = convert_color(self.normal_background_combo.get_active(), 
                              0, True)
        self.grub.set_color_normal_bg(color)

    def on_highlighted_background_combo_changed(self, widget):
        color = convert_color(self.highlighted_background_combo.get_active(), 
                              0, True)
        self.grub.set_color_highlighted_bg(color)

    def on_normal_foreground_combo_changed(self, widget):
        color = convert_color(self.normal_foreground_combo.get_active(), 
                              1, True)
        self.grub.set_color_normal_fg(color)

    def on_highlighted_foreground_combo_changed(self, widget):
        color = convert_color(self.highlighted_foreground_combo.get_active(), 
                              1, True)
        self.grub.set_color_highlighted_fg(color)

    def on_normal_blink_check_toggled(self, widget):
        active = self.normal_blink_check.get_active()
        self.grub.set_color_blink_normal(active)

    def on_highlighted_blink_check_toggled(self, widget):
        active = self.highlighted_blink_check.get_active()
        self.grub.set_color_blink_highlight(active)

    def on_grub_splash_check_toggled(self, widget):
        if self.grub_splash_combo.get_active() != -1:
            image = self.grub_splash_combo.get_active_text()
            active = self.grub_splash_check.get_active()
            self.grub.set_splash(active, image)
        else:
            self.grub_splash_check.set_active(False)

    def on_grub_splash_combo_changed(self, widget):
        active = self.grub_splash_check.get_active()
        image = self.grub_splash_combo.get_active_text()
        self.grub.set_splash(active, image)

    def on_grub_themes_button_clicked(self, widget):
        self.grub_dialog.show()

    def on_add_grub_button_clicked(self, widget):
        chooser = gtk.FileChooserDialog(title=None, parent=self.grub_dialog, 
                                        action=gtk.FILE_CHOOSER_ACTION_OPEN,
                                        buttons=(gtk.STOCK_CANCEL,
                                                 gtk.RESPONSE_CANCEL,
                                                 gtk.STOCK_OPEN,
                                                 gtk.RESPONSE_OK), 
                                        backend=None)
        chooser.set_default_response(gtk.RESPONSE_OK)
        chooser.set_current_folder(os.environ.get('HOME'))
        if os.environ.get('HOME') != '/root':
            chooser.add_shortcut_folder(os.environ.get('HOME'))

        filter = gtk.FileFilter()
        filter.set_name(_("Common formats for Grub background images"))
        filter.add_mime_type("image/png")
        filter.add_mime_type("image/jpeg")
        filter.add_mime_type("image/gif")
        filter.add_mime_type("bitmap/bmp")
        for format in self.grub.get_image_formats():
            filter.add_pattern("*." + format)
        chooser.add_filter(filter)
        filter = gtk.FileFilter()
        filter.set_name(_("All files"))
        filter.add_pattern("*")
        chooser.add_filter(filter)
        response = chooser.run()
        if response == gtk.RESPONSE_OK:
            filename = chooser.get_filename()
            if not self.grub.add_image(filename):
                dialog = gtk.MessageDialog(parent=self.main_window, 
                                           flags=gtk.DIALOG_MODAL,
                                           type=gtk.MESSAGE_ERROR, 
                                           buttons=gtk.BUTTONS_OK,
                                           message_format=_("Could not "
                                                            "add image"))
                dialog.run()
                dialog.destroy()
            self.refresh_grub_themes()
        chooser.destroy()

    def on_remove_grub_button_clicked(self, widget):
        active = self.grub_splash_combo.get_active_text()
        selected = self.grub_treeview.get_selection().get_selected()
        model = selected[0]
        iter = selected[1]
        name = model.get_value(iter, 0)
        if active == name:
            self.grub_splash_check.set_active(False)
        self.grub.remove_image(name)
        self.refresh_grub_themes()

    def on_grub_file_chooser_update_preview(self, widget):
        filename = self.grub_file_chooser.get_filename()
        self.grub.add_image(filename)

    def on_usplash_check_toggled(self, widget):
        active = self.usplash_check.get_active()
        self.grub.set_splash_active(active)

    def on_usplash_combo_changed(self, widget):
        active = self.usplash_combo.get_active_text()
        self.usplash.set_active_theme(active)

    def on_usplash_themes_button_clicked(self, widget):
        self.usplash_dialog.show()

    def on_add_usplash_button_clicked(self, widget):
        chooser = gtk.FileChooserDialog(title=None, 
                                        parent=self.usplash_dialog, 
                                        action=gtk.FILE_CHOOSER_ACTION_OPEN,
                                        buttons=(gtk.STOCK_CANCEL,
                                                 gtk.RESPONSE_CANCEL,
                                                 gtk.STOCK_OPEN,
                                                 gtk.RESPONSE_OK), 
                                        backend=None)
        chooser.set_default_response(gtk.RESPONSE_OK)
        chooser.set_current_folder(os.environ.get('HOME'))
        if os.environ.get('HOME') != '/root':
            chooser.add_shortcut_folder(os.environ.get('HOME'))

        filter = gtk.FileFilter()
        filter.set_name(_("Usplash theme files(*.so)"))
        filter.add_pattern("*.so")
        chooser.add_filter(filter)
        response = chooser.run()
        if response == gtk.RESPONSE_OK:
            filename = chooser.get_filename()
            self.usplash.add_theme(filename)
            self.refresh_usplash_themes()
        chooser.destroy()

    def on_remove_usplash_button_clicked(self, widget):
        active = self.usplash_combo.get_active_text()
        selected = self.usplash_treeview.get_selection().get_selected()
        model = selected[0]
        iter = selected[1]
        name = model.get_value(iter, 0)
        if active == name:
            dialog = gtk.MessageDialog(parent=self.main_window, 
                                       flags=gtk.DIALOG_MODAL, 
                                       type=gtk.MESSAGE_QUESTION,
                                       buttons=gtk.BUTTONS_YES_NO,
                                       message_format=_("You are removing "
                                                        "the currently "
                                                        "active usplash "
                                                        "theme. Do you "
                                                        "really want to "
                                                        "do that?"))
            dialog.set_default_response(gtk.RESPONSE_NO)
            response = dialog.run()
            dialog.destroy()
            if response == gtk.RESPONSE_NO:
                return
            elif response == gtk.RESPONSE_YES:
                dialog2 = gtk.MessageDialog(parent=self.main_window, 
                                            flags=gtk.DIALOG_MODAL, 
                                            type=gtk.MESSAGE_INFO, 
                                            buttons=gtk.BUTTONS_OK,
                                            message_format=_("Make sure "
                                                             "to select a "
                                                             "new theme "
                                                             "if you want "
                                                             "usplash to "
                                                             "function "
                                                             "properly."))
                dialog2.run()
                dialog2.destroy()
            else:
                return
        self.usplash.remove_theme(name)
        self.refresh_usplash_themes()

    def on_password_protect_check_toggled(self, widget):
        active = self.password_protect_check.get_active()
        self.grub.set_password_protection(active)

    def on_protect_rescuemode_check_toggled(self, widget):
        active = self.protect_rescuemode_check.get_active()
        self.grub.set_protect_rescuemode(active)

    def on_protect_old_check_toggled(self, widget):
        active = self.protect_old_check.get_active()
        self.grub.set_protect_oldmode(active)

    def on_update_password_button_clicked(self, widget):
        password = self.password_entry.get_text()
        active = self.password_protect_check.get_active()
        if password == self.confirm_password_entry.get_text() and \
           len(password) > 3:
            try:
                self.grub.update_password(password, active)
                self.password_notify_label.set_text(_('Password changed'))
            except:
                self.password_notify_label.set_text(_('Error while '
                                                      'changing passwords.'))

        else:
            if len(password) < 4:
                self.password_notify_label.set_text(_('Password must be '
                                                      'at least 4 characters'))
            else:
                self.password_notify_label.set_text(_('Passwords do '
                                                      'not match'))

    def on_kernel_check_toggled(self, widget):
        active = self.kernel_check.get_active()
        number = self.kernel_spinner.get_value_as_int()
        self.grub.set_kernel_limit(active, number)

    def on_kernel_spinner_value_changed(self, widget):
        active = self.kernel_check.get_active()
        number = self.kernel_spinner.get_value_as_int()
        self.grub.set_kernel_limit(active, number)

    def on_memtest_check_toggled(self, widget):
        active = self.memtest_check.get_active()
        self.grub.set_memtest(active)

    def on_recovery_check_toggled(self, widget):
        active = self.recovery_check.get_active()
        self.grub.set_create_alternative(active)

    def on_updatedefault_check_toggled(self, widget):
        active = self.updatedefault_check.get_active()
        self.grub.set_update_default(active)

    def on_rescuefloppy_button_clicked(self, widget):
        ret = 1
        dialog = gtk.MessageDialog(parent=self.main_window, 
                                   flags=gtk.DIALOG_MODAL, 
                                   type=gtk.MESSAGE_INFO,\
                                   buttons=gtk.BUTTONS_OK_CANCEL,\
                                   message_format=_("Insert a floppy. "
                                                    "Note that everything "
                                                    "on the floppy will "
                                                    "be deleted."))
        dialog.set_default_response(gtk.RESPONSE_OK)
        response = dialog.run()
        dialog.destroy()
        if not response == gtk.RESPONSE_OK:
            return
        else:
            while not ret == 0:
                dialog = gtk.Dialog(title="", parent=self.main_window, 
                                    flags=gtk.DIALOG_MODAL)
                progress=gtk.ProgressBar()
                progress.set_text(_("Formatting floppy"))
                dialog.vbox.add(progress)
                progress.show()
                dialog.show()
                thread = FloppyThread(self.grub, "format")
                thread.start()
                while(1):
                    while(gtk.events_pending()):
                        gtk.main_iteration()
                    if not thread.isAlive():
                        break
                    else:
                        progress.pulse()
                        time.sleep(0.1)
                dialog.destroy()
                ret = thread.ret
                if ret == 1:
                    dialog = gtk.MessageDialog(parent=self.main_window,
                                               flags=gtk.DIALOG_MODAL, 
                                               type=gtk.MESSAGE_QUESTION,
                                               buttons=gtk.BUTTONS_YES_NO,
                                               message_format=_("Floppy not "
                                                                "found. "
                                                                "Try again?"))
                    dialog.set_default_response(gtk.RESPONSE_YES)
                    response = dialog.run()
                    dialog.destroy()
                    if not response == gtk.RESPONSE_YES:
                        ret = 2
                if ret == 2:
                    return
                elif ret >= 3 or ret < 0:
                    print ret
                    dialog = gtk.MessageDialog(parent=self.main_window, 
                                               flags=gtk.DIALOG_MODAL, 
                                               type=gtk.MESSAGE_INFO,
                                               buttons=gtk.BUTTONS_OK,
                                               message_format=_("Error."))
                    dialog.run()
                    dialog.destroy()
                    return
        dialog = gtk.Dialog(title="", parent=self.main_window, 
                            flags=gtk.DIALOG_MODAL)
        progress = gtk.ProgressBar()
        progress.set_text(_("Creating rescue floppy"))
        dialog.vbox.add(progress)
        progress.show()
        dialog.show()
        thread = FloppyThread(self.grub, "write")
        thread.start()
        while(1):
            while(gtk.events_pending()):
                gtk.main_iteration()
            if not thread.isAlive():
                break
            else:
                progress.pulse()
                time.sleep(0.1)
        dialog.destroy()
        ret = thread.ret
        if ret == 0:
            dialog = gtk.MessageDialog(parent=self.main_window, 
                                       flags=gtk.DIALOG_MODAL, 
                                       type=gtk.MESSAGE_INFO,
                                       buttons=gtk.BUTTONS_OK,
                                       message_format=_("Rescue floppy created"
                                                        " successfully."))
            dialog.run()
            dialog.destroy()
        else:
            dialog = gtk.MessageDialog(parent=self.main_window, 
                                       flags=gtk.DIALOG_MODAL, 
                                       type=gtk.MESSAGE_INFO,
                                       buttons=gtk.BUTTONS_OK,
                                       message_format=_("Error."))
            dialog.run()
            dialog.destroy()

    def on_restore_settings_button_clicked(self, widget):
        if self.grub.restore_config():
            dialog = gtk.MessageDialog(parent=self.main_window, 
                                       flags=gtk.DIALOG_MODAL, 
                                       type=gtk.MESSAGE_INFO, 
                                       buttons=gtk.BUTTONS_OK,
                                       message_format=_("Original settings "
                                                        "restored.\n"
                                                        "Shutting down"))
            dialog.run()
            dialog.destroy()
            raise SystemExit('User requested shutdown.')
        else:
            dialog = gtk.MessageDialog(parent=self.main_window, 
                                       flags=gtk.DIALOG_MODAL, 
                                       type=gtk.MESSAGE_ERROR, 
                                       buttons=gtk.BUTTONS_OK,
                                       message_format=_("Failed to "
                                                        "restore config"))
            dialog.run()
            dialog.destroy()

    def on_help_button_clicked(self, widget):
        gnome.help_display('startupmanager', '')

    def on_grub_help_button_clicked(self, widget):
        gnome.help_display('startupmanager', 'grub')

    def on_usplash_help_button_clicked(self, widget):
        gnome.help_display('startupmanager', 'usplash')

    def on_grub_close_button_clicked(self, widget):
        self.grub_dialog.hide()

    def on_usplash_close_button_clicked(self, widget):
        self.usplash_dialog.hide()

    def on_close_button_clicked(self, widget):
        self.on_main_window_destroy(widget)

    def on_main_window_destroy(self, widget):
        if self.grub.update_grub or (self.use_usplash and 
                                     self.usplash.update_initramfs):
            self.main_window.hide()
            dialog = gtk.Dialog(title="", parent=self.main_window, 
                                flags=gtk.DIALOG_MODAL)
            progress = gtk.ProgressBar()
            progress.set_text(_("Performing post-configuration tasks"))
            dialog.vbox.add(progress)
            progress.show()
            dialog.show()
            if self.use_usplash:
                ramfs = ShutdownThread(self.grub, self.usplash)
            else:
                ramfs = ShutdownThread(self.grub)
            ramfs.start()

            while(1):
                while(gtk.events_pending()):
                    gtk.main_iteration()
                if not ramfs.isAlive():
                    break
                else:
                    progress.pulse()
                    time.sleep(0.1)
            dialog.destroy()
        gtk.main_quit()

    def refresh_grub_themes(self):
        self.grub_splash_combo.get_model().clear()
        self.grub_store.clear()
        grub_images = self.grub.get_images()
        for image in grub_images:
            self.grub_splash_combo.append_text(image)
            self.grub_store.append([image])

        default_image = self.grub.get_splash()
        if default_image != "":
            place = 0
            for grub_file in grub_images:
                if grub_file == default_image:
                    self.grub_splash_combo.set_active(place)
                place += 1

    def refresh_usplash_themes(self):
        self.usplash_combo.get_model().clear()
        self.usplash_store.clear()
        usplash_themes = self.usplash.get_all_themes()
        for theme in usplash_themes:
            self.usplash_combo.append_text(theme)
            self.usplash_store.append([theme])
        default_usplash = self.usplash.get_active_theme()
        if default_usplash != "":
            place = 0
            for usplash_theme in usplash_themes:
                if usplash_theme == default_usplash:
                    self.usplash_combo.set_active(place)
                place += 1

    def update_widgets(self):
            titles = self.grub.get_titles()
            for title in titles:
                if title != "Other operating systems:":
                    self.default_boot_combo.append_text(title)

            self.default_boot_combo.append_text(_("Last used"))
            default_boot = self.grub.get_default_boot()
            if default_boot != -1:
                self.default_boot_combo.set_active(default_boot)
            else:
                number = self.default_boot_combo.get_model()\
                         .iter_n_children(None)
                self.default_boot_combo.set_active(number - 1)
            self.timeout_check.set_active(self.grub.get_timeout_used())
            self.timeout_spinner.set_value(self.grub.get_timeout())
            self.resolution_combo.set_active(convert_vga(self.grub.get_vga(), 
                                                         0))
            self.color_depth_combo.set_active(convert_vga(self.grub.get_vga(), 
                                                          1))
            self.boot_text_check.set_active(self.grub.get_boot_text_visible())
            self.grub_menu_check.set_active(self.grub.get_menu_visible())
            self.grub_color_check.set_active(self.grub.get_use_colors())
            self.normal_background_combo.set_active(convert_color(self.grub.
                                                    get_color_normal_bg(),
                                                    0, False))
            self.highlighted_background_combo.\
                 set_active(convert_color(self.grub.get_color_highlighted_bg(),
                                          0, False))
            self.normal_foreground_combo.\
                 set_active(convert_color(self.grub.get_color_normal_fg(), 
                                          1, False))
            self.highlighted_foreground_combo.\
                 set_active(convert_color(self.grub.get_color_highlighted_fg(),
                                          1, False))
            self.normal_blink_check.set_active(self.grub.
                                               get_color_blink_normal())
            self.highlighted_blink_check.\
                 set_active(self.grub.get_color_blink_highlight())
            self.grub_splash_check.set_active(self.grub.get_splash_visible())
            self.usplash_check.set_active(self.grub.get_splash_active())

            self.refresh_grub_themes()

            self.password_protect_check.\
                 set_active(self.grub.get_password_protection())
            self.protect_rescuemode_check.\
                 set_active(self.grub.get_protect_rescuemode())
            self.protect_old_check.set_active(self.grub.get_protect_oldmode())
            self.password_notify_label.set_text("")

            self.kernel_check.set_active(self.grub.get_limit_kernel())
            self.kernel_spinner.set_value(1)
            if type(self.grub.get_kernel_limit()) == type(1):
                self.kernel_spinner.set_value(self.grub.get_kernel_limit())

            self.memtest_check.set_active(self.grub.get_memtest())
            self.recovery_check.set_active(self.grub.get_create_alternative())
            self.updatedefault_check.set_active(self.grub.get_update_default())

    def __init__(self):

        xml = gtk.glade.XML(os.path.join(self.file_path, 
                                         'startupmanager.glade'), None ,APP)

        self.main_window = xml.get_widget("main_window")

        dialog = gtk.Dialog(title="", parent=self.main_window, 
                            flags=gtk.DIALOG_MODAL)
        progress = gtk.ProgressBar()
        progress.set_text(_("Performing pre-configuration tasks"))
        dialog.vbox.add(progress)
        progress.show()
        dialog.show()
        thread = StartupThread()
        thread.start()
        while(1):
            while(gtk.events_pending()):
                gtk.main_iteration()
            if not thread.isAlive():
                break
            else:
                progress.pulse()
                time.sleep(0.1)
        dialog.destroy()
        self.grub = thread.grub
        self.default_boot_combo = xml.get_widget("default_boot_combo")
        self.default_boot_combo.get_model().clear()
        self.timeout_check = xml.get_widget("timeout_check")
        self.timeout_spinner = xml.get_widget("timeout_spinner")
        self.resolution_combo = xml.get_widget("resolution_combo")
        self.color_depth_combo = xml.get_widget("color_depth_combo")
        self.boot_text_check = xml.get_widget("boot_text_check")
        self.grub_menu_check = xml.get_widget("grub_menu_check")
        self.grub_color_check = xml.get_widget("grub_color_check")
        self.normal_background_combo = xml.get_widget\
                                       ("normal_background_combo")
        self.highlighted_background_combo = xml.get_widget\
                                            ("highlighted_background_combo")
        self.normal_foreground_combo = xml.get_widget\
                                       ("normal_foreground_combo")
        self.highlighted_foreground_combo = xml.get_widget\
                                            ("highlighted_foreground_combo")
        self.normal_blink_check = xml.get_widget("normal_blink_check")
        self.highlighted_blink_check = xml.get_widget\
                                       ("highlighted_blink_check")
        self.grub_splash_check = xml.get_widget("grub_splash_check")
        self.grub_splash_combo = xml.get_widget("grub_splash_combo")
        self.usplash_check = xml.get_widget("usplash_check")
        self.usplash_combo = xml.get_widget("usplash_combo")
        self.password_protect_check = xml.get_widget("password_protect_check")
        self.protect_rescuemode_check = xml.get_widget\
                                        ("protect_rescuemode_check")
        self.protect_old_check = xml.get_widget("protect_old_check")
        self.password_entry = xml.get_widget("password_entry")
        self.confirm_password_entry = xml.get_widget("confirm_password_entry")
        self.password_notify_label = xml.get_widget("password_notify_label")
        self.kernel_check = xml.get_widget("kernel_check")
        self.kernel_spinner = xml.get_widget("kernel_spinner")
        self.memtest_check = xml.get_widget("memtest_check")
        self.recovery_check = xml.get_widget("recovery_check")
        self.updatedefault_check = xml.get_widget("updatedefault_check")
        self.grub_dialog = xml.get_widget("grub_dialog")
        self.grub_store = gtk.ListStore(str)
        self.grub_treeview = xml.get_widget("grub_treeview")
        self.grub_treeview.append_column(gtk.TreeViewColumn(_("Image"), 
                                         gtk.CellRendererText(), text=0))
        self.grub_treeview.set_model(self.grub_store)
        self.usplash_dialog = xml.get_widget("usplash_dialog")
        self.usplash_store = gtk.ListStore(str)
        self.usplash_treeview = xml.get_widget("usplash_treeview")
        self.usplash_treeview.append_column(gtk.TreeViewColumn(_("Theme"), 
                                            gtk.CellRendererText(), text=0))
        self.usplash_treeview.set_model(self.usplash_store)

        props = { gnome.PARAM_APP_DATADIR : '/usr/share'}
        self.gnome_program = gnome.program_init('startupmanager', '1.0', 
                                                properties=props)

        self.limited_grub = False
        if not self.grub.has_update_grub():
            print "No update-grub"
            self.limited_grub = True
            xml.get_widget("color_depth_label").hide()
            xml.get_widget("color_depth_combo").hide()
            xml.get_widget("usplash_check").hide()
            xml.get_widget("boot_text_check").hide()
            xml.get_widget("frame5").hide()
            xml.get_widget("frame6").hide()
            xml.get_widget("protect_old_check").hide()
            xml.get_widget("protect_rescuemode_check").hide()
            xml.get_widget("vbox11").hide()


        if self.grub.get_user_edited_file():
            dialog = gtk.MessageDialog(parent=self.main_window, 
                                       flags=gtk.DIALOG_MODAL, 
                                       type=gtk.MESSAGE_QUESTION,
                                       buttons=gtk.BUTTONS_YES_NO,
                                       message_format=_("Your grub "
                                       "configuration file lack the line "
                                       "'### END DEBIAN AUTOMAGIC KERNELS "
                                       "LIST'.\nYou have probably "
                                       "hand-edited the configuration "
                                       "file.\n\nChoose 'yes' to have the "
                                       "file auto-generated. This means "
                                       "that any custom boot options will "
                                       "be lost.\nIf you only have one "
                                       "operating system installed and you "
                                       "do not know what I am talking "
                                       "about, it should be safe to answer "
                                       "'yes'.\n\nOtherwise, choose 'no'. "
                                       "That will stop this program.\nIt is "
                                       "however recommended to look into "
                                       "this, since system upgrades could "
                                       "also rewrite the configuration file."))
            dialog.set_default_response(gtk.RESPONSE_NO)
            response = dialog.run()
            dialog.destroy()
            if response == gtk.RESPONSE_NO:
                self.grub.restore_config()
                raise SystemExit('User requested shutdown.')
            elif response == gtk.RESPONSE_YES:
                pass
            else:
                self.grub.restore_config()
                raise SystemExit('User requested shutdown.')

        self.update_widgets()

        self.use_usplash = True
        try:
            self.usplash = Usplash()
        except:
            self.use_usplash = False

        if self.use_usplash:
            self.refresh_usplash_themes()
        else:
            if self.limited_grub:
                xml.get_widget("vbox5").hide()
            else:
                xml.get_widget("usplash_frame").hide()

        xml.signal_autoconnect(self)
        gobject.threads_init()
        gtk.main()
