/*
 * StatsLog v2.0.x
 * Copyright (c) 2002 Chris Mason
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Debian system; see the file /usr/doc/copyright/GPL. If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite
 * 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdarg.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <signal.h>
#include <time.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <sys/types.h>
#include <sys/socket.h>

#include "statslog.h"
#include "logformat.h"

static int sCount;
static char *logFile;
static ServerInfo *sInfo;

int main(int argc, char *argv[]) {
  char buffer[BUFFER_SIZE];
  struct timeval tv;
  fd_set fdset;
  fd_set wdset;
  int bytes;
  int i;

  show_version();
  check_argc(argc);

  if(argc > 2) {
    logFile = argv[2];
  }

  parse_config(argv[1]);
  if(sanity_check() != csSuccess) {
    fprintf(stdout, "Invalid Configuration\n");
    free_memory();
    return(csError);
  }

  setup_signals();
  fprintf(stdout, "Resolving Hosts...\n");
  if(resolve_servers() != csSuccess) {
    free_memory();
    return(csError);
  }
  fprintf(stdout, "---------------------------------------------------------\n");

  LOG(__LINE__, SL_LOG_VERSION, timestamp(STANDARD_TS_FORMAT), APPNAME, VERSION);

#ifndef DEBUG
  daemonize();
#endif

  connect_to_servers();

  for(;;) {
    FD_ZERO(&fdset);
    FD_ZERO(&wdset);

    tv.tv_sec = 1;
    tv.tv_usec = 0;

    if(select(select_fd_sets(&fdset, &wdset) + 1, &fdset, &wdset, NULL, &tv) > 0) {
      for(i = 0; i < sCount; i++) {
        if(sInfo[i].status == csConnecting) {
          if(FD_ISSET(sInfo[i].sockfd, &wdset)) {
            if(recv(sInfo[i].sockfd, NULL, 0, MSG_PEEK) == csError) {
              if(errno != EAGAIN) {
                close(sInfo[i].sockfd);
                LOG(__LINE__, SL_LOG_ERROR_CONNECTING, timestamp(STANDARD_TS_FORMAT), gethost(sInfo[i].addrinfo), getport(sInfo[i].addrinfo), strerror(errno));
                sInfo[i].timer = time(NULL);
                sInfo[i].status = csThrottled;
              }
            }
            else {
              fcntl(sInfo[i].sockfd, F_SETFL, 0);
              LOG(__LINE__, SL_LOG_CONNECTED, timestamp(STANDARD_TS_FORMAT), gethost(sInfo[i].addrinfo), getport(sInfo[i].addrinfo));
              sInfo[i].status = csConnected;
              send_login_sequence(i);
            }
          }
        }
        else if((sInfo[i].status == csConnected) || (sInfo[i].status == csAuthenticated)) {
          if(FD_ISSET(sInfo[i].sockfd, &fdset)) {
            if((bytes = recv(sInfo[i].sockfd, buffer, sizeof(buffer), 0)) < 1) {
              close(sInfo[i].sockfd);
              LOG(__LINE__, SL_LOG_CONNECTION_LOST, timestamp(STANDARD_TS_FORMAT), gethost(sInfo[i].addrinfo));
              close_logs(i, -1);
              sInfo[i].timer = time(NULL);
              sInfo[i].status = csThrottled;
            }
            else {
              split_lines(buffer, bytes, i);
            }
          }
        }
      }
    }
    healthcheck();
    logrotate();
  }
  return(csSuccess);
}

char *gethost(struct addrinfo *ai) {
  static char addr[128];

  memset(&addr, 0, sizeof(addr));
  switch(ai->ai_family) {
    case AF_INET:
      inet_ntop(ai->ai_family, &((struct sockaddr_in *)ai->ai_addr)->sin_addr, addr, sizeof(addr));
      break;
    case AF_INET6:
      inet_ntop(ai->ai_family, &((struct sockaddr_in6 *)ai->ai_addr)->sin6_addr, addr, sizeof(addr));
      break;
  }
  return(addr);
}

int getport(struct addrinfo *ai) {
  switch(ai->ai_family) {
    case AF_INET:
      return(htons(((struct sockaddr_in *)ai->ai_addr)->sin_port));
    case AF_INET6:
      return(htons(((struct sockaddr_in6 *)ai->ai_addr)->sin6_port));
  }
  return(0);
}

void defrag_nicklist(int i, int j) {
  int k, l;

  for(k = 0; k < sInfo[i].cInfo[j].nCount - 1; k++) {
    if(strlen(sInfo[i].cInfo[j].nInfo[k].nick) == 0) {
      for(l = k + 1; l < sInfo[i].cInfo[j].nCount - 1; l++) {
        if(strlen(sInfo[i].cInfo[j].nInfo[l].nick) != 0) {
          strncpy(sInfo[i].cInfo[j].nInfo[k].nick, sInfo[i].cInfo[j].nInfo[l].nick, sizeof(sInfo[i].cInfo[j].nInfo[k].nick));
          sInfo[i].cInfo[j].nInfo[l].nick[0] = 0;
          break;
        }
      }
      if(l == sInfo[i].cInfo[j].nCount - 1) {
        break;
      }
    }
  }
  sInfo[i].cInfo[j].nInfo = realloc(sInfo[i].cInfo[j].nInfo, sizeof(NickInfo) * (sInfo[i].cInfo[j].nCount = k + 1));
}

int resolve_servers(void) {
  int i;

  for(i = 0; i < sCount; i++) {
    if((sInfo[i].addrinfo = resolve_host(sInfo[i].server, sInfo[i].port)) == NULL) {
      return(-1);
    }
  }
  return(csSuccess);
}

void write_file(char *logfile, char *fmt, ...) {
  char str[LINE_SIZE];
  int fd;
  va_list ap;

  if(logfile != NULL) {
    va_start(ap, fmt);
    vsnprintf(str, sizeof(str), fmt, ap);
    va_end(ap);

    if((fd = open(logfile, O_CREAT | O_APPEND | O_SYNC | O_WRONLY, 0600)) >= 0) {
      write(fd, str, strlen(str));
      close(fd);
    }
  }
}

struct addrinfo *resolve_host(char *host, char *port) {
  struct addrinfo hints;
  struct addrinfo *res, *ai;
  short sockfd;
  int err;

  memset(&hints, 0, sizeof(struct addrinfo));
  hints.ai_family = AF_UNSPEC;
  hints.ai_socktype = SOCK_STREAM;
  hints.ai_protocol = IPPROTO_TCP;

  fprintf(stdout, " - %s\n", host);
  if((err = getaddrinfo(host, port, &hints, &res)) != 0) {
    fprintf(stdout, "getaddrinfo: %s\n", gai_strerror(err));
    return(NULL);
  }

  for(ai = res; ai; ai = ai->ai_next) {
    if((sockfd = socket(ai->ai_family, ai->ai_socktype, ai->ai_protocol)) != -1) {
      close(sockfd);
      return(ai);
    }
  }
  fprintf(stdout, "%s\n", strerror(errno));
  return(NULL);
}

int crontab(CronDB *cDB) {
  time_t c_time;
  struct tm *tm;

  c_time = time(NULL);
  tm = localtime(&c_time);

  if((cDB->minute != -1) && (cDB->minute != tm->tm_min)) {
    return(csError);
  }
  else if((cDB->hour != -1) && (cDB->hour != tm->tm_hour)) {
    return(csError);
  }
  else if((cDB->dom != -1) && (cDB->dom != tm->tm_mday)) {
    return(csError);
  }
  else if((cDB->month != -1) && (cDB->month != tm->tm_mon + 1)) {
    return(csError);
  }
  else if((cDB->dow != -1) && (cDB->dow != tm->tm_wday)) {
    return(csError);
  }
  return(csSuccess);
}

void LOG(long line, char *fmt, ...) {
  char str[LINE_SIZE];
  va_list ap;

#ifndef DEBUG
  int fd;
#endif

  va_start(ap, fmt);
  vsnprintf(str, sizeof(str), fmt, ap);
  va_end(ap);

#ifdef DEBUG
  fprintf(stdout, "%s(%ld): %s", __FILE__, line, str);
#else
  if(logFile != NULL) {
    if((fd = open(logFile, O_CREAT | O_APPEND | O_SYNC | O_WRONLY, 0600)) >= 0) {
      write(fd, str, strlen(str));
      close(fd);
    }
  }
#endif
}

int select_fd_sets(fd_set *fdset, fd_set *wdset) {
  int mx = 0;
  int i;

  for(i = 0; i < sCount; i++) {
    if(sInfo[i].status != csThrottled) {
      FD_SET(sInfo[i].sockfd, fdset);
      if(sInfo[i].status == csConnecting) {
        FD_SET(sInfo[i].sockfd, wdset);
      }
      mx = (sInfo[i].sockfd > mx) ? sInfo[i].sockfd : mx;
    }
  }
  return(mx);
}

void connect_to_servers(void) {
  int i;

  for(i = 0; i < sCount; i++) {
    if(sInfo[i].status == csReady) {
      initiate_connect(i);
    }
  }
}

char *strip_codes(char *p) {
  static char output[LINE_SIZE];
  int count = 0;
  int i;

  for(i = 0; i < strlen(p); i++) {
    if((p[i] != 1) && (p[i] != 2) && (p[i] != 31) && (p[i] != 15) && (p[i] != 3)) {
      output[count++] = p[i];
    }
    else if((p[i] != 1) && (p[i] != 2) && (p[i] != 31)) {
      if(p[i] == 3) {
        if((p[++i] >= '0') && (p[i] <= '9')) {
          if((p[++i] >= '0') && (p[i] <= '9')) {
            if(p[++i] == ',') {
              if((p[++i] >= '0') && (p[i] <= '9')) {
                if((p[++i] < '0') || (p[i] > '9')) {
                  i--;
                }
              }
              else {
                i--;
              }
            }
            else {
              i--;
            }
          }
          else if(p[i] == ',') {
            if((p[++i] >= '0') && (p[i] <= '9')) {
              if((p[++i] < '0') || (p[i] > '9')) {
                i--;
              }
            }
            else {
              i--;
            }
          }
          else {
            i--;
          }
        }
        else {
          i--;
        }
      }
    }
  }
  output[count] = 0;
  return(output);
}

char *timestamp(char *format) {
  time_t clock;
  struct tm *tm;
  static char ts[TIMESTAMP_SIZE];

  time(&clock);
  tm = localtime(&clock);
  strftime(ts, sizeof(ts), format, tm);
  return(ts);
}

void split_lines(char *buffer, int bytes, int i) {
  char tb[LINE_SIZE];
  int length = 0;
  int j;

  if(strlen(sInfo[i].oldline) > 0) {
    snprintf(tb, sizeof(tb), "%s", sInfo[i].oldline);
    length += strlen(tb);
  }

  for(j = 0; j < bytes; j++) {
    if((buffer[j] == '\n') || (buffer[j] == '\r')) {
      if(length > 0) {
        tb[length] = 0;
        decode(tb, i);
        length = 0;
      }
    }
    else {
      if(length < sizeof(tb)) {
        tb[length++] = buffer[j];
      }
      else {
        tb[length] = 0;
        decode(tb, i);
        length = 0;
      }
    }
  }
  if(length > 0) {
    tb[length] = 0;
    snprintf(sInfo[i].oldline, sizeof(sInfo[i].oldline), "%s", tb);
  }
  else {
    sInfo[i].oldline[0] = 0;
  }
}

void decode(char *buffer, int i) {
  char output[LINE_SIZE];
  char *cp_buffer;
  char *hostmask = NULL;
  char *knick;
  char *nick = NULL;
  char *channel;
  char *p, *q;
  int j, k;

  sInfo[i].timer = time(NULL);
  LOG(__LINE__, SL_LOG_GENERIC_CRLF, timestamp(STANDARD_TS_FORMAT), i, gethost(sInfo[i].addrinfo), getport(sInfo[i].addrinfo), buffer);

  cp_buffer = strdup(buffer);
  if(strncasecmp(buffer, "PING", 4) == 0) {
    snprintf(output, sizeof(output), PONG_CMD, &buffer[5]);
    send(sInfo[i].sockfd, output, strlen(output), 0);
    LOG(__LINE__, SL_LOG_GENERIC, timestamp(STANDARD_TS_FORMAT), i, gethost(sInfo[i].addrinfo), getport(sInfo[i].addrinfo), output);
  }
  else if(buffer[0] == ':') {
    if(strstr(cp_buffer, "!")) {
      nick = strtok(cp_buffer, "!");
      nick++;
      hostmask = strtok(NULL, " ");
    }
    else {
      nick = strtok(cp_buffer, " ");
      nick++;
    }
    if((p = strtok(buffer, " ")) != NULL) {
      if((p = strtok(NULL, " ")) != NULL) {
        if(strcasecmp(p, sInfo[i].rawcode) == 0) {
          sInfo[i].status = csAuthenticated;
          sInfo[i].timer = time(NULL);
          sInfo[i].ptimer = time(NULL);
        }
        else if(strcasecmp(p, "PRIVMSG") == 0) {
          if((channel = strtok(NULL, " ")) != NULL) {
            if((p = strtok(NULL, "\0")) != NULL) {
              for(j = 0; j < sInfo[i].cCount; j++) {
                if((strcasecmp(channel, sInfo[i].cInfo[j].channel) == 0) || (strcasecmp(channel, sInfo[i].actualnick) == 0)){
                  if((p[1] == 1) && (p[strlen(p) - 1] == 1)) {
                    p[strlen(p) - 1] = 0;
                    if(strncasecmp(&p[2], "VERSION", 7) == 0) {
                      snprintf(output, sizeof(output), VERSION_REPLY, nick, APPNAME, VERSION, COPYRIGHT);
                      send(sInfo[i].sockfd, output, strlen(output), 0);
                      LOG(__LINE__, SL_LOG_GENERIC, timestamp(STANDARD_TS_FORMAT), i, gethost(sInfo[i].addrinfo), getport(sInfo[i].addrinfo), output);
                    }
                    else if(strncasecmp(&p[2], "ACTION", 6) == 0) {
                      write_file(sInfo[i].cInfo[j].logfile, LOG_ACTION, timestamp(LOG_TS_FORMAT), nick, strip_codes(&p[9]));
                    }
                  }
                  else {
                    write_file(sInfo[i].cInfo[j].logfile, LOG_PRIVMSG, timestamp(LOG_TS_FORMAT), nick, strip_codes(&p[1]));
                  }
                  break;
                }
              }
            }
          }
        }
        else if(strcasecmp(p, "NICK") == 0) {
          if((p = strtok(NULL, "\0")) != NULL) {
            if(strcasecmp(sInfo[i].actualnick, nick) == 0) {
              strncpy(sInfo[i].actualnick, &p[1], sizeof(sInfo[i].actualnick));
            }
            else {
              for(j = 0; j < sInfo[i].cCount; j++) {
                if(sInfo[i].cInfo[j].status == csJoined) {
                  for(k = 0; k < sInfo[i].cInfo[j].nCount; k++) {
                    if(strcasecmp(sInfo[i].cInfo[j].nInfo[k].nick, nick) == 0) {
                      strncpy(sInfo[i].cInfo[j].nInfo[k].nick, &p[1], sizeof(sInfo[i].cInfo[j].nInfo[k].nick));
                      write_file(sInfo[i].cInfo[j].logfile, LOG_NICK, timestamp(LOG_TS_FORMAT), nick, &p[1]);
                      break;
                    }
                  }
                }
              }
            }
          }
        }
        else if(strcasecmp(p, CHANNEL_NAMES) == 0) {
          if((p = strtok(NULL, " ")) != NULL) {
            if((p = strtok(NULL, " ")) != NULL) {
              if((channel = strtok(NULL, " ")) != NULL) {
                if((p = strtok(NULL, "\0")) != NULL) {
                  for(j = 0; j < sInfo[i].cCount; j++) {
                    if(strcasecmp(sInfo[i].cInfo[j].channel, channel) == 0) {
                      if((q = strtok(++p, " \0")) != NULL) {
                        do {
                          if((q[0] == '@') || (q[0] == '+')) {
                            q++;
                          }
                          if(strcasecmp(sInfo[i].actualnick, q) != 0) {
                            strncpy(sInfo[i].cInfo[j].nInfo[sInfo[i].cInfo[j].nCount - 1].nick, q, sizeof(sInfo[i].cInfo[j].nInfo[sInfo[i].cInfo[j].nCount - 1].nick));
                            sInfo[i].cInfo[j].nInfo = realloc(sInfo[i].cInfo[j].nInfo, sizeof(NickInfo) * (++sInfo[i].cInfo[j].nCount));
                            sInfo[i].cInfo[j].nInfo[sInfo[i].cInfo[j].nCount - 1].nick[0] = 0;
                          }
                        }
                        while((q = strtok(NULL, " \0")) != NULL);
                      }
                      break;
                    }
                  }
                }
              }
            }
          }
        }
        else if(strcasecmp(p, "PART") == 0) {
          if((channel = strtok(NULL, " ")) != NULL) {
            for(j = 0; j < sInfo[i].cCount; j++) {
              if(strcasecmp(sInfo[i].cInfo[j].channel, channel) == 0) {
                if(strcasecmp(sInfo[i].actualnick, nick) == 0) {
                  close_logs(i, j);
                  sInfo[i].cInfo[j].status = csRotatingLogs;
                  rotate_logs(i, j);
                  sInfo[i].cInfo[j].status = csKicked;
                }
                else {
                  for(k = 0; k < sInfo[i].cInfo[j].nCount; k++) {
                    if(strcasecmp(sInfo[i].cInfo[j].nInfo[k].nick, nick) == 0) {
                      sInfo[i].cInfo[j].nInfo[k].nick[0] = 0;
                      break;
                    }
                  }

                  if(((p = strtok(NULL, "\0")) != NULL) && (strlen(&p[1]) != 0)) {
                    write_file(sInfo[i].cInfo[j].logfile, LOG_PART_MSG, timestamp(LOG_TS_FORMAT), nick, hostmask, channel, strip_codes(&p[1]));
                  }
                  else {
                    write_file(sInfo[i].cInfo[j].logfile, LOG_PART, timestamp(LOG_TS_FORMAT), nick, hostmask, channel);
                  }
                  defrag_nicklist(i, j);
                }
                break;
              }
            }
          }
        }
        else if(strcasecmp(p, "QUIT") == 0) {
          if((p = strtok(NULL, "\0")) != NULL) {
            for(j = 0; j < sInfo[i].cCount; j++) {
              if(sInfo[i].cInfo[j].status == csJoined) {
                for(k = 0; k < sInfo[i].cInfo[j].nCount; k++) {
                  if(strcasecmp(sInfo[i].cInfo[j].nInfo[k].nick, nick) == 0) {
                    sInfo[i].cInfo[j].nInfo[k].nick[0] = 0;
                    write_file(sInfo[i].cInfo[j].logfile, LOG_QUIT, timestamp(LOG_TS_FORMAT), nick, hostmask, strip_codes(&p[1]));
                    defrag_nicklist(i, j);
                    break;
                  }
                }
              }
            }
          }
        }
        else if(strcasecmp(p, "MODE") == 0) {
          if((channel = strtok(NULL, " ")) != NULL) {
            if((p = strtok(NULL, "\0")) != NULL) {
              for(j = 0; j < sInfo[i].cCount; j++) {
                if(strcasecmp(sInfo[i].cInfo[j].channel, channel) == 0) {
                  write_file(sInfo[i].cInfo[j].logfile, LOG_MODE, timestamp(LOG_TS_FORMAT), channel, p, nick);
                  break;
                }
              }
            }
          }
        }
        else if(strcasecmp(p, "TOPIC") == 0) {
          if((channel = strtok(NULL, " ")) != NULL) {
            if((p = strtok(NULL, "\0")) != NULL) {
              for(j = 0; j < sInfo[i].cCount; j++) {
                if(strcasecmp(sInfo[i].cInfo[j].channel, channel) == 0) {
                  write_file(sInfo[i].cInfo[j].logfile, LOG_TOPIC_CHANGE, timestamp(LOG_TS_FORMAT), nick, channel, strip_codes(&p[1]));
                  break;
                }
              }
            }
          }
        }
        else if(strcasecmp(p, NICK_COLLISION) == 0) {
          if((p = strtok(NULL, " ")) != NULL) {
            if(*p == '*') {
              strncat(sInfo[i].actualnick, "_", sizeof(sInfo[i].actualnick));
              snprintf(output, sizeof(output), NICK_CMD, sInfo[i].actualnick);
              send(sInfo[i].sockfd, output, strlen(output), 0);
              LOG(__LINE__, SL_LOG_GENERIC, timestamp(STANDARD_TS_FORMAT), i, gethost(sInfo[i].addrinfo), getport(sInfo[i].addrinfo), output);
            }
          }
        }
        else if(strcasecmp(p, "JOIN") == 0) {
          if((channel = strtok(NULL, "\0")) != NULL) {
            for(j = 0; j < sInfo[i].cCount; j++) {
              if(strcasecmp(&channel[1], sInfo[i].cInfo[j].channel) == 0) {
                if(strcasecmp(nick, sInfo[i].actualnick) == 0) {
                  sInfo[i].cInfo[j].nInfo = malloc(sizeof(NickInfo) * (sInfo[i].cInfo[j].nCount = 1));
                  sInfo[i].cInfo[j].nInfo[0].nick[0] = 0;
                  sInfo[i].cInfo[j].status = csJoined;
                  write_file(sInfo[i].cInfo[j].logfile, LOG_SESSION_START, timestamp(LONG_TS_FORMAT), &channel[1]);
                }
                else {
                  strncpy(sInfo[i].cInfo[j].nInfo[sInfo[i].cInfo[j].nCount - 1].nick, nick, sizeof(sInfo[i].cInfo[j].nInfo[sInfo[i].cInfo[j].nCount - 1].nick));
                  sInfo[i].cInfo[j].nInfo = realloc(sInfo[i].cInfo[j].nInfo, sizeof(NickInfo) * (++sInfo[i].cInfo[j].nCount));
                  sInfo[i].cInfo[j].nInfo[sInfo[i].cInfo[j].nCount - 1].nick[0] = 0;
                  write_file(sInfo[i].cInfo[j].logfile, LOG_JOIN, timestamp(LOG_TS_FORMAT), nick, hostmask, &channel[1]);
                  break;
                }
              }
            }
          }
        }
        else if(strcasecmp(p, "KICK") == 0) {
          if((channel = strtok(NULL, " ")) != NULL) {
            if((knick = strtok(NULL, " ")) != NULL) {
              if((p = strtok(NULL, "\0")) != NULL) {
                for(j = 0; j < sInfo[i].cCount; j++) {
                  if(strcasecmp(channel, sInfo[i].cInfo[j].channel) == 0) {
                    if(strcasecmp(knick, sInfo[i].actualnick) == 0) {
                      close_logs(i, j);
                      sInfo[i].cInfo[j].status = csKicked;
                    }
                    else {
                      for(k = 0; k < sInfo[i].cInfo[j].nCount; k++) {
                        if(strcasecmp(sInfo[i].cInfo[j].nInfo[k].nick, knick) == 0) {
                          sInfo[i].cInfo[j].nInfo[k].nick[0] = 0;
                          break;
                        }
                      }
                      write_file(sInfo[i].cInfo[j].logfile, LOG_KICK, timestamp(LOG_TS_FORMAT), knick, channel, nick, strip_codes(&p[1]));
                      defrag_nicklist(i, j);
                    }
                    break;
                  }
                }
              }
            }
          }
        }
      }
    }
  }
  free(cp_buffer);
}

void initiate_connect(int i) {
  reset_channel_status(i);
  if((sInfo[i].sockfd = socket(sInfo[i].addrinfo->ai_family, sInfo[i].addrinfo->ai_socktype, sInfo[i].addrinfo->ai_protocol)) != csError) {
    sInfo[i].status = csConnecting;
    if(strlen(sInfo[i].vhost) > 0) {
      struct addrinfo hints;
      struct addrinfo *res;
      int err;

      memset(&hints, 0, sizeof(struct addrinfo));
      hints.ai_family = AF_UNSPEC;
      hints.ai_socktype = SOCK_STREAM;
      hints.ai_protocol = IPPROTO_TCP;
      hints.ai_flags = AI_PASSIVE;

      if((err = getaddrinfo(sInfo[i].vhost, NULL, &hints, &res)) == csSuccess) {
        if(bind(sInfo[i].sockfd, res->ai_addr, res->ai_addrlen) == csError) {
          LOG(__LINE__, SL_LOG_ERROR_VHOST, timestamp(STANDARD_TS_FORMAT), gethost(res), strerror(errno));
        }
        freeaddrinfo(res);
      }
      else {
        LOG(__LINE__, SL_LOG_ERROR_GETADDRINFO, timestamp(STANDARD_TS_FORMAT), sInfo[i].vhost, gai_strerror(err));
      }
    }
    fcntl(sInfo[i].sockfd, F_SETFL, O_NONBLOCK);
    LOG(__LINE__, SL_LOG_CONNECTING, timestamp(STANDARD_TS_FORMAT), gethost(sInfo[i].addrinfo), getport(sInfo[i].addrinfo));
    sInfo[i].timer = time(NULL);

    if(connect(sInfo[i].sockfd, sInfo[i].addrinfo->ai_addr, sInfo[i].addrinfo->ai_addrlen) == csError) {
      if((errno != EAGAIN) && (errno != EINPROGRESS)) {
        close(sInfo[i].sockfd);
        LOG(__LINE__, SL_LOG_ERROR_CONNECTING, timestamp(STANDARD_TS_FORMAT), gethost(sInfo[i].addrinfo), getport(sInfo[i].addrinfo), strerror(errno));
        sInfo[i].timer = time(NULL);
      }
    }
    else {
      fcntl(sInfo[i].sockfd, F_SETFL, 0);
      LOG(__LINE__, SL_LOG_CONNECTED, timestamp(STANDARD_TS_FORMAT), gethost(sInfo[i].addrinfo), getport(sInfo[i].addrinfo));
      sInfo[i].status = csConnected;
      send_login_sequence(i);
    }
  }
  else {
    LOG(__LINE__, SL_LOG_ERROR_SOCKET, timestamp(STANDARD_TS_FORMAT), gethost(sInfo[i].addrinfo), strerror(errno));
    sInfo[i].timer = time(NULL);
    sInfo[i].status = csThrottled;
  }
}

void reset_channel_status(int i) {
  int j;

  for(j = 0; j < sInfo[i].cCount; j++) {
    sInfo[i].cInfo[j].status = csKicked;
  }
}

void send_login_sequence(int i) {
  char buffer[GENERIC_SIZE];

  LOG(__LINE__, SL_LOG_SEND_LOGIN, timestamp(STANDARD_TS_FORMAT), gethost(sInfo[i].addrinfo));
  strncpy(sInfo[i].actualnick, sInfo[i].nick, sizeof(sInfo[i].actualnick));

  snprintf(buffer, sizeof(buffer), NICK_CMD, sInfo[i].actualnick);
  send(sInfo[i].sockfd, buffer, strlen(buffer), 0);
  LOG(__LINE__, SL_LOG_GENERIC, timestamp(STANDARD_TS_FORMAT), i, gethost(sInfo[i].addrinfo), getport(sInfo[i].addrinfo), buffer);

  snprintf(buffer, sizeof(buffer), USER_CMD, sInfo[i].ident, sInfo[i].realname);
  send(sInfo[i].sockfd, buffer, strlen(buffer), 0);
  LOG(__LINE__, SL_LOG_GENERIC, timestamp(STANDARD_TS_FORMAT), i, gethost(sInfo[i].addrinfo), getport(sInfo[i].addrinfo), buffer);
}

void healthcheck(void) {
  char output[LINE_SIZE];
  int i, j;

  for(i = 0; i < sCount; i++) {
    if(sInfo[i].status == csThrottled) {
      if((time(NULL) - sInfo[i].timer) >= THROTTLE_LIMIT) {
        LOG(__LINE__, SL_LOG_THROTTLE_EXPIRE, timestamp(STANDARD_TS_FORMAT), gethost(sInfo[i].addrinfo));
        initiate_connect(i);
      }
    }
    else if(sInfo[i].status == csConnecting) {
      if((time(NULL) - sInfo[i].timer) >= CONNECT_TIMEOUT) {
        close(sInfo[i].sockfd);
        LOG(__LINE__, SL_LOG_CONNECTION_TIMEDOUT, timestamp(STANDARD_TS_FORMAT), gethost(sInfo[i].addrinfo), getport(sInfo[i].addrinfo));
        sInfo[i].status = csThrottled;
      }
    }
    else if(sInfo[i].status == csAuthenticated) {
      if((time(NULL) - sInfo[i].timer) >= atoi(sInfo[i].pingtimeout)) {
        close(sInfo[i].sockfd);
        LOG(__LINE__, SL_LOG_CONNECTION_PINGTIMEOUT, timestamp(STANDARD_TS_FORMAT), gethost(sInfo[i].addrinfo));
        sInfo[i].status = csThrottled;
        close_logs(i, -1);
      }

      if(sInfo[i].selfping == 1) {
        if((time(NULL) - sInfo[i].ptimer) >= (atoi(sInfo[i].pingtimeout) / 2)) {
          snprintf(output, sizeof(output), SPING_CMD, sInfo[i].actualnick);
          send(sInfo[i].sockfd, output, strlen(output), 0);
          LOG(__LINE__, SL_LOG_GENERIC, timestamp(STANDARD_TS_FORMAT), i, gethost(sInfo[i].addrinfo), getport(sInfo[i].addrinfo), output);
          sInfo[i].ptimer = time(NULL);
        }
      }

      if((time(NULL) - sInfo[i].stimer) >= JOIN_INTERVAL) {
        for(j = 0; j < sInfo[i].cCount; j++) {
          if(sInfo[i].cInfo[j].status == csKicked) {
            snprintf(output, sizeof(output), JOIN_CMD, sInfo[i].cInfo[j].channel);
            send(sInfo[i].sockfd, output, strlen(output), 0);
            LOG(__LINE__, SL_LOG_GENERIC, timestamp(STANDARD_TS_FORMAT), i, gethost(sInfo[i].addrinfo), getport(sInfo[i].addrinfo), output);
          }
        }

        if(strcasecmp(sInfo[i].nick, sInfo[i].actualnick) != 0) {
          snprintf(output, sizeof(output), NICK_CMD, sInfo[i].nick);
          send(sInfo[i].sockfd, output, strlen(output), 0);
          LOG(__LINE__, SL_LOG_GENERIC, timestamp(STANDARD_TS_FORMAT), i, gethost(sInfo[i].addrinfo), getport(sInfo[i].addrinfo), output);
        }

        sInfo[i].stimer = time(NULL);
      }
    }
  }
}

void rotate_logs(int i, int j) {
  char buffer[BUFFER_SIZE];
  char date_rotate_file[BUFFER_SIZE];
  int fdr, fdw;
  int bytes;

  LOG(__LINE__, SL_LOG_LOGROTATE, timestamp(STANDARD_TS_FORMAT), sInfo[i].cInfo[j].channel, gethost(sInfo[i].addrinfo));
  if((fdr = open(sInfo[i].cInfo[j].logfile, O_RDONLY)) >= 0) {
    if(sInfo[i].cInfo[j].daterotate == 1) {
      snprintf(date_rotate_file, sizeof(date_rotate_file), "%s-%s", sInfo[i].cInfo[j].logfile, timestamp(sInfo[i].cInfo[j].daterotateformat));
      if((fdw = open(date_rotate_file, O_CREAT | O_APPEND | O_SYNC | O_WRONLY, 0600)) < 0) {
        close(fdr);
        LOG(__LINE__, SL_LOG_ERROR_LOGROTATE, timestamp(STANDARD_TS_FORMAT), sInfo[i].cInfo[j].channel, gethost(sInfo[i].addrinfo));
        return;
      }
    }
    else {
      if((fdw = open(sInfo[i].cInfo[j].archivelogfile, O_CREAT | O_APPEND | O_SYNC | O_WRONLY, 0600)) < 0) {
        close(fdr);
        LOG(__LINE__, SL_LOG_ERROR_LOGROTATE, timestamp(STANDARD_TS_FORMAT), sInfo[i].cInfo[j].channel, gethost(sInfo[i].addrinfo));
        return;
      }
    }
        
    while((bytes = read(fdr, buffer, sizeof(buffer))) != 0) {
      write(fdw, buffer, bytes);
    }
    
    close(fdw);
    close(fdr);
    truncate(sInfo[i].cInfo[j].logfile, 0);
    LOG(__LINE__, SL_LOG_SUCCESS_LOGROTATE, timestamp(STANDARD_TS_FORMAT), sInfo[i].cInfo[j].channel, gethost(sInfo[i].addrinfo));
    return;
  }
  LOG(__LINE__, SL_LOG_ERROR_LOGROTATE, timestamp(STANDARD_TS_FORMAT), sInfo[i].cInfo[j].channel, gethost(sInfo[i].addrinfo));
}

void logrotate(void) {
  char output[LINE_SIZE];
  int i, j;

  for(i = 0; i < sCount; i++) {
    for(j = 0; j < sInfo[i].cCount; j++) {
      if(strlen(sInfo[i].cInfo[j].logrotate) != 0) {
        if(crontab(sInfo[i].cInfo[j].cDB) == 0) {
          if(strcasecmp(sInfo[i].cInfo[j].lastrotate, timestamp(ROTATE_TS_FORMAT)) != 0) {
            if(sInfo[i].cInfo[j].status == csJoined) {
              snprintf(output, sizeof(output), PART_CMD, sInfo[i].cInfo[j].channel, "Rotating Logs");
              send(sInfo[i].sockfd, output, strlen(output), 0);
              LOG(__LINE__, SL_LOG_GENERIC, timestamp(STANDARD_TS_FORMAT), i, gethost(sInfo[i].addrinfo), getport(sInfo[i].addrinfo), output);
            }
            else {
              rotate_logs(i, j);
            }
            snprintf(sInfo[i].cInfo[j].lastrotate, sizeof(sInfo[i].cInfo[j].lastrotate), timestamp(ROTATE_TS_FORMAT));
          }
        }
      }
    }
  }
}

void daemonize(void) {
  pid_t pid;

  fprintf(stdout, "%s v%s Daemonized...\n", APPNAME, VERSION);

  if((pid = fork()) < 0) {
    perror("fork");
    free_memory();
    exit(csError);
  }

  if(pid > 0) {
    free_memory();
    exit(csError);
  }

  if(setsid() < 0) {
    perror("setsid");
    free_memory();
    exit(csError);
  }

  umask(0);

  close(STDIN_FILENO);
  close(STDOUT_FILENO);
  close(STDERR_FILENO);
}

void msleep(unsigned long ms) {
  struct timespec rqtp;

  rqtp.tv_sec = (time_t)ms / 1000;
  rqtp.tv_nsec = (long)(ms % 1000) * 1000000;
  nanosleep(&rqtp, NULL);
}

void close_logs(int i, int j) {
  int k, l;

  if(i == -1) {
    for(k = 0; k < sCount; k++) {
      for(l = 0; l < sInfo[k].cCount; l++) {
        if(sInfo[k].cInfo[l].status == csJoined) {
          free(sInfo[k].cInfo[l].nInfo);
          write_file(sInfo[k].cInfo[l].logfile, LOG_SESSION_CLOSE, timestamp(LONG_TS_FORMAT));
        }
      }
    }
  }
  else if(j == -1) {
    for(k = 0; k < sInfo[i].cCount; k++) {
      if(sInfo[i].cInfo[k].status == csJoined) {
        free(sInfo[i].cInfo[k].nInfo);
        write_file(sInfo[i].cInfo[k].logfile, LOG_SESSION_CLOSE, timestamp(LONG_TS_FORMAT));
      }
    }
  }
  else {
    if(sInfo[i].cInfo[j].status == csJoined) {
      free(sInfo[i].cInfo[j].nInfo);
      write_file(sInfo[i].cInfo[j].logfile, LOG_SESSION_CLOSE, timestamp(LONG_TS_FORMAT));
    }
  }
}

void sig_handler(int sig, siginfo_t *sinf, void *ucon) {

  LOG(__LINE__, SL_LOG_SIGHANDLER, timestamp(STANDARD_TS_FORMAT), sig);

  if((sig == SIGINT) || (sig == SIGTERM)) {
    close_logs(-1, -1);
    free_memory();
    exit(csSuccess);
  }
}

void setup_signals(void) {
  struct sigaction sact;

  sact.sa_sigaction = sig_handler;
  sact.sa_flags = SA_SIGINFO;

  sigemptyset(&sact.sa_mask);
  sigaction(SIGINT, &sact, (struct sigaction *)NULL);
  sigaction(SIGTERM, &sact, (struct sigaction *)NULL);

  sact.sa_handler = SIG_IGN;
  sigaction(SIGPIPE, &sact, (struct sigaction *)NULL);
}

void check_argc(int argc) {
  if(argc < 2) {
#ifdef DEBUG
    fprintf(stdout, "Usage: %s <Config File>\n\n", APPNAME);
#else
    fprintf(stdout, "Usage: %s <Config File> [Log File]\n\n", APPNAME);
#endif
    exit(csError);
  }
}

int sanity_check(void) {
  int i, j;

  fprintf(stdout, "Sanity Checking........\n");

  for(i = 0; i < sCount; i++) {
    fprintf(stdout, "---------------------------------------------------------\n");
    fprintf(stdout, "Server: :%s:\n", (strlen(sInfo[i].server) == 0) ? "MISSING" : sInfo[i].server);
    fprintf(stdout, "  Port :%s:\n", (strlen(sInfo[i].port) == 0) ? "MISSING" : sInfo[i].port);
    fprintf(stdout, "  Nick :%s:\n", (strlen(sInfo[i].nick) == 0) ? "MISSING" : sInfo[i].nick);
    fprintf(stdout, "  Ident :%s:\n", (strlen(sInfo[i].ident) == 0) ? "MISSING" : sInfo[i].ident);
    fprintf(stdout, "  RealName :%s:\n", (strlen(sInfo[i].realname) == 0) ? "MISSING" : sInfo[i].realname);
    fprintf(stdout, "  PingTimeOut :%s:\n", (((strlen(sInfo[i].pingtimeout) == 0) || atoi(sInfo[i].pingtimeout) < 1)) ? "INVALID" : sInfo[i].pingtimeout);
    fprintf(stdout, "  SelfPing :%s:\n", (sInfo[i].selfping == 1) ? "On" : "Off");
    fprintf(stdout, "  RawConnectCode :%s:\n", (strlen(sInfo[i].rawcode) == 0) ? "MISSING" : sInfo[i].rawcode);
    fprintf(stdout, "  VirtualHost :%s:\n", (strlen(sInfo[i].vhost) == 0) ? "MISSING" : sInfo[i].vhost);

    for(j = 0; j < sInfo[i].cCount; j++) {
      fprintf(stdout, "Channel :%s:\n", (strlen(sInfo[i].cInfo[j].channel) == 0) ? "MISSING" : sInfo[i].cInfo[j].channel);
      fprintf(stdout, "  DateRotate :%s:\n", (sInfo[i].cInfo[j].daterotate == 1) ? "On" : "Off");
      fprintf(stdout, "  DateRotateFormat :%s:\n", (strlen(sInfo[i].cInfo[j].daterotateformat) == 0) ? "MISSING" : sInfo[i].cInfo[j].daterotateformat);
      fprintf(stdout, "  LogRotate :%s:\n", (strlen(sInfo[i].cInfo[j].logrotate) == 0) ? "MISSING" : sInfo[i].cInfo[j].logrotate);
      fprintf(stdout, "  LogFile :%s:\n", (strlen(sInfo[i].cInfo[j].logfile) == 0) ? "MISSING" : sInfo[i].cInfo[j].logfile);
      fprintf(stdout, "  ArchiveLogFile :%s:\n", (strlen(sInfo[i].cInfo[j].archivelogfile) == 0) ? "MISSING" : sInfo[i].cInfo[j].archivelogfile);;
    }
  }
  fprintf(stdout, "---------------------------------------------------------\n");

  for(i = 0; i < sCount; i++) {
    if(strlen(sInfo[i].server) < 1) {
      fprintf(stdout, "Server Option Missing/Invalid\n");
      return(csError);
    }
    else if(strlen(sInfo[i].port) < 1) {
      fprintf(stdout, "Port Option Missing/Invalid\n");
      return(csError);
    }
    else if(strlen(sInfo[i].nick) < 1) {
      fprintf(stdout, "Nick Option Missing/Invalid\n");
      return(csError);
    }
    else if(strlen(sInfo[i].ident) < 1) {
      fprintf(stdout, "Ident Option Missing/Invalid\n");
      return(csError);
    }
    else if(strlen(sInfo[i].realname) < 1) {
      fprintf(stdout, "RealName Option Missing/Invalid\n");
      return(csError);
    }
    else if(strlen(sInfo[i].pingtimeout) < 1) {
      fprintf(stdout, "PingTimeOut Option Missing/Invalid\n");
      return(csError);
    }
    else if(strlen(sInfo[i].rawcode) < 1) {
      fprintf(stdout, "RawCode Option Missing/Invalid\n");
      return(csError);
    }
    else if(sInfo[i].cCount == 0) {
      fprintf(stdout, "No Valid Channel Configuration\n");
      return(csError);
    }

    for(j = 0; j < sInfo[i].cCount; j++) {
      if(strlen(sInfo[i].cInfo[j].channel) < 1) {
        fprintf(stdout, "Channel Option Missing/Invalid\n");
        return(csError);
      }
      else if(strlen(sInfo[i].cInfo[j].logfile) < 1) {
        fprintf(stdout, "LogFile Option Missing/Invalid\n");
        return(csError);
      }
      else if((strlen(sInfo[i].cInfo[j].logrotate) < 1) && ((strlen(sInfo[i].cInfo[j].archivelogfile) > 0) || (sInfo[i].cInfo[j].daterotate == 1))) {
        fprintf(stdout, "LogRotate Option Missing/Invalid (Implied -> ArchiveLogFile || DateRotate)\n");
        return(csError);
      }
      else if((strlen(sInfo[i].cInfo[j].logrotate) > 0) && (strlen(sInfo[i].cInfo[j].archivelogfile) < 1) && (sInfo[i].cInfo[j].daterotate == 0)) {
        fprintf(stdout, "ArchiveLogFile || DateRotate Option Missing/Invalid (Implied -> LogRotate)\n");
        return(csError);
      }
      else if((strlen(sInfo[i].cInfo[j].logrotate) > 0) && (strlen(sInfo[i].cInfo[j].archivelogfile) > 0) && (sInfo[i].cInfo[j].daterotate == 1)) {
        fprintf(stdout, "ArchiveLogFile && DateRotate Option Included\n");
        return(csError);
      }
      else if((sInfo[i].cInfo[j].daterotate == 1) && (strlen(sInfo[i].cInfo[j].daterotateformat) < 1)) {
        fprintf(stdout, "DateRotateFormat Option Missing/Invalid (Implied -> DateRotate)\n");
        return(csError);
      }
      else if((sInfo[i].cInfo[j].daterotate == 0) && (strlen(sInfo[i].cInfo[j].daterotateformat) > 0)) {
        fprintf(stdout, "DateRotate Option Missing/Invalid (Implied -> DateRotateFormat)\n");
        return(csError);
      }
      else if(strcasecmp(sInfo[i].cInfo[j].logfile, sInfo[i].cInfo[j].archivelogfile) == 0) {
        fprintf(stdout, "LogFile Option == ArchiveLogFile Option\n");
        return(csError);
      }
    }
  }
  return(csSuccess);
}

void free_memory(void) {
  int i;

  for(i = 0; i < sCount; i++) {
    free(sInfo[i].cInfo);
    if(sInfo[i].addrinfo != NULL) {
      freeaddrinfo(sInfo[i].addrinfo);
    }
  }
  free(sInfo);
}

void generate_initial_info_struct(void) {
  sInfo = malloc(sizeof(ServerInfo) * (sCount = 1));
  sInfo[0].server[0] = 0;
  sInfo[0].port[0] = 0;
  sInfo[0].nick[0] = 0;
  sInfo[0].ident[0] = 0;
  sInfo[0].realname[0] = 0;
  sInfo[0].pingtimeout[0] = 0;
  sInfo[0].selfping = 0;
  sInfo[0].rawcode[0] = 0;
  sInfo[0].vhost[0] = 0;
  sInfo[0].addrinfo = NULL;
  sInfo[0].oldline[0] = 0;
  sInfo[0].actualnick[0] = 0;
  sInfo[0].status = csBlank;
  sInfo[0].cInfo = malloc(sizeof(ChannelInfo) * (sInfo[0].cCount = 1));
  sInfo[0].cInfo[0].channel[0] = 0;
  sInfo[0].cInfo[0].logrotate[0] = 0;
  sInfo[0].cInfo[0].lastrotate[0] = 0;
  sInfo[0].cInfo[0].logfile[0] = 0;
  sInfo[0].cInfo[0].archivelogfile[0] = 0;
  sInfo[0].cInfo[0].daterotate = 0;
  sInfo[0].cInfo[0].daterotateformat[0] = 0;
  sInfo[0].cInfo[0].nInfo = malloc(sizeof(NickInfo));
  sInfo[0].cInfo[0].nInfo[0].nick[0] = 0;
  sInfo[0].cInfo[0].cDB = malloc(sizeof(CronDB));
}

void generate_new_server_struct(void) {
  sInfo = realloc(sInfo, sizeof(ServerInfo) * (++sCount));
  sInfo[sCount - 1].server[0] = 0;
  sInfo[sCount - 1].port[0] = 0;
  sInfo[sCount - 1].nick[0] = 0;
  sInfo[sCount - 1].ident[0] = 0;
  sInfo[sCount - 1].pingtimeout[0] = 0;
  sInfo[sCount - 1].selfping = 0;
  sInfo[sCount - 1].realname[0] = 0;
  sInfo[sCount - 1].rawcode[0] = 0;
  sInfo[sCount - 1].vhost[0] = 0;
  sInfo[sCount - 1].addrinfo = NULL;
  sInfo[sCount - 1].oldline[0] = 0;
  sInfo[sCount - 1].actualnick[0] = 0;
  sInfo[sCount - 1].status = csBlank;
}

void generate_initial_channel_struct(void) {
  sInfo[sCount - 1].cInfo = malloc(sizeof(ChannelInfo) * (sInfo[sCount - 1].cCount = 1));
  sInfo[sCount - 1].cInfo[0].channel[0] = 0;
  sInfo[sCount - 1].cInfo[0].logrotate[0] = 0;
  sInfo[sCount - 1].cInfo[0].lastrotate[0] = 0;
  sInfo[sCount - 1].cInfo[0].logfile[0] = 0;
  sInfo[sCount - 1].cInfo[0].archivelogfile[0] = 0;
  sInfo[sCount - 1].cInfo[0].daterotate = 0;
  sInfo[sCount - 1].cInfo[0].daterotateformat[0] = 0;
  sInfo[sCount - 1].cInfo[0].nInfo = malloc(sizeof(NickInfo));
  sInfo[sCount - 1].cInfo[0].nInfo[0].nick[0] = 0;
  sInfo[sCount - 1].cInfo[0].cDB = malloc(sizeof(CronDB));
}

void generate_new_channel_struct(void) {
  sInfo[sCount - 1].cInfo = realloc(sInfo[sCount - 1].cInfo, sizeof(ChannelInfo) * (++sInfo[sCount - 1].cCount));
  sInfo[sCount - 1].cInfo[sInfo[sCount - 1].cCount - 1].channel[0] = 0;
  sInfo[sCount - 1].cInfo[sInfo[sCount - 1].cCount - 1].logrotate[0] = 0;
  sInfo[sCount - 1].cInfo[sInfo[sCount - 1].cCount - 1].lastrotate[0] = 0;
  sInfo[sCount - 1].cInfo[sInfo[sCount - 1].cCount - 1].logfile[0] = 0;
  sInfo[sCount - 1].cInfo[sInfo[sCount - 1].cCount - 1].archivelogfile[0] = 0;
  sInfo[sCount - 1].cInfo[sInfo[sCount - 1].cCount - 1].daterotate = 0;
  sInfo[sCount - 1].cInfo[sInfo[sCount - 1].cCount - 1].daterotateformat[0] = 0;
  sInfo[sCount - 1].cInfo[sInfo[sCount - 1].cCount - 1].nInfo = malloc(sizeof(NickInfo));
  sInfo[sCount - 1].cInfo[sInfo[sCount - 1].cCount - 1].nInfo[0].nick[0] = 0;
  sInfo[sCount - 1].cInfo[sInfo[sCount - 1].cCount - 1].cDB = malloc(sizeof(CronDB));
}

void parse_config(char *c) {
  char buffer[BUFFER_SIZE];
  int status = csNone;
  FILE *fd;
  char *p, *q;
  int pos;

  if((fd = fopen(c, "rt")) == NULL) {
    perror("fopen");
    exit(csError);
  }

  generate_initial_info_struct();
  while(fgets(buffer, sizeof(buffer), fd) != NULL) {
    if(strstr(buffer, "}") != NULL) {
      status = (status == csChannel) ? csServer : csNone;
      switch(status) {
        case csServer:
          generate_new_channel_struct();
          break;
        case csNone:
          free(sInfo[sCount - 1].cInfo[sInfo[sCount - 1].cCount - 1].nInfo);
          sInfo[sCount - 1].cInfo = realloc(sInfo[sCount - 1].cInfo, sizeof(ChannelInfo) * (--sInfo[sCount - 1].cCount));
          sInfo[sCount - 1].status = csReady;
          generate_new_server_struct();
          generate_initial_channel_struct();
          break;
      }
    }
    else if((p = strtok(buffer, " ")) != NULL) {
      if(strcasecmp(p, "Server") == 0) {
        status = csServer;
        if((p = strtok(NULL, " {")) != NULL) {
          strncpy(sInfo[sCount - 1].server, p, sizeof(sInfo[sCount - 1].server));
        }
      }
      else if((status == csServer) && (strcasecmp(p, "Port") == 0)) {
        if((p = strtok(NULL, " \r\n")) != NULL) {
          strncpy(sInfo[sCount - 1].port, p, sizeof(sInfo[sCount - 1].port));
        }
      }
      else if((status == csServer) && (strcasecmp(p, "Nick") == 0)) {
        if((p = strtok(NULL, " \r\n")) != NULL) {
          strncpy(sInfo[sCount - 1].nick, p, sizeof(sInfo[sCount - 1].nick));
        }
      }
      else if((status == csServer) && (strcasecmp(p, "Ident") == 0)) {
        if((p = strtok(NULL, " \r\n")) != NULL) {
          strncpy(sInfo[sCount - 1].ident, p, sizeof(sInfo[sCount - 1].ident));
        }
      }
      else if((status == csServer) && (strcasecmp(p, "RealName") == 0)) {
        if((p = strtok(NULL, "\r\n")) != NULL) {
          strncpy(sInfo[sCount - 1].realname, p, sizeof(sInfo[sCount - 1].realname));
        }
      }
      else if((status == csServer) && (strcasecmp(p, "RawConnectCode") == 0)) {
        if((p = strtok(NULL, " \r\n")) != NULL) {
          strncpy(sInfo[sCount - 1].rawcode, p, sizeof(sInfo[sCount - 1].rawcode));
        }
      }
      else if((status == csServer) && (strcasecmp(p, "PingTimeOut") == 0)) {
        if((p = strtok(NULL, " \r\n")) != NULL) {
          strncpy(sInfo[sCount - 1].pingtimeout, p, sizeof(sInfo[sCount - 1].pingtimeout));
        }
      }
      else if((status == csServer) && (strcasecmp(p, "SelfPing") == 0)) {
        if((p = strtok(NULL, " \r\n")) != NULL) {
          if(strcasecmp(p, "ON") == 0) {
            sInfo[sCount - 1].selfping = 1;
          }
        }
      }
      else if((status == csServer) && (strcasecmp(p, "BindConnect") == 0)) {
        if((p = strtok(NULL, " \r\n")) != NULL) {
          strncpy(sInfo[sCount - 1].vhost, p, sizeof(sInfo[sCount - 1].vhost));
        }
      }
      else if(strcasecmp(p, "Channel") == 0) {
        status = csChannel;
        if((p = strtok(NULL, " {")) != NULL) {
          strncpy(sInfo[sCount - 1].cInfo[sInfo[sCount - 1].cCount - 1].channel, p, sizeof(sInfo[sCount - 1].cInfo[sInfo[sCount - 1].cCount - 1].channel));
        }
      }
      else if((status == csChannel) && (strcasecmp(p, "LogRotate") == 0)) {
        if((p = strtok(NULL, "\r\n")) != NULL) {
          strncpy(sInfo[sCount - 1].cInfo[sInfo[sCount - 1].cCount - 1].logrotate, p, sizeof(sInfo[sCount - 1].cInfo[sInfo[sCount - 1].cCount - 1].logrotate));
          pos = 0;
          if((q = strtok(p, " \0")) != NULL) {
            do {
              if(pos == 0) {
                if(*q == '*') {
                  sInfo[sCount - 1].cInfo[sInfo[sCount - 1].cCount - 1].cDB->minute = -1;
                }
                else {
                  sInfo[sCount - 1].cInfo[sInfo[sCount - 1].cCount - 1].cDB->minute = atoi(q);
                }
              }
              else if(pos == 1) {
                if(*q == '*') {
                  sInfo[sCount - 1].cInfo[sInfo[sCount - 1].cCount - 1].cDB->hour = -1;
                }
                else {
                  sInfo[sCount - 1].cInfo[sInfo[sCount - 1].cCount - 1].cDB->hour = atoi(q);
                }
              }
              else if(pos == 2) {
                if(*q == '*') {
                  sInfo[sCount - 1].cInfo[sInfo[sCount - 1].cCount - 1].cDB->dom = -1;
                }
                else {
                  sInfo[sCount - 1].cInfo[sInfo[sCount - 1].cCount - 1].cDB->dom = atoi(q);
                }
              }
              else if(pos == 3) {
                if(*q == '*') {
                  sInfo[sCount - 1].cInfo[sInfo[sCount - 1].cCount - 1].cDB->month = -1;
                }
                else {
                  sInfo[sCount - 1].cInfo[sInfo[sCount - 1].cCount - 1].cDB->month = atoi(q);
                }
              }
              else if(pos == 4) {
                if(*q == '*') {
                  sInfo[sCount - 1].cInfo[sInfo[sCount - 1].cCount - 1].cDB->dow = -1;
                }
                else {
                  sInfo[sCount - 1].cInfo[sInfo[sCount - 1].cCount - 1].cDB->dow = atoi(q);
                }
              }
              pos++;
            } while((q = strtok(NULL, " \0")) != NULL);
          }
        }
      }
      else if((status == csChannel) && (strcasecmp(p, "LogFile") == 0)) {
        if((p = strtok(NULL, " \r\n")) != NULL) {
          strncpy(sInfo[sCount - 1].cInfo[sInfo[sCount - 1].cCount - 1].logfile, p, sizeof(sInfo[sCount - 1].cInfo[sInfo[sCount - 1].cCount - 1].logfile));
        }
      }
      else if((status == csChannel) && (strcasecmp(p, "ArchiveLogFile") == 0)) {
        if((p = strtok(NULL, " \r\n")) != NULL) {
          strncpy(sInfo[sCount - 1].cInfo[sInfo[sCount - 1].cCount - 1].archivelogfile, p, sizeof(sInfo[sCount - 1].cInfo[sInfo[sCount - 1].cCount - 1].archivelogfile));
        }
      }
      else if((status == csChannel) && (strcasecmp(p, "DateRotate") == 0)) {
        if((p = strtok(NULL, " \r\n")) != NULL) {
          if(strcasecmp(p, "ON") == 0) {
            sInfo[sCount - 1].cInfo[sInfo[sCount - 1].cCount - 1].daterotate = 1;
          }
        }
      }
      else if((status == csChannel) && (strcasecmp(p, "DateRotateFormat") == 0)) {
        if((p = strtok(NULL, " \r\n")) != NULL) {
          strncpy(sInfo[sCount - 1].cInfo[sInfo[sCount - 1].cCount - 1].daterotateformat, p, sizeof(sInfo[sCount - 1].cInfo[sInfo[sCount - 1].cCount - 1].daterotateformat));
        }
      }
    }
  }

  if(sCount > 1) {
    free(sInfo[sCount - 1].cInfo[0].nInfo);
    free(sInfo[sCount - 1].cInfo);
    sInfo = realloc(sInfo, sizeof(ServerInfo) * (--sCount));
  }
  fclose(fd);
}

void show_version(void) {
  fprintf(stdout, "%s v%s\n%s\n", APPNAME, VERSION, COPYRIGHT);
  fprintf(stdout, "All Rights Reserved\n\n");
}

