/* Copyright (C) 2000-2004  Thomas Bopp, Thorsten Hampel, Ludger Merkens
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 * $Id: factory.pike,v 1.3 2005/05/02 21:18:48 exodusd Exp $
 */

constant cvs_version="$Id: factory.pike,v 1.3 2005/05/02 21:18:48 exodusd Exp $";

inherit "/classes/Object";

//! This is the factory class - other factories are derived from this one.
//! See the factory pattern for factories in general. 
//! A factory is used by calling the execute() function and passing a mapping
//! of params. Each factory used the param "name" for execution in order to
//! give a new object a name. The factory for a class is retrieved by calling
//! the globally available function "get_factory(int classbit)", for example
//! get_factory(CLASS_USER).

import Attributes;

#include <macros.h>
#include <roles.h>
#include <assert.h>
#include <attributes.h>
#include <classes.h>
#include <access.h>
#include <database.h>
#include <types.h>
#include <events.h>
#include <exception.h>

static mapping mRegAttributes =([ ]); // "attribute-name":Attribute

bool check_swap() { return false; }
bool check_upgrade() { return false; }


/**
 * Init callback function sets a data storage.
 *  
  */
static void
init()
{
    ::init();
    add_data_storage(STORE_ATTREG, retrieve_attr_registration,
                     restore_attr_registration);
}

static void init_factory()
{
    register_attribute(UserAttribute(OBJ_NAME, "object name", CMD_TYPE_STRING,""));
    register_attribute(UserAttribute(OBJ_DESC, "description", CMD_TYPE_STRING,""));
    register_attribute(UserAttribute(OBJ_ICON, "icon", CMD_TYPE_OBJECT, 0));
    register_attribute(UserAttribute(OBJ_LINK_ICON,"link icon",CMD_TYPE_OBJECT,0));
    register_attribute(Attribute(OBJ_URL,"url", CMD_TYPE_STRING, 0,
				 get_module("filepath:url")));
    register_attribute(UserAttribute(OBJ_KEYWORDS,"keywords",CMD_TYPE_ARRAY,({})));
    register_attribute
      (PositionAttribute(OBJ_POSITION_X, "x-position", CMD_TYPE_FLOAT, 0.0));
    register_attribute
      (PositionAttribute(OBJ_POSITION_Y, "y-position", CMD_TYPE_FLOAT, 0.0));
    register_attribute
      (PositionAttribute(OBJ_POSITION_Z, "z-position", CMD_TYPE_FLOAT, 0.0));
}

/**
 * A factory calls initialization of factory when it is loaded.
 *  
 */
static void load_object()
{
  if ( !mappingp(mRegAttributes) )
    mRegAttributes = ([]);
  init_factory();
}

/**
 * Object constructor. Here the Attribute registration mapping is initialized.
 *  
 */
static void create_object()
{
    mRegAttributes = ([]);
    init_factory();
    require_save(STORE_ATTREG);
}

/**
 * See if a given name is valid for objects created by this factory.
 *  
 * @param string name - the name of the object
 */
void valid_name(string name)
{
    if ( !stringp(name) )
	steam_error("The name of an object must be a string !");
    if ( search(name, "/") >= 0 )
	steam_error("/ is not allowed in Object Names...");
    if ( !xml.utf8_check(name) )
	steam_error("Name of object is not utf-8 !");
}


/**
 * create a new object of 'doc_class'
 *  
 * @param string name - the name of the new object
 * @param string doc_class - the class of the new object
 * @param object env - the env the object should be moved to
 * @param mapping|int attr - attribute mapping for initial attribute settings
 * @param void|mapping attrAcq - acquired attributes
 * @param void|mapping attrLock - locked attributes initialization
 * @return pointer to the new object
 */
static object
object_create(string name, string doc_class, object env, int|mapping attr,
	      void|mapping attrAcq, void|mapping attrLock)
{
    object obj, user;
    int          res;

    user = this_user();
    doc_class = CLASS_PATH + doc_class + ".pike";
    SECURITY_LOG("New object of class:" + doc_class + " at " + ctime(time()));

    valid_name(name);

    if ( mappingp(attr) ) {
      foreach(values(attr), mixed v) {
	if ( stringp(v) && !xml.utf8_check(v) )
	  error("Cannot create user: Invalid Attribute found (utf-8 check failed).");
      }
    }



    if ( objectp(env) && !(env->get_object_class() & CLASS_CONTAINER) )
      steam_error("Failed to move object to " + env->get_object_id() +
		  " : object is no container !");

    obj = new(doc_class, name);
    if ( !objectp(obj) )
	THROW("Failed to create object !", E_ERROR);

    install_attributes(obj->this());

    if ( !stringp(name) || name == "" ) 
        THROW("No name set for object !", E_ERROR);

    if ( mappingp(attr) )
	obj->set_attributes(attr);

    obj->set_attribute(OBJ_NAME, name);
    obj->set_attribute(OBJ_CREATION_TIME, time());

    obj->set_acquire_attribute(OBJ_ICON, _Server->get_module("icons"));
    
    if ( !stringp(obj->query_attribute(OBJ_NAME)) || 
         obj->query_attribute(OBJ_NAME) == "" )
       THROW("Strange error - attribute name setting failed !", E_ERROR);
    
    SECURITY_LOG("Object " + obj->get_object_id() + " name set on " +
		 ctime(time()));
    
    if ( !objectp(user) )
	user = MODULE_USERS->lookup("root");
    obj->set_creator(user);
    
    if ( user != MODULE_USERS->lookup("root") && 
	 user != MODULE_USERS->lookup("guest") )
    {
	obj->sanction_object(user, SANCTION_ALL);
	obj->sanction_object_meta(user, SANCTION_ALL);
    }
    obj->set_acquire(obj->get_environment);
    ASSERTINFO(obj->get_acquire() == obj->get_environment,
	       "Acquire not on environment, huh?");

    obj->created();

    if ( objectp(env) ) {
	obj->move(env->this());
    }

    return obj->this();
}

static bool install_attribute(Attribute attr, object obj)
{
    if ( obj->status() < 0 )
      return true;

    mixed err = catch {
	mixed key = attr->get_key();
	mixed def = attr->get_default_value();
	if ( !zero_type(def) )
	    obj->set_attribute(key, def);

	// do not acquire attributes for the default objects set
	string|object acq = attr->get_acquire();
	if ( !objectp(acq) || acq != obj ) 
	{ 
	    if ( stringp(acq) )
		obj->set_acquire_attribute(key, obj->find_function(acq));
	    else 
		obj->set_acquire_attribute(key, acq);
	}
	return true;
    };
    FATAL("Error registering attribute: %s\n%s", 
	  err[0],
	  describe_backtrace(err[1]));
}

/**
 * register all attributes for an object
 *  
 * @param obj - the object to register attributes
 * @see register_class_attribute
 */
static void install_attributes(object obj, string|void key)
{
    Attribute attr;
    
    if ( stringp(key) ) {
	attr = mRegAttributes[key];
	if ( !objectp(attr) )
	    return;
	install_attribute(attr, obj->this());
    }
    else {
	foreach (indices(mRegAttributes), key)  {
	    attr = mRegAttributes[key];
	    install_attribute(attr, obj->this());
	}
    }
}

/**
 * register attributes for the class(es) this factory creates.
 * each newly created object will have the attributes registered here.
 *  
 * @param Attribute attr - the new attribute to register.
 * @param void|function conversion - conversion function for all objects
 *
 * @see classes/Object.set_attribute 
 * @see libraries/Attributes.pmod.Attribute
 */
void 
register_attribute(Attribute attr, void|function conversion)
{
    try_event(EVENT_REGISTER_ATTRIBUTE, CALLER, attr);
    register_class_attribute(attr, conversion);

    // register on all dependent factories too
    array(object) factories = values(_Server->get_classes());
    foreach ( factories, object factory ) {
	factory = factory->get_object();
	if ( factory->get_object_id() == get_object_id() )
	    continue;
	if ( search(Program.all_inherits(object_program(factory)),
		    object_program(this_object())) >= 0 )
	    factory->register_attribute(copy_value(attr), conversion);
    }
    run_event(EVENT_REGISTER_ATTRIBUTE, CALLER, attr);
}

/**
 * Change the registration of an attribute.
 *  
 * @param mixed key - the attribute to re-register.
 * @param object attrib - new attribute registration object.
 * @param void|bool|function conv - update all instances, might be a
 *                                  conversion function
 */
void 
re_register_attribute(mixed key, object attrib, function conv)
{
    try_event(EVENT_REGISTER_ATTRIBUTE, CALLER, key);
    Attribute attr = mRegAttributes[key];
    if ( !objectp(attr) )
	THROW("Trying to re-register unregistered attribute!", E_ERROR);
    mRegAttributes[key] = attrib;
    require_save(STORE_ATTREG);
    if ( functionp(conv) )
	update_instances(key, conv);
    array(object) factories = _Server->get_factories();
    foreach(factories, object factory) {
	if ( factory->get_class_id() > get_class_id() )
	    factory->re_register_attribute(key, conv);
    }
    run_event(EVENT_REGISTER_ATTRIBUTE, CALLER, key);
}

/**
 * Update instances of objects created by this factory when a new 
 * attribute is registered. This sets the new default value for the attribute
 * and the basic acquiring.
 *  
 * @param mixed key - the attribute key.
 * @param function|void conv - the conversion function.
 */
static void
update_instances(mixed key, function|void conv)
{
    Attribute attr = mRegAttributes[key];
    if ( !objectp(attr) )
	THROW("Unregistered Attribute !", E_ERROR);
    
    array(object) instances = _Database->get_all_objects();
    foreach(instances, object instance) {
	if ( !objectp(instance) || !functionp(instance->get_object_class) ||
	     !(instance->get_object_class() & get_class_id()) )
	    continue;
        if ( functionp(conv) ) {
	    mixed err = catch(conv(instance));
	    if ( err )
	      FATAL("Error updating instance: %O", err);
	}
	else {
	    install_attribute(attr, instance);
	}
    }
}


/**
 * Register_class_attribute is called by register_attribute,
 * this function is local and does no security checks. All instances
 * of this class are set to the default value and acquiring settings.
 *  
 * @param Attribute attr - the Attribute to register for this factories class
 * @param void|function conversion - conversion function.
 * @see register_attribute
 */
static void register_class_attribute(Attribute attr, void|function conversion)
{
    string|int key = attr->get_key();
    Attribute pattr = mRegAttributes[key];
    if ( pattr == attr ) {
	return;
    }
    
    mRegAttributes[key] = attr;
    require_save(STORE_ATTREG);

    if (!_Server->is_a_factory(this_object()) || _Server->is_a_factory(CALLER)) 
	return; // initial creation of this object or CALLER already converted

    update_instances(key, conversion);
}


/*
 * Init an attribute of this class calls registration function.  
 *
 * @see register_class_attribute
 */
static void 
init_class_attribute(mixed key, int type, string desc, 
		     int event_read, int event_write, 
		     object|int acq, int cntrl, mixed def)
{
    Attribute attr = Attribute(key, desc, type, def, acq, 
			       cntrl, event_read, event_write);
    if ( !objectp(mRegAttributes[key]) )
	register_class_attribute(attr);
}

/**
 * Check if an attributes value is going to be set correctly.
 * An objects set_attribute function calls this check and
 * throws an error if the value is incorrect.
 *  
 * @param mixed key - the attributes key.
 * @param mixed data - the new value of the attribute.
 * @param int|void regType - registration data to check, if void use factories.
 * @return true or false.
 */
bool check_attribute(mixed key, mixed data, int|void regType)
{
    object caller = CALLER;

    if ( !objectp(mRegAttributes[key]) ) 
	return true;

    // see if our factory has something about this attribute
    // if previous attribute in zero
    // this will end up in loop when installing attributes
    if ( zero_type(caller->query_attribute(key)) )
	install_attributes(CALLER, key);
    

    // value 0 should be ok
    if ( data == 0 ) return true;

    return mRegAttributes[key]->check_attribute(data);
}

/**
 * Get the registration information for one attribute of this class.
 *  
 * @param mixed key the attributes key.
 * @return The array of registered data.
 * @author Thomas Bopp (astra@upb.de) 
 */
Attribute describe_attribute(mixed key)
{
  return copy_value(mRegAttributes[key]);
}

/**
 * Get all registered attributes for this class.
 *  
 * @return the mapping of registered attributes for this class
 * @author Thomas Bopp (astra@upb.de) 
 * @see register_class_attribute
 */
mapping get_attributes()
{
    return copy_value(mRegAttributes);
}

/**
 * Get the event to fire upon reading the attribute.
 *  
 * @param mixed key - the attributes key.
 * @return read event or zero.
 * @author <a href="mailto:astra@upb.de">Thomas Bopp</a>) 
 * @see get_attribute_change_event
 */
int get_attributes_read_event(mixed key)
{
    if ( !arrayp(mRegAttributes[key]) )
	return 0;
    return mRegAttributes[key]->get_read_event();
}

/**
 * Get the event to fire upon changing an attribute.
 *  
 * @param mixed key - the attributes key.
 * @return change event or zero.
 * @author <a href="mailto:astra@upb.de">Thomas Bopp</a>) 
 * @see get_attributes_read_event
 */
int get_attributes_change_event(mixed key)
{
    if ( !mappingp(mRegAttributes) || !objectp(mRegAttributes[key]) )
	return EVENT_ATTRIBUTES_CHANGE;
    return mRegAttributes[key]->get_write_event();
}

/**
 * Get an attributes default value and acquiring.
 *  
 * @param mixed key - the attributes key.
 * @return array of default value and acquiring setting.
 */
array get_attribute_default(mixed key) 
{
    return ({ mRegAttributes[key]->get_default_value(),
		  mRegAttributes[key]->get_acquire() });
}

/**
 * Called by the _Database to get the registered attributes (saved data)
 * for this factory.
 *  
 * @return mapping of registered attributes.
 */
final mapping
retrieve_attr_registration()
{
    if ( CALLER != _Database )
	THROW("Invalid call to retrieve_data()", E_ACCESS);
    
    return ([ 
      "RegAttributes":map(mRegAttributes, save_attribute),
    ]);
}

static mapping save_attribute(Attribute attr) 
{
    return attr->save();
}

/**
 * Called by _Database to restore the registered attributes data.
 *  
 * @param mixed data - restore data.
 */
final void 
restore_attr_registration(mixed data)
{
    if ( CALLER != _Database )
	THROW("Invalid call to restore_data()", E_ACCESS);
    foreach(indices(data->RegAttributes), mixed key) { 
	if ( !stringp(key) )
	    continue;
	mixed v = data->RegAttributes[key];
	if ( arrayp(v) ) {
	    mixed acq = v[4];
	    if ( intp(acq) && acq == 1 )
		acq = REG_ACQ_ENVIRONMENT;
	    Attribute a = Attribute(key,v[1],v[0],v[6],acq,v[5],v[2],v[3]);
	    mRegAttributes[key] = a;
	}
	else {
	    Attribute a = Attribute(v->key,v->desc,v->type,v->def,v->acquire,
				    v->control, v->event_read, v->event_write);
	    mRegAttributes[key] = a;
	}
    }
}

string get_identifier() { return "factory"; }
int get_object_class() { return ::get_object_class() | CLASS_FACTORY; }
string get_class_name() { return "undefined"; }
int get_class_id() { return CLASS_OBJECT; }
















