import Visconte.SemWeb;

/**
 * <p>Class Ontology</p>
 * contoller class for the translation betweenn RDF/OWl and OWLItems.
 * Manages the input/output with a storage handler. Further Responsible for the 
 * logging of an ontology
 */

/**
 * <p>Represents the xml schema namespace</p>
 */
    public constant xml_schema_base="http://www.w3.org/2001/XMLSchema";
    
/**
 * <p>Represents the xml schema namespace</p>
 */
    public constant THING="http://www.w3.org/2002/07/owl#Thing";

/**
 * <p>Represents the xml schema namespace</p>
 */
   private string template;
/**
 * <p>Represents the storage handler to load/save this ontology</p>
 */
    private object storage_handler;

/**
 * <p>Represents the ontology version</p>
 */
    private int version=0;
   
/**
 * <p>Represents the ontology filename</p>
 */
    private string ontology_file;

/**
 * <p>Represents the ontology filename</p>
 */
    private string ontology_log_file_suffix=".log";
    
/**
 * <p>Represents the ontology filename</p>
 */
    public  Parser.XML.Tree.Node ontology_log;
    
/**
 * <p>Represents the user who init this action, default anonym</p>
 */
    protected string user = "annonym";
    
/**
 * <p>The ontology as rdfset representation</p>
 */
    private OWL owlset;
    
/**
 * <p>The backup ot the current ontology</p>
 */
    private string backup;
    
/**
 * <p>The ObjectProperty Statement</p>
 */
    private OWL.OWLResource owl_ObjectProperty;
    
/**
 * <p>The DataTypeProperty Statement</p>
 */
    private OWL.OWLResource owl_DatatypeProperty;

/**
 * <p>The DataTypeProperty Statement</p>
 */
    private OWL.OWLResource owl_Ontology;
    
/**
 * <p>The DataTypeProperty Statement</p>
 */
    private OWL.OWLResource owl_versionInfo;  

/**
 * <p>The TransitiveProperty</p>
 */
    private OWL.OWLResource owl_TransitiveProperty;
    
/**
 * <p>The SymetricProperty</p>
 */
    private OWL.OWLResource owl_SymetricProperty;

/**
 * <p>Meta object for author</p>
 */    
    private RDF.URIResource anot_creator;
    
/**
 * <p>Meta object for author</p>
 */    
    private RDF.URIResource anot_changes;
    
/**
 * <p>Meta object for date of change</p>
 */    
    private RDF.URIResource anot_created;
    
/**
 * <p>The rdfs_label statement</p>
 */
    private RDFS.RDFSResource rdfs_label;

/**
 * <p>The Visconte.Onto.Item.OWLItem Factory to produce items</p>
 */
    private Visconte.Onto.OWLItemFactory factory = Visconte.Onto.OWLItemFactory();


    
/**
 * <p>Constructor: creates a new Ontology representation. If the ontology file not exists, 
 * a new ontology will be started</p>
 * 
 * 
 * @param file - the ontology file
 * @param version - the version of the last ontology edition
 * @param storage_handler - the interface of the storage handler to load and save the ontology
 * @param user - the user which operates on the ontology
 */
    public void create(string ontology_file, int version, object storage_handler, string|void user) 
    {        
        
	    init();
	    this_program::ontology_file=ontology_file;
	    this_program::version=version;
	    this_program::storage_handler = storage_handler;
	    if(user)
		    this_program::user=user;
	    
	    load();
	    
    } 
/**
 * <p>Inits common OWLResources and loads the ontology.</p>
 * 
 * 
 */
    private void init() 
    {        
        owlset = OWL();
	//set commomn OWL Ressources
	owl_ObjectProperty = owlset->OWLResource("ObjectProperty");
	owl_DatatypeProperty = owlset->OWLResource("DatatypeProperty");
	owl_TransitiveProperty = owlset->OWLResource("TransitiveProperty");
	
	//meta Items
	owl_Ontology = owlset->OWLResource("Ontology");
	owl_versionInfo = owlset->OWLResource("versionInfo");
	anot_creator = owlset->URIResource("http://steam.upb.de/annot#creator");
	anot_created = owlset->URIResource("http://steam.upb.de/annot#created");
	anot_changes = owlset->URIResource("http://steam.upb.de/annot#changes");

	
	//set rdfs:label statement
	rdfs_label = owlset->RDFSResource("label");
	//load the ontology-file;

    } 
    
    
/**
 * <p>Get an Item with a special id</p>
 * This method will be check the type of the id and delegate it to the special functions
 * e.g. get_class etc. and finally returns a Visconte.Onto.Item.OWLItem
 * 
 * @param id - the id of the item as string
 * @return Visconte.Onto.Item.OWLItem
 */
    public Visconte.Onto.Item.OWLItem get_Item(string id) 
    {   
	
	Visconte.Onto.Item.OWLItem item;
	if (!item_exists(id))
	{
		
		throw( ({ "", backtrace(),11, id}) );
		
	}
	
	int type = get_type(id);
        switch(type)
	{
		case Visconte.Onto.Item.OWLItem.CLASS:
		item = get_Class(id);
		break;
		
		case Visconte.Onto.Item.OWLItem.OBJECT_PROPERTY:
		item = get_Property(id);
		break;
		
		case Visconte.Onto.Item.OWLItem.DATATYPE_PROPERTY:
		item = get_Property(id);
		break;
		
		case Visconte.Onto.Item.OWLItem.INDIVIDUAL:
		item = get_Indiviudal(id);
		break;
		
		default:
		 throw( ({ "",  backtrace(),601, id}) );
		break;
	}
	return item;
    } 

    
/**
 * <p>Adds an Visconte.Onto.Item.OWLItem to the ontology</p>
 * By checking the type of the item, this method delegates to specific methods lilke add_Class() etc.
 * 
 * @param item 
 * @return int success
 */
    public int add_Item(Visconte.Onto.Item.OWLItem item) 
    {    
	    
	//is this a valid item
	if(!item)
	{
		  throw( ({ "", backtrace(),11, 0}) );
		  return 0;
	}
	
        if(item_exists(item))
	{
			throw( ({ "", backtrace(),602,item->get_Id()}) );		  
			return 0;
	}

	switch(item->ITEMTYPE)
	{
		//if class
		case Visconte.Onto.Item.OWLItem.CLASS:
		return add_Class(item);
		break;
		
		case Visconte.Onto.Item.OWLItem.OBJECT_PROPERTY:
		return add_ObjectProperty(item);
		break;
		
		case Visconte.Onto.Item.OWLItem.DATATYPE_PROPERTY:
		return add_DatatypeProperty(item);
		break;
		
		case Visconte.Onto.Item.OWLItem.INDIVIDUAL:
		return add_Individual(item);
		break;
		
		default:
		 throw( ({ "", backtrace(),601,item->get_Id()}) );
		break;
		
	}
	return 1;
	
    } 

    
// Class Related
    
/**
 * <p>Adds a class to the ontology</p>
 * @param item - the OWLClass to add * 
 * @return success
 */
 private int add_Class(Visconte.Onto.Item.OWLItem item)
 {
	add_statements(create_Class_statement(item));
	set_meta(item);
	return 1;
 }
    

/**
 * <p>Get a class with a specific id</p>
 * Constructs a class of the given id and returns an Visconte.Onto.Item.OWLClass object
 *
 * @param id - the id as string
 * @return Visconte.Onto.Item.OWLClass class
 * 
 */
    public Visconte.Onto.Item.OWLClass get_Class(string id) 
    {  
	 
	    if (item_exists(id))
	    {
		    mixed parent_id;
		
		    RDF.Resource item = owlset->get_resource(id);
		    
		    //get the parent class (if one)
		    parent_id = resource_to_id(query(item,owlset->rdfs_subClassOf,0),2);
		 
		    //get the properties related to this class (only the one in the domain)
		    array properties = resources_to_ids(query(0,owlset->rdfs_domain,id), 0);
		    
		    //get the properties related to this class (only the one in the range)
		    array range_properties = resources_to_ids(query(0,owlset->rdfs_range,id), 0);
		    
		    //get the descendants ot this class
		    array childs = resources_to_ids(query(0,owlset->rdfs_subClassOf,item), 0);
		    
		    //get the diffenrent form classes
		    array different_from = resources_to_ids(query(item,owlset->owl_differentFrom,0),2);
		   
		    //get the diffenrent form classes
		    array equivalent_classes = resources_to_ids(query(item,owlset->owl_equivalentClass,0),2);
		    
		    array individuals = resources_to_ids(query(0,owlset->rdf_type,item),0);
		     
		    //construct the class
		    Visconte.Onto.Item.OWLClass owl_class = Visconte.Onto.Item.OWLClass(this_object(),id,parent_id,get_label(id),properties, range_properties, childs, individuals,equivalent_classes,different_from);
		    return owl_class;
	    }
	    else
	    {
		    throw( ({ "", backtrace(),11, id}) );
		    return 0;
	    }
	   
    }


/**
 * <p>Transforms a Visconte.Onto.Item.OWLClass to a Container of RDF-Statements</p>
 * 
 * @param item - the class to translate into statement
 * @return Visconte.Onto.StatementContainer - the statements
 */ 
 private Visconte.Onto.StatementContainer create_Class_statement(Visconte.Onto.Item.OWLItem item)
 {
	Visconte.Onto.StatementContainer stmt_container = Visconte.Onto.StatementContainer();
	 //construct Resource
	OWL.OWLResource class_resource = owlset->make_resource(item->get_Id());
	//add the class
	stmt_container->add(class_resource,owlset->rdf_type, owlset->owl_Class);
	//add the label
	stmt_container->add(class_resource,rdfs_label,owlset->LiteralResource(item->get_Label()));
	
	//add equivalent classes
	Visconte.Onto.OWLContainer c = item->get_equivalent_classes();
	Visconte.Onto.Item.OWLItem eq_class, diff_class;
	while(c->has_items())
	{
		eq_class=c->next();
		stmt_container->add(class_resource,owlset->owl_equivalentClass,owlset->get_resource(eq_class->get_Id()));
	}
	//add different classes
	Visconte.Onto.Item.OWLItem dif_class;
	c=item->get_different_from_classes();
	while(c->has_items())
	{
		diff_class=c->next();
		stmt_container->add(class_resource,owlset->owl_differentFrom,owlset->get_resource(diff_class->get_Id()));
	}
	
        
	//set as subclass if parent exists
	if(item->has_parent() && item_exists(item->get_parent()))
	{
		stmt_container->add(class_resource,owlset->rdfs_subClassOf,owlset->get_resource( item->get_parent()->get_Id() ) );
		
	}
	else
	{
		stmt_container->add(class_resource,owlset->rdfs_subClassOf,owlset->get_resource(THING));
	}
	return stmt_container;
 }



// Individual related
 
/**
 * <p>Adds an Individual to the ontology</p>
 * 
 * @param item - OWLIndividual to add
 * @return int success
 */
 private int add_Individual(Visconte.Onto.Item.OWLItem item)
 {
	
	 add_statements(create_Individual_statement(item));
	 set_meta(item);
	 return 1;
 } 

 /**
 * <p>Get a individual with a specific id</p>
 * Constructs a OWLIndividual of the given id and returns an Visconte.Onto.Item.OWLIndividual object
 *
 * @param id - the id as string
 * @return Visconte.Onto.Item.OWLIndividual individual
 * 
 */
    public Visconte.Onto.Item.OWLIndividual get_Indiviudal(string id) 
    {
	    if (item_exists(id))
	    {
	    
		     RDF.Resource individual = owlset->get_resource(id);
		     
		     string class_id = resource_to_id(query(individual,owlset->rdf_type,0),2);
		     Visconte.MVList property_values =Visconte.MVList();
		     Visconte.Onto.OWLContainer all_properties = Visconte.Onto.OWLContainer();
		     Visconte.Onto.Item.OWLClass model_class = get_Item(class_id);
		     do
		     {
			     Visconte.Onto.OWLContainer property_container = model_class->get_domain_properties();
			     
			     while(property_container->has_items())
			     {
				     Visconte.Onto.Item.OWLItem item = property_container->next();
				     //add this property to all inherited properties
				     all_properties->add(item);
				     
				     RDF.Resource property = owlset->get_resource(item->get_Id());
				     array resources;
				     if(item->ITEMTYPE==Visconte.Onto.Item.OWLItem.OBJECT_PROPERTY)
				     {
					      resources = resources_to_ids(query(individual,property,0),2);
				     }
				     else
				     {
					      resources = resources_to_literals(query(individual,property,0),2);
				     }
				     if(sizeof(resources)>0)
					       property_values->add(item->get_Id() ,resources );
			     } 
			     
			     model_class=model_class->get_parent();
		     }
		     while(model_class!=0);
		  
		    // write("RETR:"+id+"\n");
		    return Visconte.Onto.Item.OWLIndividual( this_object(), id,class_id, get_label(id), all_properties, property_values);
	     }
	    else
	    {
		    throw( ({ "", backtrace(),11, id}) );
		    return 0;
	    }
    }

/**
 * <p>Transforms a Visconte.Onto.Item.OWLIndividual to a Container of RDF-Statements</p>
 * 
 * @param item - the class to translate into statement
 * @return Visconte.Onto.StatementContainer - the statements
 */ 
 private Visconte.Onto.StatementContainer create_Individual_statement(Visconte.Onto.Item.OWLItem item)
 {

	 Visconte.Onto.StatementContainer stmt_container = Visconte.Onto.StatementContainer();
	
	 Visconte.Onto.Item.OWLClass model_class = item->get_model_class();
	 //construct Resource
	 
	 OWL.OWLResource individual_resource = owlset->make_resource(item->get_Id());
	 
	 OWL.OWLResource model_resource;
	 //test if the model class  exists 
	 if(item_exists(model_class))
		 model_resource = owlset->get_resource(model_class->get_Id());
	 else
		 model_resource = owlset->make_resource(THING);
	 
	 //add Individual
	 stmt_container->add(individual_resource,owlset->rdf_type,model_resource);
	 //add the label
	 stmt_container->add(individual_resource,rdfs_label,owlset->LiteralResource(item->get_Label()));
	
	 //add the properties by testing the defined properties in the modelclass and it parents
	 do
	 {
		 if(item_exists(model_class))
		 {
			 array(array(mixed)) property_map = item->get_property_map();
			 Visconte.Onto.OWLContainer control_properties = model_class->get_domain_properties();
			 
			 while(control_properties->has_items())
			 {
				  
				Visconte.Onto.Item.OWLItem control_property = control_properties->next();
				string control_id=control_property->get_Id();
				foreach(property_map, array prop_value)
				{
					string property_id=prop_value[0]->get_Id();
					if(property_id==control_id)
					{
						 OWL.OWLResource value_resource;
						 if(control_property->ITEMTYPE==Visconte.Onto.Item.OWLItem.OBJECT_PROPERTY)
							value_resource=owlset->get_resource(prop_value[1]);
						 else
							value_resource=owlset->LiteralResource(prop_value[1]);
							
						 stmt_container->add(individual_resource,owlset->get_resource(control_id),value_resource);
					}
				}
				 
			 }
		 }
		  model_class=model_class->get_parent();
	 }
	 while(model_class!=0);
		
	 return stmt_container;
	 
	 
 }

// Property Related



/**
 * <p>Adds a Object Property to the ontology</p>
 * 
 * 
 * @param item - the OWLObjectProperty to add
 * @return int success
 */
 private int add_ObjectProperty(Visconte.Onto.Item.OWLItem item)
 {
	
	add_statements(create_ObjectProperty_statement(item));
	set_meta(item);
	return 1;
		
 } 
 
 
/**
 * <p>Adds a Datatype Property to the ontology</p>
 * 
 * @param item - the OWLDatatypeProperty to add
 * @return int success
 */
 private int add_DatatypeProperty(Visconte.Onto.Item.OWLItem item)
 {
	add_statements(create_DatatypeProperty_statement(item));
	 set_meta(item);
	return 1;
		
 } 
 
/**
 * <p>Get a property with a specific id</p>
 *
 * @param id - the id as string
 * @return OWLProperty property
 * 
 */
    public Visconte.Onto.Item.OWLProperty get_Property(string id) 
    { 
	    Visconte.Onto.Item.OWLProperty property;
	    string domain, range, range_namespace;
	    
	     if (item_exists(id))
	     {
		RDF.Resource item = owlset->get_resource(id);
		//retrieve if its a Datatype or an Object Property by querieing the range type
		
		range=resource_to_id(query(item,owlset->rdfs_range,0),2);
		domain=resource_to_id(query(item,owlset->rdfs_domain,0),2);
		
		//test for datatype
		if(has_prefix(range,"xsd:"))
		{
				//construct the property and construct range as xsd:type statement
				property = Visconte.Onto.Item.OWLDatatypeProperty(this_object(),id,get_Property_restriction(id),get_label(id),get_Class(domain),range);
		}
		else
		{			
				property = Visconte.Onto.Item.OWLObjectProperty(this_object(),id,get_Property_restriction(id),get_label(id),get_Class(domain),get_Class(range));
		}
		
		return property;
	     }
	     else
	     {
		    
		     throw( ({ "",  backtrace(),11, id}) );		   
		     return 0;
	     }
    }    
    
/**
 * <p>Get property restrictions of a specific id</p>
 * NOTE: This method has currently no effect
 * @param id - the id as string
 * @return mapping with restrictions
 * 
 */
    private mapping get_Property_restriction(string id) 
    {     
	   // array restriction;
	    return (["transitive" : 1, "symetric" : 1]);
    }
    

/**
 * <p>Construct the resources container for the Object Property</p>
 * 
 * 
* @param item - the property (OWLObjectProperty) to translate into statement
 * @return Visconte.Onto.StatementContainer - the statements
 */
 private Visconte.Onto.StatementContainer create_ObjectProperty_statement(Visconte.Onto.Item.OWLItem item)
 {
	Visconte.Onto.StatementContainer stmt_container = Visconte.Onto.StatementContainer();

	
	OWL.OWLResource property_resource = owlset->make_resource(item->get_Id());
	OWL.OWLResource domain;
	OWL.OWLResource range;
	
	if(item_exists(item->get_domain()))
	{
		domain = owlset->get_resource(item->get_domain()->get_Id());
	}
	else
	{
		domain = owlset->make_resource(THING);
	}
	
	if(item_exists(item->get_range()))
	{
		range = owlset->get_resource(item->get_range()->get_Id());
	}
	else
	{
		range = owlset->make_resource(THING);
	}
        
	
	stmt_container->add(property_resource,owlset->rdf_type,owl_ObjectProperty);
	stmt_container->add(property_resource,owlset->rdfs_domain,domain);
	stmt_container->add(property_resource,owlset->rdfs_range,range);
	stmt_container->add(property_resource,rdfs_label,owlset->LiteralResource(item->get_Label()));
	
	return stmt_container;
		
 }

/**
 * <p>Adds a Datatype Property to the ontology</p>
 * 
 * @param item - the property (OWLDatatypeProperty) to add
 * @return success
 */
 private Visconte.Onto.StatementContainer create_DatatypeProperty_statement(Visconte.Onto.Item.OWLItem item)
 {
	Visconte.Onto.StatementContainer stmt_container = Visconte.Onto.StatementContainer();
	
	OWL.OWLResource property_resource = owlset->make_resource(item->get_Id());
	OWL.OWLResource domain;
	OWL.OWLResource range;
	
	if(item_exists(item->get_domain()))
	{
		domain = owlset->get_resource(item->get_domain()->get_Id());
	}
	else
	{
		domain = owlset->make_resource(THING);
	}
	
	//the range of the property
	range = owlset->make_resource(xml_schema_base+"#"+item->get_range());
	
	stmt_container->add(property_resource,owlset->rdf_type,owl_DatatypeProperty);
	stmt_container->add(property_resource,owlset->rdfs_domain,domain);
	stmt_container->add(property_resource,owlset->rdfs_range,range);
	stmt_container->add(property_resource,rdfs_label,owlset->LiteralResource(item->get_Label()));

	return stmt_container;
		
 }


 
 // Statement handling

 /**
 * <p>add the statement to the ontology</p>
 * 
 * @param Visconte.Onto.StatementContainer container - the statements to add 
 * @return 
 */
 private int add_statements(Visconte.Onto.StatementContainer container)
 {
	mapping stmt;
	while(container->has_statements())
	{
		stmt = container->next();
		owlset->add_statement(stmt["subj"],stmt["pred"],stmt["obj"]);
	}
	
	return 1;
 }
 
 /**
 * <p>removes statements from the ontology</p>
 * 
 * @param Visconte.Onto.StatementContainer container - the statements to remove 
 * @return 
 */
 private int remove_statements(Visconte.Onto.StatementContainer container)
 {
	mapping stmt;
	while(container->has_statements())
	{
		stmt = container->next();
		// workaround for literals
		if(stmt["obj"]->is_literal_resource)
		{
			 array lit = query(stmt["subj"],stmt["pred"],0);
			 if(!query_is_empty(lit))
			 stmt["obj"]=lit[0][2];
			
		}
		owlset->remove_statement(stmt["subj"],stmt["pred"],stmt["obj"]);
	}
 }
/**
 * <p>upadtes statements in the ontology</p>
 * 
 * 
 * @param Visconte.Onto.StatementContainer container - the statements to update 
 * @return 
 */
 private int update_statements(Visconte.Onto.StatementContainer old, Visconte.Onto.StatementContainer new)
 {
	      remove_statements(old);
	      add_statements(new);
 }



/**
 * <p>returns a label as mapping in the form "lang":"label".</p>
 * Currently this is only en for english
 * @param id - the id of the OWLItem 
 * @return label - the mapping with the label(s) 

 */
    public mapping get_label(string id)
    {
	    string label=id;
	    string lang="en";
	    array label_query = query(id,rdfs_label,0);
	    if(!query_is_empty(label_query))
	    {
		    label=label_query[0][2]->get_literal();
	    }
	    return ([lang : label]);
    }
   

/**
 * <p>Checks if a query array is empty</p>
 * 
 * 
 * @param query - the query array  
 * @return int success
 */
    public int query_is_empty(array query)
    {
	    if(sizeof(query)>0)
		    return 0;
	    else
		    return 1;
    }
    
/**
 * <p>Transforms resources to literals and returns them as array</p>
 * 
 * 
 * @param query - the query array
 * @param statement_position - 0=Subject, 1=Predicate, 2=Object
 * @return literals - array with literals
 */
    public array resources_to_literals(array query,int statement_position)
    {
	    
	string literal;
	array literals=({ });
	if(!query_is_empty(query))
	{
		foreach(query,array result)
		{
			if(result[statement_position]->is_literal_resource)
			{
				literal=result[statement_position]->get_literal();
				literals+=({literal});
			}
		}
	}
	return literals;
    }   
    
/**
 * <p>Transforms a single resources to an id and returns it</p>
 * 
 * @param query - the query array
 * @param statement_position - 0=Subject, 1=Predicate, 2=Object
 * @return id - the id as string
 */
    public string resource_to_id(array query,int statement_position)
    { 
	string id;
	if(!query_is_empty(query))
	{
		//debug(sprintf("%O",query));
		if(query[0][statement_position]->is_literal_resource)
			
			id=namespace_conversion(query[0][statement_position]->get_literal());
		else
			id=Visconte.ID(namespace_conversion(query[0][statement_position]->get_qname()))->ref;
		
	}
	
	return id;
    }
    
/**
 * <p>Transforms resources to ids and returns them as array</p>
 * 
 * @param query - the query array
 * @param statement_position - 0=Subject, 1=Predicate, 2=Object
 * @return id - array with ids
 */
    public array resources_to_ids(array query,int statement_position)
    {
	string id;
	array ids=({ });
	
	if(!query_is_empty(query))
	{	
		foreach(query,array result)
		{
			if(!result[2]->is_literal_resource)
			{
				id=Visconte.ID(namespace_conversion(result[statement_position]->get_qname()))->ref;
				ids+=({id});
			}
		}
	}
	return ids;
    }       

/**
 * <p>Checks an id for trailing basenames an returns them a modfied id</p>
 * 
 * 
 * @param id - the id to check 
 * @return the id
 */
    public string namespace_conversion(string id)
    {
	    int basename_separator_pos = search(id,":");
	    if(basename_separator_pos>=0)
	    {
		    string base = id[0..basename_separator_pos];
		    string identifier = id[basename_separator_pos+1..];                  
		    switch (base)
		    {
			    case "xsd:":
			    return id;
			    break;
			    
			    case "owl:":
			    return id;
			    break;
			    
			    case "rdfs:":
			    return id;
			    break;
			    
			    case "rdf:":
			    return id;
			    break;
			    
			    default:
			    return "#"+identifier;
			    break;
		    }
	    }
	    return id;
    }
    

/**
 * <p>Check for the existence of an item in the Ontology</p>
 * 
 * 
 * @param item (string or Visconte.Onto.Item.OWLItem)
 * @return int success 
 */
    public int item_exists(string|Visconte.Onto.Item.OWLItem item) 
    {   
	string id;
	if(!item)
		return 0;
	if (objectp(item))
	{
		id=item->get_Id();
	}
	else
		id=Visconte.ID(item)->ref;
	
        RDF.Resource test = owlset->get_resource(id);
	if (test)
		return 1;
	else
		return 0;
    } 



/**
 * <p>Replace an old item with a new one</p>
 * 
 * 
 * @param old_item - the item to replace
 * @param new_item - the exchange item
 */
    public void update(Visconte.Onto.Item.OWLItem updated_item) 
    {      
	    //the actual item on the storage
	    Visconte.Onto.Item.OWLItem  current_item;
	    
       switch(updated_item->ITEMTYPE)
	{
		//if class
		case Visconte.Onto.Item.OWLItem.CLASS:
			current_item = get_Class(updated_item->get_Id());
			update_statements(create_Class_statement(current_item),create_Class_statement(updated_item));
		break;
		
		case Visconte.Onto.Item.OWLItem.INDIVIDUAL:
			current_item = get_Indiviudal(updated_item->get_Id());
			update_statements(create_Individual_statement(current_item),create_Individual_statement(updated_item));
		break;
		
		case Visconte.Onto.Item.OWLItem.OBJECT_PROPERTY:
			current_item = get_Property(updated_item->get_Id());
			update_statements(create_ObjectProperty_statement(current_item),create_ObjectProperty_statement(updated_item));
		break;
		
		case Visconte.Onto.Item.OWLItem.DATATYPE_PROPERTY:
			current_item = get_Property(updated_item->get_Id());
			update_statements(create_DatatypeProperty_statement(current_item),create_DatatypeProperty_statement(updated_item));
		break;
	}
	
    } 

/**
 * <p>Removes an item from the ontology</p>
 * 
 * 
 * @param item - the OWLItem to remove
 * @return int success
 */
    public int remove(Visconte.Onto.Item.OWLItem owlitem) 
    {        
	Visconte.Onto.StatementContainer cont = Visconte.Onto.StatementContainer();
	remove_meta(owlitem);
	
        switch(owlitem->ITEMTYPE)
	{
		
		case Visconte.Onto.Item.OWLItem.CLASS:
			
		if(owlitem->is_root())
			return 0;
		remove_statements(create_Class_statement(owlitem));
		break;
			
		case Visconte.Onto.Item.OWLItem.OBJECT_PROPERTY:
		remove_statements(create_ObjectProperty_statement(owlitem));
		break;
		
		case Visconte.Onto.Item.OWLItem.DATATYPE_PROPERTY:
		remove_statements(create_DatatypeProperty_statement(owlitem));
		break;
		
		case Visconte.Onto.Item.OWLItem.INDIVIDUAL:
		remove_statements(create_Individual_statement(owlitem));
		break;
	}
		
    }
    
/**
 * <p>Tries to get the type of an item.</p>
 * This method tries to retrieve the type of an item as id, by checking the rdf:type statement
 * It will return the constants defined in Visconte.Onto.Item.OWLItem (see Visconte.Onto.Item.OWLItem.CLASS etc.)
 *
 * @param id 
 * @return int the constant of the  type
 * 
 */
    public int get_type(string id) 
    {  
	    int type=0;
	    if(item_exists(id))
	    {
		    RDF.Resource item = owlset->get_resource(id);
		    
		    //find the statement, to get the object type
		    array statement = owlset->find_statements(item,owlset->rdf_type,0);
		    if (sizeof(statement)>0)
		    {	
			    //object statement
			    string typename = statement[0][2]->get_qname();
			    if(typename=="Class" || typename=="owl:Class")
				    type=Visconte.Onto.Item.OWLItem.CLASS;
			    
			    else if(typename=="ObjectProperty" || typename=="owl:ObjectProperty")
				    type=Visconte.Onto.Item.OWLItem.OBJECT_PROPERTY;
			    
			    else if(typename=="DatatypeProperty" || typename=="owl:DatatypeProperty")
				    type=Visconte.Onto.Item.OWLItem.DATATYPE_PROPERTY;
			    
			    else 
				    type=Visconte.Onto.Item.OWLItem.INDIVIDUAL;
		    }
	    }
		    
	return type;
    } 

    
/**
 * <p>Generates an OWLContainer from an array of ids</p>
 * 
 * @param ids - an array with ids
 * @return OWLContainer - the container with the items 
 */
    public Visconte.Onto.OWLContainer get_Items(array ids) 
    {     
	Visconte.Onto.OWLContainer container = Visconte.Onto.OWLContainer();
	 
        foreach(ids ,string id)
	{
	   container->add(get_Item(Visconte.ID(id)->ref));
	}
	return container;
    } 

/**
 * <p>Generates an OWLContainer of a defined RDF-object</p>
 * Returns an Visconte.Onto.OWLContainer by searching in the ontology for all objects
 * that fit to the requested one. Param position indicates what should be constructed
 * for the query (0:Subject,1:Predicate,2:Object)
 * @param _object - the objects name as string
 * @param position - the position of the statment
 * @param type - the desired type of OWLItems to return (0:all, 1-4 see OWLItem.ITEMTYPE-Constant)
 * @return OWLContainer - the container with the items 
 */
    public Visconte.Onto.OWLContainer get_items_by_object(string _object, int position, int type) 
    {
	    Visconte.Onto.OWLContainer object_cnt = Visconte.Onto.OWLContainer();
	    array ids = resources_to_ids(query(0,0,_object),position);
	    foreach(ids,string id)
	    {
		      	   if(type==0 || get_type(id)==type)
			   object_cnt->add(get_Item(id));
	    }
	    return object_cnt;
	    
    }
    
/**
 * <p>Generates an OWLContainer from an array of ids</p>
 * Returns an Visconte.Onto.OWLContainer by searching in the ontology for all subjects
 * that fit to the requested one. Param position indicates what should be constructed
 * for the query (0:Subject,1:Predicate,2:Object)
 * @param _object - the objects name as string
 * @param position - the position of the statment
 * @param type - the desired type of OWLItems to return (0:all, 1-4 see OWLItem.ITEMTYPE-Constant)
 * @return OWLContainer - the container with the items 
 */
    public Visconte.Onto.OWLContainer get_items_by_subject(string subject, int position, int type) 
    {
	    Visconte.Onto.OWLContainer cnt = Visconte.Onto.OWLContainer();
	    array ids = resources_to_ids(query(subject,0,0),position);
	    foreach(ids,string id)
	    {
		    	   if(type==0 || get_type(id)==type)
			   cnt->add(get_Item(id));
	    }
	    
	    return cnt;
	    
    }
/**
 * <p>Query the ontology in a subject - predicate - object fashion</p>
 * Similar to find_statements, except the arguments are less restrictive in the 
 * RDF-Class
 * 
 * @param subject - string|RDF.Resource|int[0] - the subject
 * @param predicate - string|RDF.Resource|int[0] the predicate 
 * @param object - string|RDF.Resource|int[0] - the object
 * @return array with mathing statements
 */
    public array query(string|RDF.Resource|int subject, string|RDF.Resource|int predicate, string|RDF.Resource|int objekt) 
    {       
       Visconte.Onto.OWLContainer container = Visconte.Onto.OWLContainer();
       if(stringp(subject)) 
	       subject = owlset->make_resource(subject);
    
       if(stringp(predicate)) 
	       predicate = owlset->make_resource(predicate);
       
       if(stringp(objekt)) 
	       objekt = owlset->make_resource(objekt);
    
       return owlset->find_statements(subject,predicate,objekt);
	       
    } 

    
/**
 * <p>Returns the XML Representation of the Ontology</p>
 * 
 * 
 * @return string - xml rendering
 */
    public string get_OWL() 
    {        
        return to_string();;
    } 
	    
   

/**
 * <p>Equivalent to get_OWL.</p>
 * 
 * @return string - xml rendering
 */
    public string to_string() 
    {    
	    return owlset->get_xml();
    
    }
/**
 * <p>Returns a String with the tuple representation of the ontology.</p>
 *
 * @return string - 3tuple representation 
 */
    public string to_tuple() 
    {    
	    return owlset->get_3_tuples();
    }   

   
/**
 * <p>Set the version to the ontology</p>
 * 
 */
    private void set_version() 
    {   
	Visconte.Onto.StatementContainer remove = Visconte.Onto.StatementContainer();
	Visconte.Onto.StatementContainer cont = Visconte.Onto.StatementContainer();
	array old_version = query(owlset->get_resource("#_THIS"),owl_versionInfo,0);
	if(!query_is_empty(old_version))
	remove->add(old_version[0][0] ,old_version[0][1],old_version[0][2]);
	cont->add(owlset->make_resource("#_THIS"),owlset->rdf_type,owl_Ontology);
        cont->add(owlset->get_resource("#_THIS"),owl_versionInfo,owlset->LiteralResource((string)version));
	update_statements(remove,cont);
    }   

/**
 * <p>Get the current version of this ontology</p>
 * @return int - the versionnumber
 */
    public int get_version()
    {
	     array v_info = query(owlset->get_resource(""),owl_versionInfo,0);
	     if(!query_is_empty(v_info))
		     return (int) v_info[0][2]->get_literal();
	     else
		    return 1;
    }
    
/**
 * <p>Tries to fetchback a previous ontology</p>
 * Looks for the given version and loads the ontology. Set the version to 0
 * @param version - the version to fetchback
 * @return succes
 */
    public int fetchback_version(int version)
    {
	string old_ont = storage_handler->fetchback(ontology_file,version);
	if(old_ont)
	{
		init();
		owlset->parse_xml(old_ont);
		version=0;
		return 1;
		
	}
	else
		return 0;
    }
	    
	    
/**
 * <p>Init the versioning</p>
 * 
 */
    private void versioning() 
    {   
	version=get_version();
	version++;
        backup=owlset->get_xml();
    }     
   
/**
 * <p>Saves the Ontology File.</p>
 * Will save the ontology, the backup and the logfile with the given StorageHandler
 */
    public void save() 
    {    
       //set version and save ontology
       set_version();
       storage_handler->save(ontology_file, owlset->get_xml());
       //save backup
       if(version>1)
	       storage_handler->backup(ontology_file,backup,version);
       
       storage_handler->save(ontology_file+ontology_log_file_suffix,ontology_log->render_xml());
       
    } 

/**
 * <p>Load the ontology File</p>
 * 
 */
    private void load() 
    {     
	
	//test if the choosen file exists
	if (storage_handler->exist(ontology_file))
	{
		mixed error = catch
		{
			owlset->parse_xml(storage_handler->open(ontology_file));
			versioning();
		};
		if(error)
		{
			owlset->parse_xml(make_template());
			version=1;
			set_version();
		}
		
	}
	else
	{
		owlset->parse_xml(make_template());
		version=1;
		set_version();
	}
	if (storage_handler->exist(ontology_file+ontology_log_file_suffix))
	{
		ontology_log = Parser.XML.Tree.parse_input(storage_handler->open(ontology_file+ontology_log_file_suffix));
	}
	else
	{
		ontology_log = Parser.XML.Tree.parse_input("<ontologyLog/>");
	}
	
		
    } 
 
/**
 * <p>Makes a template for an empty file</p>
 * @return template - a XML-string 
 */
  private string make_template() 
  {
	  string template= "<?xml version='1.0'?>\n";
	  	 template+="<rdf:RDF \n";
		 template+="xmlns='http://www.w3.org/2002/07/owl#'\n";
		 template+="xmlns:rdf='http://www.w3.org/1999/02/22-rdf-syntax-ns#'\n";
		 template+="xmlns:rdfs='http://www.w3.org/2000/01/rdf-schema#'\n";
		 template+="xmlns:annot='http://steam.upb.de/annot#'\n";
		 template+="xmlns:xsd='http://www.w3.org/2001/XMLSchema#'\n";
		 template+=">\n";
		 //the xsd:int is a hack to force the usage of this namespace!
		 template+="<Class rdf:about='#Thing' rdfs:label='Abstract Thing' xsd:int='0'>\n<annot:author>Init</annot:author>\n</Class>";
		 template+="\n</rdf:RDF>\n";
		
		return template;
		


  }  
    
/**
 * <p>Add an annotation to a single item</p>
 * This annotation is either automatically added by a specific action or a normal user comment
 * 
 * @param item (string or Visconte.Onto.Item.OWLItem) - the items or the itmes id
 * @paramstring action - the action to add to the annotation
 * @param string comment - the comment of the user
 * @param string|int affected, a item which is affected to of an action (optional)
 * 
 * @return int success 
 */
    public int add_annotation(Visconte.Onto.Item.OWLItem|string item, string action, string comment, string|int affected_label) 
    { 
	        Parser.XML.Tree.Node new_annotation_node;
		mapping(string:string) anotation_data = ([]);
		anotation_data+=(["user_id":user]);
		anotation_data+=(["action": action]);
		anotation_data+=(["timestamp": (string) time()]);
		anotation_data+=(["date": (string) ctime(time())]);
		if(objectp(item))
		{
			string classContext="null";
			string classContextLabel="null";
			anotation_data+=(["about": Visconte.ID(item->get_Id())->uref]);
			anotation_data+=(["aboutLabel": item->get_Label()]);
			switch(item->ITEMTYPE)
			{
				case Visconte.Onto.Item.OWLItem.CLASS:
				classContext = item->get_Id();
				classContextLabel = item->get_Label();
				break;
				
				case Visconte.Onto.Item.OWLItem.OBJECT_PROPERTY:
				classContext = item->get_domain()->get_Id();
				classContextLabel = item->get_domain()->get_Label();
				break;
				
				case Visconte.Onto.Item.OWLItem.DATATYPE_PROPERTY:
				classContext = item->get_domain()->get_Id();
				classContextLabel = item->get_domain()->get_Label();
				break;
				
				case Visconte.Onto.Item.OWLItem.INDIVIDUAL:
				classContext = item->get_model_class()->get_Id();
				classContextLabel = item->get_model_class()->get_Label();
				break;
			}
			anotation_data+=(["classContext": Visconte.ID(classContext)->uref]);
			anotation_data+=(["classContextLabel": classContextLabel]);
		}
		else
		{
			anotation_data+=(["about": "0"]);
			anotation_data+=(["aboutLabel": (string)item ]);
			anotation_data+=(["classContext": "0"]);
		}
		
		anotation_data+=(["affectedLabel": (string) affected_label]);
		
		anotation_data+=(["ontologyVersion": (string) get_version() ]);
		anotation_data+=(["comment": comment]);
		ontology_log->get_last_child()->add_child(Parser.XML.Tree.Node(Parser.XML.Tree.XML_ELEMENT,"annotation",anotation_data,""));
		
    }    

/**
 * <p>Returns the xml representation of the log file</p>
 * 
 * @return string xml 
 */
    public string get_annotations() 
    {
	    return ontology_log->render_xml();
    }


     
/**
 * <p>Appends a author and change date to the ontology elements</p>
 * 
 * @param item (string or Visconte.Onto.Item.OWLItem)
 * @return int success 
 */
    public int set_meta(Visconte.Onto.Item.OWLItem item) 
    { 
	        RDF.Resource res = owlset->get_resource(item->get_Id());
		Visconte.Onto.StatementContainer stmt = Visconte.Onto.StatementContainer();
		//the new statements
		stmt->add(res,anot_creator,owlset->LiteralResource(user));
		stmt->add(res,anot_created,owlset->LiteralResource((string) time()));
		remove_meta(item);
		add_statements(stmt);
    }

    
/**
 * <p>Check for the existence of an item in the OWLset</p>
 * 
 * 
 * @param item (string or Visconte.Onto.Item.OWLItem)
 * @return int success 
 */
    public int remove_meta(Visconte.Onto.Item.OWLItem item) 
    { 
	        RDF.Resource res = owlset->get_resource(item->get_Id());
	    	Visconte.Onto.StatementContainer old_stmt = Visconte.Onto.StatementContainer();
		array old_author = query(res,anot_creator,0);
		array old_date = query(res,anot_created,0);
		//the old statements
		if(!query_is_empty(old_author))
		old_stmt->add(old_author[0][0], old_author[0][1], old_author[0][2] );
		if(!query_is_empty(old_date))
		old_stmt->add(old_date[0][0], old_date[0][1], old_date[0][2] );
		
		remove_statements(old_stmt);
    }
    
/**
 * <p>Check for the existence of an item in the OWLset</p>
 * 
 * 
 * @param item (string or Visconte.Onto.Item.OWLItem)
 * @return int success 
 */
    public mapping get_meta(Visconte.Onto.Item.OWLItem item) 
    { 
	        RDF.Resource res = owlset->get_resource(item->get_Id());
	    	Visconte.Onto.StatementContainer stmt = Visconte.Onto.StatementContainer();
		string author = resource_to_id(query(res,anot_creator,0),2);
		string lastChanged = resource_to_id(query(res,anot_created,0),2);
		return ([ "author" : author, "lastChanged" : lastChanged ]);
		
    }

   //++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ 
    public void debug(string dbg)
    {
	 // write("\nDebugging: "+dbg+"\n");
    }
    
