/*
* lexer.l
*
* lexical analyzer for wikicode
*/

%{

#include "boolean.h"
#include "debug.h"
#include "error.h"
#include "globals.h"
#include "transforms.h"
#include "yyexternals.h"

extern char* yyinbuffer;
int table_state = 0;

#define YY_INPUT(buf, result, max_size)\
{\
int n, len;\
len = strlen(yyinbuffer);\
result = len < max_size ? len : max_size;\
for ( n = 0; n < result; n++ )\
    buf[n] = yyinbuffer[n];\
yyinbuffer = &yyinbuffer[result];\
}

#include "output.h"


%}


ALPHANUM    ([a-z]|[A-Z]|[0-9])
LINK_TEXT   [^\]]+
PIKE_CODE   [^\@]+
EMBED_TEXT  [^\}]+
SPACE       (\t|\n|" ")
NOT_SPECIAL [^\n\[=\*#\-\{@]
BLANKLINE   ^({SPACE}*)
BLANK       \r\n\r\n
UNQUOTE     [^\"]
TABS        (\t+)
ENDDEF      [^:]+
EOL         [^\n]+
TTYPE       ("box"|"above"|"below"|"hsides"|"vsides"|"lhs"|"rhs")
TRULES      ("none"|"rows"|"cols"|"groups"|"all")
ATTR        ({SPACE}*{ALPHANUM}+{SPACE}*"="{SPACE}*[\"']{UNQUOTE}+[\"']{SPACE}*)
TATTR       {ATTR}+

HYPERLINK   "[http://"{LINK_TEXT}"]"
HYPERLINKS  "[https://"{LINK_TEXT}"]"
BARELINK    (("http")|("ftp"))"://"[^[:blank:]\n]+
IMAGE       "[["[iI]"mage:"{LINK_TEXT}"]]"
WIKILINK    "[["{LINK_TEXT}"]]"{ALPHANUM}*
ANNOTATION  "["{LINK_TEXT}"["{LINK_TEXT}"]]"
EMBED       "{"[^\|\<]{EMBED_TEXT}"}"

WIKIPIKE    "@@"{PIKE_CODE}"@@"

BOLD        '''
EMPHS       __
ITALIC      ''
LBR         \\\\
H1_START    ^=
H2_START    ^==
H3_START    ^===
H4_START    ^====
H1          =
H2          ==
H3          ===
H4          ====
BULLET      ^\*
NUMBER      ^#
INDENT      ^\:
HR          ^"----"-*
LIST        ^[\*#:]+
PRE         ^" "/[^\n]
DEFINITION  ^(\t+){ENDDEF}:\t
TAG         ("<"(.+)">")

%option noyywrap
%x table
%x td
%x caption
%x html
%x th
%x quote

%%
<INITIAL>"{|"{TATTR}* { BEGIN(table); make_table(yytext); table_state=1; }
<table>^\|\-"-"*{TATTR}* {  make_tr(yytext); }
<table>\|{TATTR}\| { output("<td %s>", yytext); BEGIN(td); }
<table>\|/[^\|\n\}\-\\+] { output("<td>"); BEGIN(td); }
<table>\|\+ { BEGIN(caption); }
<table>!   { BEGIN(th); output("<th>"); }
<th>[\n\|!] { BEGIN(table); output("</th>"); }
<th>. { output(yytext); }
<caption>[^\n\|]* { output("<caption>%s</caption>", yytext); BEGIN(table); }
<td>[\|\n] { output("</td>\n"); BEGIN(table); }
<table>^"{|"{TATTR}* { BEGIN(table); output("<td>\n"); make_table(yytext); table_state++; }
<table,td>\|\}   { table_state--; if ( table_state == 0 ) BEGIN(INITIAL); else { BEGIN(table); output("</td>\n");} output("</tr></table>\n"); }

<INITIAL>\"          { BEGIN(quote);  preformat();  }
<quote>\" { BEGIN(INITIAL); preformat(); }
<quote>. { plaintext(yytext); }
<quote,table,td,caption,th,html><<EOF>> { BEGIN(INITIAL); }
<INITIAL,td>{LIST}          { make_list(yytext); }
<INITIAL,table,td,caption,th>{BOLD}          { bold();            }
<INITIAL,table,td,caption,th>{EMPHS}         { bold();            }
<INITIAL,table,td,caption,th>{ITALIC}        { italic();          }
<INITIAL,td,quote>{TAG}           { tag(yytext);       }
<INITIAL>{PRE}           { preformat(yytext); }
<INITIAL>{H4_START}      { heading(4, TRUE);  }
<INITIAL>{H3_START}      { heading(3, TRUE);  }
<INITIAL>{H2_START}      { heading(2, TRUE);  }
<INITIAL>{H1_START}      { heading(1, TRUE);  }
<INITIAL>{H4}            { heading(4, FALSE); }
<INITIAL>{H3}            { heading(3, FALSE); }
<INITIAL>{H2}            { heading(2, FALSE); }
<INITIAL>{H1}            { heading(1, FALSE); }
<INITIAL>{HR}            { hr();              }
<INITIAL,table,td>{BARELINK}      { barelink(yytext);  }
<INITIAL,table,td>{IMAGE}         { image(yytext);     }
<INITIAL,table,td>{WIKILINK}   { linkInternal(yytext);  }
<INITIAL,table,td>{ANNOTATION}    { annotationInternal(yytext); }
<INITIAL,table,td>{HYPERLINK}     { hyperlink(yytext); }
<INITIAL,table,td>{HYPERLINKS}    { hyperlink(yytext); }
<INITIAL,table,td>{DEFINITION}    { make_def(yytext); }
<INITIAL,table,td>{EMBED}         { embed(yytext); }
<INITIAL,table,td>{WIKIPIKE}      { pi_pike(yytext); }
<INITIAL,td,th,caption>^\n             { blank_line();      }
<INITIAL,td,th,caption>{BLANKLINE}     { blank_line();      }
<INITIAL,td,th,caption>{BLANK}         { blank_line();      }
<INITIAL,td,th,caption>{LBR}           { force_br();        }
<INITIAL>^{NOT_SPECIAL}  { paragraph(yytext); }
<INITIAL,td>{SPACE}         { plaintext(yytext); }
<INITIAL,td>.               { plaintext(yytext); }
