/* Copyright (C) 2005  Thomas Bopp, Thorsten Hampel, Robert Hinn
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

static Thread.Queue saveQueue = Thread.Queue();

#include <macros.h>
#include <assert.h>
#include <attributes.h>
#include <database.h>
#include <config.h>
#include <classes.h>
#include <access.h>
#include <roles.h>
#include <events.h>
#include <exception.h>
#include <types.h>

//#define DEBUG_PERSISTENCE 1

#ifdef DEBUG_PERSISTENCE
#define PDEBUG(s, args...) werror("persistence: "+s+"\n", args)
#else
#define PDEBUG(s, args...)
#endif

#define PROXY "/kernel/proxy.pike"

static mapping namespaces = ([ ]);

static mapping pending_users = ([ ]);
static mapping pending_groups = ([ ]);

static array restricted_users = ({ "postman", "root", "service", "guest" });
static array restricted_groups = ({ "steam", "admin", "everyone" });

static object __Database;

int get_save_size()
{
  return saveQueue->size();
}

static void create() 
{
  thread_create(save_demon);
}


/** Prepends the namespace part to a given object id. May only be
 * called by namespaces (persistence layers).
 * @param small_oid the small object id (without namespace information)
 * @return the new (complete) object id (or 0 on error)
 */
int make_object_id (int small_oid)
{
  if ( ! is_namespace( CALLER ) ) {
    werror( "Persistence: make_object_id(): Caller is not a namespace: %O\n", CALLER );
    return 0;
  }
  if ( CALLER == __Database ) return small_oid & 0x7fffffff; // 32bit, first bit must be zero
  int nid = search( values(namespaces), CALLER );
  if ( nid < 0 ) {
    werror( "Persistence: make_object_id(): Caller is not registered as a namespace: %O\n", CALLER );
    return 0;
  }
  nid = indices(namespaces)[nid];
  int oid_length = (int)ceil( log( (float)small_oid ) / log( 2.0 ) );
  if ( (oid_length % 8) != 0 ) oid_length = oid_length + 8 - (oid_length % 8);
  oid_length /= 8;  // byte length, not bit length
  if ( oid_length > 0xfff ) {
    werror( "Persistence: make_object_id(): oid too long (%d bytes)\n", oid_length );
    return 0;
  }
  // build namespace part: 1[3bit-reserved][16bit-nid][12bit-oid_length]
  nid = ((0x80000000 | (nid & 0xffff)) << 12) | oid_length;
  // shift left to make room for oid:
  nid = nid << oid_length * 8;
  // cut object id to oid_length bits:
  nid = nid | small_oid;
  return nid;
}


/**
 * @return the namespace id of the persistence layer
 */
int register(object handler)
{
  if ( !objectp(handler) ) {
    werror( "Namespace tried to register, but it is not an object: %O\n", handler );
    return -1;
  }

  if ( !objectp( __Database ) ) {
    if ( !objectp( master()->get_constant("_Database") ) ) {
      werror( "Namespace tried to register, but database hasn't registered, yet!\n");
      return -1;
    }
    __Database = master()->get_constant("_Database");
    namespaces[0] = __Database;
  }

  if ( handler == __Database ) return 0;

  int nid = search( values(namespaces), handler );
  if ( nid >= 0 ) // already registered
    return nid;

  nid = 1;
  foreach( indices(namespaces), int tmp_nid )
    if ( nid <= tmp_nid ) nid = tmp_nid+1;
  namespaces[nid] = handler;
  if ( objectp(GROUP("Admin")) ) {
    GROUP("Admin")->unlock_attribute("namespaces");
    GROUP("Admin")->set_attribute("namespaces",namespaces);
    GROUP("Admin")->lock_attribute("namespaces");
  }
  return nid;
}

static void save_demon()
{
}

void postInitialization ()
{
  if ( objectp(GROUP("Admin")) ) {
    GROUP("Admin")->unlock_attribute("namespaces");
    GROUP("Admin")->set_attribute("namespaces",namespaces);
    GROUP("Admin")->lock_attribute("namespaces");
  }
  else FATAL( "Could not get \"Admin\" group to store persistence namespace ids." );
}

/**
 * @return array { nid, oid_length(bytes), oid }
 */
array split_object_id ( object|int p )
{
  int nid;
  int oid_length;
  int oid;
  if ( intp(p) ) oid = p;
  else if ( objectp(p) ) oid = p->get_object_id();
  else {
    werror("Persistence: split_object_id(): param is not int or object: %O\n", p);
    return UNDEFINED;
  }
  if ( oid < 0 ) return UNDEFINED;
  else if ( oid == 0 ) return ({ 0, 0, 0 });
  if ( (oid & 0x80000000) == 0 ) return ({ 0, 4, oid & 0xffffffff });
  oid_length = (((int)(log( (float)oid ) / log( 2.0 ))) - 31);
  nid = oid >> oid_length;
  oid = oid & ((1 << oid_length) - 1);
  return ({ nid, oid_length/8, oid });
}

object get_namespace(object|int p)
{
  mixed nid = split_object_id( p );
  if ( !arrayp(nid) ) return UNDEFINED;
  return namespaces[ nid[0] ];
}


bool user_restricted ( string user )
{
  if ( !stringp(user) ) return false;
  if ( search( restricted_users, lower_case(user) ) >= 0 ) return true;
  else return false;
}

bool group_restricted ( string group )
{
  if ( !stringp(group) ) return false;
  if ( search( restricted_groups, lower_case(group) ) >= 0 ) return true;
  else return false;
}

/**
 * creates a new persistent sTeam object.
 *
 * @param  string prog (the class to clone)
 * @return proxy and id for object
 *         note that proxy creation implies creation of associated object.
 * @see    kernel.proxy.create, register_user
 */
mixed new_object(mixed id)
{
  object proxy;
  mixed    res;
  string prog_name = master()->describe_program(object_program(CALLER));
  object obj = CALLER;
  foreach(indices(namespaces), mixed idx ) {
    if ( idx == 0 ) 
      continue;
    object handler = namespaces[idx];
    if ( res = handler->new_object(id, obj, prog_name) ) {
      return res;
    }
  }
  return namespaces[0]->new_object(obj, prog_name);
}

mapping get_namespaces() 
{
  return namespaces;
}

int get_namespace_id (object ns)
{
  mixed index = search( values(namespaces), ns );
  if ( !intp(index) ) return -1;
  return indices(namespaces)[index];
}

int is_namespace(object ns)
{
  return 1;
}

void set_proxy_status(object p, int status)
{
  if ( is_namespace(CALLER) )
    p->set_status(status);
}

bool delete_object(object p)
{
  object nid = get_namespace(p);
  if ( objectp(nid) )
    return nid->delete_object(p);
  else {
    werror( "delete_object: invalid namespace\n" );
    return false;
  }
}

int|object load_object(object proxy, int|object iOID)
{
  if ( object_program(CALLER) != (program)PROXY ) 
    error("Security Violation - caller not a proxy object !");
  
  object nid = get_namespace(iOID);
  if ( !objectp(nid) ) {
    werror("Namespace is not an object: %O (proxy: %O)\n", iOID, proxy );
    return UNDEFINED;
  }
  return nid->load_object(proxy, iOID);
}

mapping get_storage_handlers(object o)
{
    if ( !objectp(o) )
      return ([ ]);
      
    mapping m;
    mixed err = catch {
      m = o->get_data_storage();
    };
    if ( err ) {
      FATAL("Error calling get_data_storage in %O ", o);
      return ([ ]);
    }
    return m;
}

mixed call_storage_handler(function f, mixed ... params)
{
  mixed res = namespaces[0]->call_storage_handler(f, @params);
  return res;
}

void update_user ( object user )
{
  foreach (indices(namespaces), mixed idx ) {
    object handler = namespaces[idx];
    if ( functionp( handler->update_user ) )
      handler->update_user( user );
  }
}

void update_group ( object group )
{
  foreach (indices(namespaces), mixed idx ) {
    object handler = namespaces[idx];
    if ( functionp( handler->update_group ) )
      handler->update_group( group );
  }
}

object lookup(string identifier)
{
  if ( search(indices(pending_users), identifier) >= 0 )
    return pending_users[identifier];

  mixed res = 0;
  foreach(indices(namespaces), mixed idx ) {
    object handler = namespaces[idx];
    if ( res = handler->lookup(identifier) )
      break;
  }
  if ( res == 0 )
    res = namespaces[0]->lookup(identifier);
//    if ( objectp( res ) )
//      update( res );
  return res;
}


static void add_user_to_group ( object user, object group )
{
  if ( !objectp(user) || !objectp(group) )
    return;

  group->add_member( user );

  object user_workroom = user->query_attribute(USER_WORKROOM);
  object group_workroom = group->query_attribute(GROUP_WORKROOM);
  if ( !objectp(user_workroom) || !objectp(group_workroom) )
    return;

  // add exit to the group workroom if it doesn't exist:
  foreach ( user_workroom->get_inventory_by_class(CLASS_EXIT), object gate ) {
    if ( gate->get_exit() == group_workroom )
      return;  // exit already exists
  }
  object exit_factory = get_factory(CLASS_EXIT);
  if ( !objectp(exit_factory) ) {
    werror( "Persistence: Could not get ExitFactory.\n" );
    return;
  }
  object gate = exit_factory->execute( ([
                                         "name":group_workroom->get_identifier(),
                                         "exit_to":group_workroom
                                         ]) );
  if ( objectp(gate) )
    gate->move( user_workroom );
  else
    werror( "Persistence: Could not move exit to group workroom to users workarea:\nUser: %O, Group: %O\n", user, group );
}


static void remove_user_from_group ( object user, object group )
{
  if ( !objectp(user) || !objectp(group) )
    return;

  group->remove_member( user );  // not member of group anymore

  // check whether an exit to the group workroom needs to be removed:
  object user_workroom = user->query_attribute(USER_WORKROOM);
  object group_workroom = group->query_attribute(GROUP_WORKROOM);
  if ( !objectp(user_workroom) || !objectp(group_workroom) )
    return;
  foreach ( user_workroom->get_inventory_by_class(CLASS_EXIT), object gate ) {
    if ( gate->get_exit() == group_workroom )
      gate->delete();
  }
}


object lookup_user(string identifier)
{
  if ( !stringp(identifier) ) return null;

  PDEBUG( "lookup_user(%s)", identifier );

  if ( !zero_type( pending_users[identifier] ) )
    return pending_users[identifier];

  if ( user_restricted( identifier ) )
    return namespaces[0]->lookup_user(identifier);
  pending_users[identifier] = 0;

  mixed exc = catch {

  mapping attributes = ([ ]);

  object user = 0;
  bool found = false;
  // find user in persistence layers:
  foreach (indices(namespaces), mixed idx) {
    object handler = namespaces[idx];
    if ( !objectp(handler) )
      continue;
    if ( !functionp(handler->lookup_user) ) {
      FATAL("Invalid handler in Persistence: %s missing lookup_user()",
	    handler->get_identifier());
      continue; 
    }
    mixed res = handler->lookup_user(identifier);
    if ( objectp(res) ) {
      if ( !user ) user = res;
      attributes |= res->query_attributes();
      attributes["UserPassword"] = res->get_user_password();
      found = true;
    } else if ( mappingp(res) ) {
      attributes |= res;
      found = true;
    }
  }

  if ( !found ) {
    pending_users -= ([ identifier:0 ]);
    return 0;
  }

  // remove entries from attributes which are not stored as attributes:
  string password = attributes["UserPassword"];
  // Check for missing password field:
  // (Leaving password as 0 would allow access without password!
  // "*" is no valid password, but ldap authorization by ldap bind
  // will still work if enabled.)
  if ( zero_type(attributes["UserPassword"]) )
    password = "*";
  attributes -= ([ "UserPassword" : 0 ]);
  array groups = attributes["Groups"];
  attributes -= ([ "Groups" : 0 ]);
  string active_group = attributes["ActiveGroup"];
  attributes -= ([ "ActiveGroup" : 0 ]);

  if ( !objectp(user) ) {  // create new user:
    object factory = get_factory(CLASS_USER);
    user = factory->execute( ([ "name":identifier,
				"pw":password,
				"email":attributes[USER_EMAIL],
				"firstname":attributes[USER_FIRSTNAME],
				"fullname":attributes[USER_FULLNAME],
				"description":attributes[OBJ_DESC],
				]) );
    if ( !objectp(user) ) {
      pending_users -= ([ identifier:0 ]);
      werror( "Persistence: could not create user: " + identifier + "\n" );
      return 0;
    }
    user->activate_user( factory->get_activation() );
  }
  pending_users[identifier] = user;

  // sync user data:
  foreach ( indices(attributes), mixed attr )
    if ( attributes[attr] )
      call_storage_handler( user->restore_attr_data, attributes[attr], attr );
  if ( stringp(password) )
    call_storage_handler( user->restore_user_data, password, "UserPassword" );

  // sync group memberships:
  if ( arrayp(groups) ) {
    // check current group memberships:
    array current_groups = user->get_groups();
    foreach ( current_groups, object group ) {
      string group_name = group->get_name();
      if ( search( groups, group_name ) >= 0 )
	groups -= ({ group_name });  // already member of group
      else if ( group_name != "sTeam" )
        remove_user_from_group( user, group );
    }
    foreach ( groups, string group_name ) {
      object group = lookup_group(group_name);
      add_user_to_group( user, group );
    }
  }
  object active_group_object = lookup_group( active_group );
  if ( objectp(active_group_object) )
    user->set_active_group( active_group_object );

  pending_users -= ([ identifier:0 ]);

  // write back user data to persistence layers:
  update_user( user );

  return user;

  };
  if ( exc != 0 ) werror( "PERSISTENCE: lookup_user(\"%s\") : %O\n%O\n", identifier, exc[0], exc[1] );
}

object lookup_group(string identifier)
{
  if ( !stringp(identifier) ) return null;

  PDEBUG( "lookup_group(%s)", identifier );

  if ( !zero_type( pending_groups[identifier] ) )
    return pending_groups[identifier];
  if ( group_restricted( identifier ) )
    return namespaces[0]->lookup_group(identifier);

  pending_groups[identifier] = 0;

  /*
  mapping attributes = ([ OBJ_DESC:"" ]);
  */
  mapping attributes = ([ ]);

  object group = 0;
  bool found = false;
  // find group in persistence layers:
  foreach (indices(namespaces), mixed idx) {
    object handler = namespaces[idx];
    if ( !objectp( handler ) ) continue;
    if ( !functionp(handler->lookup_group) ) {
      FATAL("Invalid Handler in Persistence: %s missing lookup_user()",
	    handler->get_identifier());
      continue; 
    }
    mixed res = handler->lookup_group(identifier);
    if ( objectp(res) ) {
      if ( !group ) group = res;
      attributes |= res->query_attributes();
      found = true;
    } else if ( mappingp(res) ) {
      attributes |= res;
      found = true;
    }
  }

  if ( !found ) {
    pending_groups -= ([ identifier:0 ]);
    return 0;
  }
  pending_groups[identifier] = group;

  // remove entries from attributes which are not stored as attributes:
  array users = attributes["Users"];
  attributes -= ([ "Users" : 0 ]);

  if ( !objectp(group) ) {  // create new group:
    object factory = get_factory(CLASS_GROUP);
    group = factory->execute( ([ "name":identifier ]) );
    pending_groups -= ({ identifier });
    if ( !objectp(group) ) {
      pending_groups -= ([ identifier:0 ]);
      werror( "Persistence: could not create group: " + identifier + "\n" );
      return 0;
    }
  }
  // sync group data:
  foreach ( indices(attributes), mixed attr ) {
    group->call_storage_handler( "restore_attr_data", attr, attributes[attr] );
  }

  // sync group memberships:
  if ( arrayp(users) ) {
    foreach ( group->get_members(), object user ) {
      string user_name = user->get_name();
      if ( search( users, user_name ) >= 0 )
	users -= ({ user_name });  // already member of group
      else
        remove_user_from_group( user, group );
    }
    foreach ( users, string user_name ) {
      object user = lookup_user(user_name);
      add_user_to_group( user, group );
    }
  }

  pending_groups -= ([ identifier:0 ]);

  // write back user data to persistence layers:
  update_group( group );
  return group;
}

final object find_object(int|string iOID)
{
    object namespace;
    if ( stringp(iOID) )
	return namespaces[0]->find_object(iOID);
    namespace = get_namespace(iOID);
    if ( !objectp(namespace) ) return UNDEFINED;
    return namespace->find_object(iOID);
}

// requires saving an object
void require_save(void|string ident, void|string index)
{
  object proxy = CALLER->this();
  object nid = get_namespace(proxy);
  if ( objectp(nid) )
    nid->require_save(proxy, ident, index);
  else
    werror( "require_save: invalid namespace\n" );
}

static void
save_object(object proxy, void|string ident, void|string index)
{
}


string get_identifier() { return "PersistenceManager"; }
string describe() { return "PersistenceManager"; }
string _sprintf() { return "PersistenceManager"; }
