/**
 * cache module
 * @author: exodusd
 *
 * This module manages caches where data can be cached. You might want to use this
 * if you have to access data very frequently that takes a long time to retrieve or
 * creates cpu load or network traffic when retrieved. The data will be taken from
 * the cache if it is cached and if the cached value is not too old.
 * There is a cache collector thread that will check each cache from time to time
 * and remove entries that have expired, so that you don't run out of memory when
 * caching many different items that you only need for a short amount of time.
 *
 * Initialization: You call create_cache() and pass it a key by which to
 *   identify your cache (if a cache with this key already exists, then you get
 *   a 0 result). You also pass the time (in seconds) of how long items will remain
 *   cached. So, to create a cache that caches values for 60 seconds:
 *     object my_cache = _Server->get_module("cache")->create_cache( "MyCache", 60 );
 *
 * Usage: You usually just call the get() method, passing it the key by which the
 *   item will be cached, and a function by which to retrieve the value in case it
 *   is not cached or it's cache value has expired. You can use the lambda()
 *   syntax to wrap the retrieval code, since you don't want to actually execute
 *   if the value is in the cache:
 *     mixed value = my_cache->get( key, lambda(){ return get_some_value(); } );
 */

inherit "/kernel/module";

#include <classes.h>
#include <database.h>
#include <events.h>
#include <macros.h>

/** This is the time (in seconds) the cache collector waits between cache checks. */
int cache_collector_time = 300;  // 300 = every 5 minutes
static mapping caches = ([ ]);
static object cache_collector_thread;

string get_identifier() { return "cache"; }

class Cache {
    static mixed _id;
    /** You usually don't need to access this variable directly, use get() instead.
     * The cache data. Each key in the mapping is assigned an array. The 
     * first element is the cached value, the second is the time() when it
     * was cached. */
    mapping data;
    /** The time (in seconds) before a cached value expires. This is usually set up
     * at creation time. */
    int time_to_live;
    
    /** Don't call this directly, use the module's create_cache() method instead.
     * That way, the cache will be registered with the cache module and will be
     * handled by the cache collector. */
    void create ( mixed id, int seconds_to_live ) {
	_id = id;
	time_to_live = seconds_to_live;
	data = ([ ]);
    }

    /** Returns the id/key/name you gave the cache when creating it.
     * @return the cache's id/key/name
     */
    mixed get_id () {
	return _id;
    }

    /** Fetches a value by a given key. If the value is cached and hasn't expired,
     * then it will be taken from the cache. Otherwise the retrieval_function will
     * be called and its result will be cached and returned. If no retrieval_function
     * has been supplied and the value is not cached, then 0 will be returned (and not
     * cached of course).
     * @param key the key by which to look up the cached value
     * @param retrieval_function the function to execute to retrieve the value if it
     *   is not cached (or has expired). Use the lambda(){...} syntax if you want to
     *   write code here, so that it will only be executed when necessary.
     * @return the value (either from cache or freshly retrieved)
     */
    mixed get ( mixed key, void|function retrieval_function ) {
	array entry = data[key];
	if ( arrayp(entry) && time()- entry[1] <= time_to_live )
	    return entry[0];
	mixed value;
	if ( functionp(retrieval_function) )
	    value = retrieval_function();
	else
            value = ([ ])[0];  // make zero_type value
	if ( !zero_type(value) )
	    put( key, value );
	return value;
    }

    /** You can deliberately put data into the cache by using this method.
     * @param key the key by which to later look up the cached value
     * @param value the value to cache for that key
     */
    void put ( mixed key, mixed value ) {
	data[key] = ({ value, time() });
    }

    /** You can deliberately remove a key and value from the cache by using
     * this method, although the cache collector will do that from time to time
     * anyway.
     * @param key the key of the value that will be removed from cache
     */
    void remove ( mixed key ) {
	m_delete( data, key );
    }
}

/** This creates a new cache. You specify an id (or key, name, etc.) by which you
 * can later identify the cache. You can also specify how long data will remain
 * cached.
 * @param id the id/key/name by which to identify your cache
 * @param seconds_to_live cached data will expire after that many seconds (default value
 *    is 5 seconds)
 * @return the new cache object, or 0 if there already was a cache with that id
 *   (you can fetch that cache via the get_cache() method if you want to)
 */
object create_cache ( mixed id, void|int seconds_to_live ) {
    if ( ! zero_type(caches[id]) ) return 0;
    int time_to_live = 5;
    if ( ! zero_type(seconds_to_live) ) time_to_live = seconds_to_live;
    object cache = Cache( id, time_to_live );
    caches += ([ id : cache ]);
    return cache;
}

/** Fetches a cache by a given id/key/name.
 * @param id the id/key/name of the cache to fetch
 * @return the cache, or 0 if there is no cache by that id
 */
object get_cache ( mixed id ) {
    object cache = caches[id];
    return cache;
}

/** Fetches a cache by a given id/key/name or creates it if it doesn't exist.
 * @param id the id/key/name of the cache to fetch
 * @param seconds_to_live cached data will expire after that many seconds (default value
 * @return the cache or a newly created cache
 */
object get_or_create_cache ( mixed id, void|int seconds_to_live ) {
    object cache = get_cache( id );
    if ( objectp( cache ) ) return cache;
    int time_to_live = 5;
    if ( ! zero_type(seconds_to_live) ) time_to_live = seconds_to_live;
    return create_cache( id, time_to_live );
}

/** Destroys a cache by a given id/key/name.
 * @param id the id/key/name of the cache to destroy
 */
void remove_cache ( mixed id ) {
    m_delete( caches, id );
}

/** Returns an array of all caches registered by the cache module.
 * @return an array of all caches (objects of type Cache)
 */
array get_caches () {
    return values( caches );
}

static void cache_collector_func () {
  while ( true ) {
    foreach ( indices(caches), mixed cache_id ) {
      object cache = caches[cache_id];
      foreach ( indices(cache->data), mixed key ) {
	array entry = cache->data[key];
	if ( arrayp(entry) && time() - entry[1] > cache->time_to_live )
	  m_delete( cache->data, key );
      }
    }
    sleep( cache_collector_time );
  }
}

void init_module () {
  cache_collector_thread = Thread( cache_collector_func );
}

