package org.stegosuite.model.payload.block;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.util.Arrays;

import org.stegosuite.image.util.ByteUtils;

/**
 * This block contains the contents of a file as well as its file name
 */
public final class FileBlock
		extends Block {

	/**
	 * Unique number among all Block implementations
	 */
	public static final byte IDENTIFIER = 1;

	/**
	 * Contains the file name without path
	 */
	private String fileName = null;

	/**
	 * Contains the bytes of the file
	 */
	private byte[] fileContent = null;

	public FileBlock() {}

	public FileBlock(String fileName) {
		this.fileName = fileName;
		try {
			fileContent = Files.readAllBytes(new File(fileName).toPath());
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	@Override
	public byte getIdentifier() {
		return IDENTIFIER;
	}

	@Override
	public byte[] pack() {
		byte[] fileNameWithoutPath = new File(fileName).getName().getBytes(StandardCharsets.UTF_8);
		byte[] fileNameLength = ByteUtils.intToBytes(fileNameWithoutPath.length);
		return ByteUtils.concat(fileNameLength, fileNameWithoutPath, fileContent);
	}

	@Override
	public void unpack(byte[] data) {
		int fileNameLength = ByteBuffer.wrap(data).getInt();
		fileName = new String(Arrays.copyOfRange(data, 4, 4 + fileNameLength));
		fileContent = Arrays.copyOfRange(data, 4 + fileNameLength, data.length);
	}

	public String getFileName() {
		return fileName;
	}

	public void setFileName(String fileName) {
		this.fileName = fileName;
	}

	public byte[] getFileContent() {
		return fileContent;
	}

	public void setFileContent(byte[] fileContent) {
		this.fileContent = fileContent;
	}

}
