/**
    STEngine.m
    Scripting engine
 
    Copyright (c) 2002 Free Software Foundation

    Written by: Stefan Urbanek <urbanek@host.sk>
    Date: 2000
   
    This file is part of the StepTalk project.
 
    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.
 
    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.
 
    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 
 */

#import "STEngine.h"

#import "STEnvironment.h"
#import "STExterns.h"
#import "STFunctions.h"
#import "STLanguage.h"
#import "STMethod.h"
#import "STUndefinedObject.h"

#import <Foundation/NSDictionary.h>
#import <Foundation/NSException.h>
#import <Foundation/NSString.h>
#import <Foundation/NSZone.h>

NSZone *STMallocZone = (NSZone *)nil;

void _STInitMallocZone(void)
{
    if(!STMallocZone)
    {
        STMallocZone = NSCreateZone(NSPageSize(),NSPageSize(),NO);
    }
}

@implementation STEngine
+ (void)initialize
{
    _STInitMallocZone();

    if(!STNil)
    {
        STNil = (STUndefinedObject *)NSAllocateObject([STUndefinedObject class],
                                                      0, STMallocZone);
    }
}

/**
    Return a scripting engine for the language used in files of type 
    <var>fileType</var>
*/
+ (STEngine *) engineForFileType:(NSString *)fileType
{
    STLanguage *language = [STLanguage languageForFileType:fileType];

    return [language engine];
}

/**
    Return a scripting engine for language with specified name.
*/
+ (STEngine *) engineForLanguageWithName:(NSString *)name
{
    if(!name)
    {
        [NSException raise:@"STConversationException"
                     format:@"Unspecified language for a new engine."];
        return nil;
    }
    
    return [[STLanguage languageWithName:name] engine];
}

- (void)dealloc
{
    RELEASE(defaultEnvironment);
    
    [super dealloc];
}

/** Return the default scripting environment for the engine. */
- (STEnvironment *)defaultEnvironment
{
    NSLog(@"WARNING: -[STEngine defaultEnvironment] is deprecated. "
          @" Use STConversation object instead.");

    return defaultEnvironment;
}

/** Set the default scripting environment for the engine. */
- (void) setDefaultEnvironment:(STEnvironment *)anEnvironment
{
    NSLog(@"WARNING: -[STEngine setDefaultEnvironment:] is deprecated. "
          @" Use STConversation object instead.");

    ASSIGN(defaultEnvironment,anEnvironment);
}

/** Execude source code <var>code</var> in default scripting environment.  */
- (id)  executeCode:(NSString *)code
{
    NSLog(@"WARNING: -[STEngine executeCode:] is deprecated. "
          @" Use STConversation object instead.");

   return [self    executeCode:code 
                 inEnvironment:defaultEnvironment];
}

/** Execude source code <var>code</var> in an environment <var>env</var>. 
    This is the method, that has to be implemented by those who are writing 
    a language engine. 
    <override-subclass /> 
*/
- (id)  executeCode:(NSString *)code 
      inEnvironment:(STEnvironment *)env
{
    [self subclassResponsibility:_cmd];

    return nil;
}

- (BOOL)understandsCode:(NSString *)code 
{
    [self subclassResponsibility:_cmd];

    return NO;
}

- (STMethod *)methodFromSource:(NSString *)sourceString
                   forReceiver:(id)receiver
                 inEnvironment:(STEnvironment *)env
{
    [self subclassResponsibility:_cmd];
    return nil;
}
- (id)  executeMethod:(id <STMethod>)aMethod
          forReceiver:(id)anObject
        withArguments:(NSArray *)args
        inEnvironment:(STEnvironment *)env;
{
    [self subclassResponsibility:_cmd];
    return nil;
}
/** Set engine specific option. Refer to particuliar language engine 
    documentation for more information. */
- (void)setValue:(id)anObject forOption:(NSString *)anOption
{
    /* do nothing */
}
/** Returs a value for engine specific option */
- (id)valueForOption:(NSString *)anOption
{
    /* do nothing */
    return nil;
}

@end
