/*
 * s p o r t . c			-- String ports management
 *
 * Copyright  1993-2002 Erick Gallesio - I3S-CNRS/ESSI <eg@unice.fr>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
 * USA.
 *
 *            Author: Erick Gallesio [eg@unice.fr]
 *    Creation date: 17-Feb-1993 12:27
 * Last file update: 22-Feb-2002 12:31 (eg)
 *
 */

#include "stklos.h"

/*===========================================================================*\
 * 
 * 			Utilities
 * 
\*===========================================================================*/

static void error_bad_string(SCM s)
{
  STk_error("bad string ~S", s);
}

/*===========================================================================*\
 * 
 * Low level plugins
 * 
\*===========================================================================*/

#define START_ALLOC_SIZE  128	/* Initial size of an ouput string port */

struct sstream {
  int		cnt;
  unsigned char *ptr;
  unsigned char *base;
  int		bufsize;
  SCM		str;	/* keep a ref on original string to avoid GC problems */
};

#define PORT_CNT(x)     (((struct sstream *) (x))->cnt)
#define PORT_PTR(x)	(((struct sstream *) (x))->ptr)
#define PORT_BASE(x)	(((struct sstream *) (x))->base)
#define PORT_BUFSIZE(x) (((struct sstream *) (x))->bufsize)
#define PORT_STR(x)	(((struct sstream *) (x))->str)

static Inline int Sgetc(void *stream)
{
  return (--(PORT_CNT(stream))>=0 ? ((int) *PORT_PTR(stream)++): EOF);
}


static Inline int Seof(void * stream)
{
  return PORT_CNT(stream) <= 0;
}

static Inline int Sreadyp(void *stream)
{
  return !Seof(stream);
}

static Inline int Sclose(void * stream)
{
  return 0;
}

static Inline int Sputc(int c, void *stream)
{
  register unsigned int tmp;
  
  if (++PORT_CNT(stream) == PORT_BUFSIZE(stream)) {
    tmp	  = PORT_BUFSIZE(stream);
    tmp	 += tmp/2;
    PORT_BASE(stream) = STk_must_realloc(PORT_BASE(stream), tmp);
    PORT_PTR(stream)  = PORT_BASE(stream)+PORT_BUFSIZE(stream)-1;/* base can move */
    PORT_BUFSIZE(stream) =  tmp;
  }
 *PORT_PTR(stream)++ = (unsigned char) c;
 return 0;
}

static Inline int Sputs(char *s, void *stream)
{
  while (*s)
    Sputc(*s++, stream);
  return 0;
}

static Inline int Sflush(void *stream)
{
  return 0;
}


static Inline int Sread(void *stream, void *buffer, int count)
{
  int i, c;

  for (i = 0; i < count; i++) {
    c = Sgetc(stream);
    if (c == EOF) return i;
    *(char*)buffer++ = c;
  }
  return count;
}

static Inline int Swrite(void *stream, void *buffer, int count)
{
  int i;

  for (i = 0; i < count; i++) {
    Sputc(*(char*)buffer++, stream);
  }
  return count;
}

static void sport_print(SCM obj, SCM port)   /* Generic printing of string ports */
{
  char buffer[MAX_PATH_LENGTH + 20];

  sprintf(buffer, "#[%s-string-port %lx%s]", 
	  ISPORTP(obj) ? "input" : "output",
	  (unsigned long) obj,
	  PORT_IS_CLOSEDP(obj) ? " (closed)" : "");
  STk_puts(buffer, port);
}

static void sport_release(SCM port)
{
  /* Nothing to do */
}


/*===========================================================================*\
 * 
 * 			Input ports
 * 
\*===========================================================================*/

static struct port_obj *
make_sport(SCM str, int init_len, int flags)
{
  struct sstream  *ss = STk_must_malloc(sizeof(struct sstream));
  SCM res;

  /* Initialize the stream part */
  if (str) {
    /* this is a input string */
    PORT_BASE(ss) = STRING_CHARS(str);
    PORT_CNT(ss)  = init_len;
    PORT_STR(ss)  = str;
  } else {
    /* This is an output port */
    PORT_BASE(ss) = STk_must_malloc(init_len);
    PORT_CNT(ss)  = 0;
    PORT_STR(ss)  = STk_false;
  }
  
  PORT_PTR(ss)     = PORT_BASE(ss);
  PORT_BUFSIZE(ss) = init_len;

  /* Initialize now the port itsef */
  NEWCELL(res, port);
  PORT_STREAM(res)	= ss;
  PORT_FLAGS(res)	= flags | PORT_IS_STRING;
  PORT_UNGETC(res) 	= EOF;
  PORT_LINE(res)	= 1;
  PORT_POS(res)		= 0;
  
  PORT_PRINT(res)	= sport_print;
  PORT_RELEASE(res)	= sport_release;
  PORT_GETC(res)	= Sgetc;
  PORT_READY(res)	= Sreadyp;
  PORT_EOFP(res)	= Seof;
  PORT_CLOSE(res)	= Sclose;
  PORT_PUTC(res)	= Sputc;
  PORT_PUTS(res)	= Sputs;
  PORT_FLUSH(res)	= Sflush;
  PORT_BREAD(res)	= Sread;
  PORT_BWRITE(res)	= Swrite;
  
  return (struct port_obj *) res;
} 


/*
<doc ext open-input-string
 * (open-input-string str)
 *
 * Returns an input string port capable of delivering characters from
 * |str|.
doc>
 */
DEFINE_PRIMITIVE("open-input-string", open_input_string, subr1, (SCM s))
{
  ENTER_PRIMITIVE(open_input_string);
  if (!STRINGP(s)) error_bad_string(s);
  return (SCM) make_sport(s, STRING_SIZE(s), PORT_READ);
}


/*
<doc ext open-output-string
 * (open-output-string)
 *
 * Returns an output string port capable of receiving and collecting characters.
doc>
 */
DEFINE_PRIMITIVE("open-output-string", open_output_string, subr0,(void))
{
  return (SCM) make_sport((SCM) NULL, START_ALLOC_SIZE, PORT_WRITE);
}



/*
<doc ext get-output-string
 * (get-output-string port)
 *
 * Returns a string containing all the text that has been written on the
 * output string |port|.
 * @lisp
 *  (let ((p (open-output-string)))
 *     (display "Hello, world" p)
 *     (get-output-string p))         => "Hello, world"
 * @end lisp
doc>
 */
SCM STk_get_output_string(SCM port)
{
  struct port_obj* p;

  if (! OSPORTP(port)) STk_error("bad string port ~S", port);
#ifdef FIXME
  // Que dit SRFI-6?
  //  if (PORT_FLAGS(port) & PORT_CLOSED) 
  // Err("get-output-string: string port is closed", port);
#endif
  p = PORT_STREAM(port);
  return STk_makestring(PORT_CNT(p), PORT_BASE(p));
}


DEFINE_PRIMITIVE("get-output-string", scheme_get_output_string, subr1, (SCM port))
{
  ENTER_PRIMITIVE(scheme_get_output_string);
  return STk_get_output_string(port);
}


/*
<doc ext input-string-port? output-string-port?
 * (input-string-port? obj)
 * (output-string-port? obj)
 *
 * Returns |#t| if |obj| is an input string port or output string port 
 * respectively, otherwise returns #f.
doc>
 */
DEFINE_PRIMITIVE("input-string-port?", input_string_portp, subr1, (SCM port))
{
  return MAKE_BOOLEAN(ISPORTP(port));
}

DEFINE_PRIMITIVE("output-string-port?", output_string_portp, subr1, (SCM port))
{
  return MAKE_BOOLEAN(OSPORTP(port));
}


int STk_init_sport(void)
{
  ADD_PRIMITIVE(open_input_string);
  ADD_PRIMITIVE(open_output_string);
  ADD_PRIMITIVE(scheme_get_output_string);
  ADD_PRIMITIVE(input_string_portp);
  ADD_PRIMITIVE(output_string_portp);
  return TRUE;
}
