use core:io;
use sql;

DATABASE ProgvisDB {
	// Users in the system.
	TABLE users(
		// User id. Used in other tables.
		id INTEGER PRIMARY KEY,
		// User name.
		name TEXT ALLOW NULL,
		// User display name.
		displayName TEXT
	);

	// Known clients in the system, and what users they map to.
	TABLE clients(
		// Client ID (a long string).
		id TEXT PRIMARY KEY UNIQUE,
		// User ID.
		user INTEGER
	);

	// Problems in the system. There are two types of problems: one that only consists of a piece of
	// code, and one that has an interface + an implementation.
	TABLE problems(
		// ID of this problem.
		id INTEGER PRIMARY KEY,
		// Problem author (foreign key to users)
		author INTEGER,
		// Title of the problem.
		title TEXT,
		// What is the current state of this problem? I.e. what is the next thing to modify.
		// 0 = modify the main program, 1 = modify the implementation.
		next INTEGER,
		// Source code of the main program (to code table).
		main INTEGER,
		// Source code of the implementation (to code table).
		impl INTEGER,
		// Source code of the reference implementation (to code table).
		refimpl INTEGER,
		// Improved version of a previous problem, if any.
		improved INTEGER ALLOW NULL,
		// Created (time string in UTC)
		created TEXT
		);
	INDEX ON problems(author);
	INDEX ON problems(improved);

	// A piece of code for some problem.
	TABLE code(
		// ID of this piece of code.
		id INTEGER PRIMARY KEY,
		// Program source code.
		src TEXT,
		// Language (= file extension) of the problem.
		language TEXT
		);

	// Incorrect sequences to problems of type 1 that were found by users.
	TABLE solutions(
		// ID of this solution.
		id INTEGER PRIMARY KEY,
		// Solution author (foreign key to users)
		author INTEGER,
		// Problem (foreign key to problems)
		problem INTEGER,
		// Type of error found.
		type TEXT,
		// Sequence found.
		sequence TEXT,
		// Created (time string in UTC)
		created TEXT
		);
	INDEX ON solutions(problem);

}

class Database {
	init() {
		SQLite db(cwdUrl / "progvis.db");

		init() {
			db(db);
		}
	}

	private ProgvisDB db;

	// Find a user's identity from its client key.
	UserInfo? findUser(Str clientId) {
		if (x = WITH db: SELECT ONE users.id, users.displayName FROM clients JOIN users ON clients.user == users.id WHERE clients.id == clientId) {
			return UserInfo(x.users_id, x.users_displayName);
		}
		null;
	}

	// Find a user's name from its ID.
	Str? findUserName(Int userId) {
		if (x = WITH db: SELECT ONE displayName FROM users WHERE id == userId) {
			return x.displayName;
		} else {
			return null;
		}
	}

	// Log out a client.
	void logout(Str clientId) {
		WITH db: DELETE FROM clients WHERE id == clientId;
	}

	// Change username.
	void changeName(Int userId, Str newName) {
		WITH db: UPDATE users SET displayName = newName WHERE id == userId;
	}

	// Convert from 'currentError' to the DB representation.
	private Int toNext(Bool currentError) {
		if (currentError) {
			1;
		} else {
			0;
		}
	}

	// Convert from 'next' to CurrentError.
	private Bool fromNext(Int value) {
		value != 0;
	}

	// Get a list of problems in the system. Excludes problems that the user has already
	// solved. 'currentError' indicates whether to return problems where the implementation is
	// broken, or where the main is too weak.
	ProblemDetails[] unsolvedProblems(Int forUser, Bool currentError) {
		ProblemDetails[] result;
		Int nextV = toNext(currentError);
		var x = WITH db: SELECT problems.id AS id, displayName, title FROM problems
			JOIN users ON problems.author == users.id
			WHERE author != forUser AND next == nextV
			ORDER BY problems.id;
		for (row in x) {
			var problemId = row.id;

			var solved = (WITH db: SELECT ONE id FROM problems WHERE improved == problemId AND author == forUser);
			if (solved.empty) {
				result << ProblemDetails(problemId, row.title, row.displayName, false, currentError,
										countImproved(problemId),
										countSolved(problemId, currentError));
			}
		}

		result;
	}

	// Get users' solved problems.
	ProblemDetails[] solvedProblems(Int forUser) {
		ProblemDetails[] result;
		var x = WITH db: SELECT problems.id AS id, displayName, title, next FROM problems
			JOIN users ON problems.author == users.id
			WHERE author != forUser
			ORDER BY problems.id;
		for (row in x) {
			var problemId = row.id;

			var solved = (WITH db: SELECT ONE id FROM problems WHERE improved == problemId AND author == forUser);
			if (solved.any) {
				result << ProblemDetails(problemId, row.title, row.displayName, false, fromNext(row.next),
										countImproved(problemId),
										countSolved(problemId, fromNext(row.next)));
			}
		}

		result;
	}

	// Get a list of the user's own problems.
	ProblemDetails[] ownProblems(Int forUser) {
		ProblemDetails[] result;
		var x = WITH db: SELECT problems.id AS id, displayName, title, next FROM problems
			JOIN users ON problems.author == users.id
			WHERE author == forUser
			ORDER BY problems.id;
		for (row in x) {
			result << ProblemDetails(row.id, row.title, row.displayName, true, fromNext(row.next),
									countImproved(row.id),
									countSolved(row.id, fromNext(row.next)));
		}
		result;
	}

	private Nat countImproved(Int problemId) {
		WITH db: COUNT FROM problems WHERE improved == problemId;
	}

	private Nat countSolved(Int problemId, Bool currentError) {
		if (currentError)
			WITH db: COUNT FROM solutions WHERE problem == problemId;
		else
			0;
	}

	ProblemInfo? parentTo(Int problemId, Int userId) {
		if (p = WITH db: SELECT ONE improved FROM problems WHERE id == problemId) {
			if (improvedId = p.improved) {
				if (x = WITH db: SELECT ONE problems.id AS id, author, displayName, title, next FROM problems
					JOIN users ON problems.author == users.id
					WHERE problems.id == improvedId) {
					return ProblemInfo(x.id, x.title, x.displayName, x.author == userId, fromNext(x.next));
				}
			}
		}
		null;
	}

	Bool solvedProblem(Int userId, Int problemId) {
		var result = WITH db: SELECT ONE id FROM problems WHERE improved == problemId AND userId == author;
		result.any;
	}

	ProblemInfo[] improvementsTo(Int problemId, Int userId) {
		ProblemInfo[] result;
		var x = WITH db: SELECT problems.id AS id, author, displayName, title, next FROM problems
			JOIN users ON problems.author == users.id
			WHERE improved == problemId;
		for (row in x) {
			result << ProblemInfo(row.id, row.title, row.displayName, row.author == userId, fromNext(row.next));
		}
		result;
	}

	Solution[] solutionsTo(Int problemId) {
		Solution[] result;
		var x = WITH db: SELECT solutions.id AS id, displayName, type, sequence FROM solutions
			JOIN users ON users.id == solutions.author
			WHERE problem == problemId;
		for (row in x) {
			result << Solution(row.id, problemId, row.displayName, row.type, row.sequence);
		}
		result;
	}

	Int createProblem(Int userId, Str title, Code main, Code impl, Code refImpl, Bool currentError) {
		Int mainId = createCode(main);
		Int implId = createCode(impl);
		Int refId = createCode(refImpl);
		Int next = toNext(currentError);
		WITH db: INSERT INTO problems(author, title, next, main, impl, refimpl, created)
			VALUES (userId, title, next, mainId, implId, refId, CURRENT DATETIME);
	}

	Int improvedProblem(Int userId, Int originalId, Str title, Code? main, Code? impl, Bool currentError) {
		unless (original = WITH db: SELECT ONE main, impl, refimpl FROM problems WHERE id == originalId)
			throw ServerError("Problem ${originalId} does not exist.");

		// Check if this user has submitted a problem with similar enough code before.
		for (row in WITH db: SELECT main, impl FROM problems WHERE improved == originalId AND author == userId) {
			if (main)
				compareCode(row.main, main);
			if (impl)
				compareCode(row.impl, impl);
		}

		Int mainId = original.main;
		if (main)
			mainId = createCode(main);
		Int implId = original.impl;
		if (impl)
			implId = createCode(impl);
		Int next = toNext(currentError);
		WITH db: INSERT INTO problems(author, title, next, main, impl, refimpl, improved, created)
			VALUES (userId, title, next, mainId, implId, ${original.refimpl}, originalId, CURRENT DATETIME);
	}

	// Compare code for equality to disallow submitting multiple instances of essentially the same
	// solution to hoard points. Throws on failure.
	private void compareCode(Int original, Code newCode) {
		unless (code = WITH db: SELECT ONE src FROM code WHERE id == original)
			return;

		if (codeSignature(code.src) == newCode.signature)
			throw ServerError("This solution is almost identical to another solution you submitted to this problem.");
	}

	private Int createCode(Code code) {
		WITH db: INSERT INTO code(src, language) VALUES (${code.src}, ${code.language});
	}

	Problem problemDetails(Int problemId, Int userId) {
		var x = WITH db: SELECT ONE problems.id AS id, author, displayName, title, next, main, impl, refimpl FROM problems
			JOIN users ON problems.author == users.id
			WHERE problems.id == problemId;
		unless (x)
			throw ServerError("Problem ${problemId} does not exist.");

		Problem(x.id, x.title, x.displayName, x.author == userId, fromNext(x.next),
				getCode(x.main), getCode(x.impl), getCode(x.refimpl));
	}

	private Code getCode(Int codeId) {
		var code = WITH db: SELECT ONE src, language FROM code WHERE id == codeId;
		unless (code)
			throw ServerError("Code ${codeId} does not exist.");

		Code(code.src, code.language);
	}

	Int createSolution(Int userId, Int problemId, Str type, Str sequence) {
		unless (author = WITH db: SELECT ONE author FROM problems WHERE id == problemId)
			throw ServerError("No problem with the id ${problemId}!");

		// Note: We could check so that we don't solve our own problems here. In the new scheme,
		// this is not as important, so we skip it for now at least.
		// if (author.author == server.userId)
		// 	throw ServerError("Can not solve your own problem!");

		// Avoid duplicates.
		if (existing = WITH db: SELECT ONE id FROM solutions WHERE problem == problemId AND author == userId AND type == ${type}) {
			WITH db: UPDATE solutions SET sequence = ${sequence} WHERE id = ${existing.id};
			existing.id;
		} else {
			WITH db: INSERT INTO solutions(author, problem, type, sequence, created)
				VALUES (userId, problemId, ${type}, ${sequence}, CURRENT DATETIME);
		}
	}

	// Compute the points of all users in the database.
	Int->Int allScores() {
		Int->Int result;

		// Points given to a user who...

		// ...finds an error in a "debug" problem.
		Int debugFoundError = 10;
		// ...submits a solution to a "debug" problem.
		Int debugSubmit = 10;
		// ...submits a solution to a "debug" problem, and solves all issues.
		Int debugSolve = 50;

		// ...modifies "main" to expose an error.
		Int testFindError = 100;

		// Divider if the "parent" problem was the user's own.
		Int selfPenalty = 3; // reduce by 66% to avoid making intentionally bad solutions.

		WITH db {
			var improvements = SELECT p.author AS author, parent.author AS pAuthor, p.next AS next, parent.next AS pNext
				FROM problems p
				JOIN problems parent ON parent.id == p.improved;
			for (row in improvements) {
				// Score for this row.
				Int rowScore = 0;

				// Penalty for solving one's own problem?
				Int penalty = 1;
				if (row.author == row.pAuthor)
					penalty = selfPenalty;

				if (fromNext(row.pNext)) {
					// If the parent problem was a "debug" problem:

					// They improved it:
					rowScore += debugSubmit / penalty;

					// If it no longer has errors, they fixed all errors:
					if (!fromNext(row.next))
						rowScore += debugSolve / penalty;
				} else {
					// If the parent problem was a "test" problem:

					// The current one should contain errors, but we check for safety:
					if (fromNext(row.next))
						rowScore += testFindError / penalty;
				}

				// Update the user's score!
				result[row.author] += rowScore;
			}

			// Also look at solutions!
			var solutions = SELECT solutions.author AS sAuthor, problems.author AS pAuthor
				FROM solutions
				JOIN problems ON problems.id == solutions.problem;
			for (row in solutions) {
				Int points = debugFoundError;
				if (row.pAuthor == row.sAuthor)
					points /= selfPenalty;

				result[row.sAuthor] += points;
			}
		}

		result;
	}
}

class UserInfo {
	// User ID.
	Int id;

	// Display name of the user.
	Str name;

	init(Int id, Str name) {
		init { id = id; name = name; }
	}
}
