use core:lang;
use lang:bs;
use lang:bs:macro;
use core:asm;
use core:sync;

/**
 * Effect definition.
 *
 * Expands to a function that can be called from other languages.
 */
class Effect extends Function {
	init(SrcPos pos, Str name, Value result, ValParam[] params) {
		var vals = params.values();

		var frameType = if (found = named{}.find("EffectFrame", result, Scope()) as Type) {
			found;
		} else {
			throw InternalError("Could not found EffectFrame!");
		};


		init(pos, result, name, vals) {
			resumeType(vals);
			frameType = frameType;
		}

		setCode(LazyCode(&this.generate()));

		resumeType.parentLookup = this;
	}

	// Type for resuming the handler.
	private HandlerResumeType resumeType;

	// Type of the frame.
	private Type frameType;

	private CodeGen generate() {
		CodeGen gen(runOn, isMember, result);

		core:asm:Var[] actuals;
		for (x in params)
			actuals << gen.l.createParam(x.desc);

		gen.l << prolog();

		// Find a handler and update the resume object:
		var targetObj = gen.l.createVar(gen.block, sPtr);
		gen.l << fnParam(ptrDesc, objPtr(this));
		gen.l << fnCall(named{findHandler<Effect>}.ref, false, ptrDesc, targetObj);

		// Create our effects frame:
		var frameObj = gen.allocObject(frameType);

		// Create a handler resume frame to store the parameters we received:
		var resumeObj = gen.l.createVar(gen.block, sPtr);
		gen.l << fnParam(ptrDesc, resumeType.typeRef);
		gen.l << fnCall(BuiltIn:alloc.ref, false, ptrDesc, resumeObj);

		// Call the base class' constructor:
		var ctorParams = Value:[named{HandlerResume}, named{HandlerResume:Target}, named{EffectFrame}];
		if (ctor = named{HandlerResume}.find("__init", ctorParams, Scope()) as Function) {
			gen.l << fnParam(ptrDesc, resumeObj);
			gen.l << fnParam(ptrDesc, targetObj);
			gen.l << fnParam(ptrDesc, frameObj);
			gen.l << fnCall(ctor.ref, true);
		} else {
			throw InternalError("Failed to find resume type ctor!");
		}

		// Now, we can copy the parameters there:
		for (i, actual in actuals) {
			var v = resumeType.vars[i];
			if (v.type.isAsmType) {
				gen.l << mov(ptrA, resumeObj);
				gen.l << mov(xRel(v.type.size, ptrA, v.offset), actual);
			} else {
				gen.l << mov(ptrA, resumeObj);
				gen.l << add(ptrA, ptrConst(v.offset));
				gen.l << lea(ptrC, actual);
				gen.l << fnParam(ptrDesc, ptrA);
				gen.l << fnParam(ptrDesc, ptrC);
				gen.l << fnCall(v.type.copyCtor, true);
			}
		}

		// Finally, set the vtable properly!
		if (vtable = resumeType.vtable)
			vtable.insert(gen.l, resumeObj);


		// Finish set-up of the resume object.
		gen.l << fnParam(ptrDesc, resumeObj);
		gen.l << fnCall(named{HandlerResume:finish<HandlerResume>}.ref, true);

		// Wait for it to complete:
		gen.l << fnParam(ptrDesc, frameObj);
		gen.l << fnCall(named{EffectFrame:wait<EffectFrame>}.ref, true);

		// Load a pointer to the result if necessary:
		if (result.any) {
			unless (resultVar = frameType.find("result", frameType, Scope()) as MemberVar)
				throw InternalError("Failed to find 'result' in EffectFrame!");

			gen.l << add(frameObj, ptrConst(resultVar.offset));
			gen.l << fnRetRef(frameObj);
		} else {
			gen.l << fnRet();
		}

		gen;
	}
}

private HandlerResume:Target findHandler(Effect effect) {
	if (top = currentHandlerFrame()) {
		HandlerFrame? current = top;
		while (c = current) {
			if (h = c.effects.at(effect)) {
				return HandlerResume:Target(top, c, h);
			}

			current = c.prev;
		}
	}
	throw NoHandlerFor(effect.identifier());
}


// Data structure for storing the result to an effect invocation, and starting it again.
package class EffectFrame {
	// Ctor.
	init() {
		init {
			sema(0);
		}
	}

	// Semaphore to wait for execution to finish.
	// Note: We might need something more elaborate later.
	private Sema sema;

	// Should we throw an error to clean up?
	private Exception? throwError;

	// Wait for the result to be present.
	void wait() {
		sema.down();

		if (e = throwError) {
			throwError = null;
			throw e;
		}
	}

	// Indicate that a result is available.
	void signal() {
		sema.up();
	}

	// Signal that we should throw an exception and exit.
	void terminate(Exception e) {
		throwError = e;
		signal();
	}
}


// Generic result.
EffectFrame : generate(params) {
	if (params.count != 1)
		return null;

	var retVal = params[0].asRef(false);

	Type t("EffectFrame", [retVal], TypeFlags:typeClass);
	t.flags += NamedFlags:namedMatchNoInheritance;

	t.setSuper(named{EffectFrame});
	if (retVal.any)
		t.add(MemberVar("result", wrapMaybe(retVal), t));
	t.add(TypeDefaultCtor(t));
	t.add(TypeCopyCtor(t));
	t.add(TypeAssign(t));
	if (needsDestructor(t))
		t.add(TypeDefaultDtor(t));

	t;
}

/**
 * Message indicating that a handler block should invocate a handler.
 *
 * This class will be overloaded for each effect to provide a class with the appropriate parameters
 * and the appropriate logic inside 'call'.
 */
class HandlerResume {
	// Metadata.
	class Target {
		// Top of the handler chain.
		HandlerFrame top;

		// Target frame.
		HandlerFrame target;

		// Function to call.
		FnBase handler;

		init(HandlerFrame top, HandlerFrame target, FnBase handler) {
			init {
				top = top;
				target = target;
				handler = handler;
			}
		}
	}

	// Target information.
	Target target;

	// Information for controlling the old frame.
	EffectFrame frame;

	// Create.
	init(Target target, EffectFrame frame) {
		init { target = target; frame = frame; }
	}

	// Called to update the target frame.
	void finish() {
		target.target.resume(this);
	}

	// Called on the original UThread to invoke the handler.
	void call(HandlerFrame frame) : abstract;

	// Called at the start of 'call' to save the handler information.
	Continuation saveContinuation() {
		// Walk the chain of handler frames and collect the stacks from them.
		Continuation cont(frame);

		{
			HandlerFrame current = target.top;
			cont.add(current);

			while (current !is target.target) {
				unless (next = current.prev)
					break;
				current = next;

				cont.add(current);
			}
		}

		// Then resume the topmost handler to clean up the stack.
		Sema wait(0);
		frame.terminate(EffectUnwind(wait, target.target));

		// Wait for it to complete.
		wait.down();

		// All done!
		return cont;
	}
}


/**
 * Class that subclasses "HandlerResume" inside effects.
 *
 * Note: The type has no constructor, since it is intended to be created directly from ASM.
 */
class HandlerResumeType extends Type {
	// Create.
	init(Value[] members) {
		init("HandlerResume", TypeFlags:typeClass) {}
		setSuper(named{HandlerResume});

		for (i, m in members) {
			MemberVar v("member" + i.toS, m, this);
			vars << v;
			add(v);
		}
	}

	// Member variables, easily accessible.
	MemberVar[] vars;

	// Load functions.
	Bool loadAll() : override {
		Function callFn(Value(), "call", [this, named{HandlerFrame}]);
		callFn.setCode(LazyCode(&this.createCall));
		add(callFn);

		true;
	}

	// Create the body to 'call'
	private CodeGen createCall() {
		CodeGen gen(RunOn(), true, Value());

		var me = gen.l.createParam(ptrDesc);
		var frameParam = gen.l.createParam(ptrDesc);

		gen.l << prolog();

		// Note: We *could* pass ourselves to the handler directly instead of decomposing ourselves
		// into pieces. Then the handler itself would have to find the variables instead.

		// Save the continuation.
		var storedCont = gen.l.createVar(gen.block, sPtr);
		gen.l << fnParam(ptrDesc, me);
		gen.l << fnCall(named{HandlerResume:saveContinuation<HandlerResume>}.ref, true, ptrDesc, storedCont);

		// Find the function to call:
		gen.l << mov(ptrA, me);
		gen.l << mov(ptrC, ptrRel(ptrA, named{HandlerResume:target<HandlerResume>}.offset));
		gen.l << mov(ptrC, ptrRel(ptrC, named{HandlerResume:Target:handler<HandlerResume:Target>}.offset));

		// Compile a parameter list:
		// gen.l << mov(ptrA, me); // already done above
		gen.l << fnParam(ptrDesc, ptrC);
		gen.l << fnParam(ptrDesc, frameParam);
		for (v in vars) {
			gen.l << fnParam(v.type.desc, xRel(v.type.size, ptrA, v.offset));
		}
		gen.l << fnParam(ptrDesc, storedCont);

		// Figure out the parameters of a Fn<> object so that we can find the 'call' function:
		Value[] paramTypes;
		paramTypes << Value(); // Return
		paramTypes << named{HandlerFrame}; // Where to return
		for (x in vars)
			paramTypes << x.type;
		paramTypes << named{Continuation};

		var type = fnType(paramTypes.clone);
		paramTypes[0] = type;

		unless (toCall = type.find("call", paramTypes, Scope()) as Function)
			throw InternalError("Failed to find 'call' inside Fn!");

		gen.l << fnCall(toCall.ref, true);
		gen.l << fnRet();

		gen;
	}
}
