use core:lang;
use core:asm;
use lang:bs:macro;

class Printf extends Expr {
	init(SrcPos pos, Block block, SStr[] format, Expr[] params) {
		init(pos) {}

		Nat param = 0;
		for (f in format) {
			if (f.v.startsWith("%")) {
				if (param >= params.count)
					throw SyntaxError(f.pos, "Not enough parameters to printf to print this format placeholder.");
				addParam(f, params[param]);
				param++;
			} else {
				// Regular thing.
				items << StringConst(f.v.unescape);
			}
		}
	}

	// Items we should print.
	Item[] items;

	ExprResult result() : override {
		// No return value.
		Value();
	}

	void code(CodeGen gen, CodeResult res) : override {
		for (i in items) {
			i.computeExpr(gen);
		}
		for (i in items) {
			i.run(gen);
		}
	}

	void codePtr(CodeGen gen, Type type) : override {
		throw SyntaxError(pos, "Internal error, printf does not return anything.");
	}

	// Thing to print from printf.
	private class Item on Compiler {
		void computeExpr(CodeGen gen) : abstract;
		void run(CodeGen gen) : abstract;
	}

	// A regular string.
	private class StringConst extends Item {
		init(Str str) {
			init { value = str; }
		}

		Str value;

		void computeExpr(CodeGen gen) : override {}

		void run(CodeGen gen) : override {
			var fn = named{runtime:printStr<Str>};
			gen.l << fnParam(ptrDesc(), objPtr(value));
			gen.l << fnCall(fn.ref, false);
		}
	}

	// A formatted string from a pointer.
	private class StringPtr extends Item {
		init(Int width, Expr value) {
			init { width = width; value = value; }
		}

		Int width;
		Expr value;

		asm:Var var;

		void computeExpr(CodeGen gen) : override {
			CodeResult r(named{ConstPtr<Byte>}, gen.block);
			value.code(gen, r);
			var = r.location(gen);
		}

		void run(CodeGen gen) : override {
			Value p(named{ConstPtr<Byte>});
			var fn = named{PrintfHelpers:putStr<Int, ConstPtr<Byte>>};
			gen.l << fnParam(intDesc(), intConst(width));
			gen.l << fnParam(p.desc(), var);
			gen.l << fnCall(fn.ref, false);
		}
	}

	// An integer.
	private class Integer extends Item {
		init(Int width, Bool zeroPad, Expr value) {
			init { width = width; zeroPad = zeroPad; value = value; }
		}

		Int width;
		Bool zeroPad;
		Expr value;

		asm:Var var;

		void computeExpr(CodeGen gen) : override {
			CodeResult r(named{Int}, gen.block);
			value.code(gen, r);
			var = r.location(gen);
		}

		void run(CodeGen gen) : override {
			var fn = named{PrintfHelpers:putInt<Int, Bool, Int>};
			gen.l << fnParam(intDesc(), intConst(width));
			gen.l << fnParam(byteDesc(), byteConst(if (zeroPad) { 1b; } else { 0b; }));
			gen.l << fnParam(intDesc(), var);
			gen.l << fnCall(fn.ref, false);
		}
	}

	// A character.
	private class Character extends Item {
		init(Int width, Bool zeroPad, Expr value) {
			init { width = width; zeroPad = zeroPad; value = value; }
		}

		Int width;
		Bool zeroPad;
		Expr value;

		asm:Var var;

		void computeExpr(CodeGen gen) : override {
			CodeResult r(named{Byte}, gen.block);
			value.code(gen, r);
			var = r.location(gen);
		}

		void run(CodeGen gen) : override {
			var fn = named{PrintfHelpers:putChar<Int, Bool, Byte>};
			gen.l << fnParam(intDesc(), intConst(width));
			gen.l << fnParam(byteDesc(), byteConst(if (zeroPad) { 1b; } else { 0b; }));
			gen.l << fnParam(byteDesc(), var);
			gen.l << fnCall(fn.ref, false);
		}
	}

	// Add a call for printing a parameter.
	private void addParam(SStr str, Expr param) {
		// Escape for %
		if (str.v == "%%") {
			items << StringConst("%");
			return;
		}

		StrBuf middleBuf;
		Char lastCh;
		Nat i = 0;
		for (ch in str.v) {
			// Skip the %
			if (i++ > 1)
				middleBuf << lastCh;
			lastCh = ch;
		}

		Str middle = middleBuf.toS;
		// Check if we should pad with zero:
		Bool zeroPad = middle.any & middle.startsWith("0");
		if (zeroPad)
			middle = middle.cut(middle.begin + 1);

		if (middle.empty)
			middle = "0";
		Int width = middle.toInt;

		Str last = lastCh.toS;
		if (last == "s") {
			if (zeroPad)
				throw SyntaxError(str.pos, "Zero padding strings is not possible");
			items << StringPtr(width, param);
		} else if (last == "d") {
			items << Integer(width, zeroPad, param);
		} else if (last == "c") {
			if (zeroPad)
				throw SyntaxError(str.pos, "Zero padding strings is not possible");
			items << Character(width, zeroPad, param);
		} else {
			throw SyntaxError(str.pos, "Unsupported format specifier ${last}");
		}
	}
}


// Output functions used by the implementation.
// Due to a bug in loading the ConstPtr type, these need to be in a separate class to delay loading of the functions.
package class PrintfHelpers {
	void putInt(Int width, Bool zeroPad, Int value) : static {
		StrBuf buf;
		if (width < 0)
			buf << left;
		else
			buf << right;

		buf << core:width(abs(width).nat);
		if (zeroPad)
			buf << core:fill('0');

		buf << value;
		runtime:printStr(buf.toS);
	}

	void putChar(Int width, Bool zeroPad, Byte value) : static {
		StrBuf buf;
		if (width < 0)
			buf << left;
		else
			buf << right;

		buf << core:width(abs(width).nat);
		if (zeroPad)
			buf << core:fill('0');

		buf << Char(value.nat);
		runtime:printStr(buf.toS);
	}

	void putStr(Int width, ConstPtr<Byte> str) : static {
		Int len = runtime:strlen(str);
		Int pad = max(abs(width) - len, 0);
		Str padStr = " " * pad.nat;

		if (width > 0)
			runtime:printStr(padStr);

		runtime:putstr(str);
		if (width < 0)
			runtime:printStr(padStr);
	}
}

// Create if we don't have any parameters.
Printf printf(SrcPos pos, Block block, SStr[] format) on Compiler {
	Printf(pos, block, format, Expr[]);
}

