/* ------------------------------------------------------------------------ */
/*                                                                          */
/* [StyxScanner.C]      Styx Scanner Implementation (C++)                   */
/*                                                                          */
/*                This file belongs to the Styx Scanner & parser generator. */
/*                                                                          */
/* Copyright (c) 2003 by Doelle, Manns                                      */
/* ------------------------------------------------------------------------ */


#include "StyxScanner.h"
#include <iostream>
#include <fstream>


#define I32_HIGHVAL(v)    ( (short)(((long)(v)) >> 16) )
#define I32_LOWVAL(v)     ( (short)(((long)(v)) & 0xFFFFL) )
#define SET_I32_VAL(h,l)  ( ((((long)(h)) << 16) & 0xFFFF0000L) | \
                            (((long)(l)) & 0x0000FFFFL) )


using namespace styx;


CStyxStream::CStyxStream()
{
  cEof = 0;
}

CStyxStream::CStyxStream(string Path)
{ char* p = strrchr(Path.c_str(),'/');
  if( p == NULL )
    p = strrchr(Path.c_str(),'\\');
  if( p != NULL )
    File = p + 1;
  else
    File = Path;
  cEof = 0;
}

CStyxStream::~CStyxStream()
{
}

int CStyxStream::sgetc()
{
  return 0;
}

bool CStyxStream::eof(int c)
{
  return c <= cEof;
}

bool CStyxStream::unicode()
{
  return false;
}

string& CStyxStream::getFile() 
{
  return File;
}


CStyxStringStream::CStyxStringStream()
: CStyxStream("")
{
  this->txt = "";
  pos       = (long int)0L;
}

CStyxStringStream::CStyxStringStream(string ntxt)
: CStyxStream("")
{
  this->txt = ntxt.c_str();
  pos       = (long int)0L;
}

CStyxStringStream::~CStyxStringStream()
{
}

string& CStyxStringStream::getText()
{
  return txt;
}

void CStyxStringStream::setText(string ntxt)
{
  this->txt = ntxt.c_str();
  pos       = (long int)0L;
//printf("STREAM-TEXT=%s\n",this->txt.c_str());
}

int CStyxStringStream::sgetc()
{
  if( pos >= (long int)txt.length() )
    return 0;
  return (int)((txt[pos++])&0xff);
}

  
void CStyxScanner::setLanguage(string Language)
{
  Name = Language;
}
  
void CStyxScanner::addToken(string Token)
{
  TokId.push_back(Token);
}

void CStyxScanner::addTokenFlags(byte Flag)
{
  Flags.push_back(Flag);
}

void CStyxScanner::addFinalState(short Token)
{
  StaFin.push_back(Token);
}

void CStyxScanner::addFirstEdge(long FirstEdge)
{
  StaEdg.push_back(FirstEdge);
}

void CStyxScanner::addEdgeState(long State)
{
  EdgeS.push_back(State);
}

void CStyxScanner::addEdgeChar(wchar_t Character)
{
  EdgeC += Character;
}

void CStyxScanner::addDyckToken(short Token)
{
  dyckidx.push_back(Token);
}

void CStyxScanner::addDyckScanner(CStyxScanner& Scanner)
{
  dyckpat.push_back(Scanner);
}

void CStyxScanner::addDyckScanner()
{
  dyckpat.push_back(CStyxScanner());
}

void CStyxScanner::addSwitchToken(short Token, short Group)
{
  Switch[Token] = Group;
}

void CStyxScanner::addGroupScanner(CStyxScanner& Scanner)
{
  GrpScn.push_back(Scanner);
}

void CStyxScanner::addGroupScanner()
{
  GrpScn.push_back(CStyxScanner());
}

CStyxScanner* CStyxScanner::currentGroupScanner()
{
  return GrpScn.size() > 0
         ? &GrpScn[GrpScn.size()-1] : this;
}

CStyxScanner* CStyxScanner::currentDyckScanner()
{ CStyxScanner* curgrp = currentGroupScanner();
  return curgrp->dyckpat.size() > 0
         ? &curgrp->dyckpat[curgrp->dyckpat.size()-1] : NULL;
}

bool CStyxScanner::switchGroup(short Group, bool asgflg)
{ 
  if( !asgflg )
  {
    if( Group == -1 )
    {
      if (cTok < 0 || !(Flags[cTok]&SCN_FLG_GroupStack) ||
          GroupStack.size() <= 0) 
        return false;
      Group = GroupStack[GroupStack.size()-1];
      GroupStack.pop_back();
    }
    else
    if( Flags[cTok]&SCN_FLG_GroupStack )
    {
      GroupStack.push_back(this->Group);
    }
  }
  if( this->Group != Group )
  { 
    this->Group = Group;
    StaEdg      = GrpScn[Group].StaEdg;
    StaFin      = GrpScn[Group].StaFin;
    EdgeC       = GrpScn[Group].EdgeC;
    EdgeS       = GrpScn[Group].EdgeS;
    Flags       = GrpScn[Group].Flags;
    dscanner.clear();
    for( size_t i=0; i < GrpScn[Group].dyckidx.size(); ++i )
    {
      dscanner[GrpScn[Group].dyckidx[i]] = &GrpScn[Group].dyckpat[i];
    }
  }
  return true;
}

bool CStyxScanner::initPatternSymbol(string p, bool seq, string& result)
{ size_t len = p.length();
  result = "";
  if( len > 0 && len % 3 != 0 )
  {
    fprintf(stderr,"invalid start pattern or quotient token ID\n");
    return false;
  }
  for( size_t i=0; i < len; i+=3 )
  { unsigned char buf[4]; int c1, c2;
    buf[0] = p[i];
    buf[1] = p[i+1];
    buf[2] = p[i+2];
    buf[3] = '\0';
    c1 = atoi((char*)buf);
    if( i == 0 )
    {
      if( seq )
      {
        if( ((char)c1) != '"' && ((char)c1) != '\'' )
        {
          fprintf(stderr,"invalid quotient token ID");
          return false;
        }
      }
      else
      {
        if( c1 != 0 )
        {
          fprintf(stderr,"invalid pattern token ID");
          return false;
        }
        continue;
      }
    }
    result += (char)c1;
    if( p[i+3] == '.' )
    {
      if( i+9 >= len )
      {
        fprintf(stderr,"invalid start pattern or quotient token ID");
        return false;
      }
      i += 6;
      buf[0] = p[i];
      buf[1] = p[i+1];
      buf[2] = p[i+2];
      buf[3] = '\0';
      c2 = atoi((char*)buf);
      while( ++c1 <= c2 )
      {
        result += (char)c1;
      }
    }
  }
  if( !seq && result.length() == 0 )
  {
    fprintf(stderr,"missing pattern characters");
    return false;
  }
  return true;
}

bool CStyxScanner::initPatternMap(string p, long idx, bool pat)
{ string patsym;
  if( pat )
  { size_t pos = p.find_first_of(SCN_SEP_TOK_ID_PATTERN);
    if( pos != string::npos )
    { string p1, p2;
      p1 = p.substr(0,pos);
      p2 = p.substr(pos+1);
      if( cPatMap.find(idx) == cPatMap.end() )
      {
        if( ! initPatternSymbol(p1,false,patsym) )
          return false;
        cPatMap[idx] = patsym;
      }
      if( cQPatMap.find(idx) == cQPatMap.end() )
      {
        if( ! initPatternSymbol(p2,true,patsym) )
          return false;
        cQPatMap[idx] = patsym;
      }
    }
    else
    {
      if( !strncmp(p.c_str(),"000",3) )
      {
        if( cPatMap.find(idx) == cPatMap.end() )
        {
          if( ! initPatternSymbol(p,false,patsym) )
            return false;
          cPatMap[idx] = patsym;
        }
      }
      else
      {
        if( cQPatMap.find(idx) == cQPatMap.end() )
        {
          if( ! initPatternSymbol(p,true,patsym) )
            return false;
          cQPatMap[idx] = patsym;
        }
      }
    }
  }
  else
  { 
    if( cQPatMap.find(idx) == cQPatMap.end() )
    {
      if( ! initPatternSymbol(p,true,patsym) )
        return false;
      cQPatMap[idx] = patsym;
    }
  }
  return true;
}

bool CStyxScanner::initPattern()
/* collect patterns of quotient and pattern tokens */
{ 
  if( GrpScn.size() > 0 )
  {
    for( size_t i=0; i < GrpScn.size(); ++i )
    {
      for( size_t j=0; j < GrpScn[i].TokId.size(); j++)
      { 
        string tokid = GrpScn[i].TokId[j];
        size_t pos   = tokid.find_first_of(SCN_SEP_TOK_ID_PATTERN);
        if( pos != string::npos )
        { 
          bool ok = initPatternMap
                    (
                      tokid.substr(pos+1), SET_I32_VAL(i,j),
                      (GrpScn[i].Flags[j]&SCN_FLG_PatternToken) 
                    );
          if( !ok ) return false;
        }
      }
    }
  }
  else
  {
    for (size_t j=0; j < TokId.size(); j++)
    { string tokid = TokId[j];
      size_t pos   = tokid.find_first_of(SCN_SEP_TOK_ID_PATTERN);
      if( pos != string::npos )
      { bool ok = initPatternMap
                  (
                    tokid.substr(pos+1), SET_I32_VAL(-1,j),
                    (Flags[j]&SCN_FLG_PatternToken) 
                  );
        if( !ok ) return false;
      }
    }
  }
  return true;
}

long CStyxScanner::currentLength()
{
  if( cStream->unicode() )
    return (long)wbuffer.length();
  else
    return (long)buffer.length();
}

void CStyxScanner::beginToken()
{
  cCol = nCol; cLine = nLine;
  if( cStream->unicode() )
    wbuffer.erase();
  else
    buffer.erase();
}

bool CStyxScanner::endToken()
{
  if( cStream->unicode() )
  {
    if (cTok >= 0 && (Flags[cTok]&SCN_FLG_IgnoreCase)) 
    {
      for( size_t i=0; i < wbuffer.length(); ++i )
      {
        if( iswupper(wbuffer[i]) ) wbuffer[i] = towlower(wbuffer[i]);
      }
    }
  }
  else
  {
//printf("TOKEN-BUFFER=%s\n",buffer.c_str());
    if (cTok >= 0 && (Flags[cTok]&SCN_FLG_IgnoreCase)) 
    {
      for( size_t i=0; i < buffer.length(); ++i )
      {
        if( isupper(buffer[i]) ) buffer[i] = tolower(buffer[i]);
      }
    }
  }
  if( cTok > 0 )
  {
    if( cQPatMap.size() > 0 && !endQuotient() ) 
      return false;
    if( cTok > 0 && (Flags[cTok]&SCN_FLG_PatternToken) && !endPattern() )
      return false;
  }
  return true;
}

bool CStyxScanner::endPattern()
{ long idx = -1;
  if( GrpScn.size() > 0 )
    idx = SET_I32_VAL(this->Group,cTok);
  else
    idx = SET_I32_VAL(-1,cTok);
  if( cPatSym.length() == 0 && cPatMap.find(idx) != cPatMap.end() )
  { /* start pattern token */
    cPatTok = cTok;
    if( GrpScn.size() > 0 )
      cPatGrp = Group;
    /* get pattern set */
    string pat = cPatMap[idx];
    /* get pattern */
    if( !cStream->unicode() )
    { size_t i;
      for( i=0; i < buffer.length(); ++i )
        if( strchr(pat.c_str(),buffer[i]) != NULL )
          break;
      for( ; i < buffer.length(); ++i )
        if( strchr(pat.c_str(),buffer[i]) != NULL )
          cPatSym += buffer[i];
        else break;
    }
    else
    { char c; size_t i;
      for( i=0; i < wbuffer.length(); ++i )
      { 
        c = (char)wbuffer[i];
        if( ((wchar_t)c) == wbuffer[i] &&
            strchr(pat.c_str(),c) != NULL )
          break;
      }
      for( ; i < wbuffer.length(); ++i )
      {
        c = (char)wbuffer[i];
        if( ((wchar_t)c) == wbuffer[i] &&
            strchr(pat.c_str(),c) != NULL )
          cPatSym += c;
        else break;
      }
    }
    if( cPatSym.length() == 0 )
      cTok = 0; /* error */
  }
  else
  if( cPatSym.length() > 0 && cPatMap.find(idx) == cPatMap.end() )
  { /* end pattern token, try to match pattern */
    if( !cStream->unicode() )
    {
      if( buffer.find(cPatSym) != string::npos )
      { /* pattern found */
        cTok = cPatTok; cPatSym = "";
        if( GrpScn.size() > 0 && !switchGroup(cPatGrp,true) )
          return false;
      }
    }
    else /* multibyte mode */
    { size_t i, j, k;
      for( i=0; i < wbuffer.length(); ++i )
      { char c = (char)wbuffer[i];
        if( ((wchar_t)c) == wbuffer[i] && c == cPatSym[0] )
        { bool flg = true;
          for( k=1, j=i+1; 
               k < cPatSym.length() && j < wbuffer.length(); ++j, ++k )
          {
            c = (char)wbuffer[j];
            if( ((wchar_t)c) != wbuffer[j] || c != cPatSym[k] ) 
            {
              if( flg && c != cPatSym[0] ) i=j;
              break;
            }
            else
            if( flg && c != cPatSym[0] ) i=j;
            else flg = false;
          }
          if( cPatSym.length() == k ) /* pattern found */
          {
            cTok = cPatTok; cPatSym = "";
            if( GrpScn.size() > 0 && !switchGroup(cPatGrp,true) )
              return false;
            break;
          }
        }
      }
    }
  }
  else cTok = 0; /* error */
  return true;
}

bool CStyxScanner::endQuotient()
{ long idx = -1;
  if( GrpScn.size() > 0 )
    idx = SET_I32_VAL(this->Group,cTok);
  else
    idx = SET_I32_VAL(-1,cTok);
  if( cQPatMap.find(idx) != cQPatMap.end() )
  {
    /* get quotient pattern */
    string pat = cQPatMap[idx];
    long   len = ((long)pat.length())-1;
    /* push back quotient */
    if( len > 0 )
    {
      if( !cStream->unicode() )
      { 
        if( pat[0] == '"' ) /* sequence */
        {
          if( ((long)buffer.length()) <= len || 
              strncmp(buffer.c_str()+buffer.length()-len,pat.c_str()+1,len) )
            cTok = 0; /* error; = --> endless recursion */
          else
            backbufQuotient(buffer.length()-len);
        }
        else
        { long i;
          for( i=((long)buffer.length())-1; i >= 0; --i )
          {
            if( strchr(pat.c_str()+1,buffer[i]) == NULL )
              break;
          }
          if( i < 0 ) cTok = 0; /* error: endless recursion */
          else
          if( i < ((long)buffer.length())-1 )
            backbufQuotient(i+1);
        }
      }
      else
      {
        if( pat[0] == '"' ) /* sequence */
        {
          if( ((long)wbuffer.length()) <= len )
            cTok = 0; /* error; = --> endless recursion */
          else
          { long i, j;
            for( i=((long)wbuffer.length())-1, j=len-1; 
                 j >= 1 && i >= 0; --i, --j )
            {
              if( wbuffer[i] != ((wchar_t)pat[j]) )
                break;
            }
            if( j >= 1 ) cTok = 0; /* error */
            else
              backbufQuotient(wbuffer.length()-len);
          }
        }
        else
        { long i;
          for( i=((long)wbuffer.length())-1; i >= 0; --i )
          { char c = (char)wbuffer[i];
            if( ((wchar_t)c) != wbuffer[i] || strchr(pat.c_str()+1,c) == NULL )
              break;
          }
          if( i < 0 ) cTok = 0; /* error: endless recursion */
          else
          if( i < ((long)buffer.length())-1 )
            backbufQuotient(i+1);
        }
      }
    }
  }
  return true;
}

void CStyxScanner::getCharacter()
{
  addCharacter(cC);
  if (cC == (cStream->unicode() ? (int)L'\n' : (int)'\n'))
  { nLine += 1; nCol = 1; }
  else
  { nCol += 1; }
  if( bbuffer.size() > 0 )
  {
    cC = (int)bbuffer[bbuffer.size()-1];
    bbuffer.pop_back();
  }
  else
  {
    cC = cStream->sgetc();
    if( !skipC.empty() ) skipCharacter();
  }
}

void CStyxScanner::skipCharacter()
{
  while( skipC.find(cC) != skipC.end() )
  {
    if (cC == (cStream->unicode() ? (int)L'\n' : (int)'\n'))
    { nLine += 1; nCol = 1; }
    else
    { nCol += 1; }
    cC = cStream->sgetc();
  }
}

void CStyxScanner::addCharacter(int c)
{
  if( cStream->unicode() )
    wbuffer += (wchar_t)c;
  else
    buffer += (char)c;
}

void CStyxScanner::backbufLookahead
            (
              long cBackLen, long nBackLin, long nBackCol
            )
/* buffer character lookahead */
{ long i;
  bbuffer.push_back(cC);
  if( !cStream->unicode() )
  {
    for( i=buffer.length()-1; i > cBackLen; --i )
      bbuffer.push_back(buffer[i]);
    cC = (int)buffer[i];
    buffer.erase(i);
  }
  else
  {
    for( i=wbuffer.length()-1; i > cBackLen; --i )
      bbuffer.push_back(wbuffer[i]);
    cC = (int)wbuffer[i];
    wbuffer.erase(i);
  }
  nLine = nBackLin; nCol = nBackCol;
}

void CStyxScanner::backbufQuotient(long cBackLen)
/* buffer token quotient */
{ long i;
  bbuffer.push_back(cC);
  if( !cStream->unicode() )
  {
    for( i=buffer.length()-1; i > cBackLen; --i )
      bbuffer.push_back(buffer[i]);
    cC = (int)buffer[i];
    buffer.erase(i);
  }
  else
  {
    for( i=wbuffer.length()-1; i > cBackLen; --i )
      bbuffer.push_back(wbuffer[i]);
    cC = (int)wbuffer[i];
    wbuffer.erase(i);
  }
  for( i=0, nLine=cLine, nCol=cCol; i < cBackLen; ++i )
  {
    if( !cStream->unicode() )
    {
      if( buffer[i] == '\n' )
      { nLine += 1; nCol = 1; }
      else
      { nCol += 1; }
    }
    else
    {
      if( wbuffer[i] == L'\n' )
      { nLine += 1; nCol = 1; }
      else
      { nCol += 1; }
    }
  }
}


bool CStyxScanner::nextToken_dyck()
{ int state = 0, cnt = 1; long cLen = currentLength(); 
  short dTok = cTok;
  CStyxScanner& dyck = *dscanner[cTok];
LOOP:
  cLen    = currentLength();
  cTok = -1; /* -1 == ETX */
  if ( !cStream->eof(cC) )
  { register wchar_t* i; 
    wchar_t* s = (wchar_t*)dyck.EdgeC.data();
    state = 0;
    for (i = &s[dyck.StaEdg[state]]; *i++ > cC; );
    state = dyck.EdgeS[i-s-1]-1;
    while (state >= 0)
    {
      cTok = dyck.StaFin[state]; getCharacter();
      if ( cStream->eof(cC) )
        break;
      for (i = &s[dyck.StaEdg[state]]; *i++ > cC; );
      state = dyck.EdgeS[i-s-1]-1;
    }
    if((currentLength() - cLen) == 0 && !cStream->eof(cC)) 
    { getCharacter(); cTok = 0; }
  }
  if( cTok > 0 || ( cTok == 0 && dyck.TokId.size() == 3 ) )
  {
    if( cTok == SCN_DYCKTOKEN_PREFIX ) ++cnt;
    else
    if( cTok == SCN_DYCKTOKEN_SUFFIX ) --cnt;
    if( cnt == 0 ) cTok = dTok;
    else goto LOOP;
  }
  return true;
}

bool CStyxScanner::nextToken_default()
{ int state = 0;
  beginToken(); 
  if( GrpScn.size() > 0 && cTok >= 0 && 
      cTok == cPatTok && Group == cPatGrp && cPatSym.length() == 0 )
  { /* end pattern token switch already done */
    cPatGrp = -1; cPatTok = -1;
  }
  else
  if( GrpScn.size() > 0 && cTok >= 0 && 
      (GrpScn[Group].Switch[cTok] >= 0 || (Flags[cTok]&SCN_FLG_GroupStack)) &&
      !switchGroup(GrpScn[Group].Switch[cTok],false) )
    return false;
LOOP:
  /* '\0' (string) and -1 (file,buffer) is treated as EOF. */
  cTok = -1; /* -1 == ETX */
  if ( !cStream->eof(cC) )
  { register wchar_t* i; 
    wchar_t* s = (wchar_t*)EdgeC.data();
    // handle character lookahead ...
    long  cBackLen = -1, nBackLin = -1, nBackCol = -1;
    short cBackTok = -1;
    // ... handle character lookahead
    state = 0;
    for (i = &s[StaEdg[state]]; *i++ > (wchar_t)cC; );
    state = EdgeS[i-s-1]-1;
    while (state >= 0)
    {
      /* n character lookahead:
         jeweils den grten Endzustand merken, zusammen mit Tokenlnge.
         nach der schleife, 
         1) wert oder fehler 
         2) fehler mit vorangegangenem Wert --> 
            zurcksetzen auf ende wertposition via backbuffer und fortsetzen
      */
      cTok = StaFin[state]; getCharacter();
      // handle character lookahead ...
      if( lookahead && cTok > 0 )
      {
        cBackLen = currentLength(); 
        nBackLin = nLine; nBackCol = nCol;
        cBackTok = cTok;
      }
      // ... handle character lookahead
      if ( cStream->eof(cC) )
      {
        break;
      }
      for (i = &s[StaEdg[state]]; *i++ > (wchar_t)cC; );
      state = EdgeS[i-s-1]-1;
    }
    // handle character lookahead ...
    if( lookahead && cTok == 0 && cBackTok > 0 )
    {
      cTok = cBackTok; backbufLookahead(cBackLen,nBackLin,nBackCol);
    }
    // ... handle character lookahead
    if( currentLength() == 0 && !cStream->eof(cC) )
    { getCharacter(); cTok = 0; }
    // process dyck token ...
    if( dscanner.find(cTok) != dscanner.end() && 
        !nextToken_dyck() )
      return false;
    // ... process dyck token
    if( (Flags[cTok]&SCN_FLG_IgnoreToken) )
    { beginToken(); 
      if( GrpScn.size() > 0 && cTok >= 0 && 
          (GrpScn[Group].Switch[cTok] >= 0 || 
            (Flags[cTok]&SCN_FLG_GroupStack)) &&
          !switchGroup(GrpScn[Group].Switch[cTok],false) )
        return false;
      goto LOOP; 
    }
  }
  return endToken();
}


CStyxScanner::CStyxScanner(bool lahead)
{
  lookahead = lahead;
  cStream   = 0;
  cC        = cTok = Group = -1;
  cLine     = cCol = nLine = nCol = 1;
  next      = &CStyxScanner::nextToken_default;
  cPatTok   = cPatGrp = -1;
}

CStyxScanner::CStyxScanner(string Language, bool lahead)
{
  Name      = Language; 
  lookahead = lahead;
  cStream   = 0;
  cC        = cTok = Group = -1;
  cLine     = cCol = nLine = nCol = 1;
  next      = &CStyxScanner::nextToken_default;
  cPatTok   = cPatGrp = -1;
}

CStyxScanner::~CStyxScanner()
{
}

bool CStyxScanner::LookaheadEnabled()
{
  return lookahead;
}

string CStyxScanner::getLanguage()
{
  return Name;
}

short CStyxScanner::getGroups()
{
  return (short)GrpScn.size();
}

short CStyxScanner::getTokens(short GrpIdx)
{
  if( GrpIdx == -1 )
    return (short)TokId.size();
  else
    return GrpIdx >= 0 && ((size_t)GrpIdx) < GrpScn.size()
           ? (short)GrpScn[GrpIdx].TokId.size() : -1;
}

string CStyxScanner::getGroupID(short GrpIdx)
{
  return GrpIdx >= 0 && ((size_t)GrpIdx) < GrpScn.size()
         ? GrpScn[GrpIdx].Name : "";
}

string CStyxScanner::getTokenID(short TokIdx, short GrpIdx)
{
  if( GrpIdx == -1 )
    return TokIdx >= 0 && ((size_t)TokIdx) < TokId.size()
           ? TokId[TokIdx] : "";
  else
    return GrpIdx >= 0 && ((size_t)GrpIdx) < GrpScn.size() &&
           TokIdx >= 0 && ((size_t)TokIdx) < GrpScn[GrpIdx].TokId.size()
           ? GrpScn[GrpIdx].TokId[TokIdx] : "";
}

bool CStyxScanner::loadTableLine
     (
       string& line, CStyxScanner*& cScanner, 
       string& object, short& tokidx
     )
{
  if( line.length() > 0 && line[0] != '#' )
  { char* n = strstr(line.c_str(),"Name=");
    if( n != NULL )
    {
      cScanner->setLanguage(n+strlen("Name="));
    }
    else
    if( line == "Group" )
    {
      //GrpScn.push_back(CStyxScanner());
      //cScanner = &GrpScn[GrpScn.size()-1];
      addGroupScanner();
      cScanner = currentGroupScanner();
if( cScanner == 0 ) { printf("1!!!!!!!!!!!!!\n"); return false; }
    }
    else
    if( line == "Dyck" )
    {
      //dyckpat.push_back(CStyxScanner());
      //cScanner = &dyckpat[dyckpat.size()-1];
      cScanner->addDyckScanner();
      cScanner = currentDyckScanner();
if( cScanner == 0 ) { printf("2!!!!!!!!!!!!!\n"); return false; }
    }
    else
    if( line == "Language" )
    {
      cScanner = this;
    }
    else
    if( line[0] != '-' && !isdigit(line[0]) && !isspace(line[0]) )
    {
      object = line;
      tokidx = 0;
    }
    else
    if( isspace(line[0]) )
    {
      cScanner->addToken(line.c_str()+1);
    }
    else
    if( line[0] == '-' || isdigit(line[0]) )
    { long int v = atol(line.c_str());
      if( object == "StaEdg" )
      {
        cScanner->addFirstEdge((long)v);
      }
      else
      if( object == "StaFin" )
      {
        cScanner->addFinalState((short)v);
      }
      else
      if( object == "EdgeC" )
      {
        cScanner->addEdgeChar((wchar_t)v);
      }
      else
      if( object == "EdgeS" )
      {
        cScanner->addEdgeState((long)v);
      }
      else
      if( object == "Flags" )
      {
        cScanner->addTokenFlags((byte)v);
      }
      else
      if( object == "Switch" )
      {
        cScanner->addSwitchToken(tokidx,(short)v);
        tokidx++;
      }
      else
      if( object == "DyckToken" )
      {
        cScanner = currentGroupScanner();
if( cScanner == 0 ) { printf("3!!!!!!!!!!!!!\n"); return false; }
        cScanner->addDyckToken((short)v);
      }
    }
  }
  line = "";
  return true;
}

bool CStyxScanner::loadTableFile(string Path)
{ ifstream is(Path.c_str()); char c;
  if( !is ) 
  {
    fprintf(stderr,"couldn't open '%s'\n",Path.c_str());
    return false;
  }
  string line, object; 
  CStyxScanner* cScanner = this;
  short tokidx = 0;
  while( is.get(c) )
  {
    if( c == '\n' )
    {
      if( !loadTableLine(line,cScanner,object,tokidx) )
        return false;
    }
    else line += c;
  }
  bool result = GrpScn.size() > 0 ||
                StaFin.size() > 0 && StaEdg.size() > 0 && 
                EdgeS.size() > 0 && EdgeC.length() > 0 &&
                TokId.size() > 0 && TokId.size() == Flags.size();
  return result && initPattern();
}

bool CStyxScanner::loadTableString(string Table)
{ char c;
  string line, object; 
  CStyxScanner* cScanner = this;
  short tokidx = 0;
  for( size_t i=0; i < Table.length(); ++i )
  {
    c = Table[i];
    if( c == '\n' )
    {
      if( !loadTableLine(line,cScanner,object,tokidx) )
        return false;
    }
    else line += c;
  }
  bool result = GrpScn.size() > 0 ||
                StaFin.size() > 0 && StaEdg.size() > 0 && 
                EdgeS.size() > 0 && EdgeC.length() > 0 &&
                TokId.size() > 0 && TokId.size() == Flags.size();
  return result && initPattern();
}

bool CStyxScanner::scanStream(CStyxStream* Stream, S_int& SkipTokens)
{ 
  next    = &CStyxScanner::nextToken_default;
  cStream = Stream;
  skipC   = SkipTokens;
  if( GrpScn.size() > 0 )
  {
    Group   = 0;
    StaEdg  = GrpScn[0].StaEdg;
    StaFin  = GrpScn[0].StaFin;
    EdgeC   = GrpScn[0].EdgeC;
    EdgeS   = GrpScn[0].EdgeS;
    Flags   = GrpScn[0].Flags;
    for( size_t i=0; i < GrpScn[0].dyckidx.size(); ++i )
    {
      dscanner[GrpScn[Group].dyckidx[i]] = &GrpScn[Group].dyckpat[i];
    }
  }
  else 
  {
    Group   = -1;
    for( size_t i=0; i < dyckidx.size(); ++i )
    {
      dscanner[dyckidx[i]] = &dyckpat[i];
    }
  }
  cC      = cStream->sgetc();
  nLine   = 1;
  nCol    = 1;
  cTok    = 0;
  return true;
}

bool CStyxScanner::eos()
{
  return (cTok == -1); 
}

bool CStyxScanner::nextToken()
{
  return cStream && (this->*next)() 
         ? true : ( cStream = 0, false );
}

bool CStyxScanner::currentToken(string& Token, string& Value)
{
  if( cStream && !cStream->unicode() )
  {
    if (cTok == -1) 
    {
      Token = "[EOF]";
      return true;
    }
    else
    if( GrpScn.size() > 0 )
    {
      Token = GrpScn[Group].TokId[cTok];
    }
    else
    {
      Token = TokId[cTok];
    }
    Value = buffer;
    return true;
  }
  return false;
}

bool CStyxScanner::currentToken(string& Token, wstring& Value)
{
  if( cStream && cStream->unicode() )
  {
    if (cTok == -1)
    {
      Token = "[EOF]";
      return true;
    }
    else
    if( GrpScn.size() > 0 )
    {
      Token = GrpScn[Group].TokId[cTok];
    }
    else
    {
      Token = TokId[cTok];
    }
    Value = wbuffer;
    return true;
  }
  return false;
}

long CStyxScanner::getLine()
{
  return cLine;
}

long CStyxScanner::getCol() 
{
  return cCol;
}

string CStyxScanner::getFile() 
{
  return cStream ? cStream->getFile() : "";
}

string CStyxScanner::externalTokenID(string& Token) 
{ size_t pos = Token.find_first_of(SCN_SEP_TOK_ID_PATTERN);
  if( pos == string::npos )
    return Token;
  else
    return Token.substr(0,pos); 
}
