/*
 *	Adobe Encore DVD text script support for subtitle editor
 *      PAL version.
 *
 *	http://kitone.free.fr/subtitleeditor/
 *
 *	Copyright @ 2005-2006, kitone
 *
 *	Contact: kitone at free dot fr
 *
 *	Adobe Encore DVD text script support by Laurens Keek
 *      Created using following documentation:
 *	http://www.adobe.com/support/techdocs/329569.html
 *	
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public
 *	License as published by the Free Software Foundation; either
 *	version 2 of the License, or (at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU
 *	General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public
 *	License along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA	02111-1307	USA
 *
 *	See gpl.txt for more information regarding the GNU General Public License.
 *
 *
 *	\file
 *	\brief 
 *	\author kitone (kitone at free dot fr)
 */

#include "SubtitleEncorePAL.h"
#include "Color.h"
#include "utility.h"

#include <map>
#include <sstream>
#include <iostream>
#include <fstream>
#include <string>
#include <iomanip>
#include "RegEx.h"

/*
 *
 */
Glib::ustring SubtitleEncorePAL::get_name()
{
	return "Adobe Encore DVD PAL";
}

/*
 *
 */
Glib::ustring SubtitleEncorePAL::get_extension()
{
	return "txt";
}

/*
 *
 */
bool SubtitleEncorePAL::check(const std::string &line)
{
	/* First line should simply be: number start_time stop_time some_text*/
	static RegEx ex("^\\d+ (\\d\\d(:)\\d\\d\\2\\d\\d\\2\\d\\d ){2}");

	return ex.exec(line);
}

/*
 *	constructor
 */
SubtitleEncorePAL::SubtitleEncorePAL(Document* doc)
:SubtitleFormat(doc)
{
	se_debug(SE_DEBUG_LOADER | SE_DEBUG_SAVER);
}

/*
 *
 */
SubtitleEncorePAL::~SubtitleEncorePAL()
{
	se_debug(SE_DEBUG_LOADER | SE_DEBUG_SAVER);
}

/*
 *	read subtitle file
 */
bool SubtitleEncorePAL::open(const Glib::ustring &filename, const Glib::ustring &encoding)
{
	se_debug(SE_DEBUG_LOADER);
	
	SubtitleFormat::open(filename, encoding);
	
	std::ifstream file(filename.c_str());
	
	if(!file)
	{
		throw SubtitleException("SubtitleEncorePAL", _("I can't open this file."));
	}

	std::string line;
	int num;
	int start[4], end[4];
	char *text;
	SubtitleModifier subtitle;
	int framerate = 25;

	while(!file.eof() && std::getline(file, line))
	{
		text = (char *) malloc (line.length());
		if (sscanf(line.c_str(), "%d %d:%d:%d:%d %d:%d:%d:%d %[^\n]", &num,
					&start[0], &start[1], &start[2], &start[3], 
					&end[0], &end[1], &end[2], &end[3],
					text) == 10) {
			// we have a new subtitle
			subtitle.init( m_subtitleModel->append() );
			//convert times
			start[3] = start[3]*1000/framerate;
			end[3] = end[3]*1000/framerate;
			//set subtitle data
			subtitle.set_start_and_end(
						SubtitleTime(start[0], start[1], start[2], start[3]),
						SubtitleTime(end[0], end[1], end[2], end[3]));
			subtitle.set_text(charset_to_utf8(text));
		}
		else {
			//this is another line of the previous subtitle
			subtitle.set_text(subtitle.get_text() + get_newline() + charset_to_utf8(line));
		}
		free(text);
	}

	file.close();

	return true;
}

/*
 *	Save subtitle file
 */
bool SubtitleEncorePAL::save(const Glib::ustring &filename, const Glib::ustring &encoding)
{
	se_debug(SE_DEBUG_SAVER);

	SubtitleFormat::save(filename, encoding);
	
	std::ofstream file(filename.c_str());

	if(!file)
	{
		throw SubtitleException("SubtitleEncorePAL", _("I can't open this file."));
	}
	
	se_debug_message(SE_DEBUG_SAVER, "save Event");

	// Event
	{
		// dialog:
		Gtk::TreeNodeChildren rows = m_subtitleModel->children();
		
		for(Gtk::TreeIter it = rows.begin(); it; ++it)
		{
			SubtitleModifier subtitle(it);

			std::ostringstream oss;

			Glib::ustring text = subtitle.get_text();

			oss	<< subtitle.get_num() << " "
				<< subtitletime_to_encore_time(subtitle.get_start()) << " "
				<< subtitletime_to_encore_time(subtitle.get_end()) << " "
				<< utf8_to_charset(text) << std::endl;

			file << oss.str();
		}
	}
	
	file.close();

	return true;
}

/*
 *	convertir le temps utiliser par subtitle editor en temps valide pour le format Encore DVD
 *	0:00:00.000 -> 00:00:00:00 (last 00 are frames, not time!)
 */
Glib::ustring SubtitleEncorePAL::subtitletime_to_encore_time(const SubtitleTime &time)
{
	int framerate = 25;
	int frame = (int)(time.msecs*framerate*0.001);

	gchar *tmp = g_strdup_printf("%02i:%02i:%02i:%02i",
			time.hours, time.mins, time.secs, frame);

	Glib::ustring res(tmp);

	g_free(tmp);

	return res;
}
