/*==================================================================
 * SwamiUIGenCtrl.c - User interface generator control object
 *
 * Swami
 * Copyright (C) 1999-2003 Josh Green <jgreen@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 *
 * To contact the author of this program:
 * Email: Josh Green <jgreen@users.sourceforge.net>
 * Swami homepage: http://swami.sourceforge.net
 *==================================================================*/
#include <stdio.h>
#include <stdlib.h>
#include <gtk/gtk.h>
#include <instpatch.h>

#include <libswami/SwamiWavetbl.h>

#include "SwamiUIGenCtrl.h"
#include "SwamiUIObject.h"

#include "pixmap.h"
#include "util.h"
#include "i18n.h"

#define GENCTRL_COUNT 35

typedef struct _GenCtrl
{
  SwamiUIGenCtrl *genctrl;	/* for GTK callback convenience */
  GtkWidget *scale;		/* scale widget */
  GtkWidget *entry;		/* entry widget */
  GtkWidget *units;		/* units label */
  GtkWidget *defbtn;		/* default toggle button */
  guint16 genid;
} GenCtrl;


static void swamiui_genctrl_class_init (SwamiUIGenCtrlClass *klass);
static void swamiui_genctrl_init (SwamiUIGenCtrl *genctrl);
static void swamiui_genctrl_cb_zone_gen_change (SwamiObject *swami,
						IPZone *zone,
						SwamiUIGenCtrl *genctrl);
static void swamiui_genctrl_cb_show (GtkWidget *widg, gpointer data);

static GtkWidget *create_gen_control (SwamiUIGenCtrl *genctrl, guint16 genid);

static void genctrl_default_btn_set_active (GenCtrl *ctrl, gboolean active,
					    gboolean force);
static void genctrl_scale_set_value (GenCtrl *ctrl, int val);
static void genctrl_entry_set_value (GenCtrl *ctrl, int val);

static void cb_default_btn_toggled (GtkToggleButton *btn, GenCtrl *ctrl);
static void cb_ctrl_value_change (GtkAdjustment *adj, GenCtrl *ctrl);
static void swamiui_genctrl_cb_entry_activate (GtkWidget *entry,
					       GenCtrl *ctrl);
static void swamiui_genctrl_cb_entry_focus_out (GtkWidget *entry,
						GdkEventFocus *event,
						GenCtrl *ctrl);
/*
  static void sfgen_cb_gen_clist_button_pressed (GtkWidget * btn, gint page)
*/


guint
swamiui_genctrl_get_type (void)
{
  static guint obj_type = 0;

  if (!obj_type)
    {
      GtkTypeInfo obj_info = {
	"SwamiUIGenCtrl",
	sizeof (SwamiUIGenCtrl),
	sizeof (SwamiUIGenCtrlClass),
	(GtkClassInitFunc) swamiui_genctrl_class_init,
	(GtkObjectInitFunc) swamiui_genctrl_init,
	(GtkArgSetFunc) NULL,
	(GtkArgGetFunc) NULL,
      };
      obj_type = gtk_type_unique (gtk_scrolled_window_get_type (), &obj_info);
    }

  return obj_type;
}

static void
swamiui_genctrl_class_init (SwamiUIGenCtrlClass *klass)
{
}

static void
swamiui_genctrl_init (SwamiUIGenCtrl *genctrl)
{
  GtkWidget *notebook;
  GtkWidget *label;
  GtkWidget *ctrl;
  GtkWidget *frame;
  GtkWidget *box, *box1, *box2;

  genctrl->zone = NULL;
  genctrl->queue_update = FALSE;

  genctrl->ctrl_array = g_malloc (sizeof (GenCtrl) * GENCTRL_COUNT);
  genctrl->ctrl_counter = 0;	/* stupid counter used only during init */

  gtk_scrolled_window_set_hadjustment (GTK_SCROLLED_WINDOW (genctrl), NULL);
  gtk_scrolled_window_set_vadjustment (GTK_SCROLLED_WINDOW (genctrl), NULL);

  gtk_container_border_width (GTK_CONTAINER (genctrl), 0);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (genctrl),
    GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

  /* create notebook */
  notebook = gtk_notebook_new ();
  genctrl->notebook_widg = notebook; /* remember notebook widget pointer */
  gtk_widget_show (notebook);
  gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (genctrl),
					 notebook);
  /* create vbox for first page */
  box = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (box);

  /* Pitch frame */
  frame = gtk_frame_new (_("Pitch"));
  gtk_widget_show (frame);
  gtk_box_pack_start (GTK_BOX (box), frame, FALSE, FALSE, 2);
  box1 = gtk_hbox_new (TRUE, 0);
  gtk_widget_show (box1);
  gtk_container_add (GTK_CONTAINER (frame), box1);
  ctrl = create_gen_control (genctrl, IPGEN_COARSE_TUNE);
  gtk_box_pack_start (GTK_BOX (box1), ctrl, TRUE, TRUE, 2);
  ctrl = create_gen_control (genctrl, IPGEN_FINE_TUNE);
  gtk_box_pack_start (GTK_BOX (box1), ctrl, TRUE, TRUE, 2);
  ctrl = create_gen_control (genctrl, IPGEN_SCALE_TUNE);
  gtk_box_pack_start (GTK_BOX (box1), ctrl, TRUE, TRUE, 2);

  /* Filter frame */
  frame = gtk_frame_new (_("Filter"));
  gtk_widget_show (frame);
  gtk_box_pack_start (GTK_BOX (box), frame, FALSE, FALSE, 2);
  box1 = gtk_hbox_new (TRUE, 0);
  gtk_widget_show (box1);
  gtk_container_add (GTK_CONTAINER (frame), box1);
  ctrl = create_gen_control (genctrl, IPGEN_FILTER_Q);
  gtk_box_pack_start (GTK_BOX (box1), ctrl, TRUE, TRUE, 2);
  ctrl = create_gen_control (genctrl, IPGEN_FILTER_FC);
  gtk_box_pack_start (GTK_BOX (box1), ctrl, TRUE, TRUE, 2);

  /* Effects frame */
  frame = gtk_frame_new (_("Effects"));
  gtk_widget_show (frame);
  gtk_box_pack_start (GTK_BOX (box), frame, FALSE, FALSE, 2);
  box1 = gtk_hbox_new (TRUE, 0);
  gtk_widget_show (box1);
  gtk_container_add (GTK_CONTAINER (frame), box1);
  ctrl = create_gen_control (genctrl, IPGEN_REVERB_SEND);
  gtk_box_pack_start (GTK_BOX (box1), ctrl, TRUE, TRUE, 2);
  ctrl = create_gen_control (genctrl, IPGEN_CHORUS_SEND);
  gtk_box_pack_start (GTK_BOX (box1), ctrl, TRUE, TRUE, 2);
  ctrl = create_gen_control (genctrl, IPGEN_PAN);
  gtk_box_pack_start (GTK_BOX (box1), ctrl, TRUE, TRUE, 2);

  /* add the box to the notebook with label */
  label = gtk_label_new (_("Pitch/Effects"));
  gtk_widget_show (label);
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), box, label);


  /* create vbox for second page */
  box = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (box);

  /* Vol Envelope frame */
  frame = gtk_frame_new (_("Volume Envelope"));
  gtk_widget_show (frame);
  gtk_box_pack_start (GTK_BOX (box), frame, FALSE, FALSE, 2);
  box1 = gtk_vbox_new (TRUE, 0);
  gtk_widget_show (box1);
  gtk_container_add (GTK_CONTAINER (frame), box1);

  box2 = gtk_hbox_new (TRUE, 0);
  gtk_widget_show (box2);
  gtk_box_pack_start (GTK_BOX (box1), box2, TRUE, TRUE, 2);
  ctrl = create_gen_control (genctrl, IPGEN_VOL_ENV_DELAY);
  gtk_box_pack_start (GTK_BOX (box2), ctrl, TRUE, TRUE, 2);
  ctrl = create_gen_control (genctrl, IPGEN_VOL_ENV_ATTACK);
  gtk_box_pack_start (GTK_BOX (box2), ctrl, TRUE, TRUE, 2);
  ctrl = create_gen_control (genctrl, IPGEN_VOL_ENV_HOLD);
  gtk_box_pack_start (GTK_BOX (box2), ctrl, TRUE, TRUE, 2);

  box2 = gtk_hbox_new (TRUE, 0);
  gtk_widget_show (box2);
  gtk_box_pack_start (GTK_BOX (box1), box2, TRUE, TRUE, 2);
  ctrl = create_gen_control (genctrl, IPGEN_VOL_ENV_DECAY);
  gtk_box_pack_start (GTK_BOX (box2), ctrl, TRUE, TRUE, 2);
  ctrl = create_gen_control (genctrl, IPGEN_VOL_ENV_SUSTAIN);
  gtk_box_pack_start (GTK_BOX (box2), ctrl, TRUE, TRUE, 2);
  ctrl = create_gen_control (genctrl, IPGEN_VOL_ENV_RELEASE);
  gtk_box_pack_start (GTK_BOX (box2), ctrl, TRUE, TRUE, 2);

  box2 = gtk_hbox_new (TRUE, 0);
  gtk_widget_show (box2);
  gtk_box_pack_start (GTK_BOX (box1), box2, TRUE, TRUE, 2);
  ctrl = create_gen_control (genctrl, IPGEN_ATTENUATION);
  gtk_box_pack_start (GTK_BOX (box2), ctrl, TRUE, TRUE, 2);
  ctrl = create_gen_control (genctrl, IPGEN_KEY_TO_VOL_ENV_HOLD);
  gtk_box_pack_start (GTK_BOX (box2), ctrl, TRUE, TRUE, 2);
  ctrl = create_gen_control (genctrl, IPGEN_KEY_TO_VOL_ENV_DECAY);
  gtk_box_pack_start (GTK_BOX (box2), ctrl, TRUE, TRUE, 2);

  /* add the box to the notebook with label */
  label = gtk_label_new (_("Vol Envelope"));
  gtk_widget_show (label);
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), box, label);


  /* create vbox for third page */
  box = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (box);

  /* Mod Envelope frame */
  frame = gtk_frame_new (_("Modulation Envelope"));
  gtk_widget_show (frame);
  gtk_box_pack_start (GTK_BOX (box), frame, FALSE, FALSE, 2);
  box1 = gtk_vbox_new (TRUE, 0);
  gtk_widget_show (box1);
  gtk_container_add (GTK_CONTAINER (frame), box1);

  box2 = gtk_hbox_new (TRUE, 0);
  gtk_widget_show (box2);
  gtk_box_pack_start (GTK_BOX (box1), box2, TRUE, TRUE, 2);
  ctrl = create_gen_control (genctrl, IPGEN_MOD_ENV_DELAY);
  gtk_box_pack_start (GTK_BOX (box2), ctrl, TRUE, TRUE, 2);
  ctrl = create_gen_control (genctrl, IPGEN_MOD_ENV_ATTACK);
  gtk_box_pack_start (GTK_BOX (box2), ctrl, TRUE, TRUE, 2);
  ctrl = create_gen_control (genctrl, IPGEN_MOD_ENV_HOLD);
  gtk_box_pack_start (GTK_BOX (box2), ctrl, TRUE, TRUE, 2);

  box2 = gtk_hbox_new (TRUE, 0);
  gtk_widget_show (box2);
  gtk_box_pack_start (GTK_BOX (box1), box2, TRUE, TRUE, 2);
  ctrl = create_gen_control (genctrl, IPGEN_MOD_ENV_DECAY);
  gtk_box_pack_start (GTK_BOX (box2), ctrl, TRUE, TRUE, 2);
  ctrl = create_gen_control (genctrl, IPGEN_MOD_ENV_SUSTAIN);
  gtk_box_pack_start (GTK_BOX (box2), ctrl, TRUE, TRUE, 2);
  ctrl = create_gen_control (genctrl, IPGEN_MOD_ENV_RELEASE);
  gtk_box_pack_start (GTK_BOX (box2), ctrl, TRUE, TRUE, 2);

  box2 = gtk_hbox_new (TRUE, 0);
  gtk_widget_show (box2);
  gtk_box_pack_start (GTK_BOX (box1), box2, TRUE, TRUE, 2);
  ctrl = create_gen_control (genctrl, IPGEN_MOD_ENV_TO_PITCH);
  gtk_box_pack_start (GTK_BOX (box2), ctrl, TRUE, TRUE, 2);
  ctrl = create_gen_control (genctrl, IPGEN_MOD_ENV_TO_FILTER_FC);
  gtk_box_pack_start (GTK_BOX (box2), ctrl, TRUE, TRUE, 2);

  box2 = gtk_hbox_new (TRUE, 0);
  gtk_widget_show (box2);
  gtk_box_pack_start (GTK_BOX (box1), box2, TRUE, TRUE, 2);
  ctrl = create_gen_control (genctrl, IPGEN_KEY_TO_MOD_ENV_HOLD);
  gtk_box_pack_start (GTK_BOX (box2), ctrl, TRUE, TRUE, 2);
  ctrl = create_gen_control (genctrl, IPGEN_KEY_TO_MOD_ENV_DECAY);
  gtk_box_pack_start (GTK_BOX (box2), ctrl, TRUE, TRUE, 2);

  /* add the box to the notebook with label */
  label = gtk_label_new (_("Mod Envelope"));
  gtk_widget_show (label);
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), box, label);


  /* create vbox for fourth page */
  box = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (box);

  /* LFO frame */
  /* LFO stands for Low Frequency Oscillator */
  frame = gtk_frame_new (_("Modulation LFO"));
  gtk_widget_show (frame);
  gtk_box_pack_start (GTK_BOX (box), frame, FALSE, FALSE, 2);
  box1 = gtk_vbox_new (TRUE, 0);
  gtk_widget_show (box1);
  gtk_container_add (GTK_CONTAINER (frame), box1);

  box2 = gtk_hbox_new (TRUE, 0);
  gtk_widget_show (box2);
  gtk_box_pack_start (GTK_BOX (box1), box2, TRUE, TRUE, 2);
  ctrl = create_gen_control (genctrl, IPGEN_MOD_LFO_DELAY);
  gtk_box_pack_start (GTK_BOX (box2), ctrl, TRUE, TRUE, 2);
  ctrl = create_gen_control (genctrl, IPGEN_MOD_LFO_FREQ);
  gtk_box_pack_start (GTK_BOX (box2), ctrl, TRUE, TRUE, 2);

  box2 = gtk_hbox_new (TRUE, 0);
  gtk_widget_show (box2);
  gtk_box_pack_start (GTK_BOX (box1), box2, TRUE, TRUE, 2);
  ctrl = create_gen_control (genctrl, IPGEN_MOD_LFO_TO_PITCH);
  gtk_box_pack_start (GTK_BOX (box2), ctrl, TRUE, TRUE, 2);
  ctrl = create_gen_control (genctrl, IPGEN_MOD_LFO_TO_FILTER_FC);
  gtk_box_pack_start (GTK_BOX (box2), ctrl, TRUE, TRUE, 2);
  ctrl = create_gen_control (genctrl, IPGEN_MOD_LFO_TO_VOL);
  gtk_box_pack_start (GTK_BOX (box2), ctrl, TRUE, TRUE, 2);

  frame = gtk_frame_new (_("Vibrato LFO"));
  gtk_widget_show (frame);
  gtk_box_pack_start (GTK_BOX (box), frame, FALSE, FALSE, 2);
  box1 = gtk_vbox_new (TRUE, 0);
  gtk_widget_show (box1);
  gtk_container_add (GTK_CONTAINER (frame), box1);

  box2 = gtk_hbox_new (TRUE, 0);
  gtk_widget_show (box2);
  gtk_box_pack_start (GTK_BOX (box1), box2, TRUE, TRUE, 2);
  ctrl = create_gen_control (genctrl, IPGEN_VIB_LFO_DELAY);
  gtk_box_pack_start (GTK_BOX (box2), ctrl, TRUE, TRUE, 2);
  ctrl = create_gen_control (genctrl, IPGEN_VIB_LFO_FREQ);
  gtk_box_pack_start (GTK_BOX (box2), ctrl, TRUE, TRUE, 2);
  ctrl = create_gen_control (genctrl, IPGEN_VIB_LFO_TO_PITCH);
  gtk_box_pack_start (GTK_BOX (box2), ctrl, TRUE, TRUE, 2);

  /* add the box to the notebook with label */
  label = gtk_label_new (_("LFOs"));
  gtk_widget_show (label);
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), box, label);

  g_signal_connect (swami_object, "zone_gen_change",
		    (GCallback)swamiui_genctrl_cb_zone_gen_change, genctrl);
  gtk_signal_connect (GTK_OBJECT (genctrl), "show",
		      swamiui_genctrl_cb_show, NULL);
}

static void
swamiui_genctrl_cb_zone_gen_change (SwamiObject *swami, IPZone *zone,
				    SwamiUIGenCtrl *genctrl)
{
  if (zone != genctrl->zone)
    return;

  if (GTK_WIDGET_VISIBLE (GTK_WIDGET (genctrl)))
    swamiui_genctrl_update (genctrl);
  else genctrl->queue_update = TRUE;
}

static void
swamiui_genctrl_cb_show (GtkWidget *widg, gpointer data)
{
  SwamiUIGenCtrl *genctrl = SWAMIUI_GENCTRL (widg);

  if (genctrl->queue_update)
    {
      swamiui_genctrl_update (genctrl);
      genctrl->queue_update = FALSE;
    }
}

/**
 * Create a new generator control object
 * Returns: new widget of type SwamiUIGenCtrl
 */
GtkWidget *
swamiui_genctrl_new (void)
{
  return (GTK_WIDGET (gtk_type_new (swamiui_genctrl_get_type ())));
}

static GtkWidget *
create_gen_control (SwamiUIGenCtrl *genctrl, guint16 genid)
{
  GenCtrl *ctrl;
  GtkWidget *vbox, *hbox;
  GtkWidget *scale;
  GtkWidget *entry;
  GtkWidget *title;
  GtkWidget *alignment;
  GtkWidget *pixmap;
  GtkObject *adj;

  ctrl = &((GenCtrl *)(genctrl->ctrl_array))[genctrl->ctrl_counter++];

  ctrl->genctrl = genctrl;
  ctrl->genid = genid;

  vbox = gtk_vbox_new (FALSE, 0); /* vbox for title, entry, units & scale */
  gtk_widget_show (vbox);

  adj = gtk_adjustment_new (0.0, 0.0, 1.0, 1.0, 10.0, 0.0);
  ctrl->scale = scale = gtk_hscale_new (GTK_ADJUSTMENT (adj));

  /* don't draw current scale value (we use an entry instead) */
  gtk_scale_set_draw_value (GTK_SCALE (scale), FALSE);
  gtk_widget_show (scale);
  gtk_widget_set_sensitive (scale, FALSE); /* disable scale for now */

  /* text entry for scale value */
  ctrl->entry = entry = gtk_entry_new_with_max_length (10);
  gtk_widget_set_usize (entry, 60, 0);
  gtk_widget_show (entry);
  gtk_widget_set_sensitive (entry, FALSE); /* disable entry for now */

  /* title label (NOTE: instp_gen_info[].label are constant strings, so they
     must be translated everytime they are used) */
  title = gtk_label_new (_(instp_gen_info[genid].label));
  gtk_widget_show (title);

  /* units label */
  ctrl->units = gtk_label_new ("");
  gtk_widget_show (ctrl->units);

  alignment = gtk_alignment_new (0.5, 0.5, 0.0, 0.0);
  gtk_widget_show (alignment);

  /* default toggle button */
  ctrl->defbtn = gtk_toggle_button_new ();
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (ctrl->defbtn), TRUE);
  gtk_signal_connect (GTK_OBJECT (ctrl->defbtn), "toggled",
		      GTK_SIGNAL_FUNC (cb_default_btn_toggled), ctrl);
  gtk_widget_show (ctrl->defbtn);
  gtk_widget_set_sensitive (ctrl->defbtn, FALSE); /* disable button for now */
  gtk_container_add (GTK_CONTAINER (alignment), ctrl->defbtn);

  pixmap = swamiui_util_create_pixmap (gen_default_xpm);
  gtk_widget_show (pixmap);
  gtk_container_add (GTK_CONTAINER (ctrl->defbtn), pixmap);

  hbox = gtk_hbox_new (FALSE, 0);	/* hbox for title, entry and units */
  gtk_widget_show (hbox);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

  gtk_box_pack_start (GTK_BOX (hbox), title, FALSE, FALSE, 0);
  gtk_box_pack_end (GTK_BOX (hbox), alignment, FALSE, FALSE, 0);
  gtk_box_pack_end (GTK_BOX (hbox), entry, FALSE, FALSE, 0);
  gtk_box_pack_end (GTK_BOX (hbox), ctrl->units, FALSE, FALSE, 2);

  gtk_box_pack_start (GTK_BOX (vbox), scale, FALSE, FALSE, 2);

  /* hook adjustment value_changed signal */
  gtk_signal_connect (adj, "value_changed",
  		      GTK_SIGNAL_FUNC (cb_ctrl_value_change), ctrl);

  /* hook entry activate signal (when user presses enter on entry value) */
  gtk_signal_connect (GTK_OBJECT (entry), "activate",
		      GTK_SIGNAL_FUNC (swamiui_genctrl_cb_entry_activate),
		      ctrl);
  gtk_signal_connect (GTK_OBJECT (entry), "focus-out-event",
		      GTK_SIGNAL_FUNC (swamiui_genctrl_cb_entry_focus_out),
		      ctrl);
  return (vbox);
}

static void
genctrl_default_btn_set_active (GenCtrl *ctrl, gboolean active, gboolean force)
{
  GList *children;
  char **xpm;
  GdkPixmap *pm;
  GdkBitmap *bm;

  if (!force && gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (ctrl->defbtn))
      == active)
    return;

  xpm = active ? gen_default_xpm : gen_set_xpm;
  pixmap_get (xpm, &pm, &bm);

  children = gtk_container_children (GTK_CONTAINER (ctrl->defbtn));
  gtk_pixmap_set (GTK_PIXMAP (children->data), pm, bm);
  g_list_free (children);

  gtk_signal_handler_block_by_func (GTK_OBJECT (ctrl->defbtn),
				    GTK_SIGNAL_FUNC (cb_default_btn_toggled),
				    ctrl);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (ctrl->defbtn), active);
  gtk_signal_handler_unblock_by_func (GTK_OBJECT (ctrl->defbtn),
				      GTK_SIGNAL_FUNC (cb_default_btn_toggled),
				      ctrl);
}

static void
genctrl_scale_set_value (GenCtrl *ctrl, int val)
{
  SwamiUIGenCtrl *genctrl = ctrl->genctrl;
  GtkAdjustment *adj;

  adj = gtk_range_get_adjustment (GTK_RANGE (ctrl->scale));
  genctrl->block_handlers = TRUE;
  gtk_adjustment_set_value (adj, (float)val);
  genctrl->block_handlers = FALSE;
}

static void
genctrl_entry_set_value (GenCtrl *ctrl, int val)
{
  SwamiUIGenCtrl *genctrl = ctrl->genctrl;
  IPItem *parent;
  IPGenAmount amt;
  gboolean ispreset = FALSE;
  char *s;

  parent = instp_item_parent (INSTP_ITEM (genctrl->zone));
  if (parent && INSTP_IS_PRESET (parent))
    ispreset = TRUE;

  amt.sword = val;
  instp_units_sfont_to_user_str (ctrl->genid, amt, NULL, &s, ispreset);
  gtk_entry_set_text (GTK_ENTRY (ctrl->entry), s);
  g_free (s);
}

static void
cb_default_btn_toggled (GtkToggleButton *btn, GenCtrl *ctrl)
{
  SwamiUIGenCtrl *genctrl = ctrl->genctrl;
  IPGenAmount amt;
  gboolean active;

  if (!genctrl->zone) return;

  active = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (ctrl->defbtn));
  genctrl_default_btn_set_active (ctrl, active, TRUE);

  if (INSTP_IS_PARENT_PRESET (genctrl->zone))
    amt.sword = 0;
  else amt.sword = instp_gen_info[ctrl->genid].def;

  if (active) swami_zone_unset_gen (swami_object, genctrl->zone, ctrl->genid);
  else swami_zone_set_gen (swami_object, genctrl->zone, ctrl->genid, amt);

  genctrl_scale_set_value (ctrl, amt.sword);
  genctrl_entry_set_value (ctrl, amt.sword);

  g_signal_handlers_block_by_func (swami_object,
				   swamiui_genctrl_cb_zone_gen_change,
				   genctrl);
  g_signal_emit_by_name (swami_object, "zone_gen_change", genctrl->zone);
  g_signal_handlers_unblock_by_func (swami_object,
				     swamiui_genctrl_cb_zone_gen_change,
				     genctrl);
}

static void
cb_ctrl_value_change (GtkAdjustment *adj, GenCtrl *ctrl)
{
  SwamiUIGenCtrl *genctrl = ctrl->genctrl;
  SwamiWavetbl *wavetbl;
  IPGenAmount amt;

  if (genctrl->block_handlers || !genctrl->zone) return;

  amt.sword = (gint16) (GTK_ADJUSTMENT (adj)->value);
  swami_zone_set_gen (swami_object, genctrl->zone, ctrl->genid, amt);

  wavetbl = SWAMI_WAVETBL (swami_get_object_by_type (G_OBJECT (swami_object),
						     "SwamiWavetbl"));
  if (wavetbl)
    {
      IPItem *parent;

      parent = instp_item_parent (INSTP_ITEM (genctrl->zone));
      if (parent)
	swami_wavetbl_set_gen_realtime (wavetbl, parent,
					INSTP_ITEM (genctrl->zone),
					ctrl->genid, amt.sword);
    }

  genctrl_entry_set_value (ctrl, amt.sword);
  genctrl_default_btn_set_active (ctrl, FALSE, FALSE);

  g_signal_handlers_block_by_func (swami_object,
				   swamiui_genctrl_cb_zone_gen_change,
				   genctrl);
  g_signal_emit_by_name (swami_object, "zone_gen_change", genctrl->zone);
  g_signal_handlers_unblock_by_func (swami_object,
				     swamiui_genctrl_cb_zone_gen_change,
				     genctrl);
}

static void
swamiui_genctrl_cb_entry_activate (GtkWidget *entry, GenCtrl *ctrl)
{
  SwamiUIGenCtrl *genctrl = ctrl->genctrl;
  IPItem *parent;
  IPGenAmount amt;
  gboolean ispreset = FALSE;
  gfloat f;
  char *s, *endptr = NULL;

  if (genctrl->block_handlers || !genctrl->zone) return;

  parent = instp_item_parent (INSTP_ITEM (genctrl->zone));
  if (parent && INSTP_IS_PRESET (parent))
    ispreset = TRUE;

  /* fetch the entry text and convert it to sfont units */
  s = gtk_entry_get_text (GTK_ENTRY (entry));
  f = strtod (s, &endptr);

  if (endptr != s)		/* make sure conversion ok */
    amt.sword = instp_units_user_to_sfont (ctrl->genid, f, ispreset, TRUE);
  else				/* not good, get current value */
    swami_zone_get_gen (swami_object, genctrl->zone, ctrl->genid, &amt);


  /* set the generator value */
  swami_zone_set_gen (swami_object, genctrl->zone, ctrl->genid, amt);


  /* convert back to possibly corrected user units and place it in the entry */
  instp_units_sfont_to_user_str (ctrl->genid, amt, NULL, &s, ispreset);
  gtk_entry_set_text (GTK_ENTRY (ctrl->entry), s);
  g_free (s);

  genctrl_scale_set_value (ctrl, amt.sword);
  genctrl_default_btn_set_active (ctrl, FALSE, FALSE);

  g_signal_handlers_block_by_func (swami_object,
				   swamiui_genctrl_cb_zone_gen_change,
				   genctrl);
  g_signal_emit_by_name (swami_object, "zone_gen_change", genctrl->zone);
  g_signal_handlers_unblock_by_func (swami_object,
				     swamiui_genctrl_cb_zone_gen_change,
				     genctrl);
}

/* if the entry focus leaves, set entry text to current gen value */
static void
swamiui_genctrl_cb_entry_focus_out (GtkWidget *entry, GdkEventFocus *event,
				    GenCtrl *ctrl)
{
  SwamiUIGenCtrl *genctrl = ctrl->genctrl;
  IPGenAmount amt;
  IPItem *parent;
  gboolean ispreset = FALSE;
  char *s;

  if (!genctrl->zone) return;

  parent = instp_item_parent (INSTP_ITEM (genctrl->zone));
  if (parent && INSTP_IS_PRESET (parent))
    ispreset = TRUE;

  amt.sword = (gint16)(GTK_ADJUSTMENT (gtk_range_get_adjustment
				       (GTK_RANGE (ctrl->scale)))->value);
  instp_units_sfont_to_user_str (ctrl->genid, amt, NULL, &s, ispreset);
  gtk_entry_set_text (GTK_ENTRY (ctrl->entry), s);
  g_free (s);
}

/**
 * Set patch item to control generators of
 * @genctrl Generator control object
 * @item Patch item to control (only IPZone items used) or NULL to
 *   deactivate control object
 */
void
swamiui_genctrl_set_item (SwamiUIGenCtrl *genctrl, IPItem *item)
{
  IPZone *zone = NULL;
  GtkAdjustment *adj;
  GenCtrl *ctrl;  
  gboolean ispreset = FALSE;
  int i;

  g_return_if_fail (genctrl != NULL);
  g_return_if_fail (SWAMIUI_IS_GENCTRL (genctrl));

  if (item && !INSTP_IS_ZONE (item)) item = NULL;
  if (item) zone = INSTP_ZONE (item);

  /* check if requested zone is already active */
  if (zone == genctrl->zone) return;

  genctrl->zone = zone;

  if (genctrl->zone && INSTP_IS_PARENT_PRESET (genctrl->zone))
    ispreset = TRUE;

  /* set labels */
  genctrl->block_handlers = TRUE;
  for (i = 0; i < GENCTRL_COUNT; i++)
    {
      ctrl = &((GenCtrl *)(genctrl->ctrl_array))[i];

      if (!genctrl->zone)     /* disable and clear controls if NULL */
	{
	  gtk_label_set_text (GTK_LABEL (ctrl->units), "");

	  genctrl_default_btn_set_active (ctrl, TRUE, FALSE);
	  gtk_widget_set_sensitive (ctrl->defbtn, FALSE);

	  gtk_entry_set_text (GTK_ENTRY (ctrl->entry), "");
	  gtk_widget_set_sensitive (ctrl->entry, FALSE);

	  adj = gtk_range_get_adjustment (GTK_RANGE (ctrl->scale));
	  adj->value = 0.0;
	  adj->lower = 0.0;
	  adj->upper = 1.0;
	  gtk_adjustment_changed (adj);
	  gtk_adjustment_value_changed (adj);
	  gtk_widget_set_sensitive (ctrl->scale, FALSE);
	}
      else if (!ispreset)	/* instrument zone? */
	gtk_label_set_text (GTK_LABEL (ctrl->units),
			    _(instp_gen_conv[instp_gen_info[ctrl->genid]
					     .unit].unit_text));
      else			/* preset zone */
	gtk_label_set_text (GTK_LABEL (ctrl->units),
			    _(instp_gen_conv[instp_gen_info[ctrl->genid]
					     .unit].ofs_text));
    }
  genctrl->block_handlers = FALSE;

  swamiui_genctrl_update (genctrl);
}

/**
 * swamiui_genctrl_update:
 * @genctrl: Swami gen control object
 *
 * Synchronizes a generator control object to its assigned zone.
 */
void
swamiui_genctrl_update (SwamiUIGenCtrl *genctrl)
{
  GtkAdjustment *adj;
  GenCtrl *ctrl;  
  IPGenAmount amt;
  gboolean ispreset = FALSE;
  gboolean set;
  char *s;
  int i, x;

  if (!genctrl->zone) return;

  if (INSTP_IS_PARENT_PRESET (genctrl->zone))
    ispreset = TRUE;

  genctrl->block_handlers = TRUE;

  for (i = 0; i < GENCTRL_COUNT; i++)
    {
      ctrl = &((GenCtrl *)(genctrl->ctrl_array))[i];

      if (!ispreset)		/* instrument zone? */
	{
	  set = swami_zone_get_gen (swami_object, genctrl->zone,
				    ctrl->genid, &amt);

	  genctrl_default_btn_set_active (ctrl, !set, FALSE);
	  gtk_widget_set_sensitive (ctrl->defbtn, TRUE);

	  instp_units_sfont_to_user_str (ctrl->genid, amt, NULL, &s, ispreset);
	  gtk_entry_set_text (GTK_ENTRY (ctrl->entry), s);
	  g_free (s);
	  gtk_widget_set_sensitive (ctrl->entry, TRUE);

	  adj = gtk_range_get_adjustment (GTK_RANGE (ctrl->scale));
	  adj->value = (gfloat) amt.sword;
	  adj->lower = (gfloat) instp_gen_info[ctrl->genid].min;
	  adj->upper = (gfloat) instp_gen_info[ctrl->genid].max;
	  gtk_adjustment_changed (adj);
	  gtk_adjustment_value_changed (adj);
	  gtk_widget_set_sensitive (GTK_WIDGET (ctrl->scale), TRUE);
	}
      else			/* preset zone */
	{
	  set = swami_zone_get_gen (swami_object, genctrl->zone,
				    ctrl->genid, &amt);

	  genctrl_default_btn_set_active (ctrl, !set, FALSE);
	  gtk_widget_set_sensitive (ctrl->defbtn, TRUE);

	  instp_units_sfont_to_user_str (ctrl->genid, amt, NULL, &s, ispreset);
	  gtk_entry_set_text (GTK_ENTRY (ctrl->entry), s);
	  g_free (s);
	  gtk_widget_set_sensitive (ctrl->entry, TRUE);

	  x = instp_gen_info[ctrl->genid].max
	    - instp_gen_info[ctrl->genid].min;
	  adj = gtk_range_get_adjustment (GTK_RANGE (ctrl->scale));
	  adj->value = amt.sword;
	  adj->lower = (gfloat) (-x);
	  adj->upper = (gfloat) x;
	  gtk_adjustment_changed (adj);
	  gtk_adjustment_value_changed (adj);
	  gtk_widget_set_sensitive (GTK_WIDGET (ctrl->scale), TRUE);
	}
    }

  genctrl->block_handlers = FALSE;
}
