/*==================================================================
 * sfload.c - sound font loading functions
 *
 * libsoundfont
 * Copyright (C) 1999-2003 Josh Green <jgreen@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 *==================================================================*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "instpatch.h"
#include "i18n.h"

typedef struct _IPLoadContext
{
  IPFileContext *ctx;
  IPLoadSampleCallback *smpl_cb; /* sample data callback */
  void *cb_userptr;		/* smpl_cb user pointer */
  guint16 flags;		/* parts of sound font to be loaded */
  IPSFont *sf;			/* for loaded sound font data */
}
SFLoadContext;

#define INSTP_ERRMSG_CORRUPT_0 "Sound font is corrupt"

static int sfload_callback (IPFileContext *ctx, void *userptr);
static int sfload_infos (SFLoadContext *ld);
static int sfload_phdrs (SFLoadContext *ld);
static int sfload_pbags (SFLoadContext *ld);
static int sfload_pmods (SFLoadContext *ld);
static int sfload_pgens (SFLoadContext *ld);
static int sfload_ihdrs (SFLoadContext *ld);
static int sfload_ibags (SFLoadContext *ld);
static int sfload_imods (SFLoadContext *ld);
static int sfload_igens (SFLoadContext *ld);
static int sfload_shdrs (SFLoadContext *ld);
static int fixup_presets (SFLoadContext *ld);
static int fixup_insts (SFLoadContext *ld);

/**
 * Load a sound font file
 * @fhandle File handle to load from (file, socket, pipe, etc)
 * @load_flags Flags indicating what parts of the sound font to load
 *   see IPLOAD_* defines, #IPLOAD_ALL to load all parts
 * @smpl_cb Sample data callback or NULL to skip sample data
 * @userptr User definable pointer to pass to sample data callback
 * Returns: Loaded sound font data or NULL on error
 */
IPSFont *
instp_sfont_load (int fhandle, int load_flags, IPLoadSampleCallback *smpl_cb,
		  const void *userptr)
{
  IPSFont *sf;
  SFLoadContext *ld;
  IPFileInfo *file_info;

  ld = g_malloc0 (sizeof (SFLoadContext));

  ld->smpl_cb = smpl_cb;
  ld->cb_userptr = (void *)userptr;

  if (load_flags & IPLOAD_ALL)
    ld->flags = IPLOAD_PARTS_MASK;
  else ld->flags = load_flags & IPLOAD_PARTS_MASK;

  if (!(ld->ctx = ipfile_context_new ("r")))
    {
      free (ld);
      return (NULL);
    }

  ld->sf = instp_sfont_new ();
  if (!ld->sf)
    {
      ipfile_context_free (ld->ctx);
      free (ld);
      return (NULL);
    }

  /* create new file info structure */
  if (!(file_info = instp_file_info_new ()))
    {
      instp_item_destroy (INSTP_ITEM (ld->sf));
      ipfile_context_free (ld->ctx);
      free (ld);
      return (NULL);
    }

  file_info->fd = fhandle;
  instp_set_file_info (ld->sf, file_info); /* set sound font file info */

  ipfile_set_fhandle (ld->ctx, fhandle);
  ipfile_set_callback (ld->ctx, (IPFileCallback *)sfload_callback, ld);

  if (ipfile_start_transfer (ld->ctx) != INSTP_OK)
    {
      instp_item_destroy (INSTP_ITEM (ld->sf));
      ipfile_context_free (ld->ctx);
      free (ld);
      return (NULL);
    }

  if (fixup_presets (ld) != INSTP_OK
      || fixup_insts (ld) != INSTP_OK)
    {
      instp_item_destroy (INSTP_ITEM (ld->sf));
      ipfile_context_free (ld->ctx);
      free (ld);
      return (NULL);
    }

  sf = ld->sf;			/* store in var before freeing `ld' */

  ipfile_context_free (ld->ctx);
  free (ld);

  sf->flag_changed = FALSE;	/* instp in memory is in sync with file */
  sf->flag_saved = FALSE;	/* has not been saved yet */

  return (sf);
}

static int
sfload_callback (IPFileContext *ctx, void *userptr)
{
  SFLoadContext *ld = (SFLoadContext *)userptr;
  int retval = INSTP_OK;
  IPChunkType type;
  guint size;
  guint pos;

  if (ipfile_get_chunk_state (ctx, -1, &type, &size, &pos) != INSTP_OK)
    return (INSTP_FAIL);

  switch (type)
    {
    case IPCHUNK_INFO:		/* load sound font info? */
      if (ld->flags & IPLOAD_INFO)
	retval = sfload_infos (ld);
      break;
    case IPCHUNK_SMPL:		/* call users sample data callback? */
      /* remember offset into file of sample data (if any) */
      if (size > 0)
	{
	  ld->sf->file->sample_pos = ipfile_get_file_position (ctx);
	  ld->sf->file->sample_size = size / 2; /* save size (in samples) */
	}

      /* call user sample data callback (if any) */
      if (ld->smpl_cb)
	return ((*ld->smpl_cb)(ctx, size / 2, ld->cb_userptr));
      break;
    case IPCHUNK_PHDR:		/* load preset headers? */
      if (ld->flags & IPLOAD_PHDRS)
	retval = sfload_phdrs (ld);
      break;
    case IPCHUNK_PBAG:		/* load preset bags? */
      if (ld->flags & (IPLOAD_PMODS | IPLOAD_PGENS))
	retval = sfload_pbags (ld);
      break;
    case IPCHUNK_PMOD:		/* load preset modulators? */
      if (ld->flags & IPLOAD_PMODS)
	retval = sfload_pmods (ld);
      break;
    case IPCHUNK_PGEN:		/* load preset generators? */
      if (ld->flags & IPLOAD_PGENS)
	retval = sfload_pgens (ld);
      break;
    case IPCHUNK_IHDR:		/* load instrument headers? */
      if (ld->flags & IPLOAD_IHDRS)
	retval = sfload_ihdrs (ld);
      break;
    case IPCHUNK_IBAG:		/* load instrument bags? */
      if (ld->flags & (IPLOAD_IMODS | IPLOAD_IGENS))
	retval = sfload_ibags (ld);
      break;
    case IPCHUNK_IMOD:		/* load instrument modulators? */
      if (ld->flags & IPLOAD_IMODS)
	retval = sfload_imods (ld);
      break;
    case IPCHUNK_IGEN:		/* load instrument generators? */
      if (ld->flags & IPLOAD_IGENS)
	retval = sfload_igens (ld);
      break;
    case IPCHUNK_SHDR:		/* load sample headers? */
      if (ld->flags & IPLOAD_SHDRS)
	retval = sfload_shdrs (ld);
      break;
    default:
      break;
    }

  return (retval);
}

static int
sfload_infos (SFLoadContext *ld)
{
  char id[4];
  int size;
  int idnum;
  int retval;

  /* loop over info chunks */
  while ((retval = ipfile_load_info_chunk (ld->ctx, id, &size)) == INSTP_OK
	 && size >= 0)
    {
      idnum = ipfile_chunk_str_to_enum (id);

      if (idnum == IPINFO_VERSION) /* sound font version chunk? */
	{
	  if (ipfile_load_info_version (ld->ctx, &ld->sf->version) != INSTP_OK)
	    return (INSTP_FAIL);

	  if (ld->sf->version.major != 2)
	    {
	      g_critical (_("Sound font version is %d.%02d which"
			    " is not supported"),
			  ld->sf->version.major, ld->sf->version.minor);
	      return (INSTP_FAIL);
	    }

	  if (ld->sf->version.minor > 1)
	    g_warning (_("Sound font version is newer than 2.01,"
			 " some information might be uneditable"));
	}
      else if (idnum == IPINFO_ROM_VERSION) /* ROM version chunk? */
	{
	  if (ipfile_load_info_version (ld->ctx,
					&ld->sf->rom_version) != INSTP_OK)
	    return (INSTP_FAIL);
	}
      else
	{
	  int maxsize;
	  char *s;

	  if (idnum == IPCHUNK_UNKNOWN)
	    g_warning (_("Unknown INFO chunk \"%.4s\""), id);

	  /* make sure info chunk size is okay */
	  maxsize = instp_info_max_size (idnum);
	  if (maxsize && size > maxsize)
	    {
	      g_critical (_("Invalid INFO chunk size"));
	      return (INSTP_FAIL);
	    }

	  /* alloc for info string */
	  s = g_malloc (size);

	  if (ipfile_load_info_data (ld->ctx, s, size))
	    {
	      g_free (s);
	      return (INSTP_FAIL);
	    }

	  s[size - 1] = '\0';	/* force terminate info string */

	  if (instp_set_info_custom (ld->sf, id, s) != INSTP_OK)
	    {
	      g_free (s);
	      return (INSTP_FAIL);
	    }

	  g_free (s);
	}
    }

  return (retval);
}

/* preset header loader */
static int
sfload_phdrs (SFLoadContext *ld)
{
  int i, i2;
  IPPreset *p, *pr = NULL;	/* ptr to current & previous preset */
  guint16 zndx, pzndx;
  IPFilePHDR *phdr;

  /* loop over all preset headers (including dummy terminal record) */
  i = ipfile_get_items_left (ld->ctx);
  for (; i > 0; i--)
    {
      if (!(p = instp_preset_new ()))
	return (INSTP_FAIL);

      if (ipfile_load_phdr (ld->ctx, &phdr) != INSTP_OK)
	return (INSTP_FAIL);

      /* duplicate name string (NULL terminates even if all 20 chars used) */
      p->name = g_strndup (phdr->name, 20);
      p->psetnum = phdr->psetnum;
      p->bank = phdr->bank;
      zndx = phdr->pbagndx;
      p->library = phdr->library;
      p->genre = phdr->genre;
      p->morphology = phdr->morphology;

      if (i != 1)		/* don't add terminal record, free instead */
	instp_insert_preset (ld->sf, p, -1);
      else instp_item_destroy (INSTP_ITEM (p));

      if (pr)			/* not first preset? */
	{
	  if (zndx < pzndx)	/* make sure zone index isn't decreasing */
	    {
	      g_critical (_(INSTP_ERRMSG_CORRUPT_0));
	      return (INSTP_FAIL);
	    }

	  i2 = zndx - pzndx;	/* # of zones in last preset */
	  while (i2--)		/* create zones for last preset */
	    {
	      IPZone *zone;
	      if (!(zone = instp_zone_new ()))
		return (INSTP_FAIL);
	      instp_preset_insert_zone (pr, zone, 0);
	    }
	}
      else if (zndx > 0)	/* 1st preset, warn if ofs > 0 */
	g_warning (_("%d preset zones not referenced, discarding"), zndx);
      pr = p;			/* update preset ptr */
      pzndx = zndx;
    }

  return (INSTP_OK);
}

/* preset bag loader */
static int
sfload_pbags (SFLoadContext *ld)
{
  IPPreset *p;
  IPFileBag *bag;
  IPZone *z;
  guint16 genndx, modndx;
  guint16 pgenndx, pmodndx;
  int gc, mc;

  /* only index information, prefetch first indexes, there is a dummy bag */
  if (ipfile_load_bag (ld->ctx, &bag) != INSTP_OK)
    return (INSTP_FAIL);
  pgenndx = bag->genndx;
  pmodndx = bag->modndx;

  p = ld->sf->preset;
  while (p)			/* traverse through presets */
    {
      z = instp_preset_first_zone (p);
      while (z)			/* traverse preset's zones */
	{
	  if (ipfile_load_bag (ld->ctx, &bag) != INSTP_OK)
	    return (INSTP_FAIL);
	  genndx = bag->genndx;
	  modndx = bag->modndx;

	  if (genndx < pgenndx)
	    {
	      g_critical (_(INSTP_ERRMSG_CORRUPT_0));
	      return (INSTP_FAIL);
	    }
	  if (modndx < pmodndx)
	    {
	      g_critical (_(INSTP_ERRMSG_CORRUPT_0));
	      return (INSTP_FAIL);
	    }

	  gc = genndx - pgenndx;
	  mc = modndx - pmodndx;

	  INSTP_USERPTR (z) = GUINT_TO_POINTER (gc);

	  while (mc--)
	    {
	      IPMod *mod;
	      if (!(mod = instp_mod_new ()))
		return (INSTP_FAIL);
	      instp_zone_insert_mod (z, mod, -1);
	    }

	  pgenndx = genndx;	/* update previous zone gen index */
	  pmodndx = modndx;	/* update previous zone mod index */
	  z = instp_zone_next (z);
	}
      p = instp_preset_next (p);
    }

  return (INSTP_OK);
}

/* preset modulator loader */
static int
sfload_pmods (SFLoadContext *ld)
{
  IPPreset *p;
  IPZone *z;
  IPMod *m;
  IPFileMod *fmod;

  p = ld->sf->preset;
  while (p)			/* traverse through all presets */
    {
      z = instp_preset_first_zone (p);
      while (z)			/* traverse this preset's zones */
	{
	  m = instp_zone_first_mod (z);
	  while (m)		/* load zone's modulators */
	    {
	      if (ipfile_load_mod (ld->ctx, &fmod) != INSTP_OK)
		return (INSTP_FAIL);

	      if (!fmod)
		{
		  g_critical (_(INSTP_ERRMSG_CORRUPT_0));
		  return (INSTP_FAIL);
		}

	      m->src = fmod->src;
	      m->dest = fmod->dest;
	      m->amount = fmod->amount;
	      m->amtsrc = fmod->amtsrc;
	      m->trans = fmod->trans;

	      m = instp_mod_next (m);
	    }
	  z = instp_zone_next (z);
	}
      p = instp_preset_next (p);
    }

  return (INSTP_OK);
}

/* preset generator loader */
static int
sfload_pgens (SFLoadContext *ld)
{
  IPPreset *p;
  IPZone *z;
  IPFileGen *fgen;
  int level, discarded;
  int i;

  p = ld->sf->preset;
  while (p)			/* traverse through all presets */
    {
      discarded = FALSE;

      z = instp_preset_first_zone (p);
      while (z)			/* traverse preset's zones */
	{
	  level = 0;

	  /* retrieve our stored gen count (from load_pbag) */
	  i = GPOINTER_TO_UINT (INSTP_USERPTR (z));
	  INSTP_USERPTR (z) = NULL;

	  while (i-- > 0)	/* load zone's generators */
	    {
	      if (ipfile_load_gen (ld->ctx, &fgen) != INSTP_OK)
		return (INSTP_FAIL);

	      if (!fgen)
		{
		  g_critical (_(INSTP_ERRMSG_CORRUPT_0));
		  return (INSTP_FAIL);
		}

	      /* check validity of generator */
	      if (!instp_genid_is_valid (fgen->id, TRUE)
		  || (fgen->id == IPGEN_KEY_RANGE && level != 0)
		  || (fgen->id == IPGEN_VELOCITY_RANGE && level > 1))
		{
		  discarded = TRUE;
		  continue;
		}

	      /* IPGEN_KEY_RANGE first (if any) followed by
		 IPGEN_VELOCITY_RANGE (if any), IPGEN_INSTRUMENT_ID is last */
	      if (fgen->id == IPGEN_KEY_RANGE) level = 1;
	      else if (fgen->id == IPGEN_VELOCITY_RANGE) level = 2;
	      else if (fgen->id == IPGEN_INSTRUMENT_ID)
		{
		  /* using the userptr again, for the instrument index ID,
		     add 1 to it as NULL is used for global zones */
		  INSTP_USERPTR (z) =
		    GUINT_TO_POINTER (((int)(fgen->amount.uword)) + 1);
		  level = 3;
		  break;	/* break out of gen loop */
		}
	      else level = 2;

	      /* set the generator */
	      instp_zone_set_gen (z, fgen->id, fgen->amount);
	    }			/* generator loop */

	  /* ignore (skip) any generators following an instrument ID */
	  while (i-- > 0)
	    {
	      discarded = TRUE;
	      if (ipfile_load_gen (ld->ctx, &fgen) != INSTP_OK)
		return (INSTP_FAIL);

	      if (!fgen)
		{
		  g_critical (_(INSTP_ERRMSG_CORRUPT_0));
		  return (INSTP_FAIL);
		}
	    }

	  /* if level !=3 (no instrument ID) and not first zone, discard */
	  if (level != 3 && z != instp_preset_first_zone (p))
	    {		/* discard invalid global zone */
	      IPZone *temp = z;
	      z = instp_zone_next (z);
	      instp_item_destroy (INSTP_ITEM (temp));

	      g_warning (_("Preset \"%s\": Discarding invalid global zone"),
			 p->name);
	      continue;
	    }

	  z = instp_zone_next (z); /* next zone */
	}

      if (discarded)
	g_warning (_("Preset \"%s\": Some invalid generators were discarded"),
		   p->name);

      p = instp_preset_next (p); /* next preset */
    }

  return (INSTP_OK);
}

/* instrument header loader */
static int
sfload_ihdrs (SFLoadContext *ld)
{
  int i, i2;
  IPInst *inst, *pinst = NULL;	/* ptr to current & previous instrument */
  guint16 zndx, pzndx;
  IPFileIHDR *ihdr;

  /* loop over all instrument headers (including dummy terminal record) */
  i = ipfile_get_items_left (ld->ctx);
  for (; i > 0; i--)
    {
      if (!(inst = instp_inst_new ()))
	return (INSTP_FAIL);

      if (ipfile_load_ihdr (ld->ctx, &ihdr) != INSTP_OK)
	return (INSTP_FAIL);

      inst->name = g_strndup (ihdr->name, 20);

      zndx = ihdr->ibagndx;

      if (i != 1)		/* don't add terminal record, free instead */
	instp_add_inst (ld->sf, inst);
      else instp_item_destroy (INSTP_ITEM (inst));

      if (pinst)		/* not first instrument? */
	{
	  if (zndx < pzndx)	/* make sure zone index isn't decreasing */
	    {
	      g_critical (_(INSTP_ERRMSG_CORRUPT_0));
	      return (INSTP_FAIL);
	    }

	  i2 = zndx - pzndx;	/* # of zones in last instrument */
	  while (i2--)		/* create zones for last instrument */
	    {
	      IPZone *zone;
	      if (!(zone = instp_zone_new ()))
		return (INSTP_FAIL);
	      instp_inst_insert_zone (pinst, zone, 0);
	    }
	}
      else if (zndx > 0)	/* 1st instrument, warn if ofs > 0 */
	g_warning (_("Discarding %d unreferenced instrument zones"), zndx);
      pinst = inst;		/* update instrument ptr */
      pzndx = zndx;
    }

  return (INSTP_OK);
}

/* instrument bag loader */
static int
sfload_ibags (SFLoadContext *ld)
{
  IPInst *inst;
  IPFileBag *bag;
  IPZone *z;
  guint16 genndx, modndx;
  guint16 pgenndx, pmodndx;
  int gc, mc;

  /* only index information, prefetch first indexes, there is a dummy bag */
  if (ipfile_load_bag (ld->ctx, &bag) != INSTP_OK)
    return (INSTP_FAIL);
  pgenndx = bag->genndx;
  pmodndx = bag->modndx;

  inst = ld->sf->inst;
  while (inst)			/* traverse through instruments */
    {
      z = instp_inst_first_zone (inst);
      while (z)			/* traverse instrument's zones */
	{
	  if (ipfile_load_bag (ld->ctx, &bag) != INSTP_OK)
	    return (INSTP_FAIL);
	  genndx = bag->genndx;
	  modndx = bag->modndx;

	  if (genndx < pgenndx)
	    {
	      g_critical (_(INSTP_ERRMSG_CORRUPT_0));
	      return (INSTP_FAIL);
	    }
	  if (modndx < pmodndx)
	    {
	      g_critical (_(INSTP_ERRMSG_CORRUPT_0));
	      return (INSTP_FAIL);
	    }

	  gc = genndx - pgenndx;
	  mc = modndx - pmodndx;

	  INSTP_USERPTR (z) = GUINT_TO_POINTER (gc);

	  while (mc--)
	    {
	      IPMod *mod;
	      if (!(mod = instp_mod_new ()))
		return (INSTP_FAIL);
	      instp_zone_insert_mod (z, mod, -1);
	    }

	  pgenndx = genndx;	/* update previous zone gen index */
	  pmodndx = modndx;	/* update previous zone mod index */
	  z = instp_zone_next (z);
	}
      inst = instp_inst_next (inst);
    }

  return (INSTP_OK);
}

/* instrument modulator loader */
static int
sfload_imods (SFLoadContext *ld)
{
  IPInst *inst;
  IPZone *z;
  IPMod *m;
  IPFileMod *fmod;

  inst = ld->sf->inst;
  while (inst)			/* traverse through all instruments */
    {
      z = instp_inst_first_zone (inst);
      while (z)			/* traverse this instrument's zones */
	{
	  m = instp_zone_first_mod (z);
	  while (m)		/* load zone's modulators */
	    {
	      if (ipfile_load_mod (ld->ctx, &fmod) != INSTP_OK)
		return (INSTP_FAIL);

	      if (!fmod)
		{
		  g_critical (_(INSTP_ERRMSG_CORRUPT_0));
		  return (INSTP_FAIL);
		}

	      m->src = fmod->src;
	      m->dest = fmod->dest;
	      m->amount = fmod->amount;
	      m->amtsrc = fmod->amtsrc;
	      m->trans = fmod->trans;

	      m = instp_mod_next (m);
	    }
	  z = instp_zone_next (z);
	}
      inst = instp_inst_next (inst);
    }

  return (INSTP_OK);
}

/* instrument generator loader */
static int
sfload_igens (SFLoadContext *ld)
{
  IPInst *inst;
  IPZone *z;
  IPFileGen *fgen;
  int level, discarded;
  int i;

  inst = ld->sf->inst;
  while (inst)			/* traverse through all instruments */
    {
      discarded = FALSE;

      z = instp_inst_first_zone (inst);
      while (z)			/* traverse instruments's zones */
	{
	  level = 0;

	  /* retrieve our stored gen count (from load_pbag) */
	  i = GPOINTER_TO_UINT (INSTP_USERPTR (z));
	  INSTP_USERPTR (z) = NULL;

	  while (i--)		/* load zone's generators */
	    {
	      if (ipfile_load_gen (ld->ctx, &fgen) != INSTP_OK)
		return (INSTP_FAIL);

	      if (!fgen)
		{
		  g_critical (_(INSTP_ERRMSG_CORRUPT_0));
		  return (INSTP_FAIL);
		}

	      /* check validity of generator */
	      if (!instp_genid_is_valid (fgen->id, FALSE)
		  || (fgen->id == IPGEN_KEY_RANGE && level != 0)
		  || (fgen->id == IPGEN_VELOCITY_RANGE && level > 1))
		{
		  discarded = TRUE;
		  continue;
		}

	      /* IPGEN_KEY_RANGE first (if any) followed by
		 IPGEN_VELOCITY_RANGE (if any), IPGEN_SAMPLE_ID is last */
	      if (fgen->id == IPGEN_KEY_RANGE) level = 1;
	      else if (fgen->id == IPGEN_VELOCITY_RANGE) level = 2;
	      else if (fgen->id == IPGEN_SAMPLE_ID)
		{
		  /* using the userptr again, for the sample index ID,
		     add 1 to it as NULL is used for global zones */
		  INSTP_USERPTR (z) =
		    GUINT_TO_POINTER (((int)(fgen->amount.uword)) + 1);
		  level = 3;
		  break;	/* break out of gen loop */
		}
	      else level = 2;

	      /* set the generator */
	      instp_zone_set_gen (z, fgen->id, fgen->amount);
	    }			/* generator loop */

	  /* ignore (skip) any generators following a sample ID */
	  while (i-- > 0)
	    {
	      discarded = TRUE;
	      if (ipfile_load_gen (ld->ctx, &fgen) != INSTP_OK)
		return (INSTP_FAIL);

	      if (!fgen)
		{
		  g_critical (_(INSTP_ERRMSG_CORRUPT_0));
		  return (INSTP_FAIL);
		}
	    }

	  /* if level !=3 (no sample ID) and not first zone, discard */
	  if (level != 3 && z != instp_inst_first_zone (inst))
	    {		/* discard invalid global zone */
	      IPZone *temp = z;
	      z = instp_zone_next (z);
	      instp_item_destroy (INSTP_ITEM (temp));

	      g_warning (_("Instrument \"%s\": Discarding invalid"
			   " global zone"), inst->name);
	      continue;
	    }

	  z = instp_zone_next (z); /* next zone */
	}

      if (discarded)
	g_warning (_("Instrument \"%s\": Some invalid generators"
		     " were discarded"), inst->name);

      inst = instp_inst_next (inst); /* next instrument */
    }

  return (INSTP_OK);
}

/* sample header loader */
static int
sfload_shdrs (SFLoadContext *ld)
{
  IPSample *sample;
  IPSampleData *sampledata;
  IPSampleStore *store;
  IPFileSHDR *shdr;
  char name[21];
  int i;

  /* load all sample headers (not including terminal record) */
  i = ipfile_get_items_left (ld->ctx) - 1;
  for (; i > 0; i--)
    {
      if (ipfile_load_shdr (ld->ctx, &shdr) != INSTP_OK)
	return (INSTP_FAIL);

      if (!(sample = instp_sample_new ()))
	return (INSTP_FAIL);

      /* set the sample name, carefully */
      strncpy (name, shdr->name, 20);
      name[20] = '\0';		/* for names that use all 20 chars */
      if (instp_sample_set_name (sample, name) != INSTP_OK)
	{
	  instp_item_destroy (INSTP_ITEM (sample));
	  return (INSTP_FAIL);
	}

      /* sample NOT totally screwed? */
      if (((shdr->sampletype & IPSAMPLE_TYPE_ROM)
	   || shdr->end <= ld->sf->file->sample_size)
	  && (shdr->start - shdr->end > 4))
	{
	  if (shdr->loopend <= shdr->end
	      && shdr->loopstart < shdr->loopend
	      && shdr->start <= shdr->loopstart)
	    {			/* loop is OK */
	      sample->loopstart = shdr->loopstart - shdr->start;
	      sample->loopend = shdr->loopend - shdr->start;
	    }
	  else			/* loop is on krak */
	    {
	      int size = shdr->end - shdr->start;
	      if (size >= 48)
		{
		  sample->loopstart = 8;
		  sample->loopend = size - 8;
		}
	      else		/* sample is rather small */
		{
		  sample->loopstart = 1;
		  sample->loopend = size - 1;
		}
	    }

	  sample->samplerate = shdr->samplerate;
	  sample->origpitch = shdr->origpitch;
	  sample->pitchadj = shdr->pitchadj;
	  //    FIXME:  sample->link = shdr->link;
	  sample->sampletype = shdr->sampletype;

	  /* create new sample data object */
	  if (!(sampledata = instp_sample_data_new ()))
	    {
	      instp_item_destroy (INSTP_ITEM (sample));
	      return (INSTP_FAIL);
	    }

	  instp_sample_data_set_size (sampledata, shdr->end - shdr->start);

	  /* set the sample data object of the new sample */
	  instp_sample_set_sample_data (sample, sampledata);

	  if (!(sample->sampletype & IPSAMPLE_TYPE_ROM))
	    {
	      /* create a new sample store in the sample data object */
	      if (!(store = instp_sample_store_new (sampledata,
						    IPSAMPLE_METHOD_SFONT)))
		{
		  instp_item_destroy (INSTP_ITEM (sample));
		  return (INSTP_FAIL);
		}
	      instp_sample_method_SFONT_set_params (sampledata, store,
						    ld->sf->file, shdr->start);
	    }
	  else			/* ROM sample */
	    {
	      if (!(store = instp_sample_store_new (sampledata,
						    IPSAMPLE_METHOD_ROM)))
		{
		  instp_item_destroy (INSTP_ITEM (sample));
		  return (INSTP_FAIL);
		}
	      instp_sample_method_ROM_set_start (sampledata, store,
						 shdr->start);
	    }
	}
      else			/* sample is messed, set to blank data */
	{
	  g_warning (_("Invalid sample '%s'"), sample->name);

	  if (instp_sample_set_blank_data (sample) != INSTP_OK)
	    {
	      instp_item_destroy (INSTP_ITEM (sample));
	      return (INSTP_FAIL);
	    }
	}

      instp_add_sample (ld->sf, sample);
    }

  return (INSTP_OK);
}

/* "fixup" (inst ID -> inst ptr) instrument references in preset list */
static int
fixup_presets (SFLoadContext *ld)
{
  IPPreset *p;
  IPInst *inst;
  IPZone *z;
  IPGenAmount amt;
  int i;

  p = instp_first_preset (ld->sf);

  if (!(ld->flags & IPLOAD_IHDRS)) /* if instrument list wasn't loaded */
    {
      while (p)			/* loop over presets */
	{
	  z = instp_preset_first_zone (p);
	  while (z)		/* loop over preset zones */
	    {
	      /* retrieve instrument ID and set it in generator list */
	      if ((i = GPOINTER_TO_UINT (INSTP_USERPTR (z))))
		{
		  amt.uword = i - 1; /* instrument ID + 1, so decrement */
		  instp_zone_set_gen (z, IPGEN_INSTRUMENT_ID, amt);
		  INSTP_USERPTR (z) = NULL;
		}
	      z = instp_zone_next (z);
	    }
	  p = instp_preset_next (p);
	}

      return (INSTP_OK);
    }

  while (p)			/* loop through presets */
    {
      z = instp_preset_first_zone (p);
      while (z)			/* traverse this preset's zones */
	{
	  /* load instrument ID (actually inst ID + 1) */
	  if ((i = GPOINTER_TO_UINT (INSTP_USERPTR (z))))
	    {
	      INSTP_USERPTR (z) = NULL; /* reset user pointer */

	      inst = instp_first_inst (ld->sf);
	      while (inst && --i) /* find instrument by index (inst ID + 1) */
		inst = instp_inst_next (inst);

	      if (!inst)
		{
		  g_critical (_("Preset %03d-%03d '%s': Invalid"
				" instrument reference"),
			      p->bank, p->psetnum, p->name);
		  return (INSTP_FAIL);
		}
	      instp_zone_set_refitem (z, INSTP_ITEM (inst));
	    }
	  z = instp_zone_next (z);
	}
      p = instp_preset_next (p);
    }

  return (INSTP_OK);
}

/* "fixup" (sample ID -> sample ptr) sample references in instrument list */
static int
fixup_insts (SFLoadContext *ld)
{
  IPInst *p;
  IPSample *sam;
  IPZone *z;
  IPGenAmount amt;
  int i;

  p = instp_first_inst (ld->sf);

  if (!(ld->flags & IPLOAD_SHDRS)) /* if sample list wasn't loaded */
    {
      while (p)			/* loop over instruments */
	{
	  z = instp_inst_first_zone (p);
	  while (z)		/* loop over instrument zones */
	    {
	      /* retrieve sample ID and set it in generator list */
	      if ((i = GPOINTER_TO_UINT (INSTP_USERPTR (z))))
		{
		  amt.uword = i - 1; /* sample ID + 1, so decrement */
		  instp_zone_set_gen (z, IPGEN_SAMPLE_ID, amt);
		  INSTP_USERPTR (z) = NULL;
		}
	      z = instp_zone_next (z);
	    }
	  p = instp_inst_next (p);
	}

      return (INSTP_OK);
    }

  while (p)			/* loop through instruments */
    {
      z = instp_inst_first_zone (p);
      while (z)			/* traverse this instrument's zones */
	{
	  /* load sample ID (actually sample ID + 1) */
	  if ((i = GPOINTER_TO_UINT (INSTP_USERPTR (z))))
	    {
	      INSTP_USERPTR (z) = NULL; /* reset user pointer */

	      sam = instp_first_sample (ld->sf);
	      while (sam && --i) /* find sample by index (sample ID + 1) */
		sam = instp_sample_next (sam);

	      if (!sam)
		{
		  g_critical (_("Instrument '%s': Invalid sample reference"),
			      p->name);
		  return (INSTP_FAIL);
		}
	      instp_zone_set_refitem (z, INSTP_ITEM (sam));
	    }
	  z = instp_zone_next (z);
	}
      p = instp_inst_next (p);
    }

  return (INSTP_OK);
}
