/*
   SwingWT
   Copyright(c)2003-2005, R. Rawson-Tetley

   For more information on distributing and using this program, please
   see the accompanying "COPYING" file.

   Contact me by electronic mail: bobintetley@users.sourceforge.net

   $Log: JClosableTabbedPane.java,v $
   Revision 1.17  2005/01/05 09:22:29  bobintetley
   Updated copyright year on source

   Revision 1.16  2004/06/11 09:17:30  bobintetley
   Various fixes and things

   Revision 1.15  2004/06/08 09:24:22  dannaab
   Rename Component.getPeer() -> getSWTPeer().  added ComponentPeer and stubbed out support classes.

   Revision 1.14  2004/05/06 12:35:21  bobintetley
   Parity with Swing constants for Binary Compatibility + fixes to JDesktopPane

   Revision 1.13  2004/04/28 08:38:06  bobintetley
   Hierarchy fixes, code cleanup for base classes, additional javadocs and use of flag to identify JComponent descendants with peers

   Revision 1.12  2004/04/18 15:25:35  bobintetley
   Updates to take advantage of new features

   Revision 1.11  2004/04/18 14:59:42  bobintetley
   Virtual table support (much much faster) and removal of deprecated API
   for JClosableTabbedPane

   Revision 1.10  2004/04/18 14:41:09  bobintetley
   JClosableTabbedPane uses new CTabFolder in M8 (no longer need local copy)

   Revision 1.9  2004/03/30 10:42:44  bobintetley
   Many minor bug fixes, event improvements by Dan Naab. Full swing.Icon support

   Revision 1.8  2004/03/18 14:42:10  bobintetley
   Fix to Window hierarchy to match Swing, and fix to allow MDI apps
      to work under SWT 2.x

   Revision 1.7  2004/03/03 09:13:12  bobintetley
   JList threading fixed and top level error handling

   Revision 1.6  2004/02/27 16:52:58  bobintetley
   Scrollpane activation fixes, modern tabbedpane usage for JClosableTabbedPane

   Revision 1.5  2004/02/27 16:16:12  bobintetley
   Threading fixes

   Revision 1.4  2004/01/26 15:45:47  bobintetley
   SwingSet fixes

   Revision 1.3  2004/01/26 10:03:36  bobintetley
   Fixed component painting, added overloaded TabbedPane add() methods

   Revision 1.2  2004/01/26 09:50:11  bobintetley
   Fix to tabbed pane cache

   Revision 1.1  2004/01/20 07:38:05  bobintetley
   Bug fixes and compatibility methods

   Revision 1.7  2004/01/16 15:53:32  bobintetley
   Many compatibility methods added to Container, Component, JInternalFrame,
      UIManager, SwingUtilities, JTabbedPane, JPasswordField, JCheckBox
      and JRadioButton.

   Revision 1.6  2003/12/22 14:17:30  bobintetley
   One-off timer method and better layout management for tabs

   Revision 1.5  2003/12/18 09:37:33  bobintetley
   Enabled support for JTabbedPane and JClosableTabbedPane (SWT doesn't
   support, so a bit of magic in here)

   Revision 1.4  2003/12/16 09:19:02  bobintetley
   Various small fixes to match Swing more closely

   Revision 1.3  2003/12/15 18:29:57  bobintetley
   Changed setParent() method to setSwingWTParent() to avoid conflicts with applications

   Revision 1.2  2003/12/14 09:13:38  bobintetley
   Added CVS log to source headers

*/


package swingwtx.custom;

import org.eclipse.swt.custom.*;
import org.eclipse.swt.widgets.*;
import org.eclipse.swt.*;

import swingwtx.swing.*;
import swingwtx.swing.event.*;
import swingwtx.custom.event.*;

import java.util.*;

/**
 * Not really a swing component, but I map it here because I use it
 * for JInternalFrame/JDesktopPane. It's basically JTabbedPane, but with
 * closable buttons - something that Swing really ought to offer!
 */
public class JClosableTabbedPane extends JComponent implements SwingConstants {
    
    protected CTabFolder ppeer = null;
    
    /** Tab item cache until added to parent */
    protected Vector tabs = new Vector();
    protected Vector tabCloseListeners = new Vector();
    protected int pTabPlacement = TOP;
    protected Vector changeListeners = new Vector();
    protected int pSelectedIndex = -1;
    
    /** Thread safe accessor */
    private int iRetVal = 0;
    private String sRetVal = "";
    private Object retval;
    
    public JClosableTabbedPane() {    }
    
    public void addChangeListener(ChangeListener l) {
        changeListeners.add(l);
    }
    
    public void removeChangeListener(ChangeListener l) {
        changeListeners.remove(l);
    }
    
    public void addTab(String title, swingwt.awt.Component component) {
        addTab(title, null, component);
    }
    
    public void addTab(String title, Icon icon, swingwt.awt.Component component) {
        addTab(title, icon, component, null, true);
    }
    
    public void addTab(final String title, final Icon icon, final swingwt.awt.Component component, final String tip) {
        addTab(title, icon, component, tip, true);
    }
    
    public void addTab(final String title, final Icon icon, final swingwt.awt.Component component, final String tip, final boolean enabled) {
        addTab(title, icon, component, tip, enabled, false);
    }
    
    public void addTab(final String title, final Icon icon, final swingwt.awt.Component component, final String tip, final boolean enabled, boolean fromcache) {
        
        if (!fromcache) {
            // Create the cache entry
            TabItemCache t = new TabItemCache();
            t.title = title;
            t.icon = icon;
            t.component = component;
            t.tip = tip;
            t.enabled = enabled;
            tabs.add(t);
            t = null;
        }
        
        final JClosableTabbedPane me = this;
        SwingUtilities.invokeSync(new Runnable() {
            public void run() {
                if (!SwingWTUtils.isSWTControlAvailable(ppeer)) return;

                CTabItem t = new CTabItem(ppeer, SWT.NONE);

                // Make sure the component contained by this
                // TabItem gets created now
                try {
                    component.setSwingWTParent(me);
                }
                catch (Exception e) {
                    e.printStackTrace();
                }

                if (title != null) t.setText(title);
                if (icon != null) t.setImage(SwingWTUtils.getSWTImageFromSwingIcon(me, icon));
                
                // Only display the component if it's enabled
                if (component != null && enabled) t.setControl(component.getSWTPeer());
                
                if (tip != null) t.setToolTipText(tip);
                t.setData("enabled", (enabled ? "true" : "false"));

                // Display the thing
                postponeLayout();
            }
        });
        
    }
    
    public Icon getIconAt(final int index) {
        if (ppeer == null) {
            TabItemCache t = (TabItemCache) tabs.get(index);
            return t.icon;
        }
        else {
            SwingUtilities.invokeSync(new Runnable() {
                public void run() {
                    swingwt.awt.Image image = new swingwt.awt.Image();
                    image.image = ppeer.getItem(index).getImage();
                    retval = new ImageIcon(image);
                }
            });
            return (Icon) retval;
        }
    }
    public void setIconAt(final int index, final Icon icon) { 
        TabItemCache t = (TabItemCache) tabs.get(index);
        t.icon = icon;
        final JClosableTabbedPane pthis = this;
        if (SwingWTUtils.isSWTControlAvailable(ppeer))
            SwingUtilities.invokeSync(new Runnable() {
                public void run() {
                    ppeer.getItem(index).setImage(SwingWTUtils.getSWTImageFromSwingIcon(pthis, icon)); 
                }
            });
    }
    
    public void addTabCloseListener(TabCloseListener l) {
        tabCloseListeners.add(l);
    }
    
    public void removeTabCloseListener(TabCloseListener l) {
        tabCloseListeners.remove(l);
    }

    public void setEnabledAt(final int index, final boolean b) { 
        if (SwingWTUtils.isSWTControlAvailable(ppeer)) {
            SwingUtilities.invokeSync(new Runnable() {
                public void run() {
                    ppeer.getItem(index).setData("enabled", (b ? "true" : "false"));
                    // If we're disabled - hide the tab's control. This is about all we
                    // can do with SWT.
                    if (!b)
                        ppeer.getItem(index).setControl(null);
                    else
                        ppeer.getItem(index).setControl( ((TabItemCache) tabs.get(index)).component.getSWTPeer() );
                }
            });
        }
        else {
            TabItemCache t = (TabItemCache) tabs.get(index);
            t.enabled = b;
        }
    }
    
    public int getTabCount() { 
        if (!SwingWTUtils.isSWTControlAvailable(ppeer)) 
            return tabs.size(); 
        else {
            SwingUtilities.invokeSync(new Runnable() {
                public void run() {
                    iRetVal = ppeer.getItemCount(); 
                }
            });
            return iRetVal;
        }
    }

    public int getSelectedIndex() {
        if (SwingWTUtils.isSWTControlAvailable(ppeer)) {
            SwingUtilities.invokeSync(new Runnable() {
                public void run() {
                    iRetVal = ppeer.getSelectionIndex(); 
                }
            });
        }
        else
            iRetVal = -1;
        return iRetVal;
    }
    
    public swingwt.awt.Component getSelectedComponent() { 
        int i = getSelectedIndex();
        if (i != -1)
            return ((TabItemCache) tabs.get(i)).component;
        else
            return null;
    }
    
    public void setSelectedIndex(final int index) { 
        pSelectedIndex = index;
        if (SwingWTUtils.isSWTControlAvailable(ppeer))
            SwingUtilities.invokeSync(new Runnable() {
                public void run() {
                    ppeer.setSelection(index); 
                }
            });
    }
    
    public void removeTabAt(final int index) { 
        if (SwingWTUtils.isSWTControlAvailable(ppeer)) {
            SwingUtilities.invokeSync(new Runnable() {
                public void run() {
                    tabs.removeElementAt(index);
                    ppeer.getItem(index).dispose(); 
                }
            });
        }
        else
            tabs.removeElementAt(index);  
    }
    
    public String getTitleAt(final int index) { 
        if (SwingWTUtils.isSWTControlAvailable(ppeer)) {
            SwingUtilities.invokeSync(new Runnable() {
                public void run() {
                    sRetVal = ppeer.getItem(index).getText(); 
                }
            });
            return sRetVal;
        }
        else {
            TabItemCache t = (TabItemCache) tabs.get(index);
            return t.title;
        }
    }
    public void setTitleAt(final int index, final String title) { 
        if (SwingWTUtils.isSWTControlAvailable(ppeer)) {
            SwingUtilities.invokeSync(new Runnable() {
                public void run() {
                    ppeer.getItem(index).setText(title); 
                }
            });
        }
        else {
            TabItemCache t = (TabItemCache) tabs.get(index);
            t.title = title;
        }
    }
    public int getTabPlacement() { return pTabPlacement; }
    public void setTabPlacement(int place) { pTabPlacement = place; }
    
    public Control getSWTPeer() { return ppeer; }
    
    protected void processTabClosing(CTabFolderEvent e) {
        // See which one matches!
        CTabItem[] items = ppeer.getItems();
        for (int i = 0; i < items.length; i++) {
            if (e.item.equals(items[i])) {
                Iterator iter = tabCloseListeners.iterator();
                while (iter.hasNext()) {
                    TabCloseListener l = (TabCloseListener) iter.next();
                    e.doit = l.tabClosed(i);
                }
            }
        }
    }
    
    /**
     * Process change events - fired when the tab changes
     */
    protected void processChangeEvent(ChangeEvent e) {
        Iterator i = changeListeners.iterator();
        while (i.hasNext()) {
            ChangeListener l = (ChangeListener) i.next();
            l.stateChanged(e);
        }
    }
    
    protected void postponeLayout() {
        SwingUtilities.invokeLater(new Runnable() {
            public void run() {
                layoutTab();    
            }
        });
    }
    
    protected void layoutTab() {
        
        int i = ppeer.getSelectionIndex();
        if (i < 0 || i >= ppeer.getItems().length) return;
        
        Control p = ppeer.getItem(ppeer.getSelectionIndex()).getControl();
        if (p instanceof Composite)
            ((Composite) p).layout(true);
        else
            p.redraw();
    }
    
    
    public void setSwingWTParent(swingwt.awt.Container parent) throws Exception { 
        descendantHasPeer = true;
        ppeer = new CTabFolder(parent.getComposite(), SwingWTUtils.translateSwingAlignmentConstant(pTabPlacement));
        peer = ppeer;
        composite = ppeer;
        this.parent = parent;
        
        // Add any cached items
        if (tabs.size() > 0) {
            Object[] tabso = tabs.toArray();
            for (int i = 0; i < tabso.length; i++) {
                TabItemCache t = (TabItemCache) tabso[i];
                addTab(t.title, t.icon, t.component, t.tip, t.enabled, true);
            }
        }
        
        // Make sure the selected index is right
        if (pSelectedIndex != -1)
            ppeer.setSelection(pSelectedIndex);
        
        ppeer.setUnselectedCloseVisible(true);
        ppeer.setUnselectedImageVisible(true);
        
        // Events
        
        // -- Tab closing
        
        /* -- Correct code, but it doesn't work because the close
         *    button doesn't appear. VERY frustrating as we get
         *    deprecated warnings on compile.
         *
        ppeer.addCTabFolder2Listener(new CTabFolder2Listener() {
            public void showList(CTabFolderEvent event) {}
            public void restore(CTabFolderEvent event) {}
            public void maximize(CTabFolderEvent event) {}
            public void minimize(CTabFolderEvent event) {}
            public void close(CTabFolderEvent event) { processTabClosing(event); }
        });*/
        
        ppeer.addCTabFolderListener(new CTabFolderAdapter() {
		public void itemClosed(CTabFolderEvent event) {
			processTabClosing(event);
		}
	});

        // -- Tab is selected, may want to block it if we have it
        // flagged as disabled.
        final JClosableTabbedPane me = this;
        ppeer.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
            public void widgetSelected(org.eclipse.swt.events.SelectionEvent e) {
                // We don't send events if the tab is disabled
                if (checkTabEnabled(e)) {    
                    postponeLayout(); // Update layout of tab if we went to a new one
                    // Send change event
                    ChangeEvent ce = new ChangeEvent(me);
                    processChangeEvent(ce);
                }   
            }
            public void widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent e) {}
        });
        
        // Make sure whatever is selected gets displayed
        if (ppeer.getItemCount() > 0)
            postponeLayout();
        
    }
    
    /** Returns true if the tab is enabled according to the data set on it.
     *  This should be called from a selection event, determining whether
     *  the user is allowed to go to it or not.
     */
    protected boolean checkTabEnabled(org.eclipse.swt.events.SelectionEvent e) {
        e.doit = (e.item.getData("enabled").toString().equals("true"));
        return e.doit;
    }
    
}

