/*
   SwingWT
   Copyright(c)2003-2005, R. Rawson-Tetley
 
   For more information on distributing and using this program, please
   see the accompanying "COPYING" file.
 
   Contact me by electronic mail: bobintetley@users.sourceforge.net
 
   $Log: BoxLayout.java,v $
   Revision 1.7  2005/01/05 09:22:30  bobintetley
   Updated copyright year on source

   Revision 1.6  2004/05/05 12:43:21  bobintetley
   Patches/new files from Laurent Martell

   Revision 1.5  2004/04/20 16:36:14  bobintetley
   Code cleanup

   Revision 1.4  2004/03/30 14:22:27  bobintetley
   Fix to Component min/max sizing code, fix to JPanel insets with titled
   borders (all of which indirectly fix problems with BoxLayout). Addition
   of ComponentAdapter

   Revision 1.3  2004/01/26 08:11:00  bobintetley
   Many bugfixes and addition of SwingSet

   Revision 1.2  2004/01/06 08:28:02  bobintetley
   Header fixes

 
 */
package swingwtx.swing;

import swingwt.awt.*;

public class BoxLayout implements LayoutManager2 {
    
    public static final int X_AXIS = 0;
    public static final int Y_AXIS = 1;
    public static final int LINE_AXIS = 2;
    public static final int PAGE_AXIS = 3;
    
    protected SizeRequirements[] childWidth;
    protected SizeRequirements[] childHeight;
    protected SizeRequirements totalWidth;
    protected SizeRequirements totalHeight;
    
    private int axis;
    private Component target = null;
    
    public BoxLayout(Component target, int axis) {
        if (axis != X_AXIS && axis != Y_AXIS && axis != LINE_AXIS && axis != PAGE_AXIS) {
            throw new Error("Invalid axis");
        }
        this.axis = axis;
        this.target = target;
    }
    
    public BoxLayout(Container target, int axis) {
        this((Component) target, axis);    
    }
    
    public synchronized void invalidateLayout(Container target) {
        childWidth = null;
        childHeight = null;
        totalWidth = null;
        totalHeight = null;
    }
    
    public void addLayoutComponent(String name, Component comp) {
    }
    
    public void removeLayoutComponent(Component comp) {
    }
    
    public void addLayoutComponent(Component comp, Object constraints) {
    }
    
    public Dimension preferredLayoutSize(Container target) {

        calculate();
        Dimension size = null;
        if (totalWidth == null)
            size = new Dimension(0, 0);
        else
            size = new Dimension(totalWidth.preferred, totalHeight.preferred);
        
        Insets insets = target.getInsets();
        size.width = (int) Math.min( (long) size.width + (long) insets.left + (long) insets.right, Integer.MAX_VALUE);
        size.height = (int) Math.min( (long) size.height + (long) insets.top + (long) insets.bottom, Integer.MAX_VALUE);
        return size;
    }
    
    public Dimension minimumLayoutSize(Container target) {

        calculate();
        Dimension size = new Dimension(totalWidth.minimum, totalHeight.minimum);
        
        Insets insets = target.getInsets();
        size.width =
        (int) Math.min( (long) size.width + (long) insets.left + (long) insets.right, Integer.MAX_VALUE); 
        size.height = (int) Math.min( (long) size.height + (long) insets.top + (long) insets.bottom, Integer.MAX_VALUE);
        return size;
    }
    
    public Dimension maximumLayoutSize(Container target) {
        
        calculate();
        Dimension size = new Dimension(totalWidth.maximum, totalHeight.maximum);
        
        Insets insets = target.getInsets();
        size.width =
        (int) Math.min( (long) size.width + (long) insets.left + (long) insets.right, Integer.MAX_VALUE);
        size.height = (int) Math.min( (long) size.height + (long) insets.top + (long) insets.bottom, Integer.MAX_VALUE);
        return size;
    }
    
    public synchronized float getLayoutAlignmentX(Container target) {
        calculate();
        return totalWidth.alignment;
    }
    
    public synchronized float getLayoutAlignmentY(Container target) {
        calculate();
        return totalHeight.alignment;
    }
    
    public void layoutContainer(Container target) {
        
        int nChildren = target.getComponentCount();
        
        int[] xOffsets = new int[nChildren];
        int[] xSpans = new int[nChildren];
        int[] yOffsets = new int[nChildren];
        int[] ySpans = new int[nChildren];
        
        Dimension alloc = target.getSize();
        Insets in = target.getInsets();
        alloc.width -= in.left + in.right;
        alloc.height -= in.top + in.bottom;
        
        
        int absoluteAxis = X_AXIS;

        if (axis == LINE_AXIS)
            absoluteAxis = target.getComponentOrientation().isHorizontal() ? X_AXIS : Y_AXIS;
        else if (axis == PAGE_AXIS)
            absoluteAxis = target.getComponentOrientation().isHorizontal() ? Y_AXIS : X_AXIS;
        else
            absoluteAxis = axis;
        
        boolean ltr = (absoluteAxis != axis) ? target.getComponentOrientation().isLeftToRight() : true;
        
        // Determine where children go
        calculate();

        nChildren = childWidth.length;
        xOffsets = new int[nChildren];
        xSpans = new int[nChildren];
        yOffsets = new int[nChildren];
        ySpans = new int[nChildren];

        if (absoluteAxis == X_AXIS) {
            SizeRequirements.calculateAlignedPositions(alloc.height, totalHeight, childHeight, yOffsets, ySpans);
            SizeRequirements.calculateTiledPositions(alloc.width, totalWidth, childWidth, xOffsets, xSpans, ltr);
        } 
        else {
            SizeRequirements.calculateTiledPositions(alloc.height, totalHeight, childHeight, yOffsets, ySpans);
            SizeRequirements.calculateAlignedPositions(alloc.width, totalWidth, childWidth, xOffsets, xSpans, ltr);
        }
        
        for (int i = 0; i < nChildren; i++) {
            Component c = target.getComponent(i);
            c.setBounds(
            (int) Math.min( (long) in.left + (long) xOffsets[i], Integer.MAX_VALUE),
            (int) Math.min( (long) in.top + (long) yOffsets[i], Integer.MAX_VALUE),
            xSpans[i],
            ySpans[i]);
            
        }
    }
    
    protected void calculate() {
        if (!(target instanceof Container))
            return;
        Container tg = (Container) target;
        int n = tg.getComponentCount();
        childWidth = new SizeRequirements[n];
        childHeight = new SizeRequirements[n];
        for (int i = 0; i < n; i++) {
            Component c = tg.getComponent(i);
            if (!c.isVisible()) {
                childWidth[i] =
                new SizeRequirements(0, 0, 0, c.getAlignmentX());
                childHeight[i] =
                new SizeRequirements(0, 0, 0, c.getAlignmentY());
                continue;
            }
            Dimension min = c.getMinimumSize();
            Dimension typ = c.getPreferredSize();
            Dimension max = c.getMaximumSize();
            childWidth[i] =
            new SizeRequirements(
            min.width,
            typ.width,
            max.width,
            c.getAlignmentX());
            childHeight[i] =
            new SizeRequirements(
            min.height,
            typ.height,
            max.height,
            c.getAlignmentY());
        }

        int absoluteAxis = X_AXIS;

        if (axis == LINE_AXIS)
            absoluteAxis = tg.getComponentOrientation().isHorizontal() ? X_AXIS : Y_AXIS;
        else if (axis == PAGE_AXIS)
            absoluteAxis = tg.getComponentOrientation().isHorizontal() ? Y_AXIS : X_AXIS;
        else
            absoluteAxis = axis;

        if (absoluteAxis == X_AXIS) {
            totalWidth = SizeRequirements.getTiledSizeRequirements(childWidth);
            totalHeight = SizeRequirements.getAlignedSizeRequirements(childHeight);
        } else {
            totalWidth = SizeRequirements.getAlignedSizeRequirements(childWidth);
            totalHeight = SizeRequirements.getTiledSizeRequirements(childHeight);
        }
    }
    
}
