    # -*- coding: utf-8 -*-
"""
This module contains functions for computing 
integrals both symbolically and numerically.
"""


__author__ = "Kent-Andre Mardal and Martin Sandve Alnes"
__date__   = "2007-11-07 -- 2009-03-04"
__copyright__ = "(C) 2007-2009 Martin Sandve Alnes and Simula Resarch Laboratory"
__license__  = "GNU GPL Version 2, or (at your option) any later version"


import swiginac
import SyFi

from sfc.quadrature import find_quadrature_rule
from sfc.symbolic_utils import symbol, symbols
from sfc.geometry.UFCCell import UFCCell


def geometry_mapping(polygon):
    """
    This function computes the geometry mapping from
    the corresponding reference polygon to polygon.
    It returns the tuple (G,x0) in the
    mapping x = G xi + x0
    """

    vlen = SyFi.cvar.nsd 

    # If we get a point, return this trivial mapping
    if isinstance(polygon, swiginac.matrix):
        # x = 1 * xi + x0 = x0 because xi is always zero at the point
        assert vlen == 1
        assert polygon.rows() == 1
        assert polygon.cols() == 1
        G  = swiginac.matrix(1, 1, [1])
        x0 = polygon
        return G, x0

    Gl = []
    v0 = polygon.vertex(0)
    v1 = polygon.vertex(1)
    
    assert vlen == len(v0)

    if isinstance(polygon, SyFi.Line):
        Gl.extend(v1[k] - v0[k] for k in range(vlen))

    elif isinstance(polygon, SyFi.Triangle):
        if vlen == 2 or vlen == 3:
            v2 = polygon.vertex(2)
            Gl.extend(v1[k] - v0[k] for k in range(vlen))
            Gl.extend(v2[k] - v0[k] for k in range(vlen))
        else:
            raise RuntimeError("Unsupported vertex size.")

    elif isinstance(polygon, SyFi.Rectangle):
        if vlen == 2 or vlen == 3: # TODO: improve mapping
            v2 = polygon.vertex(2)
            v3 = polygon.vertex(3)
            Gl.extend(v1[k] - v0[k] for k in range(vlen))
            Gl.extend(v3[k] - v0[k] for k in range(vlen))
        else:
            raise RuntimeError("Unsupported vertex size.")

    elif isinstance(polygon, SyFi.Tetrahedron):
        if vlen != 3:
            raise RuntimeError("Unsupported vertex size.")
        v2 = polygon.vertex(2)
        v3 = polygon.vertex(3)
        Gl.extend(v1[k] - v0[k] for k in range(vlen))
        Gl.extend(v2[k] - v0[k] for k in range(vlen))
        Gl.extend(v3[k] - v0[k] for k in range(vlen))

    elif isinstance(polygon, SyFi.Box):
        if vlen != 3:
            raise RuntimeError("Unsupported vertex size.")
        v2 = polygon.vertex(2) # TODO: improve mapping
        v3 = polygon.vertex(3)
        v4 = polygon.vertex(4)
        Gl.extend(v1[k] - v0[k] for k in range(vlen))
        Gl.extend(v3[k] - v0[k] for k in range(vlen))
        Gl.extend(v4[k] - v0[k] for k in range(vlen))

    else:
        raise RuntimeError("Unsupported polygon type.")

    G  = swiginac.matrix(vlen, vlen, Gl).transpose()
    x0 = swiginac.matrix(vlen, 1, v0)

    #print "G =", G

    return G, x0
