#!/usr/bin/env python

__authors__ = "Martin Sandve Alnes"
__date__ = "2008-09-04 -- 2008-09-04"

import unittest
import os, sys, glob, shutil, commands

import ufl

from ufl import FiniteElement
from ufl import VectorElement
from ufl import TensorElement
from ufl import MixedElement

from ufl import BasisFunction
from ufl import TestFunction
from ufl import TrialFunction

from ufl import Function
from ufl import Constant

from ufl import dx, ds

import SyFi
import sfc as sfc

from cell_assembly import assemble_on_cell, cell2volume


def num_integrals(form):
    return (form.num_cell_integrals(), form.num_exterior_facet_integrals(), form.num_interior_facet_integrals())


_test_temp_dir = "volume_temp_dir"
_done_test_temp_dir = "done_volume_temp_dir"
class VolumeTest(unittest.TestCase):
    def __init__(self, *args, **kwargs):
        unittest.TestCase.__init__(self, *args, **kwargs)
        shutil.rmtree(_done_test_temp_dir, ignore_errors=True)
        os.mkdir(_done_test_temp_dir)
    
    def setUp(self):
        #print "Running sfc jit test in testdir"
        #print "Imported SyFi from location", SyFi.__file__
        #print "Imported sfc  from location", sfc.__file__
        self.options = sfc.default_options()
        self.options.compilation.cache_dir = os.path.abspath("test_cache")
        # Generate code in a clean directory:
        shutil.rmtree(_test_temp_dir, ignore_errors=True)
        os.mkdir(_test_temp_dir)
        os.chdir(_test_temp_dir)
    
    def tearDown(self):
        dirs = glob.glob("*")
        os.chdir("..")
        for d in dirs:
            os.rename(os.path.join(_test_temp_dir, d), os.path.join(_done_test_temp_dir, d))
    
    def testSetup(self):
        pass
    
    def _testJitVolume(self, polygon):
        "Test that the integral of 1.0 over a unit cell equals the length/area/volume of the unit cell."
        c = Constant(polygon)
        a = c*dx
        form = sfc.jit(a, options = self.options)
        self.assertTrue(form.rank() == 0)
        self.assertTrue(form.num_coefficients() == 1)
        self.assertTrue(num_integrals(form) == (1,0,0))
        A = assemble_on_cell(form, polygon, coeffs=[1.0])
        self.assertAlmostEqual(A, cell2volume[polygon])
    
    def testJitVolumeInterval(self):
        self._testJitVolume("interval")

    def testJitVolumeTriangle(self):
        self._testJitVolume("triangle")
    
    def testJitVolumeTetrahedron(self):
        self._testJitVolume("tetrahedron")
    
    def _testJitVolumeQuadrilateral(self): # Not supported by dolfin yet
        self._testJitVolume("quadrilateral")
    
    def _testJitVolumeHexahedron(self): # Not supported by dolfin yet
        self._testJitVolume("hexahedron")

    def _testJitConstant(self, polygon, degree):
        """Test that the integral of a constant coefficient over a unit 
        cell mesh equals the constant times the volume of the unit cell."""
        element = FiniteElement("CG", polygon, degree)
        f = Function(element)
        a = f*dx
        form = sfc.jit(a, options = self.options)
        self.assertTrue(form.rank() == 0)
        self.assertTrue(form.num_coefficients() == 1)
        self.assertTrue(num_integrals(form) == (1,0,0))
        const = 1.23
        A = assemble_on_cell(form, polygon, coeffs=[const])
        self.assertAlmostEqual(A, const*cell2volume[polygon])
    
    def testJitConstantInterval(self):
        polygon = "interval"
        self._testJitConstant(polygon, 1)
        self._testJitConstant(polygon, 2)

    def testJitConstantTriangle(self):
        polygon = "triangle"
        self._testJitConstant(polygon, 1)
        self._testJitConstant(polygon, 2)
    
    def testJitConstantTetrahedron(self):
        polygon = "tetrahedron"
        self._testJitConstant(polygon, 1)
        self._testJitConstant(polygon, 2)
    
    def _testJitConstantQuadrilateral(self): # Not supported by dolfin yet
        polygon = "quadrilateral"
        self._testJitConstant(polygon, 1)
        self._testJitConstant(polygon, 2)

    def _testJitConstantHexahedron(self): # Not supported by dolfin yet
        polygon = "hexahedron"
        self._testJitConstant(polygon, 1)
        self._testJitConstant(polygon, 2)


tests = [VolumeTest]

if __name__ == "__main__":
    unittest.main()

