#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
This module contains default options for SFC. 
"""

# Copyright (C) 2008-2009 Martin Sandve Alnes and Simula Resarch Laboratory
#
# This file is part of SyFi.
#
# SyFi is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 2 of the License, or
# (at your option) any later version.
#
# SyFi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with SyFi. If not, see <http://www.gnu.org/licenses/>.
#
# Modified by Kent-Andre Mardal, 2010.
#
# First added:  2008-08-13
# Last changed: 2009-04-23

from sfc.common.ParameterDict import ParameterDict


def default_parameters():
    """Construct a ParameterDict with default options for SFC,
    which can then be modified and passed to jit.

    Some options may need explanation:
    
    options.integration.representation:
        "disable"             | Generate empty tabulate_tensor functions, for debugging. 
        "symbolic"            | Integrate each element tensor entry symbolically.
        "truncated_symbolic"  | Integrate symbolically after truncating the Taylor series.
        "symbolic_quadrature" | Integrate by quadrature using symbolic expressions.
        "quadrature"          | (DEFAULT) Generate (partially) inlined quadrature rule.
        "safe_quadrature"     | Generate slow but safe quadrature code. 
    """
    
    ### General output options:
    output = ParameterDict(enable_timing       = False,
                           verbosity           = 0, # TODO: Use logging system, verbosity = logging.WARNING
                           enable_debug_prints = False,
                           store_log           = False)
    
    ### General code generation options:
    
    # General form options:
    form = ParameterDict(name = None)
    
    # Integration options:
    integral = ParameterDict(integration_method = "quadrature", # "symbolic",
                             integration_order  = None,
                             enable_debug_code  = False,
                             safemode           = False,
                             use_expand_indices2 = False)
    
    # Dof map code generation:
    dof_map = ParameterDict(enable_dof_ptv = False)
    
    # Finite element code generation:
    finite_element = ParameterDict(enable_evaluate_basis             = True,
                                   enable_evaluate_basis_derivatives = True,
                                   default_order_of_element          = 2, 
                                   evaluate_basis_derivatives_order  = 1, # highest order generated
                                   optimize_basis = False,
                                   enable_horner  = False) # TODO: use horners rule in evaluate_basis*
    
    code = ParameterDict(form            = form,
                         integral        = integral,
                         dof_map         = dof_map,
                         finite_element  = finite_element,
                         prefix          = "",
                         dolfin_wrappers = False)
    
    ### General compilation options:
    compilation = ParameterDict(enable_debug_code  = False,
                                cache_dir          = None,    # Override cache directory
                                overwrite_cache    = False,   # Force recompilation independent of cache status
                                cppargs            = ["-O2"],
                                skip               = False,   # To bypass C++ compilation phase (intended for testing and profiling) 
                                generate_interface = True,    # To bypass generation of swig interface
                                generate_setup     = True)    # To bypass generation of setup.py file
    
    ### All options collected:
    options = ParameterDict(output      = output,
                            code        = code,
                            compilation = compilation)
    return options

