/* sp-callgraph-view.c
 *
 * Copyright (C) 2016 Christian Hergert <chergert@redhat.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/* Sysprof -- Sampling, systemwide CPU profiler
 * Copyright 2004, Red Hat, Inc.
 * Copyright 2004, 2005, 2006, Soeren Sandmann
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <glib/gi18n.h>

#include "sp-callgraph-profile-private.h"
#include "sp-callgraph-view.h"

#include "util.h"

typedef struct
{
  SpCallgraphProfile  *profile;

  GtkTreeView         *callers_view;
  GtkTreeView         *functions_view;
  GtkTreeView         *descendants_view;
  GtkTreeViewColumn   *descendants_name_column;

  GQueue              *history;

  guint                profile_size;
} SpCallgraphViewPrivate;

G_DEFINE_TYPE_WITH_PRIVATE (SpCallgraphView, sp_callgraph_view, GTK_TYPE_BIN)

enum {
  PROP_0,
  PROP_PROFILE,
  N_PROPS
};

enum {
  GO_PREVIOUS,
  N_SIGNALS
};

enum {
  COLUMN_NAME,
  COLUMN_SELF,
  COLUMN_TOTAL,
  COLUMN_POINTER,
};


static void sp_callgraph_view_update_descendants (SpCallgraphView *self,
                                                  StackNode       *node);

static GParamSpec *properties [N_PROPS];
static guint signals [N_SIGNALS];

static guint
sp_callgraph_view_get_profile_size (SpCallgraphView *self)
{
  SpCallgraphViewPrivate *priv = sp_callgraph_view_get_instance_private (self);
  StackStash *stash;
  StackNode *node;
  guint size = 0;

  g_assert (SP_IS_CALLGRAPH_VIEW (self));

  if (priv->profile_size != 0)
    return priv->profile_size;

  if (priv->profile == NULL)
    return 0;

  if (NULL == (stash = sp_callgraph_profile_get_stash (priv->profile)))
    return 0;

  for (node = stack_stash_get_root (stash); node != NULL; node = node->siblings)
    size += node->total;

  priv->profile_size = size;

  return size;
}

static void
build_functions_store (StackNode *node,
                       gpointer   user_data)
{
  struct {
    GtkListStore *store;
    gdouble profile_size;
  } *state = user_data;
  GtkTreeIter iter;
  const StackNode *n;
  guint size = 0;
  guint total = 0;

  g_assert (state != NULL);
  g_assert (GTK_IS_LIST_STORE (state->store));

  for (n = node; n != NULL; n = n->next)
    {
      size += n->size;
      if (n->toplevel)
        total += n->total;
    }

  gtk_list_store_append (state->store, &iter);
  gtk_list_store_set (state->store, &iter,
                      COLUMN_NAME, (const gchar *)node->data,
                      COLUMN_SELF, 100.0 * size / state->profile_size,
                      COLUMN_TOTAL, 100.0 * total / state->profile_size,
                      COLUMN_POINTER, node,
                      -1);

}

static void
sp_callgraph_view_load (SpCallgraphView    *self,
                        SpCallgraphProfile *profile)
{
  SpCallgraphViewPrivate *priv = sp_callgraph_view_get_instance_private (self);
  GtkListStore *functions;
  StackStash *stash;
  StackNode *n;
  GtkTreeIter iter;
  struct {
    GtkListStore *store;
    gdouble profile_size;
  } state = { 0 };

  g_assert (SP_IS_CALLGRAPH_VIEW (self));
  g_assert (SP_IS_CALLGRAPH_PROFILE (profile));

  /*
   * TODO: This is probably the type of thing we want to do off the main
   *       thread. We should be able to build the tree models off thread
   *       and then simply apply them on the main thread.
   *
   *       In the mean time, we should set the state of the widget to
   *       insensitive and give some indication of loading progress.
   */

  g_set_object (&priv->profile, profile);

  if (NULL == (stash = sp_callgraph_profile_get_stash (profile)))
    return;

  for (n = stack_stash_get_root (stash); n; n = n->siblings)
    state.profile_size += n->total;

  functions = gtk_list_store_new (4, G_TYPE_STRING, G_TYPE_DOUBLE, G_TYPE_DOUBLE, G_TYPE_POINTER);

  state.store = functions;
  stack_stash_foreach_by_address (stash, build_functions_store, &state);

  gtk_tree_sortable_set_sort_column_id (GTK_TREE_SORTABLE (functions),
                                        COLUMN_TOTAL,
                                        GTK_SORT_DESCENDING);

  gtk_tree_view_set_model (priv->functions_view, GTK_TREE_MODEL (functions));
  gtk_tree_view_set_model (priv->callers_view, NULL);
  gtk_tree_view_set_model (priv->descendants_view, NULL);

  if (gtk_tree_model_get_iter_first (GTK_TREE_MODEL (functions), &iter))
    {
      GtkTreeSelection *selection;

      selection = gtk_tree_view_get_selection (priv->functions_view);
      gtk_tree_selection_select_iter (selection, &iter);
    }

  g_clear_object (&functions);
}

static void
sp_callgraph_view_unload (SpCallgraphView *self)
{
  SpCallgraphViewPrivate *priv = sp_callgraph_view_get_instance_private (self);

  g_assert (SP_IS_CALLGRAPH_VIEW (self));
  g_assert (SP_IS_CALLGRAPH_PROFILE (priv->profile));

  g_queue_clear (priv->history);
  g_clear_object (&priv->profile);
  priv->profile_size = 0;
}

void
sp_callgraph_view_set_profile (SpCallgraphView    *self,
                               SpCallgraphProfile *profile)
{
  SpCallgraphViewPrivate *priv = sp_callgraph_view_get_instance_private (self);

  g_return_if_fail (SP_IS_CALLGRAPH_VIEW (self));
  g_return_if_fail (!profile || SP_IS_CALLGRAPH_PROFILE (profile));

  if (profile != priv->profile)
    {
      if (priv->profile)
        sp_callgraph_view_unload (self);

      if (profile)
        sp_callgraph_view_load (self, profile);

      g_object_notify_by_pspec (G_OBJECT (self), properties [PROP_PROFILE]);
    }
}

static void
sp_callgraph_view_expand_descendants (SpCallgraphView *self)
{
  SpCallgraphViewPrivate *priv = sp_callgraph_view_get_instance_private (self);
  GtkTreeModel *model;
  GList *all_paths = NULL;
  GtkTreePath *first_path;
  GtkTreeIter iter;
  gdouble top_value = 0;
  gint max_rows = 40; /* FIXME */
  gint n_rows;

  g_assert (SP_IS_CALLGRAPH_VIEW (self));

  model = gtk_tree_view_get_model (priv->descendants_view);
  first_path = gtk_tree_path_new_first ();
  all_paths = g_list_prepend (all_paths, first_path);
  n_rows = 1;

  gtk_tree_model_get_iter (model, &iter, first_path);
  gtk_tree_model_get (model, &iter,
                      COLUMN_TOTAL, &top_value,
                      -1);

  while ((all_paths != NULL) && (n_rows < max_rows))
    {
      GtkTreeIter best_iter;
      GtkTreePath *best_path = NULL;
      GList *list;
      gdouble best_value = 0.0;
      gint n_children;
      gint i;

      for (list = all_paths; list != NULL; list = list->next)
        {
          GtkTreePath *path = list->data;

          g_assert (path != NULL);

          if (gtk_tree_model_get_iter (model, &iter, path))
            {
              gdouble value;

              gtk_tree_model_get (model, &iter,
                                  COLUMN_TOTAL, &value,
                                  -1);

              if (value >= best_value)
                {
                  best_value = value;
                  best_path = path;
                  best_iter = iter;
                }
            }
        }

      n_children = gtk_tree_model_iter_n_children (model, &best_iter);

      if ((n_children > 0) &&
          ((best_value / top_value) > 0.04) &&
          ((n_children + gtk_tree_path_get_depth (best_path)) / (gdouble)max_rows) < (best_value / top_value))
        {
          gtk_tree_view_expand_row (priv->descendants_view, best_path, FALSE);
          n_rows += n_children;

          if (gtk_tree_path_get_depth (best_path) < 4)
            {
              GtkTreePath *path;

              path = gtk_tree_path_copy (best_path);
              gtk_tree_path_down (path);

              for (i = 0; i < n_children; i++)
                {
                  all_paths = g_list_prepend (all_paths, path);

                  path = gtk_tree_path_copy (path);
                  gtk_tree_path_next (path);
                }

              gtk_tree_path_free (path);
            }
        }

      all_paths = g_list_remove (all_paths, best_path);

      /* Always expand at least once */
      if ((all_paths == NULL) && (n_rows == 1))
        gtk_tree_view_expand_row (priv->descendants_view, best_path, FALSE);

      gtk_tree_path_free (best_path);
    }

  g_list_free_full (all_paths, (GDestroyNotify)gtk_tree_path_free);
}

typedef struct
{
  StackNode   *node;
  const gchar *name;
  guint        self;
  guint        total;
} Caller;

static Caller *
caller_new (StackNode *node)
{
  Caller *c;

  c = g_slice_new (Caller);
  c->name = U64_TO_POINTER (node->data);
  c->self = 0;
  c->total = 0;
  c->node = node;

  return c;
}

static void
caller_free (gpointer data)
{
  Caller *c = data;
  g_slice_free (Caller, c);
}

static void
sp_callgraph_view_function_selection_changed (SpCallgraphView  *self,
                                              GtkTreeSelection *selection)
{
  SpCallgraphViewPrivate *priv = sp_callgraph_view_get_instance_private (self);
  GtkTreeModel *model = NULL;
  GtkTreeIter iter;
  GtkListStore *callers_store;
  g_autoptr(GHashTable) callers = NULL;
  g_autoptr(GHashTable) processed = NULL;
  StackNode *callees = NULL;
  StackNode *node;

  g_assert (SP_IS_CALLGRAPH_VIEW (self));
  g_assert (GTK_IS_TREE_SELECTION (selection));

  if (!gtk_tree_selection_get_selected (selection, &model, &iter))
    {
      gtk_tree_view_set_model (priv->callers_view, NULL);
      gtk_tree_view_set_model (priv->descendants_view, NULL);
      return;
    }

  gtk_tree_model_get (model, &iter,
                      COLUMN_POINTER, &callees,
                      -1);

  sp_callgraph_view_update_descendants (self, callees);

  callers_store = gtk_list_store_new (4,
                                      G_TYPE_STRING,
                                      G_TYPE_DOUBLE,
                                      G_TYPE_DOUBLE,
                                      G_TYPE_POINTER);

  callers = g_hash_table_new_full (NULL, NULL, NULL, caller_free);
  processed = g_hash_table_new (NULL, NULL);

  for (node = callees; node != NULL; node = node->next)
    {
      Caller *c;

      if (!node->parent)
        continue;

      c = g_hash_table_lookup (callers, U64_TO_POINTER (node->parent->data));

      if (c == NULL)
        {
          c = caller_new (node->parent);
          g_hash_table_insert (callers, (gpointer)c->name, c);
        }
    }

  for (node = callees; node != NULL; node = node->next)
    {
      StackNode *top_caller = node->parent;
      StackNode *top_callee = node;
      StackNode *n;
      Caller *c;

      if (!node->parent)
        continue;

      /*
       * We could have a situation where the function was called in a
       * reentrant fashion, so we want to take the top-most match in the
       * stack.
       */
      for (n = node; n && n->parent; n = n->parent)
        {
          if (n->data == node->data && n->parent->data == node->parent->data)
            {
              top_caller = n->parent;
              top_callee = n;
            }
        }

      c = g_hash_table_lookup (callers, U64_TO_POINTER (node->parent->data));

      g_assert (c != NULL);

      if (!g_hash_table_lookup (processed, top_caller))
        {
          c->total += top_callee->total;
          g_hash_table_insert (processed, top_caller, top_caller);
        }

      c->self += node->size;
    }

  {
    GHashTableIter hiter;
    gpointer key, value;
    guint size = 0;

    size = sp_callgraph_view_get_profile_size (self);

    g_hash_table_iter_init (&hiter, callers);

    while (g_hash_table_iter_next (&hiter, &key, &value))
      {
        Caller *c = value;

        gtk_list_store_append (callers_store, &iter);
        gtk_list_store_set (callers_store, &iter,
                            COLUMN_NAME, c->name,
                            COLUMN_SELF, c->self * 100.0 / size,
                            COLUMN_TOTAL, c->total * 100.0 / size,
                            COLUMN_POINTER, c->node,
                            -1);
      }
  }

  gtk_tree_view_set_model (priv->callers_view, GTK_TREE_MODEL (callers_store));
  gtk_tree_sortable_set_sort_column_id (GTK_TREE_SORTABLE (callers_store),
                                        COLUMN_TOTAL,
                                        GTK_SORT_DESCENDING);

  g_clear_object (&callers_store);
}

static void
sp_callgraph_view_set_node (SpCallgraphView *self,
                            StackNode       *node)
{
  SpCallgraphViewPrivate *priv = sp_callgraph_view_get_instance_private (self);
  GtkTreeModel *model;
  GtkTreeIter iter;

  g_assert (SP_IS_CALLGRAPH_VIEW (self));
  g_assert (node != NULL);

  model = gtk_tree_view_get_model (priv->functions_view);

  if (gtk_tree_model_get_iter_first (model, &iter))
    {
      do
        {
          StackNode *item = NULL;

          gtk_tree_model_get (model, &iter,
                              COLUMN_POINTER, &item,
                              -1);

          if (item == node)
            {
              GtkTreeSelection *selection;

              selection = gtk_tree_view_get_selection (priv->functions_view);
              gtk_tree_selection_select_iter (selection, &iter);

              break;
            }
        }
      while (gtk_tree_model_iter_next (model, &iter));
    }
}

static void
sp_callgraph_view_descendant_activated (SpCallgraphView   *self,
                                        GtkTreePath       *path,
                                        GtkTreeViewColumn *column,
                                        GtkTreeView       *tree_view)
{
  GtkTreeModel *model;
  StackNode *node = NULL;
  GtkTreeIter iter;

  g_assert (SP_IS_CALLGRAPH_VIEW (self));
  g_assert (GTK_IS_TREE_VIEW (tree_view));
  g_assert (path != NULL);
  g_assert (GTK_IS_TREE_VIEW_COLUMN (column));

  model = gtk_tree_view_get_model (tree_view);

  if (!gtk_tree_model_get_iter (model, &iter, path))
    return;

  gtk_tree_model_get (model, &iter,
                      COLUMN_POINTER, &node,
                      -1);

  if (node != NULL)
    sp_callgraph_view_set_node (self, node);
}

static void
sp_callgraph_view_caller_activated (SpCallgraphView   *self,
                                    GtkTreePath       *path,
                                    GtkTreeViewColumn *column,
                                    GtkTreeView       *tree_view)
{
  GtkTreeModel *model;
  StackNode *node = NULL;
  GtkTreeIter iter;

  g_assert (SP_IS_CALLGRAPH_VIEW (self));
  g_assert (GTK_IS_TREE_VIEW (tree_view));
  g_assert (path != NULL);
  g_assert (GTK_IS_TREE_VIEW_COLUMN (column));

  model = gtk_tree_view_get_model (tree_view);

  if (!gtk_tree_model_get_iter (model, &iter, path))
    return;

  gtk_tree_model_get (model, &iter,
                      COLUMN_POINTER, &node,
                      -1);

  if (node != NULL)
    sp_callgraph_view_set_node (self, node);
}

static void
sp_callgraph_view_tag_data_func (GtkTreeViewColumn *column,
                                 GtkCellRenderer   *cell,
                                 GtkTreeModel      *model,
                                 GtkTreeIter       *iter,
                                 gpointer           data)
{
  SpCallgraphView *self = data;
  SpCallgraphViewPrivate *priv = sp_callgraph_view_get_instance_private (self);
  StackNode *node = NULL;
  const gchar *str = NULL;

  gtk_tree_model_get (model, iter, COLUMN_POINTER, &node, -1);

  if (node && node->data)
    {
      GQuark tag;

      tag = sp_callgraph_profile_get_tag (priv->profile, GSIZE_TO_POINTER (node->data));
      if (tag != 0)
        str = g_quark_to_string (tag);
    }

  g_object_set (cell, "text", str, NULL);
}

static void
sp_callgraph_view_real_go_previous (SpCallgraphView *self)
{
  SpCallgraphViewPrivate *priv = sp_callgraph_view_get_instance_private (self);
  StackNode *node;

  g_assert (SP_IS_CALLGRAPH_VIEW (self));

  node = g_queue_pop_head (priv->history);

  if (NULL != (node = g_queue_peek_head (priv->history)))
    sp_callgraph_view_set_node (self, node);
}

static void
sp_callgraph_view_finalize (GObject *object)
{
  SpCallgraphView *self = (SpCallgraphView *)object;
  SpCallgraphViewPrivate *priv = sp_callgraph_view_get_instance_private (self);

  g_clear_pointer (&priv->history, g_queue_free);
  g_clear_object (&priv->profile);

  G_OBJECT_CLASS (sp_callgraph_view_parent_class)->finalize (object);
}

static void
sp_callgraph_view_get_property (GObject    *object,
                                guint       prop_id,
                                GValue     *value,
                                GParamSpec *pspec)
{
  SpCallgraphView *self = SP_CALLGRAPH_VIEW (object);
  SpCallgraphViewPrivate *priv = sp_callgraph_view_get_instance_private (self);

  switch (prop_id)
    {
    case PROP_PROFILE:
      g_value_set_object (value, priv->profile);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
    }
}

static void
sp_callgraph_view_set_property (GObject      *object,
                                guint         prop_id,
                                const GValue *value,
                                GParamSpec   *pspec)
{
  SpCallgraphView *self = SP_CALLGRAPH_VIEW (object);

  switch (prop_id)
    {
    case PROP_PROFILE:
      sp_callgraph_view_set_profile (self, g_value_get_object (value));
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
    }
}

static void
sp_callgraph_view_class_init (SpCallgraphViewClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);
  GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (klass);
  GtkBindingSet *bindings;

  object_class->finalize = sp_callgraph_view_finalize;
  object_class->get_property = sp_callgraph_view_get_property;
  object_class->set_property = sp_callgraph_view_set_property;

  klass->go_previous = sp_callgraph_view_real_go_previous;

  properties [PROP_PROFILE] =
    g_param_spec_object ("profile",
                         "Profile",
                         "The callgraph profile to view",
                         SP_TYPE_CALLGRAPH_PROFILE,
                         (G_PARAM_READWRITE | G_PARAM_EXPLICIT_NOTIFY | G_PARAM_STATIC_STRINGS));

  g_object_class_install_properties (object_class, N_PROPS, properties);

  signals [GO_PREVIOUS] =
    g_signal_new ("go-previous",
                  G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
                  G_STRUCT_OFFSET (SpCallgraphViewClass, go_previous),
                  NULL, NULL, NULL, G_TYPE_NONE, 0);

  gtk_widget_class_set_template_from_resource (widget_class,
                                               "/org/gnome/sysprof/ui/sp-callgraph-view.ui");

  gtk_widget_class_bind_template_child_private (widget_class, SpCallgraphView, callers_view);
  gtk_widget_class_bind_template_child_private (widget_class, SpCallgraphView, functions_view);
  gtk_widget_class_bind_template_child_private (widget_class, SpCallgraphView, descendants_view);
  gtk_widget_class_bind_template_child_private (widget_class, SpCallgraphView, descendants_name_column);

  bindings = gtk_binding_set_by_class (klass);
  gtk_binding_entry_add_signal (bindings, GDK_KEY_Left, GDK_MOD1_MASK, "go-previous", 0);
}

static void
sp_callgraph_view_init (SpCallgraphView *self)
{
  SpCallgraphViewPrivate *priv = sp_callgraph_view_get_instance_private (self);
  GtkTreeSelection *selection;
  GtkCellRenderer *cell;

  priv->history = g_queue_new ();

  gtk_widget_init_template (GTK_WIDGET (self));

  selection = gtk_tree_view_get_selection (priv->functions_view);

  g_signal_connect_object (selection,
                           "changed",
                           G_CALLBACK (sp_callgraph_view_function_selection_changed),
                           self,
                           G_CONNECT_SWAPPED);

  g_signal_connect_object (priv->descendants_view,
                           "row-activated",
                           G_CALLBACK (sp_callgraph_view_descendant_activated),
                           self,
                           G_CONNECT_SWAPPED);

  g_signal_connect_object (priv->callers_view,
                           "row-activated",
                           G_CALLBACK (sp_callgraph_view_caller_activated),
                           self,
                           G_CONNECT_SWAPPED);

  cell = g_object_new (GTK_TYPE_CELL_RENDERER_TEXT,
                       "foreground", "#666666",
                       "scale", PANGO_SCALE_SMALL,
                       "xalign", 1.0f,
                       NULL);
  gtk_tree_view_column_pack_start (priv->descendants_name_column, cell, FALSE);
  gtk_tree_view_column_set_cell_data_func (priv->descendants_name_column, cell,
                                           sp_callgraph_view_tag_data_func,
                                           self, NULL);
}

typedef struct _Descendant Descendant;

struct _Descendant
{
  const gchar *name;
  guint        self;
  guint        cumulative;
  Descendant  *parent;
  Descendant  *siblings;
  Descendant  *children;
};

static void
build_tree_cb (StackLink *trace,
               gint       size,
               gpointer   user_data)
{
  Descendant **tree = user_data;
  Descendant *parent = NULL;
  StackLink *link;

  g_assert (trace != NULL);
  g_assert (tree != NULL);

  /* Get last item */
  link = trace;
  while (link->next)
    link = link->next;

  for (; link != NULL; link = link->prev)
    {
      const gchar *address = U64_TO_POINTER (link->data);
      Descendant *prev = NULL;
      Descendant *match = NULL;

      for (match = *tree; match != NULL; match = match->siblings)
        {
          if (match->name == address)
            {
              if (prev != NULL)
                {
                  /* Move to front */
                  prev->siblings = match->siblings;
                  match->siblings = *tree;
                  *tree = match;
                }
              break;
            }
        }

      if (match == NULL)
        {
          /* Have we seen this object further up the tree? */
          for (match = parent; match != NULL; match = match->parent)
            {
              if (match->name == address)
                break;
            }
        }

      if (match == NULL)
        {
          match = g_slice_new (Descendant);
          match->name = address;
          match->cumulative = 0;
          match->self = 0;
          match->children = NULL;
          match->parent = parent;
          match->siblings = *tree;
          *tree = match;
        }

      tree = &match->children;
      parent = match;
    }

  parent->self += size;

  for (; parent != NULL; parent = parent->parent)
    parent->cumulative += size;
}

static Descendant *
build_tree (StackNode *node)
{
  Descendant *tree = NULL;

  for (; node != NULL; node = node->next)
    {
      if (node->toplevel)
        stack_node_foreach_trace (node, build_tree_cb, &tree);
    }

  return tree;
}

static void
append_to_tree_and_free (SpCallgraphView *self,
                         StackStash      *stash,
                         GtkTreeStore    *store,
                         Descendant      *item,
                         GtkTreeIter     *parent)
{
  StackNode *node = NULL;
  GtkTreeIter iter;
  guint profile_size;

  g_assert (GTK_IS_TREE_STORE (store));
  g_assert (item != NULL);

  profile_size = sp_callgraph_view_get_profile_size (self);

  gtk_tree_store_append (store, &iter, parent);

  node = stack_stash_find_node (stash, (gpointer)item->name);

  gtk_tree_store_set (store, &iter,
                      COLUMN_NAME, item->name,
                      COLUMN_SELF, item->self * 100.0 / (gdouble)profile_size,
                      COLUMN_TOTAL, item->cumulative * 100.0 / (gdouble)profile_size,
                      COLUMN_POINTER, node,
                      -1);

  if (item->siblings != NULL)
    append_to_tree_and_free (self, stash, store, item->siblings, parent);

  if (item->children != NULL)
    append_to_tree_and_free (self, stash, store, item->children, &iter);

  g_slice_free (Descendant, item);
}

static void
sp_callgraph_view_update_descendants (SpCallgraphView *self,
                                      StackNode       *node)
{
  SpCallgraphViewPrivate *priv = sp_callgraph_view_get_instance_private (self);
  GtkTreeStore *store;

  g_assert (SP_IS_CALLGRAPH_VIEW (self));

  if (g_queue_peek_head (priv->history) != node)
    g_queue_push_head (priv->history, node);

  store = gtk_tree_store_new (4,
                              G_TYPE_STRING,
                              G_TYPE_DOUBLE,
                              G_TYPE_DOUBLE,
                              G_TYPE_POINTER);

  if (priv->profile != NULL)
  {
    StackStash *stash;

    stash = sp_callgraph_profile_get_stash (priv->profile);
    if (stash != NULL)
      {
        Descendant *tree;

        tree = build_tree (node);
        if (tree != NULL)
          append_to_tree_and_free (self, stash, store, tree, NULL);
      }
  }

  gtk_tree_view_set_model (priv->descendants_view, GTK_TREE_MODEL (store));
  gtk_tree_sortable_set_sort_column_id (GTK_TREE_SORTABLE (store),
                                        COLUMN_TOTAL, GTK_SORT_DESCENDING);
  sp_callgraph_view_expand_descendants (self);

  g_clear_object (&store);
}

/**
 * sp_callgraph_view_screenshot:
 * @self: A #SpCallgraphView.
 *
 * This function will generate a text representation of the descendants tree.
 * This is useful if you want to include various profiling information in a
 * commit message or email.
 *
 * The text generated will match the current row expansion in the tree view.
 *
 * Returns: (nullable) (transfer full): A newly allocated string that should be freed
 *   with g_free().
 */
gchar *
sp_callgraph_view_screenshot (SpCallgraphView *self)
{
  SpCallgraphViewPrivate *priv = sp_callgraph_view_get_instance_private (self);
  GtkTreeView *tree_view;
  GtkTreeModel *model;
  GtkTreePath *tree_path;
  GString *str;
  GtkTreeIter iter;

  g_return_val_if_fail (SP_IS_CALLGRAPH_VIEW (self), NULL);

  tree_view = priv->descendants_view;

  if (NULL == (model = gtk_tree_view_get_model (tree_view)))
    return NULL;

  /*
   * To avoid having to precalculate the deepest visible row, we
   * put the timing information at the beginning of the line.
   */

  str = g_string_new ("      SELF CUMULATIVE    FUNCTION\n");
  tree_path = gtk_tree_path_new_first ();

  for (;;)
    {
      if (gtk_tree_model_get_iter (model, &iter, tree_path))
        {
          guint depth = gtk_tree_path_get_depth (tree_path);
          StackNode *node;
          gdouble in_self;
          gdouble total;
          guint i;

          gtk_tree_model_get (model, &iter,
                              COLUMN_SELF, &in_self,
                              COLUMN_TOTAL, &total,
                              COLUMN_POINTER, &node,
                              -1);

          g_string_append_printf (str, "[% 7.2lf%%] [% 7.2lf%%]  ", in_self, total);

          for (i = 0; i < depth; i++)
            g_string_append (str, "  ");
          g_string_append (str, GSIZE_TO_POINTER (node->data));
          g_string_append_c (str, '\n');

          if (gtk_tree_view_row_expanded (tree_view, tree_path))
            gtk_tree_path_down (tree_path);
          else
            gtk_tree_path_next (tree_path);

          continue;
        }

      if (!gtk_tree_path_up (tree_path) || !gtk_tree_path_get_depth (tree_path))
        break;

      gtk_tree_path_next (tree_path);
    }

  gtk_tree_path_free (tree_path);

  return g_string_free (str, FALSE);
}

guint
sp_callgraph_view_get_n_functions (SpCallgraphView *self)
{
  SpCallgraphViewPrivate *priv = sp_callgraph_view_get_instance_private (self);
  GtkTreeModel *model;
  guint ret = 0;

  g_return_val_if_fail (SP_IS_CALLGRAPH_VIEW (self), 0);

  if (NULL != (model = gtk_tree_view_get_model (priv->functions_view)))
    ret = gtk_tree_model_iter_n_children (model, NULL);

  return ret;
}
