#include <termios.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <sys/select.h>

#include "console.h"
#include "util.h"


struct console_t {
    int fd; /* Filedescriptor for console, or -1 if disabled */
    struct termios oldtio; /* old settings, for restore */
    int needClose; /* Does file descriptor need to be closed on reset? */
    int needRestore; /* Is the file descriptor indeed a terminal, and needs
		      * to be restored? */
};

console_t *prepareConsole(int fd) {
    struct termios newtio;
    int needClose=0;

    if(fd < 0) {
	fd = open("/dev/tty", O_RDWR);
	if(fd < 0) {
	    fprintf(stderr, "Could not open keyboard: %s\n", strerror(errno));
	    return NULL;
	}
	needClose=1;
    }

    console_t *c = MALLOC(console_t);
    if(c == NULL)
	return c;

    c->fd = fd;
    c->needClose = needClose;

    if(tcgetattr(c->fd, &c->oldtio) >= 0) {
	newtio = c->oldtio;
	newtio.c_lflag &= ~ECHO;
	newtio.c_lflag &= ~ICANON;
	newtio.c_cc[VMIN] = 1;
	newtio.c_cc[VTIME] = 0;
	tcsetattr(c->fd, 0, &newtio);
	c->needRestore = 1;

    } else {
	c->needRestore = 0;
    }

    /* Just in case, drain any pre-existing input, to avoid premature
     * starts */
    while(1) {
	fd_set read_set;
	struct timeval tv;
	char buf[80];
	int r;
	
	tv.tv_sec=0;
	tv.tv_usec=0;
	FD_ZERO(&read_set);
	FD_SET(fd, &read_set);	    
	r = select(fd+1,&read_set, NULL, NULL, &tv);
	if(r <= 0)
	    break;
	r = read(fd, buf, sizeof(buf));
	if(r <= 0)
	    break;
    }

    return c;
}

int getConsoleFd(console_t **cp) {
    console_t *c=*cp;
    if(c == NULL)
	return -1;
    return c->fd;
}

void restoreConsole(console_t **cp, int doConsume) {
    console_t *c=*cp;
    int ch;
    int n;
    
    if(c == NULL)
      return;
    
    /* If key pressed, consume it. If letter is q, quit */
    if(doConsume) {
	n = read(c->fd, &ch, 1);
	if (n == 1 && ch == 'q') {
	    exit(1);
	}
    }

    if(c->needRestore && tcsetattr(c->fd, 0, &c->oldtio) < 0) {
	perror("Restore terminal settings");
    }
    if(c->needClose)
	close(c->fd);
    free(c);
    *cp = NULL;
}
