/* TABLIX, PGA highschool timetable generator                              */
/* Copyright (C) 2002 Tomaz Solc                                           */

/* This program is free software; you can redistribute it and/or modify    */
/* it under the terms of the GNU General Public License as published by    */
/* the Free Software Foundation; either version 2 of the License, or       */
/* (at your option) any later version.                                     */

/* This program is distributed in the hope that it will be useful,         */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/* GNU General Public License for more details.                            */

/* You should have received a copy of the GNU General Public License       */
/* along with this program; if not, write to the Free Software             */
/* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA */

/* $Id: data.c,v 1.16 2004/10/17 09:37:06 avian Exp $ */

#include <stdlib.h>
#include <string.h>
#include <dlfcn.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "data.h"
#include "modsup.h"
#include "error.h"
#include "gettext.h"

int *cextbuff;                 /* Output buffers                           */
int *textbuff;

static int **conf_lookup;      /* Conflict lookup table and pointers to it */
static int ***conf_lead;
static int ***conf_start;

tuple *tuplemap;
int tuplenum=0;
int tuplemut=0;

teacher *tmap;
int tmapnum=0;

subject *smap;
int smapnum=0;

group *cmap;
int cmapnum=0;

classroom *rmap;
int rmapnum=0;

int days;
int periods;
int times; 			/* (days*periods) */

int posweight;

char *school_name;
char *school_address;
char *author;

#ifdef MORE_TEACHERS_HACK
int ***moretids;
int **moretidnums;
#endif

/* The following functions arrange a chromosome along a timeline */
/* (tuplenum == -1) means free time                              */

/* cextbuff[cid*TIMES+time]=tuplenum       */
void extend_chromo(chromo *t)
{
        int b, cid;
        int **pnt;

	int n,time,num;

	memset(cextbuff, 0xFF, sizeof(*cextbuff)*cmapnum*TIMES);
        /* for(b=0;b<cmapnum*TIMES;b++) cextbuff[b]=-1; */

        for(b=0;b<tuplenum;b++) {
                cid=tuplemap[b].cid;

		pnt=conf_start[cid];
		num=conf_lead[cid]-conf_start[cid];

		time=t->inf[b].time;
		
               	for(n=0;n<num;n++) {
	               	(*pnt)[time]=b;
			pnt++;
		}
        }
}

/* textbuff[tid*TIMES+time]=tuplenum */
void textend_chromo(chromo *t)
{
        int b;
	#ifdef MORE_TEACHERS_HACK
	int c,a;
	#endif
	
	memset(textbuff, 0xFF, sizeof(*textbuff)*tmapnum*TIMES);
        /* for(b=0;b<tmapnum*TIMES;b++) textbuff[b]=-1; */

	#ifdef MORE_TEACHERS_HACK
	if(moretids==NULL) {
	        for(b=0;b<tuplenum;b++) {
	                textbuff[tuplemap[b].tid*TIMES+t->inf[b].time]=b;
	        }
	} else {
	        for(b=0;b<tuplenum;b++) {
	                textbuff[tuplemap[b].tid*TIMES+t->inf[b].time]=b;
			for(c=0;c<(*moretidnums)[b];c++) {
				a=(*moretids)[b][c];
	                	textbuff[a*TIMES+t->inf[b].time]=b;
			}
	        }
	}
	#else
        for(b=0;b<tuplenum;b++) {
                textbuff[tuplemap[b].tid*TIMES+t->inf[b].time]=b;
        }
	#endif
}

/* See part 3 of the How-to */
outputext *oextend_chromo(chromo *t)
{
        int cid,time;
        int b,c,d;
	int mod;
	int ***conflicts;
	outputext *buf;

	/* The new way of returning results */
	buf=malloc(cmapnum*TIMES*sizeof(*buf));

        for(b=0;b<cmapnum*TIMES;b++) {
		/* We still support this for compatibility */
		cextbuff[b]=-1;

		buf[b].tuples=NULL;
		buf[b].num=0;
		buf[b].status=0;
	}

	mod=find_module("sametime.so");

	if(mod<0) {
		for(b=0;b<tuplenum;b++) {
			cid=tuplemap[b].cid;
			time=t->inf[b].time;

			for(c=0;c<cmapnum;c++)  if (cid==c) {
				d=c*TIMES+time;
				cextbuff[d]=b;

				buf[d].tuples=malloc(sizeof(*buf[d].tuples));
				buf[d].tuples[0]=b;
				buf[d].num++;
				buf[d].status=2;
			}
		}
	} else {
		conflicts=dlsym(glist[mod]->handle, "conflicts");

		if(!conflicts) fatal(_("conflicts not defined in sametime.so"));

		for(b=0;b<tuplenum;b++) {
			cid=tuplemap[b].cid;
			time=t->inf[b].time;

			for(c=0;c<cmapnum;c++)  if ((*conflicts)[cid][c]) {
				d=c*TIMES+time;
				cextbuff[d]=b;

				buf[d].tuples=realloc(buf[d].tuples, sizeof(*buf[d].tuples)*(buf[d].num+1));
				buf[d].tuples[buf[d].num]=b;
				buf[d].num++;
				buf[d].status=c==cid?2:1;
			}
		}
	}
	return(buf);
}

/* *** TUPLE HANDLING FUNCTIONS *** */

void free_tuple()
{
        free(tuplemap);
}

int add_tuple(int tid, int cid, int sid, int uid)
{
        int c;

	if (tuplenum%TUPLEMAX==0) {
		c=sizeof(*tuplemap)*(tuplenum+TUPLEMAX);
        	if (tuplenum==0) {
			tuplemap=malloc(c);
		} else {
			tuplemap=realloc(tuplemap, c);
		}
		if(tuplemap==NULL) fatal(_("Can't allocate memory"));
	}

        for(c=tuplenum;c>tuplemut;c--) tuplemap[c]=tuplemap[c-1];

        tuplemap[tuplemut].tid=tid;
        tuplemap[tuplemut].cid=cid;
        tuplemap[tuplemut].sid=sid;
        tuplemap[tuplemut].imut_time=-uid;
        tuplemap[tuplemut].imut_room=-1;

        tuplenum++;

        return(tuplemut++);
}

int add_tuple_imut(int tid, int cid, int sid, int time, int room)
{
	int c;

	if (tuplenum%TUPLEMAX==0) {
		c=sizeof(*tuplemap)*(tuplenum+TUPLEMAX);
        	if (tuplenum==0) {
			tuplemap=malloc(c);
		} else {
			tuplemap=realloc(tuplemap, c);
		}
		if(tuplemap==NULL) fatal(_("Can't allocate memory"));
	}

        tuplemap[tuplenum].tid=tid;
        tuplemap[tuplenum].cid=cid;
        tuplemap[tuplenum].sid=sid;
        tuplemap[tuplenum].imut_time=time;
        tuplemap[tuplenum].imut_room=room;

        return(tuplenum++);
}

int find_tuple(int uid)
{
        static int tid=0;
        int b;

        b=tid;

        do {
                if (tuplemap[tid].imut_time==(-uid)) return(tid);
                tid++;
                if (tid>=tuplemut) tid=0;
        } while (b!=tid);

        return(-1);
}

/* *** CLASSROOM MAP HANDLING FUNCTIONS *** */

int add_room(char *id)
{
        int c;

	if (rmapnum%ROOMS==0) {
		c=sizeof(*rmap)*(rmapnum+ROOMS);
        	if (rmapnum==0) {
			rmap=malloc(c);
		} else {
			rmap=realloc(rmap, c);
		}
		if(rmap==NULL) fatal(_("Can't allocate memory"));
	}

        rmap[rmapnum].id=id;

        return(rmapnum++);
}

int find_room(char *id)
{
        int c;

        for(c=0;c<rmapnum;c++) {
                if (!strcmp(rmap[c].id, id)) return(c);
        }

        return(-1);
}

void free_building()
{
        int c;

        for(c=0;c<rmapnum;c++) {
                free(rmap[c].id);
        }

        free(rmap);
}

/* *** TEACHER MAP HANDLING FUNCTIONS *** */

int add_teacher(char *name)
{
        int c;

	if (tmapnum%TIDS==0) {
		c=sizeof(*tmap)*(tmapnum+TIDS);
        	if (tmapnum==0) {
			tmap=malloc(c);
		} else {
			tmap=realloc(tmap, c);
		}
		if(tmap==NULL) fatal(_("Can't allocate memory"));
	}

        tmap[tmapnum].name=name;

        return(tmapnum++);
}

int find_teacher(char *name)
{
        int c;

        for(c=0;c<tmapnum;c++) {
                if (!strcmp(tmap[c].name, name)) return(c);
        }

        return(-1);
}

void free_teachers()
{
        int c;

        for(c=0;c<tmapnum;c++) {
                free(tmap[c].name);
        }

        free(tmap);
}

/* *** SUBJECT MAP HANDLING FUNCTIONS *** */

int add_subject(char *title, int tid)
{
        int c;

	if (smapnum%MAXSUBJECTS==0) {
		c=sizeof(*smap)*(smapnum+MAXSUBJECTS);
        	if (smapnum==0) {
			smap=malloc(c);
		} else {
			smap=realloc(smap, c);
		}
		if(smap==NULL) fatal(_("Can't allocate memory"));
	}

        smap[smapnum].title=title;
        smap[smapnum].tid=tid;

        return(smapnum++);
}

int find_subject(char *title, char *teacher)
{
        int c,tid;

        tid=find_teacher(teacher);

        for(c=0;c<smapnum;c++) {
                if (!strcmp(smap[c].title, title)&&smap[c].tid==tid) {
                        return(c);
                }
        }

        return(-1);
}

void free_subjects()
{
        int c;

        for(c=0;c<smapnum;c++) {
                free(smap[c].title);
        }

        free(smap);
}

/* *** CLASS MAP HANDLING FUNCTIONS *** */

int add_class(char *name, int year)
{
        int c;

	if (cmapnum%CIDS==0) {
		c=sizeof(*cmap)*(cmapnum+CIDS);
        	if (cmapnum==0) {
			cmap=malloc(c);
		} else {
			cmap=realloc(cmap, c);
		}
		if(cmap==NULL) fatal(_("Can't allocate memory"));
	}

        cmap[cmapnum].name=name;
        cmap[cmapnum].year=year;

        return(cmapnum++);
}

int find_class(char *name, int year)
{
        int c;

        for(c=0;c<cmapnum;c++) {
                if (!strcmp(cmap[c].name, name)&&cmap[c].year==year) {
                        return(c);
                }
        }

        return(-1);
}

/* must be run after xmlsup_main() */
void data_init()
{
	int c, b;
	int mod;
	int ***conflicts;

	/* Output buffers */
	cextbuff=malloc(cmapnum*TIMES*sizeof(*cextbuff));
	textbuff=malloc(tmapnum*TIMES*sizeof(*textbuff));
	
	/* Lookup tables */
	conf_lookup=malloc(cmapnum*cmapnum*sizeof(*conf_lookup));
	conf_lead=malloc(cmapnum*sizeof(*conf_lead));
	conf_start=malloc(cmapnum*sizeof(*conf_start));

	for(c=0;c<cmapnum;c++) {
		conf_start[c]=conf_lookup+c*cmapnum;
		conf_lead[c]=conf_start[c];
	}

	/* This should be done better. */

	mod=find_module("sametime.so");

	if(mod<0) {
		for(c=0;c<cmapnum;c++) {
			*conf_lead[c]=&cextbuff[c*TIMES];
			conf_lead[c]++;
		}
	} else {
		conflicts=dlsym(glist[mod]->handle, "conflicts");

		if(!conflicts) fatal(_("conflicts not defined in sametime.so"));

		for(c=0;c<cmapnum;c++) {
			for(b=0;b<cmapnum;b++) {
				if ((*conflicts)[c][b]) {
					*conf_lead[c]=&cextbuff[b*TIMES];
					conf_lead[c]++;
				}
			}
		}
	}

	#ifdef MORE_TEACHERS_HACK
	mod=find_module("more_teachers.so");
	if(mod<0) {
		moretids=NULL;
		moretidnums=NULL;
	} else {
		moretids=dlsym(glist[mod]->handle, "tids");
		if(!moretids) fatal(_("tids not defined in more_teachers.so"));
		moretidnums=dlsym(glist[mod]->handle, "tidnums");
		if(!moretidnums) fatal(_("tidnums not defined in more_teachers.so"));
	}
	#endif
}

void data_exit()
{
        free(cextbuff);
        free(textbuff);

        free(conf_lookup);
        free(conf_lead);
        free(conf_start);
}
