/* TABLIX, PGA highschool timetable generator                              */
/* Copyright (C) 2002 Tomaz Solc                                           */

/* This program is free software; you can redistribute it and/or modify    */
/* it under the terms of the GNU General Public License as published by    */
/* the Free Software Foundation; either version 2 of the License, or       */
/* (at your option) any later version.                                     */

/* This program is distributed in the hope that it will be useful,         */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/* GNU General Public License for more details.                            */

/* You should have received a copy of the GNU General Public License       */
/* along with this program; if not, write to the Free Software             */
/* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA */

/* $Id: xmlsup.c,v 1.28 2004/10/17 09:37:06 avian Exp $ */

#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <libxml/parser.h>
#include <libxml/tree.h>
#include <libxml/xmlerror.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "main.h"
#include "data.h"
#include "xmlsup.h"
#include "modsup.h"
#include "error.h"
#include "gettext.h"

int have_uids;
int uid;

xmlDocPtr config;

char xmlerr[BUFFSIZE];
int xmlerrp;

/* *** Error handling *** */

#if LIBXML_VERSION > 20411
  #define FAIL(msg,node) fatalxml(_(msg), xmlGetLineNo(node))
  #define NOPROP(prop,node) fatalxml(_("Tag <%s> without required property '%s'"), xmlGetLineNo(node), node->name, prop) 
  #define INVPROP(prop,node) fatalxml(_("Invalid value of property '%s' in tag <%s>"), xmlGetLineNo(node), prop, node->name) 
#else
  #define FAIL(msg,node) fatal(_(msg))
  #define NOPROP(prop,node) fatal(_("Tag <%s> without required property '%s'"), node->name, prop) 
  #define INVPROP(prop,node) fatalxml(_("Invalid value of property '%s' in tag <%s>"), prop, node->name) 
#endif


/* *** INFO PARSER *** */

void parse_info(xmlNodePtr cur)
{
        while (cur!=NULL) {
                if (!strcmp(cur->name, "title")) {
                        school_name=xmlNodeGetContent(cur);
                } else if (!strcmp(cur->name, "address")) {
                        school_address=xmlNodeGetContent(cur);
                } else if (!strcmp(cur->name, "author")) {
                        author=xmlNodeGetContent(cur);
                }
                cur=cur->next;
        }
}

/* *** MODULES AND GENETIC PARAMETERS PARSER *** */

void parse_option(xmlNodePtr cur, int gid)
{
	char *temp, *name;
	int c;

	while (cur!=NULL) {
                if (!strcmp(cur->name, "option")) {
                        temp=xmlGetProp(cur, "name");
                        if (temp==NULL) NOPROP("name",cur);

			name=xmlNodeGetContent(cur);

			c=call_onehandler(temp,name,0,OPTION_HND,gid);
			if(c==1) FAIL(moderror, cur);
			if(c==2) info(_("Unknown option '%s' (line %d)"), temp, xmlGetLineNo(cur));

                        free(temp);
			free(name);
                }
                cur=cur->next;
        }
}

void parse_options(xmlNodePtr cur)
{
	char *temp;
	int c;

	while (cur!=NULL) {
                if (!strcmp(cur->name, "module")) {
                        temp=xmlGetProp(cur, "name");
                        if (temp==NULL) NOPROP("name",cur);

			c=find_module(temp);

			parse_option(cur->children, c);

                        free(temp);
                }
                cur=cur->next;
        }
}

void parse_props(xmlNodePtr cur)
{
        int c, b;
        char *temp, *module;

        while (cur!=NULL) {
                if (!strcmp(cur->name, "module")) {
                        module=xmlGetProp(cur, "name");
                        if (module==NULL) NOPROP("name",cur);

                        temp=xmlGetProp(cur, "weight");
                        if (temp==NULL) NOPROP("weight",cur);

                        b=sscanf(temp, "%d", &c);
                        if (b<1) INVPROP("weight",cur);
                        free(temp);

                        if (c<=0) INVPROP("weight",cur);

                        temp=xmlGetProp(cur, "mandatory");
                        if (temp==NULL) NOPROP("mandatory",cur);

			if(strcmp(temp, "yes")&&strcmp(temp, "no")) {
				FAIL(_("Property 'mandatory' should be 'yes' or 'no'"), cur);
			}

                        b=!strcmp(temp, "yes");

			free(temp);

			load_module(module,c,b);

                        free(module);
                } else if (!strcmp(cur->name, "time")) {
                        temp=xmlGetProp(cur, "periods");
                        if (temp==NULL) NOPROP("periods",cur);

                        b=sscanf(temp, "%d", &periods);
                        if (b<1) INVPROP("periods",cur);
                        free(temp);

                        temp=xmlGetProp(cur, "days");
                        if (temp==NULL) NOPROP("days",cur);
                        b=sscanf(temp, "%d", &days);
                        if (b<1) INVPROP("days",cur);
                        free(temp);
                } else if (!strcmp(cur->name, "internal")) {
                        module=xmlGetProp(cur, "name");
                        if (module==NULL) NOPROP("name",cur);

                        temp=xmlGetProp(cur, "weight");
                        if (temp==NULL) NOPROP("weight",cur);

                        b=sscanf(temp, "%d", &c);
                        if (b<1) INVPROP("weight",cur);
                        free(temp);

                        if (c<=0) INVPROP("weight",cur);

			if (!strcmp(module, "impossible")) {
				posweight=c;
			} else {
				INVPROP("name",cur);
			}

                        free(module);
		}
                cur=cur->next;
        }
}

/* *** BUILDING PARSER *** */

/* Helper function for the parse_building_restrictions */
void one_restriction(int rid, xmlNodePtr cur)
{
	char *type, *name;
	int n;

	while (cur!=NULL) {
		if (!strcmp(cur->name, "capability")||
		    !strcmp(cur->name, "restriction")) {
			type=xmlGetProp(cur, "type");

			if (type==NULL) {
				NOPROP("type", cur);
			}

			name=xmlNodeGetContent(cur);

			n=call_handler(type, name, rid, ROOM_HND);

			if(n==1) FAIL(moderror, cur);
			if(n==2) info(_("Unknown room restriction '%s' (line %d)"), type, xmlGetLineNo(cur));

			free(type);
		}
		cur=cur->next;
	}
}

void parse_building_restrictions(xmlNodePtr cur)
{
	char *id;
	int rid;

	int start,stop,c;
	char *temp;

        while (cur!=NULL) {
                if (!strcmp(cur->name, "classroom")) {
                        id=xmlGetProp(cur, "id");

                        if (id==NULL) {
                                NOPROP("id",cur);
                        }

			rid=find_room(id);

			free(id);
	
			if (rid==-1) {
				INVPROP("id",cur);
			}

			one_restriction(rid,cur->children);
                } else if (!strcmp(cur->name, "classrooms")) {

                        temp=xmlGetProp(cur, "start");
                        if (temp==NULL) {
                                NOPROP("start", cur);
                        }
                        c=sscanf(temp, "%d", &start);
                        if (c<1) INVPROP("start", cur);
                        free(temp);

                        temp=xmlGetProp(cur, "end");
                        if (temp==NULL) {
                                NOPROP("end", cur);
                        }
                        c=sscanf(temp, "%d", &stop);
                        if (c<1) INVPROP("end", cur);
                        free(temp);

                        for(c=start;c<=stop;c++) {
                                id=malloc(10);
                                snprintf(id, 10, "%d", c);

				rid=find_room(id);

				free(id);

				if (rid==-1) {
					FAIL(_("Unknown room"), cur);
				}

				one_restriction(rid,cur->children);
                        }

                }
                cur=cur->next;
        }
}

void parse_building(xmlNodePtr cur)
{
        int a,b,c,d;
        char *temp;

        while (cur!=NULL) {
                if (!strcmp(cur->name, "classroom")) {
                        /* single classroom */
                        temp=xmlGetProp(cur, "id");

                        if (temp==NULL) {
                                NOPROP("id", cur);
                        }

			add_room(temp);
                } else if (!strcmp(cur->name, "classrooms")) {
                        /* multiple classrooms */
                        temp=xmlGetProp(cur, "start");
                        if (temp==NULL) {
                                NOPROP("start", cur);
                        }
                        d=sscanf(temp, "%d", &a);
                        if (d<1) INVPROP("start", cur);
                        free(temp);

                        temp=xmlGetProp(cur, "end");
                        if (temp==NULL) {
                                NOPROP("end", cur);
                        }
                        d=sscanf(temp, "%d", &b);
                        if (d<1) INVPROP("end", cur);
                        free(temp);

                        for(c=a;c<=b;c++) {
				temp=malloc(10);
                                snprintf(temp, 10, "%d", c);

				add_room(temp);
                        }

                }
                cur=cur->next;
        }
}

/* *** SUBJECTS & TEACHERS PARSER *** */

void parse_subjects_restrictions(xmlNodePtr cur)
{
        char *type, *temp, *title, *teacher;
        int sid;
	int n;
	xmlNodePtr ncur;

        while (cur!=NULL) {
                if (!strcmp(cur->name, "subject")) {
                        title=xmlGetProp(cur, "title");

                        if (title==NULL) {
                                NOPROP("title", cur);
                        }

                        teacher=xmlGetProp(cur, "teacher");

                        if (teacher==NULL) {
                                NOPROP("teacher", cur);
                        }

                        sid=find_subject(title,teacher);
                        if (sid==-1) {
				FAIL(_("Unknown subject"), cur);
                        }

                        ncur=cur->children;

			while (ncur!=NULL) {
				if (!strcmp(ncur->name, "restriction")) {
					type=xmlGetProp(ncur, "type");

					if (type==NULL) {
						NOPROP("type", ncur);
					}

					temp=xmlNodeGetContent(ncur);

					n=call_handler(type, temp, sid, SUBJECT_HND);
					if(n==1) FAIL(moderror, ncur);
					if(n==2) info(_("Unknown subject restriction '%s' (line %d)"), type, xmlGetLineNo(ncur));

					free(temp);
					free(type);
				}
				ncur=ncur->next;
			}
                }
                cur=cur->next;
        }
}

void parse_subjects(xmlNodePtr cur)
{
        char *title, *teacher;
        int tid;

        while (cur!=NULL) {
                if (!strcmp(cur->name, "subject")) {
                        title=xmlGetProp(cur, "title");

                        if (title==NULL) {
                                NOPROP("title", cur);
                        }

                        teacher=xmlGetProp(cur, "teacher");

                        if (teacher==NULL) {
                                NOPROP("teacher", cur);
                        }

                        tid=find_teacher(teacher);
                        if (tid==-1) {
                                /* this is a new teacher, create a new entry */
				tid=add_teacher(teacher);
			}

			add_subject(title,tid);
                }
                cur=cur->next;
        }
}

/* *** TIMETABLE & CLASSES PARSER  *** */

void parse_tuple_restrictions(xmlNodePtr cur, int tid, int perweek)
{
	int n,c,t;
	char *temp, *name;
        while (cur!=NULL) {
                if (!strcmp(cur->name, "restriction")) {
                        temp=xmlGetProp(cur, "type");
                        if (temp==NULL) NOPROP("type", cur);

			name=xmlNodeGetContent(cur);
			t=tid;

			for(n=0;n<perweek;n++) {
				c=call_handler(temp,name,t,TUPLE_HND);
				if(c==1) FAIL(moderror, cur);
				if(c==2) info(_("Unknown tuple restriction '%s' (line %d)"), temp, xmlGetLineNo(cur));
				t++;
			}

                        free(temp);
			free(name);
                }
                cur=cur->next;
        }
}

void parse_timetable_restrictions(xmlNodePtr cur)
{
        xmlNodePtr start;
        char *temp, *name;
        int c,b;
        int curcid;
	int perweek,tid;

        curcid=-1;

        start=cur;

        /* First we need to find a cid for this class */
        while (cur!=NULL) {
                if (!strcmp(cur->name, "class")) {
                        name=xmlGetProp(cur, "name");
                        temp=xmlGetProp(cur, "year");
                        b=sscanf(temp, "%d", &c);
                        if (b<1) INVPROP("year", cur);
                        curcid=find_class(name, c);
                        free(name);
                        free(temp);
                }
                cur=cur->next;
        }

        if (curcid==-1) {
                FAIL(_("Tag <timetable> without tag <class>"), start);
        }

        /* Only then can we start all over again and */
        /* realy parse those conflicts */
        cur=start;
        while (cur!=NULL) {
                if (!strcmp(cur->name, "restriction")) {
                        temp=xmlGetProp(cur, "type");
                        if (temp==NULL) NOPROP("type", cur);

			name=xmlNodeGetContent(cur);

			c=call_handler(temp,name,curcid,CLASS_HND);
			if(c==1) FAIL(moderror, cur);
			if(c==2) info(_("Unknown class restriction '%s' (line %d)"), temp, xmlGetLineNo(cur));

                        free(temp);
			free(name);
                } else if (!strcmp(cur->name, "subject")) {
			temp=xmlGetProp(cur, "perweek");
			if(temp==NULL) {
				perweek=1;
			} else {
				b=sscanf(temp, "%d", &perweek);
			}
			free(temp);
			temp=xmlGetProp(cur, "tid");
			if(temp==NULL) {
				FAIL(_("No tid found for <subject>"), cur);
			} else {
				b=sscanf(temp, "%d", &tid);
			}
			free(temp);

			/* This is an imutable tuple. */
			if(tid<0) {
				tid=tuplemut-tid-1;
			}
				
			parse_tuple_restrictions(cur->children,tid,perweek);
			xmlUnsetProp(cur, "tid");
		}
                cur=cur->next;
        }
}

void parse_timetable(xmlNodePtr cur)
{
        group *curclass;
        char *temp, *temp1,*temp2;
	char *name;
	int year;

        int cid,sid;
	int day,period,room;
	int a,b,c;

	cid=add_class("", 0);
        curclass=&cmap[cid];

        while (cur!=NULL) {
                if (!strcmp(cur->name, "class")) {
                        name=xmlGetProp(cur, "name");

                        if (name==NULL) {
                                NOPROP("name", cur);
                        }

                        temp=xmlGetProp(cur, "year");

			if (temp==NULL) {
				NOPROP("year", cur);
			}

                        b=sscanf(temp, "%d", &year);
                        if (b<1) INVPROP("year", cur);
                        free(temp);

			if(find_class(name, year)!=-1) {
				FAIL(_("Duplicate class definition"), cur);
			}

			curclass->name=name;
			curclass->year=year;
                } else if (!strcmp(cur->name, "subject")) {
                        temp=xmlGetProp(cur, "title");

			if(temp==NULL) {
				NOPROP("title", cur);
			}

                        temp1=xmlGetProp(cur, "teacher");

			if(temp1==NULL) {
				NOPROP("teacher", cur);
			}

                        sid=find_subject(temp, temp1);

                        if (sid==-1) {
                                FAIL(_("Tag <subject> in tag <timetable> without definition"), cur);
                        }

                        free(temp);
                        free(temp1);

                        if (have_uids) {
                                temp=xmlGetProp(cur, "uid");
                                b=sscanf(temp, "%d", &uid);
                                if (b<1) INVPROP("uid", cur);
                                free(temp);
                        }

                        temp=xmlGetProp(cur, "perweek");

                        if (temp==NULL) {
                                temp1=xmlGetProp(cur, "period");
                                if (temp1==NULL) NOPROP("period/perweek", cur);
                                b=sscanf(temp1, "%d", &period);
                                if (b<1) INVPROP("period", cur);
                                free(temp1);

                                temp1=xmlGetProp(cur, "day");
                                if (temp1==NULL) NOPROP("day", cur);
                                b=sscanf(temp1, "%d", &day);
                                if (b<1) INVPROP("day", cur);
                                free(temp1);

                                temp1=xmlGetProp(cur, "room");
                                if (temp1==NULL) NOPROP("room", cur);
                                room=find_room(temp1);
                                if (room==-1) INVPROP("room", cur);
                                free(temp1);

                                a=add_tuple_imut(smap[sid].tid, cid, sid, PERIODS*day+period, room);
				/* Tids of imutable tuples are not constant. We
				 * must store them as relative to tuplemut */

	                        temp2=malloc(256*sizeof(*temp2));
	                        snprintf(temp2, 256, "%d", tuplemut-a-1);
	                        xmlNewProp(cur, "tid", temp2);
	                        free(temp2);
                        } else {
                                b=sscanf(temp, "%d", &c);
                                if (b<1) INVPROP("perweek", cur);

                                free(temp);

                                if (!have_uids) {
                                        temp=malloc(256*sizeof(*temp));
                                        snprintf(temp, 256, "%d", uid);
                                        xmlNewProp(cur, "uid", temp);
                                        free(temp);
                                }
			
                                a=add_tuple(smap[sid].tid, cid, sid, uid++);
				c--;

                                for(;c>0;c--) {
                                        add_tuple(smap[sid].tid, cid, sid, uid++);
                                }

	                        temp2=malloc(256*sizeof(*temp2));
	                        snprintf(temp2, 256, "%d", a);
	                        xmlNewProp(cur, "tid", temp2);
	                        free(temp2);
                        }
                }
                cur=cur->next;
        }
}

/* *** Error reporting handler for XML parser *** */

void myhandler(void *ctx, const char *msg, ...) {
    va_list args;
    char pmsg[BUFFSIZE];
    int n;

    va_start(args, msg);
    vsnprintf(pmsg, 256, msg, args);
    n=0;
    while(n<BUFFSIZE&&pmsg[n]!=0) {
	if(pmsg[n]!='\n'&&xmlerrp<(BUFFSIZE-1)) {
		xmlerr[xmlerrp++]=pmsg[n];
	} else {
		xmlerr[xmlerrp]=0;
    		error(xmlerr);
		xmlerrp=0;
	}
	n++;
    }
    va_end(args);
}


/* *** MAIN *** */

void xmlsup_main(char *what)
{
        xmlNodePtr cur, building, subjects, population, props, info;

        int b,c;
	char *oldmodule;

        xmlLineNumbersDefault(1);
	xmlerrp=0;
        xmlSetGenericErrorFunc(NULL, myhandler);

	oldmodule=curmodule;
	curmodule="xmlsup";

        config=xmlParseFile(what);
        if (config==NULL) fatal("Parse error");

        cur=xmlDocGetRootElement(config);

        if (strcmp(cur->name, "school")) {
                FAIL(_("Root element mismatch"), cur);
        }

        cur=cur->children;

        building=NULL;
        subjects=NULL;
        population=NULL;
        props=NULL;
        info=NULL;

        while (cur!=NULL) {
                if (!strcmp(cur->name, "building")) {
                        building=cur->children;
                } else if (!strcmp(cur->name, "subjects")) {
                        subjects=cur->children;
                } else if (!strcmp(cur->name, "population")) {
                        population=cur->children;
                } else if (!strcmp(cur->name, "parameters")) {
                        props=cur->children;
                } else if (!strcmp(cur->name, "info")) {
                        info=cur->children;
                }

                cur=cur->next;
        }

        if (building==NULL) {
                fatal("no <building>");
        }
        if (subjects==NULL) {
                fatal("no <subjects>");
        }
	if (props==NULL) {
		fatal("no <parameters>");
	}
        if (population==NULL) {
                have_uids=0;
                uid=1;
        } else {
                have_uids=1;
        }

        /* Default weights, parameters, info */
        days=5;
        periods=7;

	posweight=25;

        parse_props(props);

	times=days*periods;

	if (gnum<1) {
		fatal("no modules loaded.");
	}

        school_name=NULL;
        school_address=NULL;
        author=NULL;

        if (!(info==NULL)) {
                parse_info(info);
        }

	/* building, subjects, teachers */

        parse_building(building);
        parse_subjects(subjects);

        /* begin parsing timetables */

        cur=xmlDocGetRootElement(config);
        cur=cur->children;

        while (cur!=NULL) {
                if (!strcmp(cur->name, "timetable")) {
                        parse_timetable(cur->children);
                }

                cur=cur->next;
        }

        /* first initialize modules */

	for(c=0;c<gnum;c++) if (glist[c]->initfunc!=NULL) {
		b=(*glist[c]->initfunc)();
		if (b) {
			tell(glist[c]->name, MSG_ERROR, moderror);
			fatal(_("Module failed to initialize"));
		}
	}

	parse_options(props);

	/* parse restrictions (capabilities) */

        parse_building_restrictions(building); 
        parse_subjects_restrictions(subjects);
	
        cur=xmlDocGetRootElement(config);
        cur=cur->children;

        while (cur!=NULL) {
                if (!strcmp(cur->name, "timetable")) {
                        parse_timetable_restrictions(cur->children);
                }

                cur=cur->next;
        }

	/* precalculate data in modules */

	for(c=0;c<gnum;c++) if (glist[c]->precalcfunc!=NULL) {
		b=(*glist[c]->precalcfunc)();
		if (b) {
			tell(glist[c]->name, MSG_ERROR, moderror);
			fatal("Module failed to precalculate");
		}
	}

        /* done parsing */

	curmodule=oldmodule;
}

void xmlsup_getchromo(chromo *pop)
{
        xmlNodePtr cur, chromo;
        int period, day, uid, tid, b;
        char *room;
        char *temp;

        chromo=NULL;

        cur=xmlDocGetRootElement(config);
        cur=cur->children;

        while (cur!=NULL) {
                if (!strcmp(cur->name, "chromo")) {
                        chromo=cur;
                }
                cur=cur->next;
        }

        if (chromo==NULL) fatal("no <chromo>");

        temp=xmlGetProp(chromo, "grade");
        if (temp==NULL) NOPROP("grade", chromo);
        b=sscanf(temp, "%d", &(pop->grade));
        if (b<1) INVPROP("grade", chromo);
        free(temp);

        cur=chromo->children;

        while (cur!=NULL) {
                if (!strcmp(cur->name, "gen")) {
                        temp=xmlGetProp(cur, "period");
                        if (temp==NULL) NOPROP("period", cur);
                        b=sscanf(temp, "%d", &period);
                        if (b<1) INVPROP("period", cur);
                        free(temp);

                        temp=xmlGetProp(cur, "day");
                        if (temp==NULL) NOPROP("day", cur);
                        b=sscanf(temp, "%d", &day);
                        if (b<1) INVPROP("day", cur);
                        free(temp);

                        room=xmlGetProp(cur, "room");
                        if (room==NULL) NOPROP("room", cur);

                        temp=xmlGetProp(cur, "uid");
                        if (temp==NULL) NOPROP("uid", cur);
                        b=sscanf(temp, "%d", &uid);
                        if (b<1) INVPROP("uid", cur);
                        free(temp);

                        tid=find_tuple(uid);
                        if (tid==-1) INVPROP("uid", cur);
                        pop->inf[tid].time=day*PERIODS+period;
                        pop->inf[tid].room=find_room(room);

                        free(room);
                }
                cur=cur->next;
        }
	
	/* bug fix by Jaume Obrador */
	for(b=tuplemut;b<tuplenum;b++) {
		pop->inf[b].time=tuplemap[b].imut_time;
		pop->inf[b].room=tuplemap[b].imut_room;
	}
}

void xmlsup_addchromo(chromo *pop)
{
        xmlNodePtr cur, chromo, gen;
        int b;
        char temp[256];

        cur=xmlDocGetRootElement(config);

        xmlNodeAddContent(cur, "\t");
        chromo=xmlNewChild(cur, NULL, "chromo", NULL);
        xmlNodeAddContent(cur, "\n");

        snprintf(temp, 256, "%d", pop->grade);
        xmlNewProp(chromo, "grade", temp);

        for(b=0;b<tuplemut;b++) {
                xmlNodeAddContent(chromo, "\n\t\t");
                gen=xmlNewChild(chromo, NULL, "gen", NULL);

                snprintf(temp, 256, "%d", pop->inf[b].time%PERIODS);
                xmlNewProp(gen, "period", temp);

                snprintf(temp, 256, "%d", pop->inf[b].time/PERIODS);
                xmlNewProp(gen, "day", temp);

                xmlNewProp(gen, "room", rmap[pop->inf[b].room].id);

                snprintf(temp, 256, "%d", -tuplemap[b].imut_time);
                xmlNewProp(gen, "uid", temp);
        }
        xmlNodeAddContent(chromo, "\n\t");
}

void xmlsup_dump(FILE *f)
{
        xmlDocDump(f, config);
}

void xmlsup_exit()
{
        xmlFreeDoc(config);

        free_teachers(); /* :) */
        free_subjects();
        free_building();
        free_tuple();
}
