#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <gtk/gtk.h>
#include <glade/glade.h>

#include "str_util.h"
#include "str_convert.h"
#include "file_util.h"
#include "file_list.h"
#include "progress_dlg.h"
#include "cursor.h"
#include "audio_file.h"
#ifdef ENABLE_MP3
#  include "mpeg_file.h"
#endif

#include "clear_tab.h"


enum {
	APPLY_TO_ALL = 0,
	APPLY_TO_SELECTED = 1
};


/* widgets */
static GtkButton *b_clear_go = NULL;
static GtkComboBox *combo_clear_apply = NULL;
static GtkToggleButton *rb_clear_all = NULL;
static GtkToggleButton *rb_clear_id3v1 = NULL;
static GtkToggleButton *rb_clear_id3v2 = NULL;


/*** private functions ******************************************************/

static void clear_tag(audio_file *af)
{
#ifdef ENABLE_MP3

	if (af->type == AF_MPEG && !gtk_toggle_button_get_active(rb_clear_all)) {
		int version;
		if (gtk_toggle_button_get_active(rb_clear_id3v1))
			version = ID3TT_ID3V1;
		else
			version = ID3TT_ID3V2;
		mpeg_file_remove_tag_v((mpeg_file*)af, version);
	}
	else {
		audio_file_remove_tag(af);
	}

#else

	audio_file_remove_tag(af);

#endif
}


static void clear_tags(GEList *file_list)
{
	GList *iter;
	audio_file *af;
	gchar *last_path = "";
	gchar *curr_path;
	gchar *utf8;
	int count_total, count_tagged;
	int save_errno, res;

	pd_start("Clearing Tags");
	pd_printf(PD_ICON_INFO, "Starting in directory \"%s\"", fl_get_working_dir_utf8());

	count_total = 0;
	count_tagged = 0;
	for (iter = g_elist_first(file_list); iter; iter = g_list_next(iter)) {
		/* flush pending gtk operations so the UI doesn't freeze */
		pd_scroll_to_bottom();
		while (gtk_events_pending()) gtk_main_iteration();
		if (pd_stop_requested()) {
			pd_printf(PD_ICON_WARN, "Operation stopped at user's request");
			break;
		}

		count_total++;

		curr_path = (gchar *)iter->data;

		if (!fu_compare_file_paths(last_path, curr_path)) {
			gchar *p;
			utf8 = str_filename_to_utf8(curr_path, "(UTF8 conversion error)");
			p = g_utf8_strrchr(utf8, -1, '/');

			pd_printf(PD_ICON_INFO, "Entering directory \"%.*s\"", (gint)(p-utf8), utf8);
			free(utf8);
		}
		last_path = curr_path;

		res = audio_file_new(&af, curr_path, TRUE);
		if (res != AF_OK) {
			utf8 = str_filename_to_utf8(g_basename(curr_path), "(UTF8 conversion error)");
			pd_printf(PD_ICON_FAIL, "Error in file \"%s\"", utf8);
			free(utf8);

			if (res == AF_ERR_FILE)
				pd_printf(PD_ICON_NONE, "Couldn't open file for writing");
			else if (res == AF_ERR_FORMAT)
				pd_printf(PD_ICON_NONE, "Audio format not recognized");
			else 
				pd_printf(PD_ICON_NONE, "Unknown error (%d)", res);

			goto _continue;
		}

		clear_tag(af);
		res = audio_file_write_changes(af);
		if (res != AF_OK) {
			save_errno = errno;
			utf8 = str_filename_to_utf8(g_basename(curr_path), "(UTF8 conversion error)");
			pd_printf(PD_ICON_FAIL, "Error in file \"%s\"", utf8);
			pd_printf(PD_ICON_NONE, "%s (errno %d)", strerror(save_errno), save_errno);
			free(utf8);

			goto _continue;
		}

		utf8 = str_filename_to_utf8(g_basename(curr_path), "(UTF8 conversion error)");
		pd_printf(PD_ICON_OK, "Cleared tag from \"%s\"", utf8);
		free(utf8);
		count_tagged++;

	_continue:
		if (af) {
			audio_file_delete(af);
			af = NULL;
		}
	}

	pd_printf(PD_ICON_INFO, "Done (Cleared %d of %d files)", count_tagged, count_total);
	pd_end();
}


static void start_operation()
{
	GEList *file_list;

	/*
	int button;
	button = message_box(w_main, "Remove all tags", 
			     "This will remove tags from all selected files. Proceed?", 0, 
			     GTK_STOCK_CANCEL, GTK_STOCK_YES, NULL);
	if (button == 0)
		return;
	*/

	if (gtk_combo_box_get_active(combo_clear_apply) == APPLY_TO_ALL)
		file_list = fl_get_all_files();
	else
		file_list = fl_get_selected_files();

	if (g_elist_length(file_list) == 0) {
		pd_start("Clearing Tags");
		pd_printf(PD_ICON_FAIL, "No files selected");
		pd_end();

		g_elist_free(file_list);
		return;
	}

	cursor_set_wait();
	gtk_widget_set_sensitive(GTK_WIDGET(b_clear_go), FALSE);
	clear_tags(file_list);
	gtk_widget_set_sensitive(GTK_WIDGET(b_clear_go), TRUE);
	cursor_set_normal();

	g_elist_free(file_list);
}


/*** UI callbacks ***********************************************************/

void cb_clear_go(GtkButton *button, gpointer user_data)
{
	start_operation();
}

static void cb_file_selection_changed(GtkTreeSelection *selection, gpointer data)
{
	if (fl_count_selected() > 0)
		gtk_combo_box_set_active(combo_clear_apply, APPLY_TO_SELECTED);
	else
		gtk_combo_box_set_active(combo_clear_apply, APPLY_TO_ALL);
}


/*** public functions *******************************************************/

void ct_init(GladeXML *xml)
{
	GtkStyle *style;
	GtkWidget *w;

	/*
	 * get the widgets from glade
	 */

	b_clear_go = GTK_BUTTON(glade_xml_get_widget(xml, "b_clear_go"));
	combo_clear_apply = GTK_COMBO_BOX(glade_xml_get_widget(xml, "combo_clear_apply"));

	rb_clear_all = GTK_TOGGLE_BUTTON(glade_xml_get_widget(xml, "rb_clear_all"));
	rb_clear_id3v1 = GTK_TOGGLE_BUTTON(glade_xml_get_widget(xml, "rb_clear_id3v1"));
	rb_clear_id3v2 = GTK_TOGGLE_BUTTON(glade_xml_get_widget(xml, "rb_clear_id3v2"));

	/* initialize some widgets' state */
	gtk_combo_box_set_active(combo_clear_apply, APPLY_TO_ALL);

	/* connect signals */
	g_signal_connect(gtk_tree_view_get_selection(GTK_TREE_VIEW(glade_xml_get_widget(xml, "tv_files"))),
			 "changed", G_CALLBACK(cb_file_selection_changed), NULL);

#ifndef ENABLE_MP3
	gtk_widget_hide(glade_xml_get_widget(xml, "tab_clear_options"));
#endif


	/*
	 * set the title colors
	 */

	w = glade_xml_get_widget(xml, "lab_clear_title");
	gtk_widget_ensure_style(w);
	style = gtk_widget_get_style(w);

	gtk_widget_modify_fg(w, GTK_STATE_NORMAL, &style->text[GTK_STATE_SELECTED]);

	w = glade_xml_get_widget(xml, "box_clear_title");
	gtk_widget_modify_bg(w, GTK_STATE_NORMAL, &style->base[GTK_STATE_SELECTED]);

}

