/*
 * Tapioca library
 * Copyright (C) 2006 INdT.
 * @author  Luiz Augusto von Dentz <luiz.dentz@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with self library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <string.h>

#include "tpa-avatar.h"

#define DEBUG_DOMAIN TPA_DOMAIN_CONNECTION

#include <tapioca/base/tpa-debug.h>
#include <tapioca/base/tpa-connection-bindings.h>

struct _TpaAvatarPrivate
{
    GArray *image;
    gchar *mimetype;
    gchar *token;
    gchar *file;
    gboolean disposed;
};

enum
{
    ARG_0,
    ARG_IMAGE,
    ARG_MIMETYPE,
    ARG_TOKEN,
    ARG_FILE
};

G_DEFINE_TYPE(TpaAvatar, tpa_avatar, G_TYPE_OBJECT)

static GObject*
tpa_avatar_constructor (GType type,
                        guint n_construct_params,
                        GObjectConstructParam *construct_params)
{
    GObject *object;
    TpaAvatar *self;
    GIOChannel *io;
    GError *error = NULL;

    object = G_OBJECT_CLASS (tpa_avatar_parent_class)->constructor
                            (type, n_construct_params, construct_params);

    self  = TPA_AVATAR (object);

    if (self->priv->file) {
        gchar *buffer;
        gsize length;

        io = g_io_channel_new_file (self->priv->file, "r", &error);

        if (!io)
            goto out;

        if (g_io_channel_set_encoding (io, NULL, &error) != G_IO_STATUS_NORMAL)
            goto out;

        if (g_io_channel_read_to_end (io, &buffer, &length, &error) != G_IO_STATUS_NORMAL)
            goto out;

        self->priv->image = g_array_new (FALSE, FALSE, sizeof (gchar));
        self->priv->image = g_array_insert_vals (self->priv->image, 0, buffer, length);
    }

out:
    return object;
}

static void
tpa_avatar_dispose (GObject *object)
{
    TpaAvatar *self = TPA_AVATAR(object);

    if (self->priv->disposed)
    /* If dispose did already run, return. */
       return;

    /* Make sure dispose does not run twice. */
    self->priv->disposed = TRUE;

    if(self->priv->image)
        g_array_free (self->priv->image, TRUE);
    if(self->priv->mimetype)
        g_free (self->priv->mimetype);
    if(self->priv->token)
        g_free (self->priv->token);

    G_OBJECT_CLASS (tpa_avatar_parent_class)->dispose (object);
}

static void
tpa_avatar_set_property (GObject *object,
                         guint prop_id,
                         const GValue *value,
                         GParamSpec *pspec)
{
    TpaAvatar *self = TPA_AVATAR (object);

    switch (prop_id) {
        case ARG_IMAGE:
           self->priv->image = g_value_get_pointer (value);
           break;
        case ARG_MIMETYPE:
            self->priv->mimetype = g_value_dup_string (value);
            break;
        case ARG_TOKEN:
            self->priv->token = g_value_dup_string (value);
            break;
        case ARG_FILE:
            self->priv->file = g_value_dup_string (value);
            break;
       default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
            break;
    }
}

static void
tpa_avatar_get_property (GObject *object,
                         guint prop_id,
                         GValue *value,
                         GParamSpec *pspec)
{
    TpaAvatar *self = TPA_AVATAR (object);

    switch (prop_id) {
        case ARG_IMAGE:
            g_value_set_pointer (value, self->priv->image);
            break;
         case ARG_MIMETYPE:
            g_value_set_string (value, self->priv->mimetype);
            break;
        case ARG_TOKEN:
            g_value_set_string (value, self->priv->token);
            break;
        case ARG_FILE:
            g_value_set_string (value, self->priv->file);
            break;
        default:
          G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
          break;
    }
}

static void
tpa_avatar_class_init (TpaAvatarClass *klass)
{
    GObjectClass *gobject_class;

    gobject_class = (GObjectClass *) klass;
    tpa_avatar_parent_class = g_type_class_peek_parent (klass);

    g_type_class_add_private (klass, sizeof (TpaAvatarPrivate));

    gobject_class->constructor = tpa_avatar_constructor;
    gobject_class->dispose = tpa_avatar_dispose;
    gobject_class->set_property = tpa_avatar_set_property;
    gobject_class->get_property = tpa_avatar_get_property;

    g_object_class_install_property (gobject_class,
                                     ARG_IMAGE,
                                     g_param_spec_pointer ("image",
                                     "image",
                                     "image",
                                     G_PARAM_READWRITE));

    g_object_class_install_property (gobject_class,
                                     ARG_MIMETYPE,
                                     g_param_spec_string ("mimetype",
                                     "mimetype",
                                     "mimetype",
                                     NULL,
                                     G_PARAM_READWRITE));

    g_object_class_install_property (gobject_class,
                                     ARG_TOKEN,
                                     g_param_spec_string ("token",
                                     "token",
                                     "token",
                                     NULL,
                                     G_PARAM_READWRITE));

    g_object_class_install_property (gobject_class,
                                     ARG_FILE,
                                     g_param_spec_string ("file",
                                     "file",
                                     "file",
                                     NULL,
                                     G_PARAM_READWRITE));
}

static void
tpa_avatar_init (TpaAvatar *self)
{
    self->priv = TPA_AVATAR_GET_PRIVATE (self);
    self->priv->image = NULL;
    self->priv->mimetype = NULL;
    self->priv->token = NULL;
    self->priv->file = NULL;
}

TpaAvatar *
tpa_avatar_new_from_file (const gchar *file,
                          const gchar *mimetype)
{
    TpaAvatar *self;

    self = TPA_AVATAR (g_object_new (TPA_TYPE_AVATAR,
                       "file", file,
                       "mimetype", mimetype,
                       NULL));

    return self;
}

TpaAvatar *
tpa_avatar_new (GArray *image,
                const gchar *mimetype,
                const gchar *token)
{
    TpaAvatar *self;

    self = TPA_AVATAR (g_object_new (TPA_TYPE_AVATAR,
                       "image", image,
                       "mimetype", mimetype,
                       "token", token,
                       NULL));

    return self;
}

const GArray *
tpa_avatar_get_image (TpaAvatar *self)
{
    return self->priv->image;
}

const gchar *
tpa_avatar_get_mime_type (TpaAvatar *self)
{
    return self->priv->mimetype;
}

const gchar *
tpa_avatar_get_token (TpaAvatar *self)
{
    return self->priv->token;
}
