/*
 * Tapioca library
 * Copyright (C) 2006 INdT.
 * @author  Luiz Augusto von Dentz <luiz.dentz@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>

#include "tpa-object.h"
#include "tpa-enums.h"
#include "tpa-thread.h"

#define DEBUG_DOMAIN TPA_DOMAIN_CONNECTION

#include <tapioca/base/tpa-debug.h>
#include <tapioca/base/tpa-channel-bindings.h>
#include <tapioca/base/tpa-connection-bindings.h>
#include <tapioca/base/tpa-signals-marshal.h>

typedef struct _TpaSignals TpaSignals;
typedef struct _TpaSignal TpaSignal;
typedef struct _TpaSignalThread TpaSignalThread;

struct _TpaObjectPrivate {
    DBusGProxy *proxy;
    gchar *service;
    gchar *path;
    gchar *interface;
    gboolean reference;
    gboolean disposed;
};

struct _TpaSignals {
    DBusGProxy *proxy;
    GHashTable *table;
};

struct _TpaSignalThread {
    GCallback callback;
    GPtrArray *pool;
};

struct _TpaSignal{
    DBusGProxy *proxy;
    gchar *name;
    GCallback callback;
    gpointer data;
};

enum
{
    ARG_0,
    ARG_SERVICE,
    ARG_PATH,
    ARG_INTERFACE,
    ARG_OBJECT
};

G_DEFINE_TYPE(TpaObject, tpa_object, G_TYPE_OBJECT)

static void
_3_string (DBusGProxy *proxy,
           const gchar *arg0,
           const gchar *arg1,
           const gchar *arg2,
           TpaSignal *signal)
{
    TpaThread *thread;
    GValue *garg0;
    GValue *garg1;
    GValue *garg2;

    thread = g_new0 (TpaThread, 1);

    thread->invoke = (TpaCallback) signal->callback;
    thread->proxy = signal->proxy;
    thread->data = signal->data;
    thread->args = g_value_array_new (3);

    garg0 = g_new0 (GValue, 1);
    g_value_init (garg0, G_TYPE_POINTER);
    g_value_set_pointer (garg0, (gpointer) g_strdup (arg0));
    g_value_array_append (thread->args, garg0);

    garg1 = g_new0 (GValue, 1);
    g_value_init (garg1, G_TYPE_POINTER);
    g_value_set_pointer (garg1, (gpointer) g_strdup (arg1));
    g_value_array_append (thread->args, garg1);

    garg2 = g_new0 (GValue, 1);
    g_value_init (garg2, G_TYPE_POINTER);
    g_value_set_pointer (garg2, (gpointer) g_strdup (arg2));
    g_value_array_append (thread->args, garg2);

    tpa_thread_push (thread);
}

static void
_2_uint (DBusGProxy *proxy,
         guint arg0,
         guint arg1,
         TpaSignal *signal)
{
    TpaThread *thread;
    GValue *garg0;
    GValue *garg1;

    thread = g_new0 (TpaThread, 1);

    thread->invoke = (TpaCallback) signal->callback;
    thread->proxy = signal->proxy;
    thread->data = signal->data;
    thread->args = g_value_array_new (2);

    garg0 = g_new0 (GValue, 1);
    g_value_init (garg0, G_TYPE_POINTER);
    g_value_set_pointer (garg0, GUINT_TO_POINTER (arg0));
    g_value_array_append (thread->args, garg0);

    garg1 = g_new0 (GValue, 1);
    g_value_init (garg1, G_TYPE_POINTER);
    g_value_set_pointer (garg1, GUINT_TO_POINTER (arg1));
    g_value_array_append (thread->args, garg1);

    tpa_thread_push (thread);
}

static void
_2_string_3_uint (DBusGProxy *proxy,
                const gchar *arg0,
                const gchar *arg1,
                guint arg2,
                guint arg3,
                gboolean arg4,
                TpaSignal *signal)
{
    TpaThread *thread;
    GValue *garg0;
    GValue *garg1;
    GValue *garg2;
    GValue *garg3;
    GValue *garg4;

    thread = g_new0 (TpaThread, 1);

    thread->invoke = (TpaCallback) signal->callback;
    thread->proxy = signal->proxy;
    thread->data = signal->data;
    thread->args = g_value_array_new (5);

    garg0 = g_new0 (GValue, 1);
    g_value_init (garg0, G_TYPE_POINTER);
    g_value_set_pointer (garg0, (gpointer) g_strdup (arg0));
    g_value_array_append (thread->args, garg0);

    garg1 = g_new0 (GValue, 1);
    g_value_init (garg1, G_TYPE_POINTER);
    g_value_set_pointer (garg1, (gpointer) g_strdup (arg1));
    g_value_array_append (thread->args, garg1);

    garg2 = g_new0 (GValue, 1);
    g_value_init (garg2, G_TYPE_POINTER);
    g_value_set_pointer (garg2, GUINT_TO_POINTER (arg2));
    g_value_array_append (thread->args, garg2);

    garg3 = g_new0 (GValue, 1);
    g_value_init (garg3, G_TYPE_POINTER);
    g_value_set_pointer (garg3, GUINT_TO_POINTER (arg3));
    g_value_array_append (thread->args, garg3);

    garg4 = g_new0 (GValue, 1);
    g_value_init (garg4, G_TYPE_POINTER);
    g_value_set_pointer (garg4, GUINT_TO_POINTER (arg4));
    g_value_array_append (thread->args, garg4);

    tpa_thread_push (thread);
}

static void
_uint_string (DBusGProxy *proxy,
              guint arg0,
              const gchar *arg1,
              TpaSignal *signal)
{
    TpaThread *thread;
    GValue *garg0;
    GValue *garg1;

    thread = g_new0 (TpaThread, 1);

    thread->invoke = (TpaCallback) signal->callback;
    thread->proxy = signal->proxy;
    thread->data = signal->data;
    thread->args = g_value_array_new (2);

    garg0 = g_new0 (GValue, 1);
    g_value_init (garg0, G_TYPE_POINTER);
    g_value_set_pointer (garg0, GUINT_TO_POINTER (arg0));
    g_value_array_append (thread->args, garg0);

    garg1 = g_new0 (GValue, 1);
    g_value_init (garg1, G_TYPE_POINTER);
    g_value_set_pointer (garg1, (gpointer) g_strdup (arg1));
    g_value_array_append (thread->args, garg1);

    tpa_thread_push (thread);
}

/*static void
_pointer (DBusGProxy *proxy,
          gpointer arg0,
          TpaSignal *signal)
{
    TpaThread *thread;
    GValue *garg0;

    thread = g_new0 (TpaThread, 1);

    thread->invoke = (TpaCallback) signal->callback;
    thread->proxy = signal->proxy;
    thread->data = signal->data;
    thread->args = g_value_array_new (1);

    garg0 = g_new0 (GValue, 1);
    g_value_init (garg0, G_TYPE_POINTER);
    g_value_set_pointer (garg0, arg0);
    g_value_array_append (thread->args, garg0);

    tpa_thread_push (thread);
}

static void
_string_4_pointer_2_uint (DBusGProxy *proxy,
                          const gchar *arg0,
                          gpointer arg1,
                          gpointer arg2,
                          gpointer arg3,
                          gpointer arg4,
                          guint arg5,
                          guint arg6,
                          TpaSignal *signal)
{
    TpaThread *thread;
    GValue *garg0;
    GValue *garg1;
    GValue *garg2;
    GValue *garg3;
    GValue *garg4;
    GValue *garg5;
    GValue *garg6;

    thread = g_new0 (TpaThread, 1);

    thread->invoke = (TpaCallback) signal->callback;
    thread->proxy = signal->proxy;
    thread->data = signal->data;
    thread->args = g_value_array_new (7);

    garg0 = g_new0 (GValue, 1);
    g_value_init (garg0, G_TYPE_POINTER);
    g_value_set_pointer (garg0, (gpointer) g_strdup (arg0));
    g_value_array_append (thread->args, garg0);

    garg1 = g_new0 (GValue, 1);
    g_value_init (garg1, G_TYPE_POINTER);
    g_value_set_pointer (garg0, arg1);
    g_value_array_append (thread->args, garg1);

    garg2 = g_new0 (GValue, 1);
    g_value_init (garg2, G_TYPE_POINTER);
    g_value_set_pointer (garg2, arg2);
    g_value_array_append (thread->args, garg2);

    garg3 = g_new0 (GValue, 1);
    g_value_init (garg3, G_TYPE_POINTER);
    g_value_set_pointer (garg3, arg3);
    g_value_array_append (thread->args, garg3);

    garg4 = g_new0 (GValue, 1);
    g_value_init (garg4, G_TYPE_POINTER);
    g_value_set_pointer (garg4, arg4);
    g_value_array_append (thread->args, garg4);

    garg5 = g_new0 (GValue, 1);
    g_value_init (garg5, G_TYPE_POINTER);
    g_value_set_pointer (garg5, GUINT_TO_POINTER (arg5));
    g_value_array_append (thread->args, garg5);

    garg6 = g_new0 (GValue, 1);
    g_value_init (garg6, G_TYPE_POINTER);
    g_value_set_pointer (garg6, GUINT_TO_POINTER (arg6));
    g_value_array_append (thread->args, garg6);

    tpa_thread_push (thread);
}*/

static void
_void (DBusGProxy *proxy,
       TpaSignal *signal)
{
    TpaThread *thread;

    thread = g_new0 (TpaThread, 1);

    thread->invoke = (TpaCallback) signal->callback;
    thread->proxy = signal->proxy;
    thread->data = signal->data;
    thread->args = g_value_array_new (0);

    tpa_thread_push (thread);
}

static void
_5_uint_string (DBusGProxy *proxy,
                guint arg0,
                guint arg1,
                guint arg2,
                guint arg3,
                guint arg4,
                const gchar *arg5,
                TpaSignal *signal)
{
    TpaThread *thread;
    GValue *garg0;
    GValue *garg1;
    GValue *garg2;
    GValue *garg3;
    GValue *garg4;
    GValue *garg5;

    thread = g_new0 (TpaThread, 1);

    thread->invoke = (TpaCallback) signal->callback;
    thread->proxy = signal->proxy;
    thread->data = signal->data;
    thread->args = g_value_array_new (6);

    garg0 = g_new0 (GValue, 1);
    g_value_init (garg0, G_TYPE_POINTER);
    g_value_set_pointer (garg0, GUINT_TO_POINTER (arg0));
    g_value_array_append (thread->args, garg0);

    garg1 = g_new0 (GValue, 1);
    g_value_init (garg1, G_TYPE_POINTER);
    g_value_set_pointer (garg1, GUINT_TO_POINTER (arg1));
    g_value_array_append (thread->args, garg1);

    garg2 = g_new0 (GValue, 1);
    g_value_init (garg2, G_TYPE_POINTER);
    g_value_set_pointer (garg2, GUINT_TO_POINTER (arg2));
    g_value_array_append (thread->args, garg2);

    garg3 = g_new0 (GValue, 1);
    g_value_init (garg3, G_TYPE_POINTER);
    g_value_set_pointer (garg3, GUINT_TO_POINTER (arg3));
    g_value_array_append (thread->args, garg3);

    garg4 = g_new0 (GValue, 1);
    g_value_init (garg4, G_TYPE_POINTER);
    g_value_set_pointer (garg4, GUINT_TO_POINTER (arg4));
    g_value_array_append (thread->args, garg4);

    garg5 = g_new0 (GValue, 1);
    g_value_init (garg5, G_TYPE_POINTER);
    g_value_set_pointer (garg5, (gpointer) g_strdup (arg5));
    g_value_array_append (thread->args, garg5);

    tpa_thread_push (thread);
}

static void
_3_uint (DBusGProxy *proxy,
         guint arg0,
         guint arg1,
         guint arg2,
         TpaSignal *signal)
{
    TpaThread *thread;
    GValue *garg0;
    GValue *garg1;
    GValue *garg2;
    GValue *garg3;

    thread = g_new0 (TpaThread, 1);

    thread->invoke = (TpaCallback) signal->callback;
    thread->proxy = signal->proxy;
    thread->args = g_value_array_new (1);

    garg0 = g_new0 (GValue, 1);
    g_value_init (garg0, G_TYPE_POINTER);
    g_value_set_pointer (garg0, GUINT_TO_POINTER (arg0));
    g_value_array_append (thread->args, garg0);

    garg1 = g_new0 (GValue, 1);
    g_value_init (garg1, G_TYPE_POINTER);
    g_value_set_pointer (garg1, GUINT_TO_POINTER (arg1));
    g_value_array_append (thread->args, garg1);

    garg2 = g_new0 (GValue, 1);
    g_value_init (garg2, G_TYPE_POINTER);
    g_value_set_pointer (garg2, GUINT_TO_POINTER (arg2));
    g_value_array_append (thread->args, garg2);

    garg3 = g_new0 (GValue, 1);
    g_value_init (garg3, G_TYPE_POINTER);
    g_value_set_pointer (garg3, signal->data);
    g_value_array_append (thread->args, garg3);

    tpa_thread_push (thread);
}

static void
_2_uint_string (DBusGProxy *proxy,
                guint arg0,
                guint arg1,
                const gchar *arg2,
                TpaSignal *signal)
{
    TpaThread *thread;
    GValue *garg0;
    GValue *garg1;
    GValue *garg2;
    GValue *garg3;

    thread = g_new0 (TpaThread, 1);

    thread->invoke = (TpaCallback) signal->callback;
    thread->proxy = signal->proxy;
    thread->args = g_value_array_new (1);

    garg0 = g_new0 (GValue, 1);
    g_value_init (garg0, G_TYPE_POINTER);
    g_value_set_pointer (garg0, GUINT_TO_POINTER (arg0));
    g_value_array_append (thread->args, garg0);

    garg1 = g_new0 (GValue, 1);
    g_value_init (garg1, G_TYPE_POINTER);
    g_value_set_pointer (garg1, GUINT_TO_POINTER (arg1));
    g_value_array_append (thread->args, garg1);

    garg2 = g_new0 (GValue, 1);
    g_value_init (garg2, G_TYPE_POINTER);
    g_value_set_pointer (garg2, (gpointer) g_strdup (arg2));
    g_value_array_append (thread->args, garg2);

    garg3 = g_new0 (GValue, 1);
    g_value_init (garg3, G_TYPE_POINTER);
    g_value_set_pointer (garg3, signal->data);
    g_value_array_append (thread->args, garg3);

    tpa_thread_push (thread);
}

static void
_uint (DBusGProxy *proxy,
       guint arg0,
       TpaSignal *signal)
{
    TpaThread *thread;
    GValue *garg0;
    GValue *garg1;

    thread = g_new0 (TpaThread, 1);

    thread->invoke = (TpaCallback) signal->callback;
    thread->proxy = signal->proxy;
    thread->args = g_value_array_new (3);

    garg0 = g_new0 (GValue, 1);
    g_value_init (garg0, G_TYPE_POINTER);
    g_value_set_pointer (garg0, GUINT_TO_POINTER (arg0));
    g_value_array_append (thread->args, garg0);

    garg1 = g_new0 (GValue, 1);
    g_value_init (garg1, G_TYPE_POINTER);
    g_value_set_pointer (garg1, signal->data);
    g_value_array_append (thread->args, garg1);

    tpa_thread_push (thread);
}

 static void
_disconnect_signals (gpointer data)
 {
    TpaSignals *signals = (TpaSignals *) data;

    if (signals->table)
        g_hash_table_destroy (signals->table);
    if (signals->proxy)
        g_object_unref (signals->proxy);
    g_free (signals);
}

static void
_disconnect_signal (gpointer data)
{
    TpaSignalThread *thread = data;
    guint i;

    if (thread->pool) {
        for (i = 0; i < thread->pool->len; i++) {
            TpaSignal *signal = g_ptr_array_index (thread->pool, i);
            dbus_g_proxy_disconnect_signal (signal->proxy,
                                            signal->name,
                                            signal->callback,
                                            signal->data);
            g_free (signal->name);
            g_object_unref (signal->proxy);
            g_free (signal);
	}
        g_ptr_array_free (thread->pool, TRUE);
    }
    g_free (thread);
}

static void
tpa_object_add_signals (TpaObject *self,
                        DBusGProxy *proxy,
                        const gchar *name)
{
    TpaSignals *signals;
    TpaSignalThread *signal;

    VERBOSE ("(%p, %p, %s)", self, proxy, name);
    g_assert (self);
    g_return_if_fail (proxy);
    g_return_if_fail (name);
    g_return_if_fail ((signals = g_hash_table_lookup (self->proxies, name)));

    if (g_str_equal (TPA_INTERFACE_CM, name)) {
        if (!g_hash_table_lookup (signals->table, "NewConnection")) {
            dbus_g_proxy_add_signal (proxy, "NewConnection",
                                     G_TYPE_STRING,
                                     DBUS_TYPE_G_OBJECT_PATH,
                                     G_TYPE_STRING,
                                     G_TYPE_INVALID);
            signal = g_new0 (TpaSignalThread, 1);
            signal->callback = (GCallback) _3_string;
            signal->pool = g_ptr_array_new ();
            g_hash_table_insert (signals->table,
                                 g_strdup ("NewConnection"),
                                 signal);
        }
    }
    else if (g_str_equal (TPA_INTERFACE_CONNECTION, name)) {
        if (!g_hash_table_lookup (signals->table, "StatusChanged")) {
            dbus_g_proxy_add_signal (proxy, "StatusChanged",
                                     G_TYPE_UINT,
                                     G_TYPE_UINT,
                                     G_TYPE_INVALID);
            signal = g_new0 (TpaSignalThread, 1);
            signal->callback = (GCallback) _2_uint;
            signal->pool = g_ptr_array_new ();
            g_hash_table_insert (signals->table,
                                 g_strdup ("StatusChanged"),
                                 signal);
        }
        if (!g_hash_table_lookup (signals->table, "NewChannel")) {
           dbus_g_proxy_add_signal (proxy, "NewChannel",
                                     DBUS_TYPE_G_OBJECT_PATH,
                                     G_TYPE_STRING,
                                     G_TYPE_UINT,
                                     G_TYPE_UINT,
                                     G_TYPE_BOOLEAN,
                                     G_TYPE_INVALID);
            signal = g_new0 (TpaSignalThread, 1);
            signal->callback = (GCallback) _2_string_3_uint;
            signal->pool = g_ptr_array_new ();
            g_hash_table_insert (signals->table,
                                 g_strdup ("NewChannel"),
                                 signal);
        }
    }
    else if (g_str_equal (TPA_INTERFACE_AVATARS, name)) {
        if (!g_hash_table_lookup (signals->table, "AvatarUpdated")) {
            dbus_g_proxy_add_signal (proxy, "AvatarUpdated",
                                     G_TYPE_UINT,
                                     G_TYPE_STRING,
                                     G_TYPE_INVALID);
            signal = g_new0 (TpaSignalThread, 1);
            signal->callback = (GCallback) _uint_string;
            signal->pool = g_ptr_array_new ();
            g_hash_table_insert (signals->table,
                                 g_strdup ("AvatarUpdated"),
                                 signal);
        }
    }
    else if (g_str_equal (TPA_INTERFACE_ALIASING, name)) {
        if (!g_hash_table_lookup (signals->table, "AliasesChanged")) {
            dbus_g_proxy_add_signal (proxy, "AliasesChanged",
                                     dbus_g_type_get_collection ("GPtrArray",
                                     dbus_g_type_get_struct ("GValueArray",
                                     G_TYPE_UINT,
                                     G_TYPE_STRING,
                                     G_TYPE_INVALID)),
                                     G_TYPE_INVALID);
            signal = g_new0 (TpaSignalThread, 1);
            signal->callback = NULL;
//            signal->callback = (GCallback) _pointer;
            signal->pool = g_ptr_array_new ();
            g_hash_table_insert (signals->table,
                                 g_strdup ("AliasesChanged"),
                                 signal);
        }
    }
    else if (g_str_equal (TPA_INTERFACE_CAPABILITIES, name)) {
        if (!g_hash_table_lookup (signals->table, "CapabilitiesChanged")) {
            dbus_g_proxy_add_signal (proxy, "CapabilitiesChanged",
                                     dbus_g_type_get_collection ("GPtrArray",
                                     dbus_g_type_get_struct ("GValueArray",
                                     G_TYPE_UINT,
                                     G_TYPE_STRING,
                                     G_TYPE_UINT,
                                     G_TYPE_UINT,
                                     G_TYPE_UINT,
                                     G_TYPE_UINT,
                                     G_TYPE_INVALID)),
                                     G_TYPE_INVALID);
            signal = g_new0 (TpaSignalThread, 1);
            signal->callback = NULL;
//            signal->callback = (GCallback) _pointer;
            signal->pool = g_ptr_array_new ();
            g_hash_table_insert (signals->table,
                                 g_strdup ("CapabilitiesChanged"),
                                 signal);
        }
    }
    else if (g_str_equal (TPA_INTERFACE_PRESENCE, name)) {
        if (!g_hash_table_lookup (signals->table, "PresenceUpdate")) {
            dbus_g_proxy_add_signal (proxy, "PresenceUpdate",
                                     (dbus_g_type_get_map ("GHashTable",
                                     G_TYPE_UINT,
                                     (dbus_g_type_get_struct ("GValueArray",
                                     G_TYPE_UINT,
                                     (dbus_g_type_get_map ("GHashTable",
                                     G_TYPE_STRING,
                                     (dbus_g_type_get_map ("GHashTable",
                                     G_TYPE_STRING,
                                     G_TYPE_VALUE)))),
                                     G_TYPE_INVALID)))),
                                     G_TYPE_INVALID);
            signal = g_new0 (TpaSignalThread, 1);
            signal->callback = NULL;
//            signal->callback = (GCallback) _pointer;
            signal->pool = g_ptr_array_new ();
            g_hash_table_insert (signals->table,
                                 g_strdup ("PresenceUpdate"),
                                 signal);
        }
    }
    else if (g_str_equal (TPA_INTERFACE_GROUP, name) ||
        g_str_has_prefix (name, TPA_INTERFACE_CONTACT_LIST)) {
        if (!g_hash_table_lookup (signals->table, "MembersChanged")) {
            dbus_g_proxy_add_signal (proxy, "MembersChanged",
                                     G_TYPE_STRING,
                                     DBUS_TYPE_G_UINT_ARRAY,
                                     DBUS_TYPE_G_UINT_ARRAY,
                                     DBUS_TYPE_G_UINT_ARRAY,
                                     DBUS_TYPE_G_UINT_ARRAY,
                                     G_TYPE_UINT,
                                     G_TYPE_UINT,
                                     G_TYPE_INVALID);
            signal = g_new0 (TpaSignalThread, 1);
            signal->callback = NULL;
//            signal->callback = (GCallback) _string_4_pointer_2_uint;
            signal->pool = g_ptr_array_new ();
            g_hash_table_insert (signals->table,
                                 g_strdup ("MembersChanged"),
                                 signal);
        }

        if (!g_hash_table_lookup (signals->table, "GroupFlagsChanged")) {
            dbus_g_proxy_add_signal (proxy, "GroupFlagsChanged",
                                     G_TYPE_UINT,
                                     G_TYPE_UINT,
                                     G_TYPE_INVALID);
            signal = g_new0 (TpaSignalThread, 1);
            signal->callback = (GCallback) _2_uint;
            signal->pool = g_ptr_array_new ();
            g_hash_table_insert (signals->table,
                                 g_strdup ("GroupFlagsChanged"),
                                 signal);
        }
    }
    else if (g_str_equal (TPA_INTERFACE_CHANNEL, name)) {
        if (!g_hash_table_lookup (signals->table, "Closed")) {
            dbus_g_proxy_add_signal (proxy, "Closed", G_TYPE_INVALID);
            signal = g_new0 (TpaSignalThread, 1);
            signal->callback = (GCallback) _void;
            signal->pool = g_ptr_array_new ();
            g_hash_table_insert (signals->table,
                                 g_strdup ("Closed"),
                                 signal);
        }
    }
    else if (g_str_equal (TPA_INTERFACE_TEXT, name)) {
        if (!g_hash_table_lookup (signals->table, "Received")) {
            dbus_g_proxy_add_signal (proxy, "Received",
                                     G_TYPE_UINT,
                                     G_TYPE_UINT,
                                     G_TYPE_UINT,
                                     G_TYPE_UINT,
                                     G_TYPE_UINT,
                                     G_TYPE_STRING,
                                     G_TYPE_INVALID);
            signal = g_new0 (TpaSignalThread, 1);
            signal->callback = (GCallback) _5_uint_string;
            signal->pool = g_ptr_array_new ();
            g_hash_table_insert (signals->table,
                                 g_strdup ("Received"),
                                 signal);
        }

        if (!g_hash_table_lookup (signals->table, "Sent")) {
            dbus_g_proxy_add_signal (proxy, "Sent",
                                     G_TYPE_UINT,
                                     G_TYPE_UINT,
                                     G_TYPE_STRING,
                                     G_TYPE_INVALID);
            signal = g_new0 (TpaSignalThread, 1);
            signal->callback = (GCallback) _2_uint_string;
            signal->pool = g_ptr_array_new ();
            g_hash_table_insert (signals->table,
                                 g_strdup ("Sent"),
                                 signal);
        }
    }
    else if (g_str_equal (TPA_INTERFACE_STREAMED_MEDIA, name)) {
        if (!g_hash_table_lookup (signals->table, "StreamStateChanged")) {
            dbus_g_proxy_add_signal (proxy, "StreamStateChanged",
                                     G_TYPE_UINT,
                                     G_TYPE_UINT,
                                     G_TYPE_INVALID);
            signal = g_new0 (TpaSignalThread, 1);
            signal->callback = (GCallback) _2_uint;
            signal->pool = g_ptr_array_new ();
            g_hash_table_insert (signals->table,
                                 g_strdup ("StreamStateChanged"),
                                 signal);
        }

        if (!g_hash_table_lookup (signals->table, "StreamError")) {
            dbus_g_proxy_add_signal (proxy, "StreamError",
                                     G_TYPE_UINT,
                                     G_TYPE_UINT,
                                     G_TYPE_STRING,
                                     G_TYPE_INVALID);
            signal = g_new0 (TpaSignalThread, 1);
            signal->callback = (GCallback) _2_uint_string;
            signal->pool = g_ptr_array_new ();
            g_hash_table_insert (signals->table,
                                 g_strdup ("StreamError"),
                                 signal);
        }

        if (!g_hash_table_lookup (signals->table, "StreamAdded")) {
            dbus_g_proxy_add_signal (proxy, "StreamAdded",
                                     G_TYPE_UINT,
                                     G_TYPE_UINT,
                                     G_TYPE_UINT,
                                     G_TYPE_INVALID);
            signal = g_new0 (TpaSignalThread, 1);
            signal->callback = (GCallback) _3_uint;
            signal->pool = g_ptr_array_new ();
            g_hash_table_insert (signals->table,
                                 g_strdup ("StreamAdded"),
                                 signal);
        }

        if (!g_hash_table_lookup (signals->table, "StreamRemoved")) {
            dbus_g_proxy_add_signal (proxy, "StreamRemoved",
                                     G_TYPE_UINT,
                                     G_TYPE_INVALID);
            signal = g_new0 (TpaSignalThread, 1);
            signal->callback = (GCallback) _uint;
            signal->pool = g_ptr_array_new ();
            g_hash_table_insert (signals->table,
                                 g_strdup ("StreamRemoved"),
                                 signal);
        }
    }
    VERBOSE ("return");
}

static void
tpa_object_load_interfaces (TpaObject *self,
                            gchar **ifaces)
{
    DBusGProxy *proxy;
    guint i;

    for (i = 0; ifaces[i]; i++) {
        proxy = dbus_g_proxy_new_from_proxy (self->priv->proxy, ifaces[i], NULL);
        tpa_object_add_proxy (self, proxy);
        INFO ("interface %s loaded", ifaces[i]);
        g_object_unref (proxy);
    }
}

static GObject*
tpa_object_constructor (GType type,
                        guint n_construct_params,
                        GObjectConstructParam *construct_params)
{
    GObject *object;
    TpaObject *self;
    DBusGProxy *proxy;
    DBusGConnection *bus;

    object = G_OBJECT_CLASS (tpa_object_parent_class)->constructor
                            (type, n_construct_params, construct_params);
    self = TPA_OBJECT (object);

    if (self->priv->service && self->priv->path && self->priv->interface) {
        bus = tpa_thread_get_bus ();
        if (!bus) {
            ERROR ("failed to open connection to dbus");
            return NULL;
        }

        proxy = dbus_g_proxy_new_for_name (bus,
                                           self->priv->service,
                                           self->priv->path,
                                           self->priv->interface);

        tpa_object_add_proxy (TPA_OBJECT (self), proxy);
    }

    return object;
}

static void
tpa_object_set_property (GObject *object,
                         guint prop_id,
                         const GValue *value,
                         GParamSpec *pspec)
{
    TpaObject *self = TPA_OBJECT (object);
    TpaObject *obj;

    switch (prop_id) {
        case ARG_OBJECT:
            obj = TPA_OBJECT (g_value_get_pointer (value));
            if (obj)
                self->proxies = obj->proxies;
//                self->proxies = g_hash_table_ref (obj->proxies);
            break;
        case ARG_SERVICE:
            self->priv->service = g_value_dup_string (value);
            break;
        case ARG_PATH:
            self->priv->path = g_value_dup_string (value);
            break;
        case ARG_INTERFACE:
            self->priv->interface = g_value_dup_string (value);
            break;
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
            break;
    }
}

static void
tpa_object_get_property (GObject *object,
                         guint prop_id,
                         GValue *value,
                         GParamSpec *pspec)
{
    TpaObject *self = TPA_OBJECT (object);

    switch (prop_id) {
        case ARG_SERVICE:
            g_value_set_string (value, self->priv->service);
            break;
        case ARG_PATH:
            g_value_set_string (value, self->priv->path);
            break;
        case ARG_INTERFACE:
            g_value_set_string (value, self->priv->interface);
            break;
        case ARG_OBJECT:
            g_value_set_pointer (value, self);
            break;
        default:
          G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
          break;
    }
}

static void
tpa_object_dispose (GObject *object)
{
    TpaObject *self = TPA_OBJECT (object);

    if (self->priv->disposed)
       /* If dispose did already run, return. */
       return;

//    g_hash_table_unref (self->proxies);
    if (!self->priv->reference)
        g_hash_table_destroy (self->proxies);

    if (self->priv->proxy)
        g_object_unref (self->priv->proxy);

    /* Make sure dispose does not run twice. */
    self->priv->disposed = TRUE;

    G_OBJECT_CLASS (tpa_object_parent_class)->dispose (object);
}

static void
tpa_object_class_init (TpaObjectClass *klass)
{
    GObjectClass *gobject_class;

    gobject_class = (GObjectClass *) klass;
    tpa_object_parent_class = g_type_class_peek_parent (klass);

    g_type_class_add_private (klass, sizeof (TpaObjectPrivate));

    gobject_class->dispose = tpa_object_dispose;
    gobject_class->constructor = tpa_object_constructor;
    gobject_class->get_property = tpa_object_get_property;
    gobject_class->set_property = tpa_object_set_property;

    g_object_class_install_property (gobject_class,
                                     ARG_OBJECT,
                                     g_param_spec_pointer ("object",
                                     "object",
                                     "object",
                                     G_PARAM_CONSTRUCT_ONLY | G_PARAM_READWRITE));

    g_object_class_install_property (gobject_class,
                                     ARG_SERVICE,
                                     g_param_spec_string ("service",
                                     "service",
                                     "service",
                                     NULL,
                                     G_PARAM_CONSTRUCT_ONLY | G_PARAM_READWRITE));

    g_object_class_install_property (gobject_class,
                                     ARG_PATH,
                                     g_param_spec_string ("path",
                                     "path",
                                     "path",
                                     NULL,
                                     G_PARAM_CONSTRUCT_ONLY | G_PARAM_READWRITE));

    g_object_class_install_property (gobject_class,
                                     ARG_INTERFACE,
                                     g_param_spec_string ("interface",
                                     "interface",
                                     "interface",
                                     NULL,
                                     G_PARAM_CONSTRUCT_ONLY | G_PARAM_READWRITE));
}

static void
tpa_object_init (TpaObject *self)
{
    self->priv = TPA_OBJECT_GET_PRIVATE (self);
    self->proxies = NULL;
    self->name = NULL;
    self->priv->disposed = FALSE;
    self->priv->reference = TRUE;
}

DBusGProxy *
tpa_object_get_proxy (TpaObject *self,
                      const gchar *name)
{
    TpaSignals *signals;

    VERBOSE ("(%p, %s)", self, name);
    g_assert (self);
    g_return_val_if_fail (self->proxies, NULL);

    signals = g_hash_table_lookup (self->proxies, name);

    if (!signals) {
        VERBOSE ("return %p", NULL);
        return NULL;
    }

    VERBOSE ("return %p", signals->proxy);
    return signals->proxy ? g_object_ref (signals->proxy) : NULL;
}

GPtrArray *
tpa_object_get_all_proxies (TpaObject *self)
{
    GPtrArray *proxies;

    VERBOSE ("(%p)", self);
    g_assert (self);

    proxies = g_ptr_array_new ();

    VERBOSE ("return %p", proxies);
    return proxies;
}

void
tpa_object_add_proxy (TpaObject *self,
                      DBusGProxy *proxy)
{
    tpa_object_add_proxy_with_name (self,
                                    dbus_g_proxy_get_interface (proxy),
                                    proxy);
}

void
tpa_object_add_proxy_with_name (TpaObject *self,
                                const gchar *name,
                                DBusGProxy *proxy)
{
    gchar **ifaces;
    gchar *type_iface;
    TpaSignals *signals;
    GHashTable *table;
    GError *error = NULL;

    VERBOSE ("(%p, %s, %p)", self, name, proxy);
    g_assert (self);
    g_return_if_fail (name);
    g_return_if_fail (proxy);

    if (!self->proxies) {
        self->proxies = g_hash_table_new_full (g_str_hash,
                                               g_str_equal,
                                               g_free,
                                               _disconnect_signals);
        self->priv->proxy = g_object_ref (proxy);
        self->priv->reference = FALSE;
    }

    if (!g_hash_table_lookup (self->proxies, name)) {
        table = g_hash_table_new_full (g_str_hash,
                                       g_str_equal,
                                       g_free,
                                       _disconnect_signal);

        signals = g_new0 (TpaSignals, 1);
        signals->proxy = g_object_ref (proxy);
        signals->table = table;

        g_hash_table_insert (self->proxies,
                             g_strdup (name),
                             signals);

        tpa_object_add_signals (self, proxy, name);
    }


    if ((g_str_equal (name, TPA_INTERFACE_CONNECTION))) {
        if (!org_freedesktop_Telepathy_Connection_get_interfaces (proxy, &ifaces, &error)
            || error) {
            ERROR ("%s", error->message);

            g_error_free (error);
        }
        else {
            tpa_object_load_interfaces (self, ifaces);
            g_strfreev (ifaces);
        }
    }
    else if ((g_str_equal (name, TPA_INTERFACE_CHANNEL))) {
        if (!org_freedesktop_Telepathy_Channel_get_interfaces (proxy, &ifaces, &error)
            || error) {
            ERROR ("%s", error->message);

            g_error_free (error);
        }
        else {
            tpa_object_load_interfaces (self, ifaces);
            g_strfreev (ifaces);

            /* Object type interface is not available via GetInterface */
            if (!org_freedesktop_Telepathy_Channel_get_channel_type (proxy, &type_iface, &error)
                || error) {
                ERROR ("%s", error->message);

                g_error_free (error);
            }
            else {
                proxy = dbus_g_proxy_new_from_proxy (proxy, type_iface, NULL);
                tpa_object_add_proxy (self, proxy);
                INFO ("interface %s loaded", type_iface);
                g_object_unref (proxy);
                g_free (type_iface);
            }
        }
    }
    VERBOSE ("return");
}

void
tpa_object_remove_proxy (TpaObject *self,
                         const gchar *name)
{
    VERBOSE ("(%p, %s)", self, name);
    g_assert (self);

    g_hash_table_remove (self->proxies, name);

    VERBOSE ("remove");
}

gboolean
tpa_object_has_proxy (TpaObject *self,
                      const gchar *name)
{
    gboolean ret;

    VERBOSE ("(%p, %s)", self, name);
    g_assert (self);

    ret = (g_hash_table_lookup (self->proxies, name) != NULL);
    VERBOSE ("return %s", ret ? "true" : "false");
    return ret;
}

void
tpa_object_connect_signal (TpaObject *self,
                           const gchar *name,
                           const gchar *signal_name,
                           GCallback callback,
                           gpointer data)
{
    TpaSignals *signals;
    TpaSignalThread *thread;
    TpaSignal *signal;

    VERBOSE ("(%p, %s, %s, %p, %p)", self, name, signal_name, callback, data);
    g_assert (self);

    signals = g_hash_table_lookup (self->proxies, name);
    if (!signals) {
        ERROR ("could not find any proxy named '%s'", name);
        VERBOSE ("return");
        return;
    }

    thread = g_hash_table_lookup (signals->table, signal_name);
    if (!thread) {
        ERROR ("could not find any signal named '%s'", signal_name);
        VERBOSE ("return");
        return;
    }

    signal = g_new0 (TpaSignal, 1);
    signal->name = g_strdup (signal_name);
    signal->proxy = g_object_ref (signals->proxy);
    signal->callback = callback;
    signal->data = data;

    dbus_g_proxy_connect_signal (signal->proxy,
                                 signal->name,
                                 thread->callback ? thread->callback : signal->callback,
                                 thread->callback ? signal : signal->data,
                                 NULL);

    g_ptr_array_add (thread->pool, signal);

    VERBOSE ("return");
}

