/*
 * Tapioca library
 * Copyright (C) 2006 INdT.
 * @author  Luiz Augusto von Dentz <luiz.dentz@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>
#include <time.h>

#include "tpa-text-message.h"

#define DEBUG_DOMAIN TPA_DOMAIN_CHANNEL

#include <tapioca/base/tpa-debug.h>

struct _TpaTextMessagePrivate {
    guint id;
    gchar *uri;
    gchar *contents;
    guint timestamp;
    TpaTextMessageType type;
    gboolean disposed;
};

enum
{
    ARG_0,
    ARG_ID,
    ARG_URI,
    ARG_CONTENTS,
    ARG_TIMESTAMP,
    ARG_TYPE
};

G_DEFINE_TYPE(TpaTextMessage, tpa_text_message, G_TYPE_OBJECT)

static GObject*
tpa_text_message_constructor (GType type,
                              guint n_construct_params,
                              GObjectConstructParam *construct_params)
{
    GObject *object;

    object = G_OBJECT_CLASS (tpa_text_message_parent_class)->constructor
                            (type, n_construct_params, construct_params);

    return object;
}

static void
tpa_text_message_dispose (GObject *object)
{
    TpaTextMessage *self = TPA_TEXT_MESSAGE (object);

    if (self->priv->disposed)
       /* If dispose did already run, return. */
       return;

    /* Make sure dispose does not run twice. */
    self->priv->disposed = TRUE;

    if (self->priv->uri)
        g_free (self->priv->uri);

    if (self->priv->contents)
        g_free (self->priv->contents);

    G_OBJECT_CLASS (tpa_text_message_parent_class)->dispose (object);
}

static void
tpa_text_message_set_property (GObject *object,
                               guint prop_id,
                               const GValue *value,
                               GParamSpec *pspec)
{
    TpaTextMessage *self = TPA_TEXT_MESSAGE (object);

    switch (prop_id) {
        case ARG_ID:
           self->priv->id = g_value_get_int (value);
           break;
        case ARG_URI:
            self->priv->uri = g_value_dup_string (value);
            break;
        case ARG_CONTENTS:
            self->priv->contents = g_value_dup_string (value);
            break;
        case ARG_TIMESTAMP:
           self->priv->timestamp = g_value_get_uint (value);
           break;
        case ARG_TYPE:
           self->priv->type = g_value_get_uint (value);
           break;
      default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
            break;
    }
}

static void
tpa_text_message_get_property (GObject *object,
                               guint prop_id,
                               GValue *value,
                               GParamSpec *pspec)
{
    TpaTextMessage *self = TPA_TEXT_MESSAGE (object);

    switch (prop_id) {
        case ARG_ID:
            g_value_set_int (value, self->priv->id);
            break;
         case ARG_URI:
            g_value_set_string (value, self->priv->uri);
            break;
        case ARG_CONTENTS:
            g_value_set_string (value, self->priv->contents);
            break;
        case ARG_TIMESTAMP:
            g_value_set_uint (value, self->priv->timestamp);
            break;
        case ARG_TYPE:
            g_value_set_uint (value, self->priv->type);
            break;
        default:
          G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
          break;
    }
}

static void
tpa_text_message_class_init (TpaTextMessageClass *klass)
{
    GObjectClass *gobject_class;

    gobject_class = (GObjectClass *) klass;
    tpa_text_message_parent_class = g_type_class_peek_parent (klass);

    g_type_class_add_private (klass, sizeof (TpaTextMessagePrivate));

    gobject_class->dispose = tpa_text_message_dispose;
    gobject_class->constructor = tpa_text_message_constructor;
    gobject_class->set_property = tpa_text_message_set_property;
    gobject_class->get_property = tpa_text_message_get_property;

    g_object_class_install_property (gobject_class,
                                     ARG_ID,
                                     g_param_spec_int ("id",
                                     "id",
                                     "id",
                                     0,
                                     G_MAXINT,
                                     0,
                                     G_PARAM_READWRITE));

    g_object_class_install_property (gobject_class,
                                     ARG_URI,
                                     g_param_spec_string ("uri",
                                     "uri",
                                     "uri",
                                     NULL,
                                     G_PARAM_READWRITE));

    g_object_class_install_property (gobject_class,
                                     ARG_CONTENTS,
                                     g_param_spec_string ("contents",
                                     "contents",
                                     "contents",
                                     NULL,
                                     G_PARAM_READWRITE));

    g_object_class_install_property (gobject_class,
                                     ARG_TIMESTAMP,
                                     g_param_spec_uint ("timestamp",
                                     "timestamp",
                                     "timestamp",
                                     0,
                                     G_MAXUINT,
                                     0,
                                     G_PARAM_READWRITE));

    g_object_class_install_property (gobject_class,
                                     ARG_TYPE,
                                     g_param_spec_uint ("type",
                                     "type",
                                     "type",
                                     TPA_MESSAGE_TYPE_NORMAL,
                                     TPA_MESSAGE_TYPE_ANY,
                                     TPA_MESSAGE_TYPE_NORMAL,
                                     G_PARAM_READWRITE));
}

static void
tpa_text_message_init (TpaTextMessage *self)
{
    self->priv = TPA_TEXT_MESSAGE_GET_PRIVATE (self);
    self->priv->disposed = FALSE;
    self->priv->id = 0;
    self->priv->uri = NULL;
    self->priv->contents = NULL;
    self->priv->timestamp = 0;
    self->priv->type = 0;
}

TpaTextMessage *
tpa_text_message_new (guint id,
                      const gchar *uri,
                      const gchar *contents,
                      guint timestamp,
                      TpaTextMessageType type)
{
    TpaTextMessage *self;

    g_return_val_if_fail (contents != NULL, NULL);

    /* Pass givin arguments to constructor */
    self = TPA_TEXT_MESSAGE (g_object_new (TPA_TYPE_TEXT_MESSAGE,
                                           "id", id,
                                           "uri", uri,
                                           "contents", contents,
                                           "type", type,
                                           NULL));

    return self;
}

guint
tpa_text_message_get_id (TpaTextMessage *self)
{
    return self->priv->id;
}

const gchar *
tpa_text_message_get_uri (TpaTextMessage *self)
{
    return self->priv->uri;
}

const gchar *
tpa_text_message_get_contents (TpaTextMessage *self)
{
    return self->priv->contents;
}

guint
tpa_text_message_get_time_stamp (TpaTextMessage *self)
{
    return self->priv->timestamp;
}

TpaTextMessageType
tpa_text_message_type (TpaTextMessage *self)
{
    return self->priv->type;
}
